// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000-2003 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// event.cpp
////////////////////////////////////////////////////////////////////////

#include "SDL_endian.h"

#include "event.h"
#include "history.h"
#include "extwindow.h"

////////////////////////////////////////////////////////////////////////
// NAME       : Event::Event
// DESCRIPTION: Load an event from a file.
// PARAMETERS : file    - file descriptor
//              players - pointer to the array of players
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

Event::Event( SDL_RWops *file, Player **players ) {
  int i;
  unsigned char pid;

  SDL_RWread( file, &e_id, 1, 1 );
  SDL_RWread( file, &e_type, 1, 1 );
  SDL_RWread( file, &e_trigger, 1, 1 );
  SDL_RWread( file, &e_depend, 1, 1 );
  for ( i = 0; i < 3; ++i ) e_tdata[i] = SDL_ReadLE16( file );
  for ( i = 0; i < 3; ++i ) e_data[i] = SDL_ReadLE16( file );
  e_title = SDL_ReadLE16( file );
  e_message = SDL_ReadLE16( file );
  e_flags = SDL_ReadLE16( file );

  SDL_RWread( file, &pid, 1, 1 );
  e_player = players[pid];
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::Save
// DESCRIPTION: Save the event data to a file.
// PARAMETERS : file - data file descriptor
// RETURNS    : 0 on success, -1 on error
////////////////////////////////////////////////////////////////////////

int Event::Save( SDL_RWops *file ) const {
  int i;
  unsigned char pid = e_player->ID();

  SDL_RWwrite( file, &e_id, 1, 1 );
  SDL_RWwrite( file, &e_type, 1, 1 );
  SDL_RWwrite( file, &e_trigger, 1, 1 );
  SDL_RWwrite( file, &e_depend, 1, 1 );

  for ( i = 0; i < 3; ++i ) SDL_WriteLE16( file, e_tdata[i] );
  for ( i = 0; i < 3; ++i ) SDL_WriteLE16( file, e_data[i] );
  SDL_WriteLE16( file, e_title );
  SDL_WriteLE16( file, e_message );
  SDL_WriteLE16( file, e_flags );

  SDL_RWwrite( file, &pid, 1, 1 );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::Check
// DESCRIPTION: Check whether the event can be executed. The trigger
//              conditions and event dependencies must be met.
// PARAMETERS : -
// RETURNS    : 0 if the event cannot be activated, 1 if it can, and 2
//              if it can be activated and requires a message window to
//              pop up
////////////////////////////////////////////////////////////////////////

short Event::Check( void ) {
  short rc = CheckTrigger();
 
  if ( rc != 0 ) {
    TLWList deps;
    deps.AddHead( new TLWNode( "", this, ID() ) );

    if ( !CheckDependencies( deps ) ) rc = 0;
  }
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::CheckTrigger
// DESCRIPTION: Check whether the event trigger conditions are met.
// PARAMETERS : -
// RETURNS    : 0 if the trigger is inactivate, 1 if active, and 2
//              if activate and execution requires a message window to
//              pop up
////////////////////////////////////////////////////////////////////////

short Event::CheckTrigger( void ) {
  short rc = 0;

  if ( !Disabled() ) {
    switch ( e_trigger ) {
    case ETRIGGER_TURN:
      if ( (Gam->Turn() >= e_tdata[0]) &&
           (Gam->CurrentPlayer() == e_player) ) rc = 1;
      break;
    case ETRIGGER_UNIT_DESTROYED:
      if ( e_tdata[0] == -1 ) {                               // destroy all enemy units to score
        if ( Gam->GetPlayer( e_tdata[1] )->Units(0) == 0 ) rc = 1;
      } else {                                                // trigger if
        Unit *u = Gam->GetUnitByID( e_tdata[0] );             //  * unit not found
        if ( !u || !u->IsAlive() ||                           //  * unit found, but already dead
           (u->Owner() && (u->Owner()->ID() != e_tdata[1])) ) //  * owner != original owner (captured)
          rc = 1;
      }
      break;
    case ETRIGGER_HAVE_BUILDING:
      if ( (e_tdata[2] == -1) || (e_tdata[2] == Gam->Turn()) ) {
        Building *b = Gam->GetBuildingByID( e_tdata[0] );
        if ( b && b->Owner() && (b->Owner()->ID() == e_tdata[1]) ) rc = 1;
      }
      break;
    case ETRIGGER_HAVE_UNIT:
      if ( (e_tdata[2] == -1) || (e_tdata[2] == Gam->Turn()) ) {
        Unit *u = Gam->GetUnitByID( e_tdata[0] );
        if ( u && u->Owner() && (u->Owner()->ID() == e_tdata[1]) ) rc = 1;
      }
      break;
    case ETRIGGER_UNIT_POSITION:
      if ( e_tdata[0] == -1 ) {
        Unit *u = Gam->GetMapWindow()->GetMapView()->GetMap()->GetUnit( Point(e_tdata[1], e_tdata[2]) );
        if ( u && (u->Owner() == e_player) ) rc = 1;
      } else {
        Unit *u = Gam->GetUnitByID( e_tdata[0] );
        if ( u && (u->Owner() == e_player) &&
           (u->Position() == Point(e_tdata[1], e_tdata[2])) ) rc = 1;
      }
      break;
    }

    if ( rc && (e_player->Type() == HUMAN) &&
         (e_player == Gam->CurrentPlayer()) && (e_message != -1) ) rc = 2;
  }
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::CheckDependencies
// DESCRIPTION: Check whether event dependencies are met.
// PARAMETERS : deps - list of dependent events already checked (with
//                     positive results)
// RETURNS    : TRUE if all dependencies are met, FALSE otherwise
////////////////////////////////////////////////////////////////////////

bool Event::CheckDependencies( TLWList &deps ) {
  bool rc = true;

  if ( e_depend != -1 ) {

    // if the event does not exist anymore it has already been executed
    Event *dep = Gam->GetEventByID( e_depend );
    if ( dep ) {

      // did we already check this event earlier in the cycle?
      if ( !deps.GetNodeByID( dep->ID() ) ) {
        if ( dep->CheckTrigger() ) {
          deps.AddTail( new TLWNode( "", dep, dep->ID() ) );
          rc = dep->CheckDependencies( deps );
        } else rc = false;
      }
    }

  }

  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::Execute
// DESCRIPTION: Execute this event. What this means depends on the event
//              type.
// PARAMETERS : msgwin  - pointer to a message window to show messages
//                        (may be NULL for events with a return code of
//                        1 from Event::Check())
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Event::Execute( MessageWindow *msgwin ) {
  const char *tstr;
  if ( e_title == -1 ) tstr = e_player->Name();
  else tstr = Gam->GetMessage(e_title);

  bool show_msg = true;

  switch ( e_type ) {
  case EVENT_MESSAGE:
    break;
  case EVENT_CREATE_UNIT: {
    Building *b = Gam->GetBuildingByID( e_data[1] );
    if ( b && (b->Owner() == e_player) ) {
      Unit *u = Gam->CreateUnit( Gam->GetUnitType( e_data[0] ), b->Position(), e_player );
      if ( u ) u->UnsetFlags( U_DONE );
    } else show_msg = false;
    break; }
  case EVENT_MINING: {
    Building *b = Gam->GetBuildingByID( e_data[0] );
    if ( b ) {
      if ( e_data[2] == 0 ) b->SetCrystals( e_data[1] );
      else if ( e_data[2] == 1 ) b->ModifyCrystals( e_data[1] );
      else if ( e_data[2] == 2 ) b->SetCrystalProduction( e_data[1] );
      else if ( e_data[2] == 3 ) b->ModifyCrystalProduction( e_data[1] );
    } else show_msg = false;
    break; }
  case EVENT_RESEARCH: {
    Building *b = Gam->GetBuildingByID( e_data[0] );
    if ( b && (b->Owner() == e_player) ) b->SetUnitProduction( 1 << e_data[1] );
    else show_msg = false;
    break; }
  case EVENT_SCORE:
    e_player->Success( (signed char)e_data[0] );
    break;
  case EVENT_NEXT_MAP:
    Gam->SetNextMap( e_data[0] );
    break;
  case EVENT_MANIPULATE_EVENT: {
    Event *e = Gam->GetEventByID( e_data[0] );
    if ( e ) {
      if ( e_data[2] == 0 ) e->SetFlags( e_data[1] );
      else if ( e_data[2] == 1 ) e->UnsetFlags( e_data[1] );
      else if ( e_data[2] == 2 ) e->ToggleFlags( e_data[1] );
    }
    break; }
  }

  if ( (e_message == -1) || (e_player->Type() != HUMAN) ) show_msg = false;
  if ( show_msg ) {
    if ( Gam->CurrentPlayer() == e_player )
      DisplayMessage( msgwin, tstr, Gam->GetMessage(e_message) );
    else {
      // if the other player is human (ie. we have a history) put the message
      // into the turn replay queue
      History *hist = Gam->GetHistory();
      if ( hist ) hist->RecordMsgEvent( e_title, e_message );
    }
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::DisplayMessage
// DESCRIPTION: Pop up a MessageWindow with a message.
// PARAMETERS : title - title string
//              body  - body text
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Event::DisplayMessage( MessageWindow *win, const char *title,
                            const char *body ) const {
  win->SetTitle( title );
  win->textscroll->SetText( body );
  win->Draw();
  win->Show();
}

