"======================================================================
"cream-spell.vim
"
" Cream -- An easy-to-use configuration of the famous Vim text editor
" [ http://cream.sourceforge.net ] Copyright (C) 2002-2004  Steve Hall
"
" License:
" This program is free software; you can redistribute it and/or modify
" it under the terms of the GNU General Public License as published by
" the Free Software Foundation; either version 2 of the License, or
" (at your option) any later version.
" [ http://www.gnu.org/licenses/gpl.html ]
"
" This program is distributed in the hope that it will be useful, but
" WITHOUT ANY WARRANTY; without even the implied warranty of
" MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
" General Public License for more details.
"
" You should have received a copy of the GNU General Public License
" along with this program; if not, write to the Free Software
" Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
" 02111-1307, USA.
"

" TODO:
" o Resolve persistent dialect selection, Cancel or other selection
"   should clear existing, not append as appears to be happening.
"   => FIXED?
"

" Description: {{{1
"
" Origins:
" This file originated from Dr. Charles E. Campbell, Jr.'s wonderful
" spell checking work for Vim, available all over the web, but
" specifically available from:
"   http://vim.sourceforge.net/scripts/script.php?script_id=195
"
" At this point, modifications include many essential changes to work
" with Cream and from insertmode, as well as many other major and
" minor adjustments that we've integrated over time. The original
" concept still lives on in the work, but now with so many changes Dr.
" Chip might not even recognize it.
"
" Notes:
"
" o g:CREAM_SPELL_LANG is an multi-dimensional "array" of selected
"   dictionaries:
"
"       "eng,s,US,CA,GB,;ger,l,DE,CH,;"
"
"        xxx,                <-- language code (specified by ISO 639-2)
"            x,              <-- dictionary size (s,m,l,x)
"              xx,           <-- dialect code (specified by ISO 3166-1)
"                 xx,        <--           "
"                    xx      <--           "
"                      ;     <-- semi-colon language separator
"
"                        ... <-- (next language string)
"
" o Names of available dictionaries would then be in the format:
"   cream-spell-dict-eng-s-US_1.vim -- case insensitive
"               "            _2     -- case sensitive
"               "            _3     -- case insensitive, multi-word
"               "            _4     -- case sensitive, multi-word
"
" o g:CREAM_SPELL_MULTIDICT holds the option value to use the default
"   dictionary(0), a single language dictionary(1) or multiple
"   dictionaries(2) at time.
"
" o Test words:
"   - English: correct
"   - French:  assemblais assembleuse assemblée fidéjussion
"   - German:  abschweifendem
"   - Spanish: acurrucarive
"
" o Note: General highlighting is/must be defined elsewhere so as to
"   be themeable

" 1}}}
" Initialization
" Cream_spell_init() {{{1
function! Cream_spell_init()
" initialize spelling setup (called via autocmd)

	" spell check options (if don't exist)
	if !exists("g:CREAM_SPELL_MULTIDICT")
		" default no ('0' = off, '1' = Single, '2' = Multiple)
		let g:CREAM_SPELL_MULTIDICT = 0
	endif

	" language (list of sourced dictionaries)
	" * default is empty ("")
	" * if g:CREAM_SPELL_MULTIDICT == 2, allowed to be multiple
	if !exists("g:CREAM_SPELL_LANG") || g:CREAM_SPELL_MULTIDICT == 0
		let g:CREAM_SPELL_LANG = "eng,s,US,GB,CA;"
	endif

	" set spell check dictionary path
	let g:cream_spelldicts = $CREAM . "spelldicts/"

	" load abstracted lang-specific module
	call Cream_source(g:cream_spelldicts . "cream-spell-dict.vim")

endfunction

" 1}}}
" Mapping Targets
" Cream_spellcheck() {{{1
function! Cream_spellcheck()
" toggle error highlighting on/off
	if !exists("b:cream_spell")
		" set
		let b:cream_spell = "1"
		" load main function
		call s:Cream_spell()
		"" refresh colors
		"call Cream_colors()
	else
		" unset
		unlet b:cream_spell
		" clear sets
		silent! syntax clear BadWord
		silent! syntax clear GoodWord
		" reset original highlighting
		call Cream_filetype()
		" refresh color theme
		call Cream_colors()
	endif
endfunction


" Cream_spell_next() {{{1
function! Cream_spell_next()
" Next word (and turn on if not on)
	" if highlighting not already on, turn on
	if !exists("b:cream_spell")
		call Cream_spellcheck()
	endif
	call s:Cream_SpchkNxt()
endfunction

" s:Cream_SpchkNxt() {{{1
function! s:Cream_SpchkNxt()
" Next spelling error
	let errid = synIDtrans(hlID("BadWord"))
	let lastline = line("$")
	let curcol = 0
	normal w

	" skip words until we find next error
	while synIDtrans(synID(line("."),col("."),1)) != errid
		normal w
		if line(".") == lastline
			let prvcol = curcol
			let curcol = col(".")
			if curcol == prvcol
				break
			endif
		endif
	endwhile
endfunction

" Cream_spell_prev() {{{1
function! Cream_spell_prev()
" Previous word (and turn on if not on)
	" if highlighting not already on, turn on
	if !exists("b:cream_spell")
		call Cream_spellcheck()
	endif
	call s:Cream_SpchkPrv()
endfunction

" s:Cream_SpchkPrv() {{{1
function! s:Cream_SpchkPrv()
" Previous spelling error
	let errid = synIDtrans(hlID("BadWord"))
	let curcol= 0
	normal b

	" skip words until we find previous error
	while synIDtrans(synID(line("."),col("."),1)) != errid
		normal b
		if line(".") == 1
			let prvcol=curcol
			let curcol=col(".")
			if curcol == prvcol
				break
			endif
		endif
	endwhile
endfunction

" 1}}}
" Save Word
" Cream_spell_saveword() {{{1
function! Cream_spell_saveword()
" Save word to dictionary
	" only perform if spell check turned on
	if exists("b:cream_spell")
		let myword = expand("<cword>")
		call Cream_spell_saveword_dialog(myword)
	else
		call confirm("Please turn on spell check before adding words to dictionary.", "&Ok", 1, "Warning")
	endif
endfunction

" Cream_spell_saveword_v() {{{1
function! Cream_spell_saveword_v()

	" assign selection (on register "x") to variable
	let myword = @x

	" only perform if spell check turned on
	if exists("b:cream_spell")
		call Cream_spell_saveword_dialog(myword)
	else
		call confirm("Please turn on spell check before adding words to dictionary.", "&Ok", 1, "Warning")
	endif

	" reselect
	normal gv

endfunction

" Cream_spell_saveword_dialog() {{{1
function! Cream_spell_saveword_dialog(myword)
	" allow editing
	let myreturn = "bad"
	while myreturn == "bad"

		let myword = inputdialog("Verify word to be saved:", a:myword)
		if myword == ""
			break
		endif

		" get valid chars for lang-encoding
		let mychars = Cream_spell_dict_lang_chars()

		" prohibit if other matches
		if match(myword, "\@!" . "[" . mychars . "]") != -1
			call confirm("Sorry, only the following characters are permitted:\n\n" . mychars . "\n\n", "&Ok", 1, "Info")
		else
			call s:Cream_SpchkSave(myword)
			break
		endif

	endwhile

endfunction

" s:Cream_SpchkSave() {{{1
function! s:Cream_SpchkSave(newword)
" save a new word to a user dictionary ($CREAM_SPELLDICTS/cream-spell-dict-usr_?.vim).
" * must verify valid characters prior to using

	" determine which dictionary it goes in

	" case sensitive?
	" no uppercase characters found
	if     match(a:newword, "[A-Z]") == -1
		let case = "no"
	" no lowercase characters found
	elseif match(a:newword, "[a-z]") == -1
		let case = "yes"
	" mixed--verify with dialog
	else
		let myreturn = confirm("Is this word, \"" . a:newword . "\", case sensitive?", "&Yes\n&\No", 1, "Info")
		if myreturn == 1
			let case = "yes"
		elseif myreturn == 2
			let case = "no"
		else
			" use must have hit esc... quit.
			return -1
		endif
	endif

	" multi-word?
	if   match(a:newword, " ") != -1
	 " || match(a:newword, "-") != -1
	 " || match(a:newword, "&") != -1
		let multword = "yes"
	else
		let multword = "no"
	endif

	" determine correct dictionary part
	" * white space == 3 or 4
	" * mixed case, all upper == 1 or 3
	if     multword == "no" && case == "no"
		let myext = "_1"
	elseif multword == "no" && case == "yes"
		let myext = "_2"
	elseif multword == "yes" && case == "no"
		let myext = "_3"
	elseif multword == "yes" && case == "yes"
		let myext = "_4"
	endif

	" set path
	let myfile = g:cream_user . "spelldicts/cream-spell-dict-usr" . myext . ".vim"

	" open correct file (use split to preserve existing splits)
	execute "silent new " . myfile

	" insert syntax string plus word line at end of document
	if multword == "yes"
		" final form:  syntax match GoodWord "\<\(naive\|passe\)\>"
		let @x = "syntax match GoodWord \"\\\<\\(" . a:newword . "\\)\\>\""
	else
		let @x = "syntax keyword GoodWord transparent " . a:newword
	endif
	$put x

	" write and quit
	silent! write
	silent! bwipeout!

	" now execute the same statement to add it to the current
	" dictionary in memory (much faster than toggling off/on!
	execute @x

endfunction

" 1}}}
"" Alternate Word Suggesting
"" Cream_spell_altword() {{{1
"function! Cream_spell_altword()
"" Show alternate words
"" TODO: Broken. (Module uses the external agrep)
"	" only perform if spell check turned on
"	if exists("b:cream_spell")
"		let myword = expand("<cword>")
"		call s:SpchkAlternate(myword)
"	endif
"endfunction
"
"" s:Cream_SpchkAlternate() {{{1
"function! s:Cream_SpchkAlternate(wrd)
"*** This routine uses agrep, not available on typical Windows systems!
" * suggest words that are close in spelling
"
"    if exists("g:esc_alternate")
"        " re-use wordlist in bottom window
"        execute "normal \<c-w>bG0DAAlternate<" . a:wrd . ">: \<Esc>"
"
"     " *** don't forget to fix g:cream_user stuff! ***
"    elseif filereadable(expand("g:cream_spelldicts/cream-spell-dict-" . g:CREAM_SPELL_LANG . ".wordlist"))
"        " read in previously generated <engspchk.wordlist>
"        let g:esc_alternate= 1
"        bo 1new
"        setlocal lz
"        setlocal winheight=1
"        silent execute "normal GoAlternate<" . a:wrd . ">: \<Esc>"
"    else
"        " generate <engspchk.wordlist> from <engspchk.vim>
"        let g:esc_alternate= 1
"        echo "Building <cream-spell-dict-" . g:CREAM_SPELL_LANG . ".wordlist>..."
"        bo 1new
"        setlocal lz
"        setlocal winheight=1
"        execute "silent 0r $CREAM_SPELLDICTS/cream-spell-dict-" . g:CREAM_SPELL_LANG . ".vim"
"        silent 1,/The Dictionary/+1d
"        silent /The Raison D/-1,$d
"        %s/^syntax keyword GoodWord transparent\t//
"        silent execute "%s/\\s\\+/\<CR>/g"
"        execute "w! $CREAM_SPELLDICTS/cream-spell-dict-" . g:CREAM_SPELL_LANG . ".wordlist"
"        silent %d
"        silent execute "normal $oAlternate<" . a:wrd . ">: \<Esc>"
"    endif
"
"    nnoremap <buffer> <CR> <C-W>khea#<Esc>bde<C-W>blbye<C-W>kPlxb
"    cnoremap <buffer> q :unlet g:esc_alternate<CR>:q!<CR>
"    setlocal nomodified nowrap ic
"
"    " let's add a wee bit of color...
"    syntax match altLeader "^Alternate"
"    syntax match altAngle "[<>]"
"    highlight def link altLeader Statement
"    highlight def link altAngle Delimiter
"
"    " Build patterns based on permutations of up to first 3 letters
"    execute "normal \<c-w>b"
"    "+++ Cream: DUDE! we don't have agrep on Windows!
"    if strlen(a:wrd) <= 4
"        execute "read! agrep -2 -i -w -S2 " . a:wrd . " " . expand("$CREAM_SPELLDICTS/cream-spell-dict-" . g:CREAM_SPELL_LANG . ".wordlist")
"    else
"        execute "read! agrep -2 -i -w -S2 " . a:wrd . " " . expand("$CREAM_SPELLDICTS/cream-spell-dict-" . g:CREAM_SPELL_LANG . ".wordlist")
"    endif
"    "+++
"    %j
"    normal 04w
"    setlocal nomod
"
"endfunction

" 1}}}
" Language Selection
" Cream_spell_lang() {{{1
function! Cream_spell_lang()
" dialog spellcheck options

	" save guioptions environment
	let myguioptions = &guioptions
	" do not use a vertical layout for dialog buttons
	set guioptions-=v

	" verify initialization
	if !exists("g:CREAM_SPELL_MULTIDICT")
		let g:CREAM_SPELL_MULTIDICT = 0
	endif

	" verify use of additional language dictionaries
	if g:CREAM_SPELL_MULTIDICT == 0
		let g:CREAM_SPELL_MULTIDICT = 1
	endif

	" get options
	let valid = '0'
	while valid == '0'
		" get find, replace and path/filename info
		let myreturn = s:Cream_spell_lang_dialog()
		" if quit code returned
		if myreturn == '1' || myreturn == '-1'
			break
		endif
	endwhile

	" refresh if spell check on
	if exists("b:cream_spell") && b:cream_spell == "1"
		syntax enable
		call s:Cream_spell()
	endif

	" refresh autocorrect
	call Cream_autocorrect_init()

	" restore guioptions
	execute "set guioptions=" . myguioptions

endfunction

" s:Cream_spell_lang_dialog() {{{1
function! s:Cream_spell_lang_dialog()
" Called by s:Cream_spell_lang() to set globals for language
" selection(s). Returns 1 if user selects "Ok", -1 if "Cancel"

	" get list of available dictionary languages
	let avail = s:Cream_spell_dict_avail()
	if avail == ""
		call confirm(
			\"You do not have multiple language dictionaries available to reference.\n" .
			\"\n" .
			\"Please download additional optional language dictionaries from http://cream.sourceforge.net !\n" .
			\"\n", "&Ok", 1, "Info")
		let g:CREAM_SPELL_MULTIDICT = 0
		let g:CREAM_SPELL_LANG = ""
		return -1
	endif

	" if multidict status to use default dictionary, reset...
	if g:CREAM_SPELL_MULTIDICT == 0
		" ...clear...
		let g:CREAM_SPELL_LANG = ""
		" ...and quit.
		return 2
	endif

	" obtain names for ISO639 abbreviations
	let temp = ""
	let i = 0
	let cnt = MvNumberOfElements(avail, ",")
	while i < cnt
		" use newline separator since some names have commas
		let temp = MvAddElement(temp, "\n", Cream_iso639(MvElementAt(avail, ",", i)))
		let i = i + 1
	endwhile
	" sort alphabetically
	let temp = MvQSortElements(temp, "\n", 'CmpByString', 1)
	let avail = temp

	" set dialog buttons for each language
	let buttons = ""
	let i = 0
	let cnt = MvNumberOfElements(avail, "\n")
	while i < cnt
		" use newline separator since some names have commas
		let buttons = MvAddElement(buttons, "\n", MvElementAt(avail, "\n", i))
		let i = i + 1
	endwhile

	" set up dialog box based on multidict status
	if g:CREAM_SPELL_MULTIDICT == 1

		let msg = "Select the spelling dictionary language...\n\n"

		let buttons = buttons . "&Cancel"

	elseif g:CREAM_SPELL_MULTIDICT == 2

		let msg = "Toggle the active spelling dictionaries...\n"
		let msg = msg . "\n"
		let msg = msg . "\n"

		" loop through each avail language
		let i = 0
		while i < cnt
			let lang = MvElementAt(avail, "\n", i)
			" set dialog based on if lang is global lang string
			if MvContainsElement(avail, "\n", lang) == 1
				if match(g:CREAM_SPELL_LANG, lang) >= 0
					let msg = msg . "               [X] Yes   [_] No          " . lang . "\n"
				else
					let msg = msg . "               [_] Yes   [X] No          " . lang . "\n"
				endif
			endif
			let i = i + 1
		endwhile

		let msg = msg . "\n"
		let buttons = buttons . "&Ok"

	endif

    " display dialog
    " default last button (Cancel or Ok))
	let n = confirm(msg, buttons, cnt + 1, "Info")

	" iterate through potential choices to find one selected
	let max = MvNumberOfElements(avail, "\n")
	if n <= max
		let i = 1
		while i <= max
			" act upon one selected
			if n == i
				let mylang = MvElementAt(avail, "\n", i - 1)
				break
			endif
			let i = i + 1
		endwhile
	else
		" user canceled, abort
		return -1
	endif

	" convert back to abbr
	let mylang = Cream_iso639(mylang)

	" get position of language in existing string
	let mylangpos = s:Cream_spell_getlangstrpos(mylang)

	" only process language specifics *if* being added (usually, but
	" not when multi-dict toggling off)
	if g:CREAM_SPELL_MULTIDICT == 1 ||
	\  g:CREAM_SPELL_MULTIDICT == 2 && mylangpos == -1

		" get size (will return at least 1 except on error/cancel)
		let mysize = s:Cream_spell_size(mylang)
		if mysize == -1
			" quit
			return -1
		endif

		" get dialects
		" useful if language isn't changing but dialects/size are
		let mylangstr = s:Cream_spell_getlangstr(mylang)
		let mydialect = s:Cream_spell_dialect(mylangstr)

		" cat full individual language string
		" we return something other than "" (cancel) or "0" (doesn't exist)
		if strlen(mydialect) > 1
			let mynewlang = mylang . "," . mysize . "," . mydialect
		else
			let mynewlang = mylang . "," . mysize
		endif

	endif

	" set dictionary based on multidict status
	if g:CREAM_SPELL_MULTIDICT == 1
		let g:CREAM_SPELL_LANG = ""
		let g:CREAM_SPELL_LANG = MvInsertElementAt(g:CREAM_SPELL_LANG, ";", mynewlang, 0)
		return 1
	elseif g:CREAM_SPELL_MULTIDICT == 2
		" if didn't exist, add
		if mylangpos == -1
			let g:CREAM_SPELL_LANG = MvInsertElementAt(g:CREAM_SPELL_LANG, ";", mynewlang, 0)
		" else remove
		else
			let g:CREAM_SPELL_LANG = MvRemoveElementAt(g:CREAM_SPELL_LANG, ";", mylangpos)
		endif
		return 2
	endif

endfunction

" s:Cream_spell_size() {{{1
function! s:Cream_spell_size(lang)
" return size selected, -1 if none selected

	let temp = ""
	" get list of available sizes in language
	let avail = s:Cream_spell_size_avail(a:lang)

	" setup dialog
	" sort available into the same order as selection (reverse order)
	if MvContainsElement(avail, ",", "h") == 1
		let temp = MvInsertElementAt(temp, ",", "h", 0)
	endif
	if MvContainsElement(avail, ",", "l") == 1
		let temp = MvInsertElementAt(temp, ",", "l", 0)
	endif
	if MvContainsElement(avail, ",", "m") == 1
		let temp = MvInsertElementAt(temp, ",", "m", 0)
	endif
	if MvContainsElement(avail, ",", "s") == 1
		let temp = MvInsertElementAt(temp, ",", "s", 0)
	endif
	let avail = temp

	let mylang = s:Cream_spell_getlangstr(a:lang)
	if mylang == -1
		" start with this language if empty
		let mylang = a:lang
	endif

	" if available, add to dialog
	let buttons = ""
	if MvContainsElement(avail, ",", "s") == 1
		let buttons = buttons . "&Small\n"
	endif
	if MvContainsElement(avail, ",", "m") == 1
		let buttons = buttons . "&Medium\n"
	endif
	if MvContainsElement(avail, ",", "l") == 1
		let buttons = buttons . "&Large\n"
	endif
	if MvContainsElement(avail, ",", "h") == 1
		let buttons = buttons . "&Huge\n"
	endif

	" prepare dialog (radio button-like -- never allow multiple dialects)
	let buttons = buttons . "&Cancel"

	" dialog
	let n = confirm(
		\"Please select the prefered dictionary size. Please note:\n" .
		\" * The larger the dictionary, the slower the error highlighting.\n" .
		\" * Huge size indicates a dictionary which includes such rare words\n" .
		\"   that common mis-spellings could be passed (such as \"ort\" a common\n" .
		\"   mis-spelling of \"or\".\n" .
		\"\n", buttons, MvNumberOfElements(avail, ",") + 1, "Info")

	let max = MvNumberOfElements(avail, ",")
	if 0 < n && n <= max
		let i = 1
		let max1 = MvNumberOfElements(avail, ",")
		while i <= max1
			if n == i
				let mysize = MvElementAt(avail, ",", i - 1)
			endif
			let i = i + 1
		endwhile
		return mysize
	else
		return -1
	endif

endfunction

" s:Cream_spell_dialect() {{{1
function! s:Cream_spell_dialect(lang)
" Return comma-separated list of dialect selection(s) from a:lang's
" dictionaries available.
" {a:lang}  language code string including lang, size and dialects

	" get three-char language code
	let mylang = MvElementAt(a:lang, ",", 0)

	" get list of available dialects
	let availd = s:Cream_spell_dialect_avail(mylang)
	" get count of avail dialects
	let availcnt = MvNumberOfElements(availd, ",")

	" handle special situations
	if     availcnt == 0
		return ""
	" if only one available
	elseif availcnt == 1
		" return it (don't bother user)
		return MvElementAt(availd, ",", 0)
	endif

	" get currently active dialects (in a:lang position 0-based 2+)
	let curdialects = ""
	let mycnt = MvNumberOfElements(a:lang, ",")
	" start at third element
	let i = 2
	while i < mycnt
		let curdialects = MvAddElement(curdialects, ",", MvElementAt(a:lang, ",", i))
		let i = i + 1
	endwhile

	let myreturn = 0
	while myreturn == 0

		" compose msg and buttons based on available
		let msg = "Select active dialects available for " . Cream_iso639(mylang) . ":\n\n\n"
		let buttons = ""
		let i = 0
		while i < availcnt
			" get dialect code from available
			let myabbr = MvElementAt(availd, ",", i)
			" set dialog dialect according to current
			if MvContainsElement(curdialects, ",", myabbr) == 1
				let msg = msg . "               [X] Yes   [_] No          " . Cream_iso3166_1(myabbr) . "\n"
			else
				let msg = msg . "               [_] Yes   [X] No          " . Cream_iso3166_1(myabbr) . "\n"
			endif
			" set dialect as button
			let buttons = buttons . Cream_iso3166_1(myabbr) . "\n"
			let i = i + 1
		endwhile
		let msg = msg . "\n"
		" look ma, no Cancel button!
		let buttons = buttons . "&Ok"

		" call dialog
		let n = confirm(msg, buttons, availcnt + 1, "Info")

		" evaluate if not error, Cancel or Ok
		if n != 0
		\ && n <= availcnt

			" each possibility separately
			let i = 1
			while i <= availcnt
				if n == i
					" get dialect of avail at corresponding position
					let mydialect = MvElementAt(availd, ",", i - 1)
					" if already current, remove
					if MvContainsElement(curdialects, ",", mydialect) == 1
						let curdialects = MvRemoveElement(curdialects, ",", mydialect)
					" if not current, add
					else
						let curdialects = MvAddElement(curdialects, ",", mydialect)
					endif
				endif
				let i = i + 1
			endwhile

		else
			" finished
			break
		endif

	endwhile

	return curdialects

endfunction

" 1}}}
" Options
" Cream_spell_options() {{{1
function! Cream_spell_options()
" prompt for spell check options (default, single or multiple dictionaries)

	" save guioptions environment
	let myguioptions = &guioptions
	" do not use a vertical layout for dialog buttons
	set guioptions-=v

	let avail = s:Cream_spell_dict_avail()
	" if optional dictionary(s) not available, quit
	let max = MvNumberOfElements(avail, ",")
	if max == 0
		call confirm(
			\"You do not have any language dictionaries available to reference.\n" .
			\"\n" .
			\"Please download optional language dictionaries from http://cream.sourceforge.net !\n" .
			\"\n", "&Ok", 1, "Info")
		if exists("g:CREAM_SPELL_LANG")
			let g:CREAM_SPELL_LANG = ""
		endif
	else
		" get options
		let mystatus = g:CREAM_SPELL_MULTIDICT
		let valid = 0
		while valid == 0
			" get find, replace and path/filename info
			let myreturn = s:Cream_spell_options_dialog(mystatus)
			" if enter quit code returned
			if myreturn == -1
				let g:CREAM_SPELL_MULTIDICT = mystatus
				break
			endif
			" increment
			if mystatus == 0
				let mystatus = 1
			elseif mystatus == 1
				let mystatus = 2
			elseif mystatus == 2
				let mystatus = 0
			endif
		endwhile
	endif

	" condition environment based on g:CREAM_SPELL_MULTIDICT
	if g:CREAM_SPELL_MULTIDICT == 0
		let g:CREAM_SPELL_LANG = ""
	elseif g:CREAM_SPELL_MULTIDICT == 1
		" strip languages above first
		" (We rebuild to ensure single element is properly terminated.
		" Cream prior to v.13 did not use a multi-dimensional array,
		" and separated language elements with commas. By re-building,
		" we can clean up the existing variable state.)

		" get first lang
		let tempstr = MvElementAt(g:CREAM_SPELL_LANG, ";", 0)
		" clear all rest
		let g:CREAM_SPELL_LANG = ""
		" reduce to one lang if not empty
		if tempstr != ""
			" re-define array as just first lang
			let g:CREAM_SPELL_LANG = MvInsertElementAt(g:CREAM_SPELL_LANG, ";", tempstr, 0)
		endif

	elseif g:CREAM_SPELL_MULTIDICT == 2
		" can be empty or full
	endif

	" select dictionary if none present and dictionaries selected
	if g:CREAM_SPELL_LANG == "" && g:CREAM_SPELL_MULTIDICT > 0
		call Cream_spell_lang()
	endif

	" refresh if spell check on
	if exists("b:cream_spell") && b:cream_spell == "1"
		syntax enable
		" reload dictionaries
		call s:Cream_spell()
	endif

	" restore guioptions
	execute "set guioptions=" . myguioptions

endfunction

" s:Cream_spell_options_dialog() {{{1
function! s:Cream_spell_options_dialog(status)

	let mychoice = 0
	let msg = "Options:\n"
	let msg = msg . "\n"
	let msg = msg . "\n"
	let msg = msg . "    Dictionary referencing:\n"
	if a:status == 0
		let msg = msg . "               [X] Basic\n"
		let msg = msg . "               [_] Single Language\n"
		let msg = msg . "               [_] Multiple Language\n"
	elseif a:status == 1
		let msg = msg . "               [_] Basic\n"
		let msg = msg . "               [X] Single Language\n"
		let msg = msg . "               [_] Multiple Language\n"
	elseif a:status == 2
		let msg = msg . "               [_] Basic\n"
		let msg = msg . "               [_] Single Language\n"
		let msg = msg . "               [X] Multiple Language\n"
	endif
	let msg = msg . "\n"
	let mychoice = confirm(msg, "Dictionary Referencing (toggle)\n&Ok", 1)
	if mychoice == 1
		" increment and return
		if a:status == 0
			return 1
		elseif a:status == 1
			return 2
		elseif a:status == 2
			return 0
		endif
	else
		return -1
	endif

endfunction

" 1}}}
" Spell Check
" s:Cream_spell() {{{1
function! s:Cream_spell()
" load on demand, too slow to load on startup

	"----------------------------------------------------------------------
	" establish BadWord

	" The Raison D'Etre!
	" BadWord (misspelling) highlighting done here.  Plus comment
	" support -- do spell-checking inside comments. This can be done
	" only for those syntax files' comment blocks that
	" contains=@cluster. The code below adds GoodWord and BadWord to
	" various clusters.  If your favorite syntax comments are not
	" included, send a note to your syntax file's maintainer about it!
	if     &filetype == "amiga"
		syntax cluster amiCommentGroup add=GoodWord,BadWord
		syntax match BadWord contained "\<\a\+\>"
	elseif &filetype == "bib"
		syntax cluster bibVarContents contains=GoodWord,BadWord
		syntax cluster bibCommentContents contains=GoodWord,BadWord
		syntax match BadWord contained "\<\a\+\>"
	elseif &filetype == "c" || &filetype == "cpp"
		syntax cluster cCommentGroup add=GoodWord,BadWord
		syntax match BadWord contained "\<\a\+\>"
	elseif &filetype == "csh"
		syntax cluster cshCommentGroup add=GoodWord,BadWord
		syntax match   BadWord contained "\<\a\+\>"
	elseif &filetype == "dcl"
		syntax cluster dclCommentGroup add=GoodWord,BadWord
		syntax match BadWord contained "\<\a\+\>"
	elseif &filetype == "fortran"
		syntax cluster fortranCommentGroup	add=GoodWord,BadWord
		syntax match BadWord contained "\<\a\+\>"
		syntax match fortranGoodWord contained	"^[Cc]\>"
		syntax cluster fortranCommentGroup add=fortranGoodWord
		highlight link fortranGoodWord fortranComment
	elseif &filetype == "sh" || &filetype == "ksh" || &filetype == "bash"
		syntax cluster shCommentGroup add=GoodWord,BadWord
		syntax match BadWord contained "\<\a\+\>"
	elseif &filetype == "tex"
		syntax match GoodWord "{[a-zA-Z|@]\+}"lc=1,me=e-1
		syntax match GoodWord "\[[a-zA-Z]\+]"lc=1,me=e-1
		syntax match texGoodWord "\\[a-zA-Z]\+"lc=1,me=e-1 contained
		highlight link texGoodWord texComment
		syntax cluster texCommentGroup add=texGoodWord
	elseif &filetype == "vim"
		syntax cluster vimCommentGroup add=GoodWord,BadWord
		syntax match BadWord contained "\<\a\+\>"
	"else
	"    " highlight all words as BadWords (redeem below)
	"    syntax match BadWord "\<\a\+\>"
	endif


	" remainder are in BadWord
	" OPTION 0
	"syntax match BadWord "\<[\a\+\>"
	" OPTION 1
	"" get valid chars for lang-encoding
	"let mychars = Cream_spell_dict_lang_chars()
	"" match BadWord is any word composed with these chars
	"execute "syntax match BadWord \"\\<[" . mychars . "]\\+\\>"
	" OPTION 2
	" from engspck 2004-09-11
	syntax match BadWord "\<[^[:punct:][:space:][:digit:]]\{2,}\>"
	"" used by non-english languages
	"syntax match BadWord "\<[^[!@#$%^&*()_+=[\]{};:",<>./?\\|[:space:][:digit:]]\{2,}\>"


	"----------------------------------------------------------------------
	" establish GoodWord
	" Note: *matches* need to follow the BadWord so that they take
	" priority!

	" load dictionaries
	" -----------------
	call s:Cream_spell_dictload()


	" case sensitive
	" --------------
	syntax case match

	" ignores people's middle-name initials
	syntax match GoodWord "\<[A-Z]\."


	" case insensitive
	" ----------------
	syntax case ignore

	""" work better with LaTeX
	""if &filetype == "tex"
	""    syntax match GoodWord "{[a-zA-Z|@]\+}"lc=1,me=e-1
	""    syntax match GoodWord "\[[a-zA-Z]\+]"lc=1,me=e-1
	""    syntax match texGoodWord "\\[a-zA-Z]\+"lc=1,me=e-1 contained
	""    highlight link texGoodWord texComment
	""    syntax cluster texCommentGroup add=texGoodWord
	""endif

	" qualify prefixes or endings not in dictionaries
	call Cream_spell_dict_lang_wordparts()

	" exist in proper match dictionaries now
	"" these are words but have special meaning in vim files, so they can't be keywords :o
	"syntax match GoodWord "\<\(transparent\|contained\|contains\|display\|extend\|fold\|skip\)\>"

	" ignore web addresses and \n for newlines (case insensitive)
	syntax match GoodWord "\<http://www\.\S\+"
	syntax match GoodWord "\\n"

	" Chase Tingley patch: prevents Error highlighting of words while
	" one is typing them.  \%* is a new magic atom that matches
	" zero-length if that is where the cursor currently is.
	syntax match GoodWord "\<\k\+\%#\>"
	syntax match GoodWord "\<\k\+'\%#"

	" numbers
	syntax match GoodWord "\<\d\+\>"
	" TEST: 8848

	"" mixed numbers and letters
	"" TEST: 8848a8 a8848 8848a
	"" preceding alpha
	"syntax match GoodWord "\<\a\+\d\+\>"
	"" trailing alpha
	"syntax match GoodWord "\<\d\+\a\+\>"
	"" embedded alpha
	"syntax match GoodWord "\<\d\+\a\+\d\+\>"
	""" combined (broken)
	""syntax match GoodWord "\<\a\*\d\+\a\*\d\*\>"


	" multi- or cross-word errors
	" ---------------------------

	" catch errant case across punctuation
	call Cream_spell_dict_lang_punctcase()

	" double word checking (TEST: the the coww )
	" Bram Moolenaar:
	"syntax match DoubleWord "\(\<\k\+\>\)\s\+\1"
	" Preben Peppe Guldberg:
	"syntax match DoubleWord "\(\<\k\+\>\)\s\+\1\>"
	" Gary Holloway:
	"syntax match DoubleWord "\(\<\k\+\>\)\(\_s\+\<\1\>\)\+"
	" Rober Montante:
	"syntax match DoubleWord "\(\<\w\+\>\)\_s*\1\>"
	" Benji Fisher:
	"syntax match DoubleWord "\<\(\k\+\)\_s\+\1\>"
	" Chip Campbell: (ahead of keyword recognition)
	"syntax match DoubleWord "\s\(\k\+\)\_s\+\1\>"ms=s+1
	" Testing:
	"syntax match DoubleWord "\(\<\k\+\>\)\_s\+\1"

endfunction

" s:Cream_spell_dictload() {{{1
function! s:Cream_spell_dictload()
" load spell check dictionaries based on user preferences (set elsewhere)

	" user (previously discovered and validated in creamrc)
	syntax case ignore
	call Cream_source(g:cream_user . "spelldicts/cream-spell-dict-usr_1.vim")
	call Cream_source(g:cream_user . "spelldicts/cream-spell-dict-usr_3.vim")
	syntax case match
	call Cream_source(g:cream_user . "spelldicts/cream-spell-dict-usr_2.vim")
	call Cream_source(g:cream_user . "spelldicts/cream-spell-dict-usr_4.vim")

	" languages (read above for explanation of g:CREAM_SPELL_LANG array)
	let i = 0
	" iterate through each language
	let max = MvNumberOfElements(g:CREAM_SPELL_LANG, ";")
	while i < max

		" get entire language element
		let mylangstr = MvElementAt(g:CREAM_SPELL_LANG, ";", i)
		" read language item from it (item 0)
		let mylang = MvElementAt(mylangstr, ",", 0)

		" if empty, load default and quit
		if mylang == "" && i == 0
			call s:Cream_spell_dictload_default()
			return
		endif

		call s:Cream_spell_dictload_lang(mylangstr)

		let i = i + 1

	endwhile

endfunction

" s:Cream_spell_dictload_default() {{{1
function! s:Cream_spell_dictload_default()
" default dictionary loading (loaded these if no language selected)
	let n = 0
	syntax case ignore
	let n = n + Cream_source(g:cream_spelldicts . "cream-spell-dict-eng-s_1.vim")
	let n = n + Cream_source(g:cream_spelldicts . "cream-spell-dict-eng-s_3.vim")
	syntax case match
	let n = n + Cream_source(g:cream_spelldicts . "cream-spell-dict-eng-s_2.vim")
	let n = n + Cream_source(g:cream_spelldicts . "cream-spell-dict-eng-s_4.vim")
	if n != 4
		call confirm("Warning, default dictionary not found.", "&Ok", 1, "Warning")
	endif
endfunction

" s:Cream_spell_dictload_lang() {{{1
function! s:Cream_spell_dictload_lang(lang)
" load all dictionary components of {lang} (should be the entire
" language's string!)

	" get dict size
	let mydictsize = MvElementAt(a:lang, ",", 1)
	" add all that are smaller (dictionaries are cumulative)
	if     mydictsize == ""
		" we need an element to iterate once if empty
		let mydictsize = "nosize"
	elseif mydictsize == "s"
		" we're okey dokey
	elseif mydictsize == "m"
		let mydictsize = "s,m,"
	elseif mydictsize == "l"
		let mydictsize = "s,m,l,"
	elseif mydictsize == "h"
		let mydictsize = "s,m,l,h,"
	else
		call confirm("Error determining sizes in s:Cream_spell_dictload_lang()", "&Ok", 1)
		return
	endif

	" get dialects (items remaining)
	let j = 2
	let mydialects = ""
	let max = MvNumberOfElements(a:lang, ",")
	while j < max
		let myelement = MvElementAt(a:lang, ",", j)
		let mydialects = MvAddElement(mydialects, ",", myelement)
		let j = j + 1
	endwhile

	" create file name (minus part indicator)
	" language
	let mylangfile = g:cream_spelldicts . "cream-spell-dict-" . MvElementAt(a:lang, ",", 0)

	" compose list of all dictionary components to be loaded
	let loadfiles = ""
	let filesizes = 1	" avoid divide by 0 ;)
	" sizes
	let i = 0
	let max = MvNumberOfElements(mydictsize, ",")
	while i < max

		" append size (if present and not empty)
		if mydictsize != "" && mydictsize != "nosize"
			let mylangfilesize = mylangfile . "-" . MvElementAt(mydictsize, ",", i)
		else
			let mylangfilesize = ""
		endif

		" dialects (-1 includes default/dialect-free first)
		let j = -1
		let max2 = MvNumberOfElements(mydialects, ",")
		while j < max2

			" append dialects (if exist)
			if j == -1
				" do base language (dialect-free) first (ignore dialect extension)
				let myfile = mylangfilesize
			else
				" add dialects (if exist)
				let mydialect_ = MvElementAt(mydialects, ",", j)
				if MvNumberOfElements(mydialects, ",") > 0 && mydialect_ != ""
					let myfile = mylangfilesize . "-" . mydialect_
				endif
			endif

			" append 4 part names
			let k = 1		" 4 dict parts (upper | lower && single | multiword)
			while k <= 4

				" append individual number and extension
				let fullfile = myfile . "_" . k . ".vim"

				" accumulate total file sizes
				let filesizes = filesizes + getfsize(fullfile)

				" add to list
				let loadfiles = MvAddElement(loadfiles, "\n", fullfile)

				" add 1 for all, 2 to omit case-sensitive (not slow, don't bother)
				let k = k + 1

			endwhile
			let j = j + 1

		endwhile
		let i = i + 1

	endwhile

	" load list
	let myfilesizepart = ""
	let i = 0
	let max = MvNumberOfElements(loadfiles, "\n")
	while i < max

		" get file
		let myfile = MvElementAt(loadfiles, "\n", i)

		" show progress bar
		" get this file's size
		let myfilesize = getfsize(myfile)
		" calculate this file's size percentage of total
		let myfilesizeparttmp = (myfilesize * 100) / filesizes
		let myfilesizepart = myfilesizepart + myfilesizeparttmp

		" set case sensitivity for specific load
		let k = strpart(myfile, strlen(myfile) - 5, 1)
		if k == 1 || k == 3
			syntax case ignore
		elseif k == 2 || k == 4
			syntax case match
		endif

		" echo progress
		call ProgressBar(myfilesizepart, "    Loading dictionaries... ", "=", 0)

		" source dictionary of choice
		call Cream_source(myfile)

		let i = i + 1

	endwhile

endfunction

" s:Cream_spell_dict_avail() {{{1
function! s:Cream_spell_dict_avail()
" Return comma-separated list of ISO639-2 abbreviations for available
" language dictionaries. Returns "" if none found.
" o The entire dictionary is said to be available even if only one
"   part of a multi-part exists.

	let myavail = ""
	" speed up: we only search for "_1" so we have 1/4 to deal with
	let mylangs = Cream_getfilelist(g:cream_spelldicts . "cream-spell-dict-*_1.vim")

	" spool through dicts found and summarize
	let i = 0
	let max = MvNumberOfElements(mylangs, "\n")
	while i < max
		let myfile = MvElementAt(mylangs, "\n", i)
		" get ISO639-2 lang code (first 3 letters after "cream-spell-dict-")
		let mylang = matchstr(myfile, 'cream-spell-dict-\zs\a\a\a')
		" if not already in avail, add
		if matchstr(myavail, mylang) == ""
			let myavail = myavail . mylang . ","
		endif
		let i = i + 1
	endwhile

	return myavail

endfunction

" s:Cream_spell_dialect_avail() {{{1
function! s:Cream_spell_dialect_avail(lang)
" return sorted, comma-separated list of dialects available for {lang}

		let myavail = ""
		" directory the language (*** ASSUMPTION: "_1" is representative of available! ***)
		let mylang = Cream_getfilelist(g:cream_spelldicts . "cream-spell-dict-" . a:lang . "*_1.vim")
		" parse filenames for available dialects
		let i = 0
		let max = MvNumberOfElements(mylang, "\n")
		while i < max

			let myfile = MvElementAt(mylang, "\n", i)
			" get dialect code (two uppercase chars immediately preceeding "_#.vim")
			let mydialect = matchstr(myfile, '\u\u\ze_\d\.vim')

			" if filename has dialect (exists)
			if mydialect != ""
				" add if doesn't already exist
				if MvContainsElement(myavail, ",", mydialect) == 0
					let myavail = MvAddElement(myavail, ",", mydialect)
				endif
			endif
			let i = i + 1

		endwhile

		" sort alphabetically
		let myavail = MvQSortElements(myavail, ",", 'CmpByString', 1)
		return myavail

endfunction

" s:Cream_spell_size_avail() {{{1
function! s:Cream_spell_size_avail(lang)
" return comma-separated list of dict sizes available for a language

	let myavail = ""
	" directory the language (Hack: assume "_1" represents available.)
	let mylangfiles = Cream_getfilelist(g:cream_spelldicts . "cream-spell-dict-" . a:lang . "*_1.vim")
	" find/parse out the available sizes of each filename
	let i = 0
	let max = MvNumberOfElements(mylangfiles, "\n")
	while i < max

		let myfile = MvElementAt(mylangfiles, "\n", i)
		let mysize = matchstr(myfile, "cream-spell-dict-" . a:lang . '-\zs[smlh]\{1}\ze.*\.vim')

		" add if doesn't already exist
		if match(myavail, mysize) == -1
			let myavail = MvAddElement(myavail, ",", mysize)
		endif
		let i = i + 1

	endwhile


	return myavail

endfunction

" s:Cream_spell_getlangstr() {{{1
function! s:Cream_spell_getlangstr(lang)
" return {lang}'s elements out of g:CREAM_SPELL_LANG
" return {lang} if doesn't exist
	" iterate through language array,
	let i = 0
	let mylang = ""
	let max = MvNumberOfElements(g:CREAM_SPELL_LANG, ";")
	while i < max
		let mylangstr = MvElementAt(g:CREAM_SPELL_LANG, ";", i)
		" if first element matches language
		if MvElementAt(mylangstr, ",", 0) == a:lang
			return mylangstr
		endif
		let i = i + 1
	endwhile
	return a:lang
endfunction

" s:Cream_spell_getlangstrpos() {{{1
function! s:Cream_spell_getlangstrpos(lang)
" return language string's position in array; 0 based; -1 if doesn't
" exist
	" iterate through language array
	let i = 0
	let max = MvNumberOfElements(g:CREAM_SPELL_LANG, ";")
	while i <= max
		let mylang = MvElementAt(g:CREAM_SPELL_LANG, ";", i)
		" if first element matches language
		if MvElementAt(mylang, ",", 0) == a:lang
			return i
		endif
		let i = i + 1
	endwhile
	return -1
endfunction

" 1}}}
" vim:foldmethod=marker
