/*
 * cowbell
 * Copyright (c) 2005 Brad Taylor
 *
 * This file is part of cowbell.
 *
 * cowbell is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * cowbell is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with cowbell; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.IO;
using System.Collections;

namespace Cowbell.Base
{
	public class Batch
	{
		/* public methods */
		public Batch (string[] paths)
		{
			Ask = true;		// TODO: Make this a runtime option

			ArrayList search_dirs = new ArrayList ();
			foreach (string path in paths)
			{
				string dir = Path.Combine (Directory.GetCurrentDirectory (), path);

				search_dirs.AddRange (GetDirectories (dir));
			}

			foreach (string dir in search_dirs)
			{
				Console.WriteLine (Catalog.GetString ("Searching {0}..."), dir);
				ProcessAlbum (dir);
				Console.WriteLine ();
				Console.WriteLine ();
			}

			Console.WriteLine (Catalog.GetString ("Done!"));
		}

		/* private fields */
		private bool Ask;
	
		/* private methods */	
		private string[] GetDirectories (string path)
		{
			if (!Directory.Exists (path)) {
				Console.WriteLine (Catalog.GetString ("Error: \"{0}\" does not exist."), path);
				return new string[0];
			}

			// Don't add hidden directories
			string[] folders = path.Split (Path.DirectorySeparatorChar);
			if (folders.Length > 0) {
				if (folders[folders.Length - 1].StartsWith (".")) {
					return new string[0];
				}
			}

			ArrayList subdirs = new ArrayList ();
			string[] dirs = Directory.GetDirectories (path);
			if (dirs.Length < 1) {
				subdirs.Add (path);
			}

			foreach (string dir in dirs)
			{
				subdirs.AddRange (GetDirectories (dir));
			}

			return (string[])subdirs.ToArray (typeof (string));
		}

		private void ProcessAlbum (string dir)
		{
			Console.WriteLine (Catalog.GetString ("Found {0}"), dir);

			Filesystem.Import (Directory.GetFiles (dir));

			// Guess song info
			Console.WriteLine (Catalog.GetString ("Guessing Song Information..."));
			AmazonMetadataProxy p = new AmazonMetadataProxy ();
			p.Block = true;
			p.Import ();

			ArrayList songs = (ArrayList)Runtime.Database.Clone ();	
			songs.Sort ();

			// normalize case
			foreach (Song s in songs)
			{
				s.Title = Utils.ToSmartTitleCase (s.Title);
			}

			Console.WriteLine ("======================================");
			Console.WriteLine (Catalog.GetString ("{0}- {1}"),
			                   Runtime.Database.GlobalData.Artist,
			                   Runtime.Database.GlobalData.Album);

			foreach (Song s in songs)
			{
				Console.WriteLine (Catalog.GetString ("{0}. {1}"), s.TrackNumber, s.Title);
			}

			Console.WriteLine ("======================================");
			if (Ask) {
				if (!CliUtils.QueryWithYesNo (Catalog.GetString ("Is this information correct?"), true)) {
					foreach (Song s in songs)
					{
						Console.WriteLine ();			
						Console.WriteLine (Catalog.GetString ("Song: {0,3}. {1}"),
								   s.TrackNumber, s.Title);
						string title = CliUtils.Query (Catalog.GetString ("New name: "));
						if (title != String.Empty) {
							s.Title = title;
						}
					}
				}

				if (!CliUtils.QueryWithYesNo (String.Format (Catalog.GetString ("Is the current genre ({0}) correct?"),
				                                             Runtime.Database.GlobalData.Genre), true)) {
					string genre = CliUtils.Query (Catalog.GetString ("Enter the new genre: "));
					Runtime.Database.GlobalData.Genre = genre;
				}
			}

			Console.WriteLine (Catalog.GetString ("Saving..."));

			foreach (Song s in songs)
			{
				s.Save ();
			}

			if (Runtime.Preferences.RenameFiles) {
				Console.WriteLine (Catalog.GetString ("Renaming files..."));
				Runtime.Database.RenameAll (Runtime.Preferences.RenameFilesPattern);
			}

			if (Runtime.Preferences.ExportPlaylist) {
				Console.WriteLine (Catalog.GetString ("Creating album-order playlist..."));
				Filesystem.ExportPlaylist (Filesystem.GetBasedir ());
			}
		}
	}
}
