#!/usr/bin/env python

#****************************************************************************
# unitedit.py, provides a line edit for unit entry
#
# ConvertAll, a units conversion program
# Copyright (C) 2003, Douglas W. Bell
#
# This is free software; you can redistribute it and/or modify it under the
# terms of the GNU General Public License, Version 2.  This program is
# distributed in the hope that it will be useful, but WITTHOUT ANY WARRANTY.
#*****************************************************************************

from qt import *


class UnitEdit(QLineEdit):
    """Text line editor for unit entry"""
    def __init__(self, unitGroup, parent=None, name=None):
        QLineEdit.__init__(self, parent, name)
        self.unitGroup = unitGroup
        if qVersion()[0] == '2':
            self.connect(self, SIGNAL('textChanged(const QString &)'), \
                         self.updateGroup)

    def unitUpdate(self):
        """Update text from unit group"""
        newText = self.unitGroup.unitString()
        cursorPos = len(newText) - self.text().length() + self.cursorPosition()
        if cursorPos < 0:      # cursor set to same distance from right end
            cursorPos = 0
        self.blockSignals(1)
        self.setText(newText)
        self.setCursorPosition(cursorPos)
        self.blockSignals(0)
        self.emit(PYSIGNAL('unitChanged'), ())  # update numEdit

    def updateGroup(self):
        """Update unit group based on edit text change (except spacing change)"""
        if str(self.text()).replace(' ', '') \
           != self.unitGroup.unitString().replace(' ', ''):
            self.unitGroup.update(str(self.text()), self.cursorPosition())
            self.emit(PYSIGNAL('currentChanged'), ())  # update listView
            self.unitUpdate()   # replace text with formatted text

    def updateCurrentUnit(self):
        """Change current unit based on cursor movement"""
        self.unitGroup.updateCurrentUnit(str(self.text()), \
                                         self.cursorPosition())
        self.emit(PYSIGNAL('currentChanged'), ())  # update listView

    def keyPressEvent(self, event):
        """Keys for cursor movement, return and up/down"""
        if event.key() == Qt.Key_Up:
            self.unitGroup.moveToNext(1)
            self.emit(PYSIGNAL('currentChanged'), ())  # update listView
            self.unitUpdate()
        elif event.key() == Qt.Key_Down:
            self.unitGroup.moveToNext(0)
            self.emit(PYSIGNAL('currentChanged'), ())  # update listView
            self.unitUpdate()
        elif event.key() in (Qt.Key_Return, Qt.Key_Enter):
            self.unitGroup.completePartial()
            self.emit(PYSIGNAL('currentChanged'), ())  # update listView
            self.unitUpdate()
        elif event.key() in (Qt.Key_Left, Qt.Key_Right, \
                             Qt.SHIFT + Qt.Key_Left, Qt.SHIFT + Qt.Key_Right, \
                             Qt.Key_Home, Qt.Key_End, Qt.CTRL + Qt.Key_A, \
                             Qt.CTRL + Qt.Key_B, Qt.CTRL + Qt.Key_E, \
                             Qt.CTRL + Qt.Key_F):
            QLineEdit.keyPressEvent(self, event)
            self.updateCurrentUnit()
        else:
            QLineEdit.keyPressEvent(self, event)
            if qVersion()[0] >= '3':
                self.updateGroup()  # update on key in Qt3 (signal cursor bug)

    def paste(self):
        """Override paste to get change in Qt3 (not virtual in Qt2)"""
        QLineEdit.paste(self)
        self.updateGroup()

    def cut(self):
        """Override cut to get change in Qt3 (not virtual in Qt2)"""
        QLineEdit.cut(self)
        self.updateGroup()

    def event(self, event):
        """Catch tab press to complete unit"""
        if event.type() == QEvent.KeyPress and event.key() == Qt.Key_Tab:
            self.unitGroup.completePartial()
            self.emit(PYSIGNAL('currentChanged'), ())  # update listView
            self.unitUpdate()
        return QLineEdit.event(self, event)

    def mouseReleaseEvent(self, event):
        """Catch cursor movement"""
        QLineEdit.mouseReleaseEvent(self, event)
        self.updateCurrentUnit()

    def wheelEvent(self, event):
        """Catch wheel for list view"""
        self.unitGroup.moveToNext(event.delta() > 0 and 1 or 0)
        self.emit(PYSIGNAL('currentChanged'), ())  # update listView
        self.unitUpdate()
