# Cohoba - a GNOME client for Telepathy
#
# Copyright (C) 2006 Collabora Limited
#
# This package is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This package is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with this package; if not, write to the Free Software
# Foundation, 51 Franklin Street, Fifth Floor, Boston, MA, 02110-1301 USA.

import gtk
from gettext import gettext as _

from telepathy.constants import (CONNECTION_STATUS_CONNECTED,
	CONNECTION_STATUS_CONNECTING, CONNECTION_STATUS_DISCONNECTED,

	CONNECTION_STATUS_REASON_NONE_SPECIFIED,
	CONNECTION_STATUS_REASON_REQUESTED,
	CONNECTION_STATUS_REASON_NETWORK_ERROR,
	CONNECTION_STATUS_REASON_AUTHENTICATION_FAILED,
	CONNECTION_STATUS_REASON_ENCRYPTION_ERROR,
	CONNECTION_STATUS_REASON_NAME_IN_USE,
	CONNECTION_STATUS_REASON_CERT_NOT_PROVIDED,
	CONNECTION_STATUS_REASON_CERT_UNTRUSTED,
	CONNECTION_STATUS_REASON_CERT_EXPIRED,
	CONNECTION_STATUS_REASON_CERT_NOT_ACTIVATED,
	CONNECTION_STATUS_REASON_CERT_HOSTNAME_MISMATCH,
	CONNECTION_STATUS_REASON_CERT_FINGERPRINT_MISMATCH,
	CONNECTION_STATUS_REASON_CERT_SELF_SIGNED,
	CONNECTION_STATUS_REASON_CERT_OTHER_ERROR)

import cohoba
from os.path import join, splitext, exists
import logging

ICON_THEME = gtk.icon_theme_get_default()

def load_icon(icon, width=-1, height=-1):
	pixbuf = None
	if icon != None and icon != "":
		try:
			our_icon = join(cohoba.ART_DATA_DIR, icon)
			print our_icon
			if exists(our_icon):
				pixbuf = gtk.gdk.pixbuf_new_from_file_at_size(our_icon, width, height)
			elif icon.startswith("/"):
				pixbuf = gtk.gdk.pixbuf_new_from_file_at_size(icon, width, height)
			else:
				pixbuf = ICON_THEME.load_icon(splitext(icon)[0], width, gtk.ICON_LOOKUP_USE_BUILTIN)
		except Exception, msg1:
			try:
				pixbuf = ICON_THEME.load_icon(icon, width, gtk.ICON_LOOKUP_USE_BUILTIN)
			except Exception, msg2:
				print 'Error:load_icon:Icon Load Error:%s (or %s)' % (msg1, msg2)
				
	return pixbuf
	
def global_error_handler(e, msg="-"):
	get_logger("DBus").error('Unexpected dbus error (%s): %s', msg, e)

def ignore_error_handler(e, msg="-"):
	get_logger("DBus").debug("Ignoring dbus error (%s): %s", msg, e)
	
def presence_status_to_string(name):
	if name == 'available':
		msg = _('Available')
	elif name == 'away':
		msg = _('Away')
	elif name == 'brb':
		msg = _('Be Right Back')
	elif name == 'busy':
		msg = _('Busy')
	elif name == 'dnd':
		msg = _('Do Not Disturb')
	elif name == 'xa':
		msg = _('Extended Away')
	elif name == 'hidden':
		msg = _('Hidden')
	elif name == 'offline':
		msg = _('Offline')
	else:
		# translators: Unkown online status
		msg = _('Unknown')
		name = 'offline'

	return msg

def presence_status_message(parameters):
	if "message" in parameters and parameters["message"] and parameters["message"] != "":
		return parameters['message']
		
def conn_status_to_string(status):
	if status == CONNECTION_STATUS_CONNECTED:
		return _("Connected")
	elif status == CONNECTION_STATUS_CONNECTING:
		return _("Connecting")
	elif status == CONNECTION_STATUS_DISCONNECTED:
		return _("Disconnected")
	else:
		# translators: Unkown connection status
		return _("Unknown")
		
def conn_reason_to_string(reason):
	if reason == CONNECTION_STATUS_REASON_NONE_SPECIFIED or reason == None:
		# translators: No (dis)connection reason specified
		return _("None")
	elif reason == CONNECTION_STATUS_REASON_REQUESTED:
		return _("User Request")
	elif reason == CONNECTION_STATUS_REASON_NETWORK_ERROR:
		return _("Network Error")
	elif reason == CONNECTION_STATUS_REASON_AUTHENTICATION_FAILED:
		return _("Authentication Failed")
	elif reason == CONNECTION_STATUS_REASON_ENCRYPTION_ERROR:
		return _("Encryption Error")
	elif reason == CONNECTION_STATUS_REASON_NAME_IN_USE:
		return _("Someone is already connected with that name")
	elif reason == CONNECTION_STATUS_REASON_CERT_NOT_PROVIDED:
		return _("The server did not provide a SSL certificate")
	elif reason == CONNECTION_STATUS_REASON_CERT_UNTRUSTED:
		return _("The server's SSL certificate could not be trusted")
	elif reason == CONNECTION_STATUS_REASON_CERT_EXPIRED:
		return _("The server's SSL certificate has expired")
	elif reason == CONNECTION_STATUS_REASON_CERT_NOT_ACTIVATED:
		return _("The server's SSL certificate is not yet valid")
	elif reason == CONNECTION_STATUS_REASON_CERT_HOSTNAME_MISMATCH:
		return _("The server's SSL certificate did not match its hostname")
	elif reason == CONNECTION_STATUS_REASON_CERT_FINGERPRINT_MISMATCH:
		return _("The server's SSL certificate does not have the expected fingerprint")
	elif reason == CONNECTION_STATUS_REASON_CERT_SELF_SIGNED:
		return _("The server's SSL certificate is self-signed")
	elif reason == CONNECTION_STATUS_REASON_CERT_OTHER_ERROR:
		return _("Error validating the server's SSL certificate.")
	else:
		# translators: Unknown connection error reason
		return _("Unknown Error")

def protocol_to_description(protocol):
	if protocol == "jabber":
		return "Jabber"
	elif protocol == "aim":
		return "AOL Instant Messenger"
	elif protocol == "gadugadu":
		return "Gadu-Gadu"
	elif protocol == "groupwise":
		return "Novell Groupwise"
	elif protocol == "icq":
		return "ICQ"
	elif protocol == "irc":
		return "Internet Relay Chat"
	elif protocol == "msn":
		return "MSN Messenger"
	elif protocol == "napster":
		return "Napster"
	elif protocol == "silc":
		return "SILC"
	elif protocol == "sip":
		return "Session Initiation Protocol (SIP)"
	elif protocol == "trepia":
		return "Trepia"
	elif protocol == "yahoo":
		return "Yahoo! Messenger"
	elif protocol == "zephyr":
		return "Zephyr"
	else:
		print 'Warning:Unknown protocol name:', protocol
		return protocol

def dbus_type_to_python_type (dbus_type):
	if dbus_type == "s":
		return str
	elif dbus_type == "b":
		def to_bool(string):
			if string.lower() == "false":
				return False
			else:
				return True
		return to_bool
	else:
		return int

# A better version of this functionality should eventually go into python-dbus.
class SignalConnection(object):
	"""A workaround for the inability to disconnect from DBUS signals.

	Encapsulates the connection of a DBus signal to a Python callable,
	with the following behaviour:

	- When the signal is received, the callable is called with appropriate
	  parameters.
	- The callable is referenced by the SignalConnection (and so can't
	  be garbage-collected) until the disconnect() method is called.
	- The disconnect() method drops the reference to the callable, and
	  if the signal is received after this, nothing happens.
	"""
	__slots__ = ('_ref',)

	def __init__(self, interface, signal, callable):
		"""Constructor.

		:Parameters:
			interface : DBus Interface object
				The Interface proxy
			signal : string
				The signal name
			callable : Python object
				Any callable object (usually a bound method)
		"""
		assert callable is not None
		self._ref = callable
		interface.connect_to_signal(signal, self._handler)

	def _handler(self, *args, **kwargs):
		if self._ref is not None:
			self._ref(*args, **kwargs)

	def disconnect(self):
		"""Cause this object's associated callable not to receive
		any more signals from this connection.
		"""
		self._ref = None

logging.basicConfig(level=cohoba.LOGGING_LEVEL, format="%(name)s:%(levelname)s:%(message)s")
def get_logger(name):
	return logging.getLogger(name)
	
#Decorator
def singleton(cls):
	instances = {}
	def getinstance():
		if cls not in instances:
			instances[cls] = cls()
		return instances[cls]
	return getinstance
