# Cohoba - a GNOME client for Telepathy
#
# Copyright (C) 2006 Collabora Limited
#
# This package is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This package is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with this package; if not, write to the Free Software
# Foundation, 51 Franklin Street, Fifth Floor, Boston, MA, 02110-1301 USA.

import ConfigParser, os
import dircache

from telepathy.interfaces import (CHANNEL_TYPE_TEXT, CHANNEL_TYPE_CONTACT_LIST,
	CHANNEL_TYPE_STREAMED_MEDIA)

"""
The registry of channel handlers takes the form of any number of .chandler files, which
are searched for in /usr/share/telepathy/managers or in ~/.telepathy.

.chandler files should have an initial stanza of the form:

[ChannelHandler]
Name = value
BusName = value
ObjectPath = value
ChannelType = value

where:
'Name' sets the given name of the Channel Handler, must be an unique name amongst other chandlers
'BusName' sets the D-Bus bus name of this connection manager.
'ObjectPath' sets the D-Bus object path to the ConnectionManager object under this service.
'ChannelType' sets the Telepathy Channel Type that the channel handler can handle

When the Mission Control receives a new channel, it will call all matching channel
handlers for the matching channel type applying the following method to the ObjectPath object:

HandleChannel(
	s: connection_name,
	o: connection_object_path,
	s: telepathy_channel_type,
	o: chanbel_object_path,
	u: handle_type,
	u: handle) -> void
"""

class ChandlersRegistry:
	def __init__(self):
		self.chandlers = {}
		self.LoadChandlers()

	def LoadChandler(self, path):
		config = ConfigParser.SafeConfigParser()
		config.read(path)
		chandler = dict(config.items("ChannelHandler"))
		
		if "busname" not in chandler:
			raise ConfigParser.NoOptionError("busname", "ChannelHandler")
		elif "objectpath" not in chandler:
			raise ConfigParser.NoOptionError("objectpath", "ChannelHandler")
		elif "channeltype" not in chandler:
			raise ConfigParser.NoOptionError("channeltype", "ChannelHandler")
		
		# If name can't be found, fallback to use dbus name
		if "name" in chandler:
			self.chandlers[chandler["name"]] = chandler
		else:
			self.chandlers[chandler["busname"]] = chandler

	def LoadChandlers(self):
		"""
		Searches local and system wide configurations

		Can raise all ConfigParser errors. Generally filename member will be
		set to the name of the erronous file.
		"""

		all_paths = (
			'/usr/share/telepathy/managers/',
			'/usr/local/share/telepathy/managers/',
			os.path.expanduser('~/.telepathy'),
			)

		for path in all_paths:
			if os.path.exists(path):
				for name in dircache.listdir(path):
					if name.endswith('.chandler'):
						self.LoadChandler(os.path.join(path, name))

	def GetChandlers(self, channel_type):
		"""
		Returns names of chandlers that can handle the given channel type.
		"""
		return [name for name, chandler in self.chandlers.items() if channel_type == chandler["channeltype"]]

	def GetBusName(self, chandler):
		assert(chandler in self.chandlers)
		assert('busname' in self.chandlers[chandler])
		return self.chandlers[chandler]['busname']

	def GetObjectPath(self, chandler):
		assert(chandler in self.chandlers)
		assert('objectpath' in self.chandlers[chandler])
		return self.chandlers[chandler]['objectpath']

if __name__ == "__main__":
	reg = ChandlersRegistry()
	reg.LoadChandlers()
	
	for chan_type in (CHANNEL_TYPE_CONTACT_LIST, CHANNEL_TYPE_STREAMED_MEDIA, CHANNEL_TYPE_TEXT):
		chandlers = reg.GetChandlers(chan_type)
		print '--Available chandlers for:', chan_type, ':', chandlers
		for chandler in chandlers:
			name = reg.GetBusName(chandler)
			path = reg.GetObjectPath(chandler)
			print '----Chandler:', chandler, name, path
