;;; xref.lisp -- a cross-reference facility for CMUCL
;;;
;;; Author: Eric Marsden <emarsden@laas.fr>
;;;
(ext:file-comment
  "$Header: /project/cmucl/cvsroot/src/compiler/xref.lisp,v 1.2 2003/02/24 16:40:05 emarsden Exp $")
;;
;; This code was written as part of the CMUCL project and has been
;; placed in the public domain.
;;
;;
;; The cross-referencing facility provides the ability to discover
;; information such as which functions call which other functions and
;; in which program contexts a given global variables may be used. The
;; cross-referencer maintains a database of cross-reference
;; information which can be queried by the user to provide answers to
;; questions like:
;;
;;  - the program contexts where a given function may be called,
;;    either directly or indirectly (via its function-object).
;;
;;  - the program contexts where a global variable (ie a dynamic
;;    variable or a constant variable -- something declared with
;;    DEFVAR or DEFPARAMETER or DEFCONSTANT) may be read, or bound, or
;;    modified.
;;
;; More details are available in "Cross-Referencing Facility" chapter
;; of the CMUCL User's Manual.
;;
;;
;; Missing functionality:
;;
;;   - there is no mechanism to save cross-reference information in a
;;     FASL file, so you need to recompile all files in the current image
;;
;;   - maybe add macros EXT:WITH-XREF, and an :xref option to
;;     COMPILE-FILE, as per ACL.
;;
;;   - add WHO-MACROEXPANDS
;;
;;   - in (defun foo (x) (flet ((bar (y) (+ x y))) (bar 3))), we want to see
;;     FOO calling (:internal BAR FOO)
;;
;; The cross-reference facility is implemented by walking the IR1
;; representation that is generated by CMUCL when compiling (for both
;; native and byte-compiled code, and irrespective of whether you're
;; compiling from a file, from a stream, or interactively from the
;; listener).


;; types of names that we can see in IR1 LEAF nodes:
;;
;;   - symbol that is a function-name or a macro name
;;   - strings of the form "DEFINE-COMPILER-MACRO ~A"
;;   - strings of the form "DEFMETHOD FOOBAR (SPECIALIZER1 SPECIALIZER2)"
;;   - strings of the form "defstruct foo"
;;   - strings of the form "Creation Form for #<kernel::class-cell struct-two>"
;;
;; and it would be nice to change this to
;;
;;   - (:method foobar (specializer1 specializer2))
;;   - (:flet external-function internal-function)
;;   - (:labels external-function internal-function)
;;   - (:internal containing-function 0) for an anonymous lambda
;;   - (:internal (flet external internal) 0) for anonymous lambda inside an FLET/LABELS


(in-package :xref)

(export '(init-xref-database
          register-xref
          who-calls
          who-references
          who-binds
          who-sets
          #+pcl who-subclasses
          #+pcl who-superclasses
          make-xref-context
          xref-context-name
          xref-context-file
          xref-context-source-path))


(defstruct (xref-context (:print-function %print-xref-context))
  name
  (file *compile-file-truename*)
  (source-path nil))

(defun %print-xref-context (s stream d)
  (declare (ignore d))
  (format stream "#<xref-context ~S~@[ in ~S~]>"
          (xref-context-name s)
          (xref-context-file s)))


;; program contexts where a globally-defined function may be called at runtime
(defvar *who-calls* (make-hash-table :test #'eq))

;; program contexts where a global variable may be referenced
(defvar *who-references* (make-hash-table :test #'eq))

;; program contexts where a global variable may be bound
(defvar *who-binds* (make-hash-table :test #'eq))

;; program contexts where a global variable may be set
(defvar *who-sets* (make-hash-table :test #'eq))

;; you can print these conveniently with code like
;; (maphash (lambda (k v) (format t "~S <-~{ ~S~^,~}~%" k v)) xref::*who-sets*)
;; or
;; (maphash (lambda (k v) (format t "~S <-~%   ~@<~@;~S~^~%~:>~%" k v)) xref::*who-calls*)


(defun register-xref (type target context)
  (declare (type xref-context context))
  (let ((database (ecase type
                    (:calls *who-calls*)
                    (:references *who-references*)
                    (:binds *who-binds*)
                    (:sets *who-sets*))))
    (if (gethash target database)
        (pushnew context (gethash target database) :test 'equal)
        (setf (gethash target database) (list context)))))

;; INIT-XREF-DATABASE -- interface
;;
(defun init-xref-database ()
  "Reinitialize the cross-reference database."
  (setf *who-calls* (make-hash-table :test #'eq))
  (setf *who-references* (make-hash-table :test #'eq))
  (setf *who-binds* (make-hash-table :test #'eq))
  (setf *who-sets* (make-hash-table :test #'eq))
  (values))


;; WHO-CALLS -- interface
;;
(defun who-calls (function &key (reverse nil))
  "Return a list of those program contexts where a globally-defined
function may be called at runtime."
  (declare (type (or symbol function) function))
  (let ((name (etypecase function
                (symbol function)
                ;; FIXME what about MACRO-FUNCTION? 
                (function (multiple-value-bind (ignore ignored fname)
                              (function-lambda-expression function)
                            (declare (ignore ignore ignored))
                            (unless fname
                              (error "Function ~a has no name" function))
                            fname))))
        (fun (etypecase function
               (symbol (symbol-function function))
               (function function))))
    (if reverse
        ;; this depends on the function having been loaded (rather than only
        ;; on it having been compiled), and on the source file that it was
        ;; compiled from being accessible.
        (let ((debug-fun (di::function-debug-function fun))
              (called (list)))
          (unless debug-fun
            (error "Function ~a has no debug information" function))
          (di::do-debug-function-blocks (block debug-fun)
            (di::do-debug-block-locations (loc block)
              (di::fill-in-code-location loc)
              (when (eq :call-site (di::compiled-code-location-kind loc))
                (let* ((loc (debug::maybe-block-start-location loc))
                       (form-num (di:code-location-form-number loc)))
                  (multiple-value-bind (translations form)
                      (debug::get-top-level-form loc)
                    (unless (< form-num (length translations))
                      (error "Source path no longer exists"))
                    (pushnew (car (di:source-path-context form (aref translations form-num) 0))
                             called))))))
          called)
        (gethash name *who-calls*))))

;; WHO-REFERENCES -- interface
;;
(defun who-references (global-variable)
  "Return a list of those program contexts where GLOBAL-VARIABLE
may be referenced at runtime."
  (declare (type symbol global-variable))
  (gethash global-variable *who-references*))

;; WHO-BINDS -- interface
;;
(defun who-binds (global-variable)
  "Return a list of those program contexts where GLOBAL-VARIABLE may
be bound at runtime."
  (declare (type symbol global-variable))
  (gethash global-variable *who-binds*))

;; WHO-SETS -- interface
;;
(defun who-sets (global-variable)
  "Return a list of those program contexts where GLOBAL-VARIABLE may
be set at runtime."
  (declare (type symbol global-variable))
  (gethash global-variable *who-sets*))


;; introspection functions from the CLOS metaobject protocol

;; WHO-SUBCLASSES -- interface
;;
#+pcl
(defun who-subclasses (class)
  (declare (type class class))
  (pcl::class-direct-subclasses class))

;; WHO-SUPERCLASSES -- interface
;;
#+pcl
(defun who-superclasses (class)
  (declare (type class class))
  (pcl::class-direct-superclasses class))

;; generic functions defined for this class
#+pcl
(defun who-specializes (class)
  (declare (type class class))
  (let ((pcl-class (etypecase class
                     (lisp:class (pcl::coerce-to-pcl-class class))
                     (pcl::class class))))
    (pcl::specializer-direct-methods pcl-class)))



(in-package :compiler)


(defun prettiest-caller-name (lambda-node toplevel-name)
  (cond
    ((not lambda-node)
     (list :anonymous toplevel-name))

    ;; for DEFMETHOD forms
    ((eql 0 (search "defmethod" toplevel-name :test 'char-equal))
     ;; FIXME this probably won't handle FLET/LABELS inside a
     ;; defmethod properly ...
     (let* ((readable (substitute #\? #\# toplevel-name))
            (listed (concatenate 'string "(" readable ")"))
            (*read-eval* nil)
            (list (ignore-errors (read-from-string listed))))
       (cons :method (rest list))))

    ;; LET and FLET bindings introduce new unnamed LAMBDA nodes.
    ;; If the home slot contains a lambda with a nice name, we use
    ;; that; otherwise fall back on the toplevel-name.
    ((or (not (eq (lambda-home lambda-node) lambda-node))
         (lambda-calls lambda-node))
     (let ((home (lambda-name (lambda-home lambda-node)))
           (here (lambda-name lambda-node)))
       (cond ((and home here)
              (list :internal home here))
             ((symbolp here) here)
             ((symbolp home) home)
             (t
              (or here home toplevel-name)))))

    ;; LET and FLET bindings introduce new unnamed LAMBDA nodes.
    ;; If the home slot contains a lambda with a nice name, we use
    ;; that; otherwise fall back on the toplevel-name.
    #+nil
    ((not (lambda-name lambda-node))
     (let ((home (lambda-home lambda-node)))
       (or (and home (lambda-name home))
           toplevel-name)))

    ((and (listp (lambda-name lambda-node))
          (eq :macro (first (lambda-name lambda-node))))
     (lambda-name lambda-node))

    ;; a reference from a macro is named (:macro name)
    ((eql 0 (search "defmacro" toplevel-name :test 'char-equal))
     (list :macro (subseq toplevel-name 9)))

    ;; probably "Top-Level Form"
    ((stringp (lambda-name lambda-node))
     (lambda-name lambda-node))

    ;; probably (setf foo)
    ((listp (lambda-name lambda-node))
     (lambda-name lambda-node))

    (t
     ;; distinguish between nested functions (FLET/LABELS) and
     ;; global functions by checking whether the node has a HOME
     ;; slot that is different from itself. Furthermore, a LABELS
     ;; node at the first level inside a lambda may have a
     ;; self-referential home slot, but still be internal. 
     (cond ((not (eq (lambda-home lambda-node) lambda-node))
            (list :internal
                  (lambda-name (lambda-home lambda-node))
                  (lambda-name lambda-node)))
           ((lambda-calls lambda-node)
            (list :internal/calls
                  (lambda-name (lambda-home lambda-node))
                  (lambda-name lambda-node)))
           (t (lambda-name lambda-node))))))


;; RECORD-NODE-XREFS -- internal
;;
;; TOPLEVEL-NAME is an indication of the name of the COMPONENT that
;; contains this node, or NIL if it was really "Top-Level Form". 
(defun record-node-xrefs (node toplevel-name)
  (declare (type node node))
  (let ((context (xref:make-xref-context)))
    (when *compile-file-truename*
      (setf (xref:xref-context-source-path context)
            (reverse
             (source-path-original-source
              (node-source-path node)))))
    (typecase node
      (ref
       (let* ((leaf (ref-leaf node))
              (lexenv (ref-lexenv node))
              (lambda (lexenv-lambda lexenv))
              (caller (prettiest-caller-name lambda toplevel-name)))
         
         (setf (xref:xref-context-name context) caller)
         (typecase leaf
           ;; a reference to a LEAF of type GLOBAL-VAR
           (global-var
            (let ((called (global-var-name leaf)))
              ;; a reference to #'C::%SPECIAL-BIND means that we are
              ;; binding a special variable. The information on which
              ;; variable is being bound, and within which function, is
              ;; available in the ref's LEXENV object.
              (cond ((eq called 'c::%special-bind)
                     (setf (xref:xref-context-name context) (caar (lexenv-blocks lexenv)))
                     (xref:register-xref :binds (caar (lexenv-variables lexenv)) context))
                    ;; we're not interested in lexical environments
                    ;; that have no name; they are mostly due to code
                    ;; inserted by the compiler (eg calls to %VERIFY-ARGUMENT-COUNT)
                    ((not caller)
                     nil)
                    ;; we're not interested in lexical environments
                    ;; named "Top-Level Form".
                    ((and (stringp caller)
                          (string= "Top-Level Form" caller))
                     t)
                    ((not called)
                     nil)
                    ((eq :global-function (global-var-kind leaf))
                     (xref:register-xref :calls called context))
                    ((eq :special (global-var-kind leaf))
                     (xref:register-xref :references called context)))))
           ;; a reference to a LEAF of type CONSTANT
           (constant
            (let ((called (constant-name leaf)))
              (and called
                   (not (eq called t))    ; ignore references to trivial variables
                   caller
                   (not (and (stringp caller) (string= "Top-Level Form" caller)))
                   (xref:register-xref :references called context)))))))

      ;; a variable is being set
      (cset
       (let* ((variable (set-var node))
              (lexenv (set-lexenv node)))
         (and (global-var-p variable)
              (eq :special (global-var-kind variable))
              (let* ((lblock (first (lexenv-blocks lexenv)))
                     (user (or (and lblock (car lblock)) toplevel-name))
                     (used (global-var-name variable)))
                (setf (xref:xref-context-name context) user)
                (and user used (xref:register-xref :sets used context))))))

      ;; nodes of type BIND are used to bind symbols to LAMBDA objects
      ;; (including for macros), but apparently not for bindings of
      ;; variables.
      (bind
       t))))


;; RECORD-COMPONENT-XREFS -- internal
;;
(defun record-component-xrefs (component)
  (declare (type component component))
  (do ((block (block-next (component-head component)) (block-next block)))
      ((null (block-next block)))
    (let ((fun (block-home-lambda block))
          (name (component-name component))
          (this-cont (block-start block))
          (last (block-last block)))
      (unless (eq :deleted (functional-kind fun))
        (loop
         (let ((node (continuation-next this-cont)))
           (record-node-xrefs node name)
           (let ((cont (node-cont node)))
             (when (eq node last) (return))
             (setq this-cont cont))))))))

;; EOF
