/*
 * Time-stamp: <99/03/23 18:21:58 panic>
 * Author:	The C-Mix Project <cmix@diku.dk>
 *              Arne John Glenstrup  <panic@diku.dk>
 * Contents:	Saving ray traced images to a TIFF file.
 */

#ifdef HAVE_LIBTIFF
#include <tiffio.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "tiff.h"

#define BITPLANES 8
#define pix_t uint8

static TIFF* tif = NULL;
static tif_color;
static tif_compress;
static pix_t* line;

int tiffOpenFile256colors(const char* prg_name, const char* filename,
		 const char* title,
		 int width, int height, int use_color, int compress) {
  static uint16 rcolormap[1 << 8];
  static uint16 gcolormap[1 << 8];
  static uint16 bcolormap[1 << 8];
  int m;
  time_t now; char* nowstr; char datetimestr[21];
  line = (pix_t*)calloc(width, sizeof(pix_t));
  tif_color = use_color; tif_compress = compress;
  if (!(tif = TIFFOpen(filename, "w"))) {
    fprintf(stderr, "%s: Couldn't open file `%s'\n", prg_name, filename);
    return -1;
  } else {
    TIFFSetField(tif, TIFFTAG_BITSPERSAMPLE, 8);
    
    rcolormap[0] = 0; gcolormap[0] = 0; bcolormap[0] = 0;
    rcolormap[1] = 65535; gcolormap[1] = 65535; bcolormap[1] = 65535;
    if (use_color) {
      for(m = 0; m < 216; m++) {
	rcolormap[m + 2] = 13107*( m     % 6);
	gcolormap[m + 2] = 13107*((m/6)  % 6);
	bcolormap[m + 2] = 13107*((m/36) % 6);
      }
      for(m = 0; m < 27; m++) {
	rcolormap[m + 220] = 3277*( m    % 3);
	gcolormap[m + 220] = 3277*((m/3) % 3);
	bcolormap[m + 220] = 3277*((m/9) % 3);
      }
    }
    else {
      for(m = 0; m < (1 << 8) - 2; m++) {
	rcolormap[m + 2] = (256 >> 8) * 256 * m;
	gcolormap[m + 2] = (256 >> 8) * 256 * m;
	bcolormap[m + 2] = (256 >> 8) * 256 * m;
      }
    }
    
    TIFFSetField(tif, TIFFTAG_COLORMAP,
		 &rcolormap, &gcolormap, &bcolormap);
    switch (tif_compress) {
    case TIFF_COMPRESS_PACKBITS:
      TIFFSetField(tif, TIFFTAG_COMPRESSION, COMPRESSION_PACKBITS);
      break;
    case TIFF_COMPRESS_LZW:
      TIFFSetField(tif, TIFFTAG_COMPRESSION, COMPRESSION_LZW);
      break;
    case TIFF_COMPRESS_NONE:
    default:
      TIFFSetField(tif, TIFFTAG_COMPRESSION, COMPRESSION_NONE);
    }
    time(&now); nowstr = ctime(&now);
    strncpy(datetimestr, nowstr + 4, 12);
    strncpy(datetimestr + 12, nowstr + 19, 5);
    strcpy(datetimestr + 17, "  ");
    TIFFSetField(tif, TIFFTAG_DATETIME, datetimestr);
    TIFFSetField(tif, TIFFTAG_DOCUMENTNAME, filename);
    TIFFSetField(tif, TIFFTAG_IMAGEDESCRIPTION, title);
    TIFFSetField(tif, TIFFTAG_IMAGELENGTH, height);
    TIFFSetField(tif, TIFFTAG_IMAGEWIDTH, width);
    TIFFSetField(tif, TIFFTAG_ORIENTATION, ORIENTATION_BOTLEFT);
    TIFFSetField(tif, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
    TIFFSetField(tif, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_PALETTE);
    TIFFSetField(tif, TIFFTAG_SAMPLEFORMAT, SAMPLEFORMAT_UINT);      
    TIFFSetField(tif, TIFFTAG_SOFTWARE, "C-Mix Project Ray Tracer");
  }
  return 0;
}

int tiffPlotColor256colors(int x, myfloat red, myfloat green, myfloat blue) {
  short r,g,b;
  static double r_err, g_err, b_err;
  
  r_err += red * 6.0;
  g_err += green * 6.0;
  b_err += blue * 6.0;
  
  if (r_err<1.0 && g_err<1.0 && b_err<1.0) {
    r = (short) (r_err * 3.0); if (r<0) r = 0;
    g = (short) (g_err * 3.0); if (g<0) g = 0;
    b = (short) (b_err * 3.0); if (b<0) b = 0;
    
    r_err -= (r/3.0);
    g_err -= (g/3.0);
    b_err -= (b/3.0);

    line[x] = r+3*g+9*b+220;

  }  else {   
    r = (short) (r_err + 0.5); if (r>=6) r = 5;
    g = (short) (g_err + 0.5); if (g>=6) g = 5;
    b = (short) (b_err + 0.5); if (b>=6) b = 5;

    r_err -= r;
    g_err -= g;
    b_err -= b;

    line[x] = r+6*g+36*b+2;
  }
  return 0;
}

int tiffPlotGrey256colors(int x, myfloat red, myfloat green, myfloat blue) {
  int i;
  
  i = (red * 0.299 + green * 0.587 + blue * 0.114) *
    (1 << BITPLANES) + (RANDOM()%100000)/100000.0;
  
  if (i <= 0)
    i = 0;
  else if (i >= (1 << BITPLANES) - 1)
    i = 1;
  else
    i++;

  line[x] = i;
  return 0;
}

int tiffPlot256colors(int x, myfloat red, myfloat green, myfloat blue) {
  if (tif_color)
    return tiffPlotColor256colors(x, red, green, blue);
  else
    return tiffPlotGrey256colors(x, red, green, blue);
}

int tiffOpenFile(const char* prg_name, const char* filename,
		 const char* title,
		 int width, int height, int use_color, int compress) {
  int m;
  time_t now; struct tm* nowstruct;
  char* nowstr; char datetimestr[20];
  tif_color = use_color; tif_compress = compress;
  if (!(tif = TIFFOpen(filename, "w"))) {
    fprintf(stderr, "%s: Couldn't open file `%s'\n", prg_name, filename);
    return -1;
  } else {
    TIFFSetField(tif, TIFFTAG_BITSPERSAMPLE, 8);
    
    switch (tif_compress) {
    case TIFF_COMPRESS_PACKBITS:
      TIFFSetField(tif, TIFFTAG_COMPRESSION, COMPRESSION_PACKBITS);
      break;
    case TIFF_COMPRESS_LZW:
      TIFFSetField(tif, TIFFTAG_COMPRESSION, COMPRESSION_LZW);
      break;
    case TIFF_COMPRESS_NONE:
    default:
      TIFFSetField(tif, TIFFTAG_COMPRESSION, COMPRESSION_NONE);
    }
    time(&now); nowstruct = localtime(&now);
    sprintf(datetimestr, "%4d:%02d:%02d %02d:%02d:%02d",
	    nowstruct->tm_year + 1900,
	    nowstruct->tm_mon + 1, nowstruct->tm_mday,
	    nowstruct->tm_hour, nowstruct->tm_min, nowstruct->tm_sec);
    TIFFSetField(tif, TIFFTAG_DATETIME, datetimestr);
    TIFFSetField(tif, TIFFTAG_DOCUMENTNAME, filename);
    TIFFSetField(tif, TIFFTAG_IMAGEDESCRIPTION, title);
    TIFFSetField(tif, TIFFTAG_IMAGELENGTH, height);
    TIFFSetField(tif, TIFFTAG_IMAGEWIDTH, width);
    TIFFSetField(tif, TIFFTAG_ORIENTATION, ORIENTATION_BOTLEFT);
    TIFFSetField(tif, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
    TIFFSetField(tif, TIFFTAG_ROWSPERSTRIP, 1);      
    TIFFSetField(tif, TIFFTAG_SAMPLEFORMAT, SAMPLEFORMAT_UINT);      
    TIFFSetField(tif, TIFFTAG_SOFTWARE, "C-Mix Project Ray Tracer");
    if (use_color) {
      TIFFSetField(tif, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_RGB);
      TIFFSetField(tif, TIFFTAG_SAMPLESPERPIXEL, 3);      
      line = (pix_t*)calloc(width * 3, sizeof(pix_t));
    } else {
      TIFFSetField(tif, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_MINISBLACK);
      line = (pix_t*)calloc(width, sizeof(pix_t));
    }      
  }
  return 0;
}

int tiffPlotColor(int x, myfloat red, myfloat green, myfloat blue) {
  line[3 * x]     = red   * 256.0;
  line[3 * x + 1] = green * 256.0;
  line[3 * x + 2] = blue  * 256.0;
  return 0;
}

int tiffPlotGrey(int x, myfloat red, myfloat green, myfloat blue) {
  int i;
  
  i = (red + green + blue) / 3 * 256.0;
  line[x]     = i;
  return 0;
}

int tiffPlot(int x, myfloat red, myfloat green, myfloat blue) {
  if (tif_color)
    return tiffPlotColor(x, red, green, blue);
  else
    return tiffPlotGrey(x, red, green, blue);
}

int tiffFlushLine(int y) {
  if (tif)
    return TIFFWriteScanline(tif, line, y, 0);
  else
    return -1;
}

int tiffCloseFile() {
  TIFFClose(tif);
}

#endif
