#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <clutter/clutter.h>

typedef struct
{
  ClutterActor *stage;
  ClutterActor *label;

  ClutterTimeline *timeline;

  volatile gboolean cancelled;
} TestThreadData;

static TestThreadData *
test_thread_data_new (void)
{
  TestThreadData *data;

  data = g_new0 (TestThreadData, 1);

  return data;
}

static void
test_thread_data_free (TestThreadData *data)
{
  g_object_unref (data->label);
  g_object_unref (data->stage);
  g_object_unref (data->timeline);

  g_free (data);
}

static gboolean
test_thread_done_idle (gpointer user_data)
{
  TestThreadData *data = user_data;

  g_print ("Thread completed\n");

  clutter_label_set_text (CLUTTER_LABEL (data->label), "Completed");
  clutter_timeline_stop (data->timeline);

  test_thread_data_free (data);

  return FALSE;
}

static GStaticPrivate test_thread_data = G_STATIC_PRIVATE_INIT;

typedef struct
{
  gint count;
  TestThreadData *thread_data;
} TestUpdate;

static gboolean
update_label_idle (gpointer data)
{
  TestUpdate *update = data;
  gchar *text;

  text = g_strdup_printf ("Count to %d", update->count);

  clutter_label_set_text (CLUTTER_LABEL (update->thread_data->label), text);

  g_free (text);
  g_free (update);
  return FALSE;
}

static void
do_something_very_slow (void)
{
  TestThreadData *data;
  gint i;

  data = (TestThreadData *) g_static_private_get (&test_thread_data);
  if (data->cancelled)
    return;

  for (i = 0; i < 100; i++)
    {
      /* sleep for a while */
      sleep(rand() / (RAND_MAX / 3) + 1);

      if ((i % 10) == 0)
        {
          TestUpdate *update;

          update = g_new (TestUpdate, 1);
          update->count = i;
          update->thread_data = data;
          clutter_threads_add_idle (update_label_idle, update);
        }
    }
}

static gpointer
test_thread_func (gpointer user_data)
{
  TestThreadData *data;

  data = user_data;
  g_static_private_set (&test_thread_data, data, NULL);

  do_something_very_slow ();

  clutter_threads_add_idle (test_thread_done_idle, data);

  return NULL;
}

static ClutterTimeline *timeline = NULL;
static ClutterActor *count_label = NULL;

static void
on_key_press_event (ClutterStage    *stage,
                    ClutterKeyEvent *event,
                    gpointer         user_data)
{
  TestThreadData *data;

  switch (clutter_key_event_symbol (event))
    {
    case CLUTTER_s:
      clutter_timeline_start (timeline);
      
      data = test_thread_data_new ();
      data->stage = g_object_ref (stage);
      data->label = g_object_ref (count_label);
      data->timeline = g_object_ref (timeline);
      g_thread_create (test_thread_func, data, FALSE, NULL);
      break;
    case CLUTTER_q:
      clutter_main_quit ();
      break;
    }
}

int
main (int argc, char *argv[])
{
  ClutterActor *stage;
  ClutterActor *label;
  ClutterActor *rect;
  ClutterColor stage_color = { 0xcc, 0xcc, 0xcc, 0xff };
  ClutterColor rect_color = { 0xee, 0x55, 0x55, 0x99 };
  ClutterBehaviour *behaviour;

  g_thread_init (NULL);
  clutter_threads_init ();
  clutter_threads_enter ();

  clutter_init (&argc, &argv);

  stage = clutter_stage_get_default ();
  clutter_stage_set_color (CLUTTER_STAGE (stage), &stage_color);
  clutter_actor_set_size (stage, 800, 600);
  
  count_label = clutter_label_new_with_text ("Mono 16", "Counter");
  clutter_actor_set_position (count_label, 350, 50);
  clutter_actor_show (count_label);

  label = clutter_label_new_with_text ("Mono 16", "Press 's' to start");
  clutter_actor_set_position (label, 50, 50);
  clutter_actor_show (label);

  rect = clutter_rectangle_new_with_color (&rect_color);
  clutter_actor_set_position (rect, 150, 150);
  clutter_actor_set_size (rect, 25, 25);
  clutter_actor_show (rect);

  clutter_container_add (CLUTTER_CONTAINER (stage),
                         count_label, label,
                         rect,
                         NULL);

  timeline = clutter_timeline_new (150, 50);
  clutter_timeline_set_loop (timeline, TRUE);
  behaviour = clutter_behaviour_rotate_new (clutter_alpha_new_full (timeline,
                                                                    CLUTTER_ALPHA_SINE,
                                                                    NULL, NULL),
                                            CLUTTER_Z_AXIS,
                                            CLUTTER_ROTATE_CW,
                                            0.0, 360.0);
  clutter_behaviour_apply (behaviour, rect);

  g_signal_connect (stage,
                    "button-press-event", G_CALLBACK (clutter_main_quit),
                    NULL);
  g_signal_connect (stage,
                    "key-press-event", G_CALLBACK (on_key_press_event),
                    NULL);

  clutter_actor_show (stage);

  clutter_main ();
  clutter_threads_leave ();

  g_object_unref (behaviour);
  g_object_unref (timeline);

  return EXIT_SUCCESS;
}
