package Charset::EBCDIC;

# Convert between EBCDIC and ASCII

# This file is part of CLC-INTERCAL.

# Copyright (C) 1999 Claudio Calvelli <lunatic@assurdo.com>, all rights reserved

# WARNING - do not operate heavy machinery while using CLC-INTERCAL

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

use vars qw($VERSION);

$VERSION = '0.05';

sub import {
    my $class = shift;
    my ($callpack, $callfile, $calline) = caller;
    my @EXPORT;
    if (@_) {
    	@EXPORT = @_;
    } else {
    	@EXPORT = qw(ascii2ebcdic ebcdic2ascii);
    }
    my $sym;
    foreach $sym (@EXPORT) {
	if ($sym eq 'ascii2ebcdic' ||
	    $sym eq 'ebcdic2ascii' ||
	    $sym eq 'ebcdic2cards')
	{
	    *{"$callpack\::$sym"} = \&{"$class\::$sym"};
	} else {
	    die "015 Cannot export $sym at $callfile line $calline\n";
	}
    }
}

sub ebcdic2ascii {
    @_ == 1 or do {
    	my ($callpack, $callfile, $calline) = caller;
	die "013 Usage: ebcdic2ascii(STRING) at $callfile line $calline\n";
    };
    my $string = shift;
    $string =~ tr[\000-\037\100\112\113\114\115\116\117\120\132\133\134\135\136\137\140\141\145\152\153\154\155\156\157\172\173\174\175\176\177\201-\211\221-\231\234\236\241\242-\251\260\301-\311\321-\331\334\336\342-\351\360-\371\377]
    		 [\000-\037\040\242\056\074\050\053\041\046\135\044\052\051\073\254\055\057\245\174\054\045\137\076\077\072\043\100\047\075\042\141-\151\152-\162\173\133\176\163-\172\136\101-\111\112-\122\175\134\123-\132\060-\071\177]
	== length($string)
	or do {
	    my ($callpack, $callfile, $calline) = caller;
	    # die "109 Illegal character in string at $callfile line $calline\n";
	};
    $string;
}

sub ascii2ebcdic {
    @_ == 1 or do {
    	my ($callpack, $callfile, $calline) = caller;
	die "013 Usage: ascii2ebcdic(STRING) at $callfile line $calline\n";
    };
    my $string = shift;
    $string =~ tr[\000-\037\040\242\056\074\050\053\041\046\135\044\052\051\073\254\055\057\245\174\054\045\137\076\077\072\043\100\047\075\042\141-\151\152-\162\173\133\176\163-\172\136\101-\111\112-\122\175\134\123-\132\060-\071\177]
    		 [\000-\037\100\112\113\114\115\116\117\120\132\133\134\135\136\137\140\141\145\152\153\154\155\156\157\172\173\174\175\176\177\201-\211\221-\231\234\236\241\242-\251\260\301-\311\321-\331\334\336\342-\351\360-\371\377]
	== length($string)
	or do {
	    my ($callpack, $callfile, $calline) = caller;
	    # die "109 Illegal character in string at $callfile line $calline\n";
	};
    $string;
}

sub ebcdic2cards {
    @_ == 1 or do {
    	my ($callpack, $callfile, $calline) = caller;
	die "013 Usage: ascii2ebcdic(STRING) at $callfile line $calline\n";
    };
    local $" = '';
    # The following code is too painfully self-evident. If you have a
    # less readable version, feel free to replace it.
    "@{[ map s/^(.{80}).*$/$1/ && $_ || $_ . '@' x (80 - length),
	     split /\r?\n/, $_[0] ]}";
}

1;

__END__

=head1 NAME

Charset::EBCDIC - allows to use EBCDIC string constants in ASCII programs (and v.v.)

=head1 SYNOPSIS

    use Charset::EBCDIC;

    my $a = ebcdic2ascii "(EBCDIC text)";

=head1 DESCRIPTION

I<Charset::EBCDIC> defines functions to convert between a subset of ASCII and a
subset of nonstandard EBCDIC (since there isn't such a thing as a standard
EBCDIC we defined our own variant which is guaranteed to be incompatible
with all versions of EBCDIC used by IBM hardware - however, when we have
chosen a code for a character, we have made sure that at least one - but
certainly not all - IBM models used that same code, so the choice cannot
be criticised). If you really want to know, several variants of EBCDIC
are listed in RFC 1345, which is available from the usual sources.

By default, both functions I<ebcdic2ascii> and I<ascii2ebcdic> are
imported in your namespace. If you don't want that, you know how to
avoid it. They do the obvious thing to their first argument and
return the transformed string.

There is another function I<ebcdic2cards>, not exported by default.
It takes any string of EBCDIC characters, containing newlines (or
maybe carriage return-newline pairs), and returns a string of punched
cards (that is, each line is padded to 80 characters with spaces, and
the newlines are removed; everything from column 81 is lost).

Well, that's all.

=head1 EBCDIC CHARACTER TABLE

The following are the characters recognised. The ones shown as 2 letter
abbreviations cannot be translated to ASCII (except for the control
characters, which do have an ASCII equivalent).

     +   0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f   Notes
    00                          OV TA LF       CR         OV=overstrike
    10                                                    TA=tab
    20                                                    LF=linefeed
    30                                                    CR=carr-return
    40  SP                            CT  .  <  (  +  !   SP=space
    50   &                             ]  $  *  )  ; NO   CT=cents
    60   -  /          XO              |  ,  %  _  >  ?   NO=not-sign
    70                                 :  #  @  '  =  "   XO=XOR(1)
    80      a  b  c  d  e  f  g  h  i                  
    90      j  k  l  m  n  o  p  q  r        {     [   
    a0      ~  s  t  u  v  w  x  y  z                RE   RE=registered
    b0   ^ PO       CO                                    PO=pound
    c0      A  B  C  D  E  F  G  H  I                     CO=copyright
    d0      J  K  L  M  N  O  P  Q  R        }     \   
    e0         S  T  U  V  W  X  Y  Z                  
    f0   0  1  2  3  4  5  6  7  8  9                DE   DE=delete

(1) The symbol for the INTECAL XOR operator, "V overstrike -".

=head1 NOTES

This module reimplements some of the functionality of two other modules
(see L<Exporter>, L<Carp>, I<Reinventing the Wheel>). This is intentional,
as it will leave larger scope for obfuscation in a future release.

=head1 COPYRIGHT

This module is part of CLC-INTERCAL.

Copyright (c) 1999 by Claudio Calvelli E<lt>C<lunatic@assurdo.com>E<gt>,
all (f)rights reserved.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

=head1 SEE ALSO

A qualified psychiatrist.

