#!/usr/bin/env python

########################################################################
# Chris's Lame Filebrowser 4 (File Information Gathering)
# Copyright 2004, Gabe Ginorio <gabe@zevallos.com.br>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
######################################################################


import os, os.path, string, stat, commands

def getStats(location):
	'''This functions get the file information (size, permissions ...) for a file passed to it'''

	# If this is a non-existant file or dead link, then let's return some bugus values
	if not os.path.exists(location):		
		return "0 bytes", 'Dead Link', "Unknown"	
		
	# Get a list of users on the system
	users = __getUsers__()
	
	permissions=""
	
	# Get the file information
	try:
		fileInfo = os.lstat(location)
	
		# Get the UID and then use the username dictionary to get the user name
		UID = fileInfo[stat.ST_UID]
		owner = users[str(UID)]
	
		# Check to make sure the user is valid
		if owner == None or owner == "":
			owner = "unknown"

		# Get the file size
		if fileInfo[stat.ST_SIZE] > 1024:
			file_size = str(fileInfo[stat.ST_SIZE]/1024) + " Kbs"
		else:
			file_size = str(fileInfo[stat.ST_SIZE]) + " bytes"

		# Get the permissions in octal form (UNIX numeric [777,755,530] form)
		mode = oct(fileInfo[stat.ST_MODE])

		# This dictionary contains the permission types
		permissionTypes = {7:'rwx', 6:'rw-', 5:'r-x', 4:'r--', 3:'-wx', 2:'-w-', 1:'--x', 0:'---'}

		# Determine the permissions
		for number in mode[-3:]:
			permissions += permissionTypes[int(number)]
	except:
		# If we can't get the file information from stat, we'll get it the old fashioned way
		owner = "Unknown"
		file_size = os.path.getsize(location)
				
		# Get the file size
		if file_size > 1024:
			file_size = str(file_size/1024) + " Kbs"
		else:
			file_size = str(file_size) + " bytes"	
	
		# This is the old way of getting permissions
		if os.access(location, os.R_OK):
			permissions += "r"
		else:
			permissions += "-"
		if os.access(location, os.W_OK):
			permissions += "w"
		else:
			permissions += "-"
		if os.access(location, os.X_OK):
			permissions += "x"
		else:
			permissions += "-"

	# Directories need total size, not inode size
	if os.path.isdir(location):
		file_size = commands.getoutput("du -cxsh " + __returnClean__(location))
		file_size = string.split(file_size)[-2]

		file_number = str(len(os.listdir(location)))

		# Return whatever we get
		return file_size, file_number, permissions, owner
	else:
		# Return whatever we get
		return file_size, permissions, owner

#########################################
# This function returns the system users from /etc/passwd
#########################################		
def __getUsers__(): 

	infile = file('/etc/passwd', 'r').readlines()

	users = {}
	for line in infile:
		line = string.split(line,':')
		users[line[2]] = line[0]

	return users

#########################################
# This function returns BASH friendly text
#########################################	
def __returnClean__(text):
		
	# Here are the bad characters
	badChars = [" ","\'","&","(",")"]
			
	# Here are the replacement characters
	goodChars=["\ ","\\'","\&","\(","\)"]
			
	# Replace the bad with the good
	for x in range(len(badChars)):
		text = string.replace(text, badChars[x], goodChars[x])		
		
	# Return the "cleaned", BASH friendly text
	return text
		
