;; CMUCL GLUT interface
;; Knut Arild Erstad 2001

;; Lisp and C glue for GLUT callbacks.

;; The idea is something like this:
;; A Lisp function CALL-GLUT-CALLBACK is called from the C callbacks.
;; Parameters (arguments) are converted to fixnums and placed in the Lisp vector
;; *GLUT-PARAMETERS* instead of being passed directly using funcall*() or
;; call_into_lisp(), which turned out to not work portably for 4+ arguments.
;; (The new solution hopefully works.)
;; The C addresses of the lisp function and vector are updated after GC.
;;
;; There is lots of messy Lisp and C glue here, good luck figuring it out.  :-)

(in-package :gl)

(eval-when (:compile-toplevel :load-toplevel)
  (defparameter *output-c* nil
    "Boolean; true if C code should be output by the GLUT-CALLBACK macro.
This only has to be done once, and since the resulting C file is distributed
along with this file, the value should be false (NIL).")

  (defparameter *c-filename* "glut-callbacks-c.c"))

(defparameter *glut-parameters*
  (make-array 10 :element-type 'fixnum :initial-element 0)
  "Vector for passing (fixnum) parameters from C to Lisp.
The first value is the number of parameters.")

(defparameter *glut-functions*
  (make-array 100)
  "Vector of glut functions.")

(defparameter *glut-function-counter* 0
  "Counter for glut-callback callbacks.")

#+cmu
(defun update-c-variables ()
  "Update the C variables pointing to lisp objects."
  (alien:alien-funcall (alien:extern-alien
			"set_lisp_variables"
			(function c-call:void c-call:unsigned-int
				  c-call:unsigned-int))
		       (system:sap-int (system:vector-sap *glut-parameters*))
		       (kernel:get-lisp-obj-address #'call-glut-callback)))

#+sbcl
(defun update-c-variables ()
  "Update the C variables pointing to lisp objects."
  (sb-alien:alien-funcall (sb-alien:extern-alien
			"set_lisp_variables"
			(function sb-alien:void sb-alien:unsigned-int
				  sb-alien:unsigned-int))
		       (sb-sys:sap-int (sb-sys:vector-sap *glut-parameters*))
		       (sb-kernel:get-lisp-obj-address #'call-glut-callback)))

(defun call-glut-callback (function-number)
  "Call a Lisp GLUT callback by getting the function from *GLUT-FUNCTIONS*
and the parameters from *GLUT-PARAMETERS*.  Meant to be called from C."
  ;;(format t "CALL-GLUT-CALLBACK ~A: ~A~%" function-number *glut-parameters*)
  (system:without-gcing
   (let ((func (svref *glut-functions* function-number)))
     (when (null func)
       (error "GLUT function ~A not defined?" function-number))
     (let* ((n (aref *glut-parameters* 0))
	    (params (loop for i from 1 to n collect
			  (aref *glut-parameters* i))))
       (apply func params)))))

(eval-when (:compile-toplevel :load-toplevel)
  (defun c-typename (type)
    (ecase type
      (int "int")
      (unsigned-char "unsigned char")))

  (defun c-arglist (argtypes)
    (if (null argtypes)
	"void"
	(loop for i below (length argtypes)
	      for x in argtypes
	      with str = ""
	      do (setq str (format nil "~A, ~A arg~A"
				   str (c-typename x) i))
	      finally (return (subseq str 2)))))
  
  (defun c-setargs (n)
    (loop for i below n
	  with str = (format nil "glut_parameters[0] = make_fixnum(~A);" n)
	  do (setq str (format nil "~A
  glut_parameters[~A] = make_fixnum(arg~A);"
			       str (1+ i) i))
	  finally (return str)))
  )

(defmacro glut-callback (name argtypes c-name)
  (let ((function-number (1- (incf *glut-function-counter*)))
	(c-set-callback-function (gensym)))
    ;; C code
    (when *output-c*
      (with-open-file (file *c-filename*
			    :direction :output :if-exists :append)
	(format file "
/*** ~A ***/
void ~A_callback (~A) {
  if ((glut_parameters == 0) || (call_glut_callback == 0)) {
    printf(\"~A_callback: \"
           \"Lisp parameters or callback not initialized?\\n\"
           \"%d %d\\n\",
           (int)glut_parameters, (int)call_glut_callback);
    fflush(stdout);
    return;
  }
  ~A
  funcall1(call_glut_callback, make_fixnum(~A));
}
int ~A_check = 0;
void set_~A_callback(int set) {
  if (set==0 && ~A_check==0)
    return;
  if (set==0)
    ~A(0);
  else {
    ~A_check = 1;
    ~A(~A_callback);
  }
}"
		c-name c-name
		;; argument list
		(c-arglist argtypes)
		c-name
		;; set glut arguments
		(c-setargs (length argtypes))
		function-number
		c-name c-name c-name c-name c-name c-name c-name)))
    ;; Lisp code
    #+cmu
    `(eval-when (:compile-toplevel :load-toplevel)
      (alien:def-alien-variable
	  (,(format nil "set_~A_callback" c-name) ,c-set-callback-function)
	  (function c-call:void c-call:int))
      (defun ,name (lisp-function)
	(setf (svref *glut-functions* ,function-number) lisp-function)
	;;(format t "calling set_~A_callback~%" ,c-name)
	(alien:alien-funcall ,c-set-callback-function
			     (if (null lisp-function) 0 1))
	;;(format t "finished calling set_~A_callback~%" ,c-name)
	)
      (export ',name))
    #+sbcl
    `(eval-when (:compile-toplevel :load-toplevel)
      (sb-alien:define-alien-variable
	  (,(format nil "set_~A_callback" c-name) ,c-set-callback-function)
	  (function sb-alien:void sb-alien:int))
      (defun ,name (lisp-function)
	(setf (svref *glut-functions* ,function-number) lisp-function)
	;;(format t "calling set_~A_callback~%" ,c-name)
	(sb-alien:alien-funcall ,c-set-callback-function
			     (if (null lisp-function) 0 1))
	;;(format t "finished calling set_~A_callback~%" ,c-name)
	)
      (export ',name))))

;; init values before compiling/loading
(eval-when (:compile-toplevel :load-toplevel)
  (setq *glut-function-counter* 0))

(eval-when (:load-toplevel)
  (fill *glut-functions* nil)
  (update-c-variables)
  (pushnew #'update-c-variables ext:*after-gc-hooks*)
  )

;; Maybe generate C "header"
(eval-when (:compile-toplevel)
  (when *output-c*
    (with-open-file (file *c-filename*
			  :direction :output :if-exists :supersede)
      (format file "
/* CMUCL GLUT callback glue: generated from glut-callbacks.lisp */
typedef unsigned int lispobj; /* safe assumtion? */
#define make_fixnum(n) ((lispobj)(((int)n)<<2))
extern lispobj funcall1(lispobj function, lispobj arg);

/* lisp variables */
lispobj *glut_parameters = 0;
lispobj call_glut_callback = 0;

#include <stdio.h>

void set_lisp_variables(lispobj params, lispobj func) {
  glut_parameters = (lispobj *)params;
  call_glut_callback = func;
}
/* debug printing */
void print_vars (void) {
  printf(\"Lisp vars: %d %d\\n\", (int)glut_parameters,
         (int)call_glut_callback);
  fflush(stdout);
}

#include <GL/glut.h>
/* GLUT/lisp callback code follows */"))))

;; Define callbacks, and maybe generate C code
(glut-callback glutdisplayfunc () "glutDisplayFunc")
(glut-callback glutreshapefunc (int int) "glutReshapeFunc")
(glut-callback glutkeyboardfunc (unsigned-char int int) "glutKeyboardFunc")
(glut-callback glutmousefunc (int int int int) "glutMouseFunc")
(glut-callback glutmotionfunc (int int) "glutMotionFunc")
(glut-callback glutpassivemotionfunc (int int) "glutPassiveMotionFunc")
(glut-callback glutentryfunc (int) "glutEntryFunc")
(glut-callback glutvisibilityfunc (int) "glutVisibilityFunc")
(glut-callback glutidlefunc () "glutIdleFunc")
;;(glut-callback gluttimerfunc (unsigned-int (function int) int) "glutTimerFunc")
(glut-callback glutmenustatefunc (int) "glutMenuStateFunc")
(glut-callback glutspecialfunc (int int int) "glutSpecialFunc")
(glut-callback glutspaceballmotionfunc (int int int) "glutSpaceballMotionFunc")
(glut-callback glutspaceballrotatefunc (int int int) "glutSpaceballRotateFunc")
(glut-callback glutspaceballbuttonfunc (int int) "glutSpaceballButtonFunc")
(glut-callback glutbuttonboxfunc (int int) "glutButtonBoxFunc")
(glut-callback glutdialsfunc (int int) "glutDialsFunc")
(glut-callback gluttabletmotionfunc (int int) "glutTabletMotionFunc")
(glut-callback gluttabletbuttonfunc (int int int int) "glutTabletButtonFunc")
(glut-callback glutmenustatusfunc (int int int) "glutMenuStatusFunc")
(glut-callback glutoverlaydisplayfunc () "glutOverlayDisplayFunc")
(glut-callback glutwindowstatusfunc (int) "glutWindowStatusFunc")
