;;;; -*- mode: lisp; indent-tabs-mode: nil -*-
#+sbcl
(in-package :sb-vm)

#+(and sbcl (or x86 x86-64))
(progn
(define-vop (fill-block-ub8)
  (:policy :fast-safe)
  (:args (block :scs (descriptor-reg))
         (buffer :scs (descriptor-reg))
         (offset :scs (unsigned-reg immediate) :target buffer-index))
  (:variant-vars big-endian-p bytes-to-copy 64-bit-p)
  (:temporary (:sc unsigned-reg) temp buffer-index block-index)
  (:generator 50
    (let* ((data-offset (- (* n-word-bytes vector-data-offset)
                             other-pointer-lowtag))
           (block-disp (+ data-offset bytes-to-copy))
           (ea-size #+x86 :dword #+x86-64 :qword)
           (immediate-offset (sc-is offset immediate))
           (unroll (if immediate-offset 2 1))
           (loop (gen-label)))
      (flet ((ea-for-buffer (&optional (offset 0))
               (if immediate-offset
                   (make-ea ea-size :base buffer
                            :index block-index
                            :scale n-word-bytes
                            :disp (+ block-disp offset))
                   (make-ea ea-size :base buffer
                            :index buffer-index :disp data-offset)))
             (ea-for-block (&optional (offset 0))
               (make-ea ea-size :base block
                        :index block-index
                        :scale n-word-bytes
                        :disp (+ block-disp offset)))
             (handle-endianness (x)
               (when big-endian-p
                 (inst bswap x)
                 #+x86-64
                 (unless 64-bit-p
                   (inst rol x 32)))))
        (unless immediate-offset
          (move buffer-index offset))
        (inst mov block-index (- (truncate bytes-to-copy n-word-bytes)))
        (emit-label loop)
        (inst mov temp (ea-for-buffer 0))
        (when immediate-offset
          (inst mov buffer-index (ea-for-buffer n-word-bytes)))
        (handle-endianness temp)
        (when immediate-offset
          (handle-endianness buffer-index))
        (inst mov (ea-for-block) temp)
        (when immediate-offset
          (inst mov (ea-for-block n-word-bytes) buffer-index))
        (unless immediate-offset
          (inst add buffer-index n-word-bytes))
        (inst add block-index unroll)
        (inst jmp :nz loop)))))

(define-vop (fill-block-ub8-le fill-block-ub8)
  (:translate ironclad::fill-block-ub8-le)
  (:arg-types simple-array-unsigned-byte-32
              simple-array-unsigned-byte-8
              positive-fixnum)
  (:variant nil 64 nil))

(define-vop (fill-block-ub8-be fill-block-ub8)
  (:translate ironclad::fill-block-ub8-be)
  (:arg-types simple-array-unsigned-byte-32
              simple-array-unsigned-byte-8
              positive-fixnum)
  (:variant t 64 nil))

#+x86-64
(define-vop (fill-block-ub8-le/64 fill-block-ub8)
  (:translate ironclad::fill-block-ub8-le/64)
  (:arg-types simple-array-unsigned-byte-64
              simple-array-unsigned-byte-8
              positive-fixnum)
  ;; Yes.  Really.  Inconsistent naming FTW.
  (:variant nil 64 t))

#+x86-64
(define-vop (fill-block-ub8-be/64 fill-block-ub8)
  (:translate ironclad::fill-block-ub8-be/64)
  (:arg-types simple-array-unsigned-byte-64
              simple-array-unsigned-byte-8
              positive-fixnum)
  (:variant t 128 t))

(define-vop (expand-block)
  (:translate ironclad::expand-block)
  (:policy :fast-safe)
  (:args (block :scs (descriptor-reg)))
  (:arg-types simple-array-unsigned-byte-32)
  (:temporary (:sc unsigned-reg) temp count)
  (:generator 100
    (flet ((block-word (elem-offset)
             (make-ea :dword :base block
                      :index count
                      :scale 4
                      :disp (+ (- (* n-word-bytes vector-data-offset)
                                  other-pointer-lowtag)
                               (* 4 elem-offset)))))
      (let ((loop (gen-label))
            #+x86-64 (temp (reg-in-size temp :dword)))
        (inst mov count 16)
        (emit-label loop)
        (inst mov temp (block-word -3))
        (inst xor temp (block-word -8))
        (inst xor temp (block-word -14))
        (inst xor temp (block-word -16))
        (inst rol temp 1)
        (inst mov (block-word 0) temp)
        (inst add count 1)
        (inst cmp count 79)
        (inst jmp :le loop)))))

(define-vop (sha256-expand-block)
  (:translate ironclad::sha256-expand-block)
  (:policy :fast-safe)
  (:args (block :scs (descriptor-reg)))
  (:arg-types simple-array-unsigned-byte-32)
  (:temporary (:sc unsigned-reg) t1 t2 t3 t4 count)
  (:generator 100
    (flet ((block-word (elem-offset)
             (make-ea :dword :base block
                      :index count
                      :scale 4
                      :disp (+ (- (* n-word-bytes vector-data-offset)
                                  other-pointer-lowtag)
                               (* 4 elem-offset)))))
      (let ((loop (gen-label))
            #+x86-64 (t1 (reg-in-size t1 :dword))
            #+x86-64 (t2 (reg-in-size t2 :dword))
            #+x86-64 (t3 (reg-in-size t3 :dword))
            #+x86-64 (t4 (reg-in-size t4 :dword)))
        ;; The code could definitely be improved for x86-64 by using
        ;; more temporaries, but this version works on both 32- and
        ;; 64-bit and eliminates many of the stupidities in the modular
        ;; arithmetic version (mostly on 64-bit, but some lameness in
        ;; the 32-bit version as well).
        (inst mov count 16)
        (emit-label loop)
        (inst mov t1 (block-word -2))
        ;; sigma1
        (inst mov t2 t1)
        (inst rol t2 15)
        (inst mov t3 t1)
        (inst rol t3 13)
        (inst xor t2 t3)
        (inst shr t1 10)
        (inst xor t1 t2)
        (inst mov t2 (block-word -15))
        ;; sigma0
        (inst mov t3 t2)
        (inst rol t3 25)
        (inst mov t4 t2)
        (inst rol t4 14)
        (inst xor t3 t4)
        (inst shr t2 3)
        (inst xor t2 t3)
        (inst add t1 (block-word -7))
        (inst add t2 (block-word -16))
        (inst add t1 t2)
        (inst mov (block-word 0) t1)
        (inst add count 1)
        (inst cmp count 63)
        (inst jmp :le loop)))))

;;; Implementing this for x86 would require nasty hacks with
;;; pseudo-atomic.  Might just be worth it for the speed increase,
;;; though.  The code is also probably not scheduled optimally.
#+x86-64
(define-vop (update-sha1-block)
  (:translate ironclad::%update-sha1-block)
  (:policy :fast-safe)
  (:args (regs :scs (descriptor-reg) :target result)
         (block :scs (descriptor-reg)))
  (:arg-types simple-array-unsigned-byte-32 simple-array-unsigned-byte-32)
  (:results (result :scs (descriptor-reg)))
  (:result-types simple-array-unsigned-byte-32)
  (:temporary (:sc unsigned-reg) a b c d e t1 t2)
  (:generator 1000
    (let ((a (reg-in-size a :dword))
          (b (reg-in-size b :dword))
          (c (reg-in-size c :dword))
          (d (reg-in-size d :dword))
          (e (reg-in-size e :dword))
          (t1 (reg-in-size t1 :dword))
          (t2 (reg-in-size t2 :dword))
          (k1 #x5a827999)
          (k2 #x6ed9eba1)
          (k3 #x-70e44324)
          (k4 #x-359d3e2a))
      (labels ((block/reg-ea (base index)
                 (make-ea :dword
                          :base base
                          :disp (+ (- (* n-word-bytes vector-data-offset)
                                      other-pointer-lowtag)
                                   (* 4 index))))
               (f1 (a b c d e n)
                 (inst mov t2 a)
                 (inst mov t1 c)
                 (inst rol t2 5)
                 (inst xor t1 d)
                 (inst add t2 (block/reg-ea block n))
                 (inst and t1 b)
                 (inst xor t1 d)
                 (inst lea e (make-ea :dword :base t1 :index e :disp k1))
                 (inst rol b 30)
                 (inst add e t2))
               (f2/4 (a b c d e n k)
                 (inst mov t2 a)
                 (inst mov t1 d)
                 (inst rol t2 5)
                 (inst xor t1 c)
                 (inst add t2 (block/reg-ea block n))
                 (inst xor t1 b)
                 (inst lea e (make-ea :dword :base t1 :index e :disp k))
                 (inst rol b 30)
                 (inst add e t2))
               (f2 (a b c d e n)
                 (f2/4 a b c d e n k2))
               (f4 (a b c d e n)
                 (f2/4 a b c d e n k4))
               (f3 (a b c d e n)
                 (inst mov t2 c)
                 (inst mov t1 c)
                 (inst and t2 b)
                 (inst or t1 b)
                 (inst and t1 d)
                 (inst or t1 t2)
                 (inst mov t2 a)
                 (inst rol t2 5)
                 (inst add t2 (block/reg-ea block n))
                 (inst rol b 30)
                 (inst lea e (make-ea :dword :base t1 :index e :disp k3))
                 (inst add e t2))
               (sha1-rounds (start end f)
                 (let ((xvars (ironclad::make-circular-list a b c d e)))
                   (loop for i from start upto end
                         for vars on xvars by #'cddddr
                         do (multiple-value-bind (a b c d e)
                                (apply #'values (ironclad::circular-list-subseq vars 0 5))
                              (funcall f a b c d e i))))))
        (inst mov a (block/reg-ea regs 0))
        (inst mov b (block/reg-ea regs 1))
        (inst mov c (block/reg-ea regs 2))
        (inst mov d (block/reg-ea regs 3))
        (inst mov e (block/reg-ea regs 4))
        (sha1-rounds 0 19 #'f1)
        (sha1-rounds 20 39 #'f2)
        (sha1-rounds 40 59 #'f3)
        (sha1-rounds 60 79 #'f4)
        (inst add (block/reg-ea regs 0) a)
        (inst add (block/reg-ea regs 1) b)
        (inst add (block/reg-ea regs 2) c)
        (inst add (block/reg-ea regs 3) d)
        (inst add (block/reg-ea regs 4) e)
        (move result regs)))))

#+x86-64
(define-vop (salsa-core-fast)
  (:translate ironclad::x-salsa-core)
  (:policy :fast-safe)
  (:args (buffer :scs (descriptor-reg))
         (state :scs (descriptor-reg)))
  (:info n-rounds)
  (:arg-types (:constant (signed-byte 61)) simple-array-unsigned-byte-8
              simple-array-unsigned-byte-32)
  (:temporary (:sc double-reg) x0 x1 x2 x3)
  (:temporary (:sc unsigned-reg) r0 r1 r2 r3 temp count)
  (:generator 1000
    (labels ((nth-xmm-mem (base i)
               (make-ea :qword :base base
                               :disp (+ (- (* n-word-bytes vector-data-offset)
                                           other-pointer-lowtag)
                                        (* 16 i))))
             (ea (i)
               (make-ea :dword :base buffer
                        :disp (+ (- (* n-word-bytes vector-data-offset)
                                    other-pointer-lowtag)
                                 (* 4 i))))
             (quarter-round (y0 y1 y2 y3)
               (let ((r0 (reg-in-size r0 :dword))
                     (r1 (reg-in-size r1 :dword))
                     (r2 (reg-in-size r2 :dword))
                     (r3 (reg-in-size r3 :dword))
                     (temp (reg-in-size temp :dword)))
                 ;; x[y0] = XOR(x[y0],ROTATE(PLUS(x[y3],x[y2]), 7));
                 ;; x[y1] = XOR(x[y1],ROTATE(PLUS(x[y0],x[y3]), 9));
                 ;; x[y2] = XOR(x[y2],ROTATE(PLUS(x[y1],x[y0]),13));
                 ;; x[y3] = XOR(x[y3],ROTATE(PLUS(x[y2],x[y1]),18));
                 (inst mov r2 (ea y2))
                 (inst mov r3 (ea y3))

                 (inst lea r0 (make-ea :dword :base r3 :index r2))
                 (inst rol r0 7)
                 (inst xor r0 (ea y0))

                 (inst lea r1 (make-ea :dword :base r0 :index r3))
                 (inst rol r1 9)
                 (inst xor r1 (ea y1))

                 (inst lea temp (make-ea :dword :base r1 :index r0))
                 (inst rol temp 13)
                 (inst xor r2 temp)

                 (inst lea temp (make-ea :dword :base r2 :index r1))
                 (inst rol temp 18)
                 (inst xor r3 temp)

                 (inst mov (ea y0) r0)
                 (inst mov (ea y1) r1)
                 (inst mov (ea y2) r2)
                 (inst mov (ea y3) r3))))
      ;; copy state to the output buffer
      (inst movdqa x0 (nth-xmm-mem state 0))
      (inst movdqa x1 (nth-xmm-mem state 1))
      (inst movdqa x2 (nth-xmm-mem state 2))
      (inst movdqa x3 (nth-xmm-mem state 3))
      (inst movdqa (nth-xmm-mem buffer 0) x0)
      (inst movdqa (nth-xmm-mem buffer 1) x1)
      (inst movdqa (nth-xmm-mem buffer 2) x2)
      (inst movdqa (nth-xmm-mem buffer 3) x3)

      (let ((repeat (gen-label)))
        (inst mov count n-rounds)
        (emit-label repeat)
        (quarter-round 4 8 12 0)
        (quarter-round 9 13 1 5)
        (quarter-round 14 2 6 10)
        (quarter-round 3 7 11 15)

        (quarter-round 1 2 3 0)
        (quarter-round 6 7 4 5)
        (quarter-round 11 8 9 10)
        (quarter-round 12 13 14 15)
        (inst sub count 1)
        (inst jmp :nz repeat))

      (inst paddd x0 (nth-xmm-mem buffer 0))
      (inst paddd x1 (nth-xmm-mem buffer 1))
      (inst paddd x2 (nth-xmm-mem buffer 2))
      (inst paddd x3 (nth-xmm-mem buffer 3))
      (inst movdqa (nth-xmm-mem buffer 0) x0)
      (inst movdqa (nth-xmm-mem buffer 1) x1)
      (inst movdqa (nth-xmm-mem buffer 2) x2)
      (inst movdqa (nth-xmm-mem buffer 3) x3))))

#+x86-64
(define-vop (chacha-core-fast)
  (:translate ironclad::x-chacha-core)
  (:policy :fast-safe)
  (:args (buffer :scs (descriptor-reg))
         (state :scs (descriptor-reg)))
  (:info n-rounds)
  (:arg-types (:constant (signed-byte 61)) simple-array-unsigned-byte-8
              simple-array-unsigned-byte-32)
  (:temporary (:sc double-reg) x0 x1 x2 x3 y0 y1 y2 y3 t0)
  (:temporary (:sc unsigned-reg) count)
  (:generator 1000
    (flet ((nth-xmm-mem (base i)
             (make-ea :qword :base base
                             :disp (+ (- (* n-word-bytes vector-data-offset)
                                         other-pointer-lowtag)
                                      (* 16 i)))))
      (macrolet ((chacha-round ()
                   `(progn
                      (inst paddd y0 y1)
                      (inst pxor y3 y0)
                      (inst movdqa t0 y3)
                      (inst pslld-imm y3 16)
                      (inst psrld-imm t0 16)
                      (inst por y3 t0)

                      (inst paddd y2 y3)
                      (inst pxor y1 y2)
                      (inst movdqa t0 y1)
                      (inst pslld-imm y1 12)
                      (inst psrld-imm t0 20)
                      (inst por y1 t0)

                      (inst paddd y0 y1)
                      (inst pxor y3 y0)
                      (inst movdqa t0 y3)
                      (inst pslld-imm y3 8)
                      (inst psrld-imm t0 24)
                      (inst por y3 t0)

                      (inst paddd y2 y3)
                      (inst pxor y1 y2)
                      (inst movdqa t0 y1)
                      (inst pslld-imm y1 7)
                      (inst psrld-imm t0 25)
                      (inst por y1 t0))))
        (inst movdqa x0 (nth-xmm-mem state 0))
        (inst movdqa x1 (nth-xmm-mem state 1))
        (inst movdqa x2 (nth-xmm-mem state 2))
        (inst movdqa x3 (nth-xmm-mem state 3))
        (inst movdqa y0 x0)
        (inst movdqa y1 x1)
        (inst movdqa y2 x2)
        (inst movdqa y3 x3)

        (let ((repeat (gen-label)))
          (inst mov count n-rounds)
          (emit-label repeat)

          (chacha-round)
          (inst pshufd y1 y1 #b00111001)
          (inst pshufd y2 y2 #b01001110)
          (inst pshufd y3 y3 #b10010011)

          (chacha-round)
          (inst pshufd y1 y1 #b10010011)
          (inst pshufd y2 y2 #b01001110)
          (inst pshufd y3 y3 #b00111001)

          (inst sub count 1)
          (inst jmp :nz repeat))

        (inst paddd x0 y0)
        (inst paddd x1 y1)
        (inst paddd x2 y2)
        (inst paddd x3 y3)
        (inst movdqa (nth-xmm-mem buffer 0) x0)
        (inst movdqa (nth-xmm-mem buffer 1) x1)
        (inst movdqa (nth-xmm-mem buffer 2) x2)
        (inst movdqa (nth-xmm-mem buffer 3) x3)))))

#+x86-64
(define-vop (aes-ni-support-p)
  (:translate ironclad::aes-ni-support-p)
  (:policy :fast-safe)
  (:conditional :c)
  (:temporary (:sc unsigned-reg :offset rax-offset) eax)
  (:temporary (:sc unsigned-reg :offset rcx-offset) ecx)
  (:generator 10
    (inst mov eax 1)
    (inst cpuid)
    (inst bt ecx 25)))

#+x86-64
(define-vop (aes-ni-generate-round-keys)
  (:translate ironclad::aes-ni-generate-round-keys)
  (:policy :fast-safe)
  (:args (key :scs (descriptor-reg))
         (key-length :scs (unsigned-reg))
         (encryption-keys :scs (descriptor-reg))
         (decryption-keys :scs (descriptor-reg)))
  (:arg-types simple-array-unsigned-byte-8
              unsigned-num
              simple-array-unsigned-byte-32
              simple-array-unsigned-byte-32)
  (:temporary (:sc double-reg) x0 x1 x2 x3 x4 x5 x6)
  (:generator 1000
    (flet ((ea (base i)
             (make-ea :qword
                      :base base
                      :disp (+ (- (* n-word-bytes vector-data-offset)
                                  other-pointer-lowtag)
                               (* 16 i)))))
      (macrolet ((expand-key-128 ()
                   `(progn
                      (inst pshufd x1 x1 #b11111111)
                      (inst shufps x2 x0 #b00010000)
                      (inst pxor x0 x2)
                      (inst shufps x2 x0 #b10001100)
                      (inst pxor x0 x2)
                      (inst pxor x0 x1)))
                 (expand-key-192a ()
                   `(progn
                      (inst pshufd x1 x1 #b01010101)
                      (inst shufps x2 x0 #b00010000)
                      (inst pxor x0 x2)
                      (inst shufps x2 x0 #b10001100)
                      (inst pxor x0 x2)
                      (inst pxor x0 x1)
                      (inst movdqa x4 x3)
                      (inst movdqa x5 x3)
                      (inst pslldq x4 4)
                      (inst pshufd x6 x0 #b11111111)
                      (inst pxor x3 x6)
                      (inst pxor x3 x4)
                      (inst movdqa x1 x0)
                      (inst shufps x5 x0 #b01000100)
                      (inst shufps x1 x3 #b01001110)))
                 (expand-key-192b ()
                   `(progn
                      (inst pshufd x1 x1 #b01010101)
                      (inst shufps x2 x0 #b00010000)
                      (inst pxor x0 x2)
                      (inst shufps x2 x0 #b10001100)
                      (inst pxor x0 x2)
                      (inst pxor x0 x1)
                      (inst movdqa x4 x3)
                      (inst pslldq x4 4)
                      (inst pshufd x5 x0 #b11111111)
                      (inst pxor x3 x5)
                      (inst pxor x3 x4)))
                 (expand-key-256a ()
                   `(expand-key-128))
                 (expand-key-256b ()
                   `(progn
                      (inst pshufd x1 x1 #b10101010)
                      (inst shufps x2 x3 #b00010000)
                      (inst pxor x3 x2)
                      (inst shufps x2 x3 #b10001100)
                      (inst pxor x3 x2)
                      (inst pxor x3 x1))))
        (let ((key-192 (gen-label))
              (key-128 (gen-label))
              (end (gen-label)))
          (inst pxor x2 x2)
          (inst movdqa x0 (ea key 0))
          (inst cmp key-length 24)
          (inst jmp :b key-128)
          (inst jmp :z key-192)

          (inst movdqa x3 (ea key 1))
          (inst movdqa (ea encryption-keys 0) x0)
          (inst movdqa (ea decryption-keys 14) x0)
          (inst movdqa (ea encryption-keys 1) x3)
          (inst aesimc x6 x3)
          (inst movdqa (ea decryption-keys 13) x6)
          (inst aeskeygenassist x1 x3 1)
          (expand-key-256a)
          (inst movdqa (ea encryption-keys 2) x0)
          (inst aesimc x6 x0)
          (inst movdqa (ea decryption-keys 12) x6)
          (inst aeskeygenassist x1 x0 1)
          (expand-key-256b)
          (inst movdqa (ea encryption-keys 3) x3)
          (inst aesimc x6 x3)
          (inst movdqa (ea decryption-keys 11) x6)
          (inst aeskeygenassist x1 x3 2)
          (expand-key-256a)
          (inst movdqa (ea encryption-keys 4) x0)
          (inst aesimc x6 x0)
          (inst movdqa (ea decryption-keys 10) x6)
          (inst aeskeygenassist x1 x0 2)
          (expand-key-256b)
          (inst movdqa (ea encryption-keys 5) x3)
          (inst aesimc x6 x3)
          (inst movdqa (ea decryption-keys 9) x6)
          (inst aeskeygenassist x1 x3 4)
          (expand-key-256a)
          (inst movdqa (ea encryption-keys 6) x0)
          (inst aesimc x6 x0)
          (inst movdqa (ea decryption-keys 8) x6)
          (inst aeskeygenassist x1 x0 4)
          (expand-key-256b)
          (inst movdqa (ea encryption-keys 7) x3)
          (inst aesimc x6 x3)
          (inst movdqa (ea decryption-keys 7) x6)
          (inst aeskeygenassist x1 x3 8)
          (expand-key-256a)
          (inst movdqa (ea encryption-keys 8) x0)
          (inst aesimc x6 x0)
          (inst movdqa (ea decryption-keys 6) x6)
          (inst aeskeygenassist x1 x0 8)
          (expand-key-256b)
          (inst movdqa (ea encryption-keys 9) x3)
          (inst aesimc x6 x3)
          (inst movdqa (ea decryption-keys 5) x6)
          (inst aeskeygenassist x1 x3 16)
          (expand-key-256a)
          (inst movdqa (ea encryption-keys 10) x0)
          (inst aesimc x6 x0)
          (inst movdqa (ea decryption-keys 4) x6)
          (inst aeskeygenassist x1 x0 16)
          (expand-key-256b)
          (inst movdqa (ea encryption-keys 11) x3)
          (inst aesimc x6 x3)
          (inst movdqa (ea decryption-keys 3) x6)
          (inst aeskeygenassist x1 x3 32)
          (expand-key-256a)
          (inst movdqa (ea encryption-keys 12) x0)
          (inst aesimc x6 x0)
          (inst movdqa (ea decryption-keys 2) x6)
          (inst aeskeygenassist x1 x0 32)
          (expand-key-256b)
          (inst movdqa (ea encryption-keys 13) x3)
          (inst aesimc x6 x3)
          (inst movdqa (ea decryption-keys 1) x6)
          (inst aeskeygenassist x1 x3 64)
          (expand-key-256a)
          (inst movdqa (ea encryption-keys 14) x0)
          (inst movdqa (ea decryption-keys 0) x0)
          (inst jmp end)

          (emit-label key-192)
          (inst movdqa x3 (ea key 1))
          (inst movdqa (ea encryption-keys 0) x0)
          (inst movdqa (ea decryption-keys 12) x0)
          (inst aeskeygenassist x1 x3 1)
          (expand-key-192a)
          (inst movdqa (ea encryption-keys 1) x5)
          (inst aesimc x6 x5)
          (inst movdqa (ea decryption-keys 11) x6)
          (inst movdqa (ea encryption-keys 2) x1)
          (inst aesimc x6 x1)
          (inst movdqa (ea decryption-keys 10) x6)
          (inst aeskeygenassist x1 x3 2)
          (expand-key-192b)
          (inst movdqa (ea encryption-keys 3) x0)
          (inst aesimc x6 x0)
          (inst movdqa (ea decryption-keys 9) x6)
          (inst aeskeygenassist x1 x3 4)
          (expand-key-192a)
          (inst movdqa (ea encryption-keys 4) x5)
          (inst aesimc x6 x5)
          (inst movdqa (ea decryption-keys 8) x6)
          (inst movdqa (ea encryption-keys 5) x1)
          (inst aesimc x6 x1)
          (inst movdqa (ea decryption-keys 7) x6)
          (inst aeskeygenassist x1 x3 8)
          (expand-key-192b)
          (inst movdqa (ea encryption-keys 6) x0)
          (inst aesimc x6 x0)
          (inst movdqa (ea decryption-keys 6) x6)
          (inst aeskeygenassist x1 x3 16)
          (expand-key-192a)
          (inst movdqa (ea encryption-keys 7) x5)
          (inst aesimc x6 x5)
          (inst movdqa (ea decryption-keys 5) x6)
          (inst movdqa (ea encryption-keys 8) x1)
          (inst aesimc x6 x1)
          (inst movdqa (ea decryption-keys 4) x6)
          (inst aeskeygenassist x1 x3 32)
          (expand-key-192b)
          (inst movdqa (ea encryption-keys 9) x0)
          (inst aesimc x6 x0)
          (inst movdqa (ea decryption-keys 3) x6)
          (inst aeskeygenassist x1 x3 64)
          (expand-key-192a)
          (inst movdqa (ea encryption-keys 10) x5)
          (inst aesimc x6 x5)
          (inst movdqa (ea decryption-keys 2) x6)
          (inst movdqa (ea encryption-keys 11) x1)
          (inst aesimc x6 x1)
          (inst movdqa (ea decryption-keys 1) x6)
          (inst aeskeygenassist x1 x3 128)
          (expand-key-192b)
          (inst movdqa (ea encryption-keys 12) x0)
          (inst movdqa (ea decryption-keys 0) x0)
          (inst jmp end)

          (emit-label key-128)
          (inst movdqa (ea encryption-keys 0) x0)
          (inst movdqa (ea decryption-keys 10) x0)
          (inst aeskeygenassist x1 x0 1)
          (expand-key-128)
          (inst movdqa (ea encryption-keys 1) x0)
          (inst aesimc x6 x0)
          (inst movdqa (ea decryption-keys 9) x6)
          (inst aeskeygenassist x1 x0 2)
          (expand-key-128)
          (inst movdqa (ea encryption-keys 2) x0)
          (inst aesimc x6 x0)
          (inst movdqa (ea decryption-keys 8) x6)
          (inst aeskeygenassist x1 x0 4)
          (expand-key-128)
          (inst movdqa (ea encryption-keys 3) x0)
          (inst aesimc x6 x0)
          (inst movdqa (ea decryption-keys 7) x6)
          (inst aeskeygenassist x1 x0 8)
          (expand-key-128)
          (inst movdqa (ea encryption-keys 4) x0)
          (inst aesimc x6 x0)
          (inst movdqa (ea decryption-keys 6) x6)
          (inst aeskeygenassist x1 x0 16)
          (expand-key-128)
          (inst movdqa (ea encryption-keys 5) x0)
          (inst aesimc x6 x0)
          (inst movdqa (ea decryption-keys 5) x6)
          (inst aeskeygenassist x1 x0 32)
          (expand-key-128)
          (inst movdqa (ea encryption-keys 6) x0)
          (inst aesimc x6 x0)
          (inst movdqa (ea decryption-keys 4) x6)
          (inst aeskeygenassist x1 x0 64)
          (expand-key-128)
          (inst movdqa (ea encryption-keys 7) x0)
          (inst aesimc x6 x0)
          (inst movdqa (ea decryption-keys 3) x6)
          (inst aeskeygenassist x1 x0 128)
          (expand-key-128)
          (inst movdqa (ea encryption-keys 8) x0)
          (inst aesimc x6 x0)
          (inst movdqa (ea decryption-keys 2) x6)
          (inst aeskeygenassist x1 x0 27)
          (expand-key-128)
          (inst movdqa (ea encryption-keys 9) x0)
          (inst aesimc x6 x0)
          (inst movdqa (ea decryption-keys 1) x6)
          (inst aeskeygenassist x1 x0 54)
          (expand-key-128)
          (inst movdqa (ea encryption-keys 10) x0)
          (inst movdqa (ea decryption-keys 0) x0)
          (emit-label end))))))

#+x86-64
(define-vop (aes-ni-encrypt)
  (:translate ironclad::aes-ni-encrypt)
  (:policy :fast-safe)
  (:args (plaintext :scs (descriptor-reg))
         (plaintext-start :scs (unsigned-reg))
         (ciphertext :scs (descriptor-reg))
         (ciphertext-start :scs (unsigned-reg))
         (round-keys :scs (descriptor-reg))
         (n-rounds :scs (unsigned-reg)))
  (:arg-types simple-array-unsigned-byte-8
              unsigned-num
              simple-array-unsigned-byte-8
              unsigned-num
              simple-array-unsigned-byte-32
              unsigned-num)
  (:temporary (:sc double-reg) x0 x1)
  (:generator 1000
     (flet ((buffer-mem (base offset)
              (make-ea :qword
                       :base base
                       :index offset
                       :disp (- (* n-word-bytes vector-data-offset)
                                other-pointer-lowtag)))
            (round-key (i)
              (make-ea :qword
                       :base round-keys
                       :disp (+ (- (* n-word-bytes vector-data-offset)
                                   other-pointer-lowtag)
                                (* 16 i)))))
       (let ((last-round (gen-label)))
         (inst movdqa x0 (buffer-mem plaintext plaintext-start))
         (inst movdqa x1 (round-key 0))
         (inst pxor x0 x1)
         (inst movdqa x1 (round-key 1))
         (inst aesenc x0 x1)
         (inst movdqa x1 (round-key 2))
         (inst aesenc x0 x1)
         (inst movdqa x1 (round-key 3))
         (inst aesenc x0 x1)
         (inst movdqa x1 (round-key 4))
         (inst aesenc x0 x1)
         (inst movdqa x1 (round-key 5))
         (inst aesenc x0 x1)
         (inst movdqa x1 (round-key 6))
         (inst aesenc x0 x1)
         (inst movdqa x1 (round-key 7))
         (inst aesenc x0 x1)
         (inst movdqa x1 (round-key 8))
         (inst aesenc x0 x1)
         (inst movdqa x1 (round-key 9))
         (inst aesenc x0 x1)
         (inst movdqa x1 (round-key 10))
         (inst cmp n-rounds 10)
         (inst jmp :z last-round)
         (inst aesenc x0 x1)
         (inst movdqa x1 (round-key 11))
         (inst aesenc x0 x1)
         (inst movdqa x1 (round-key 12))
         (inst cmp n-rounds 12)
         (inst jmp :z last-round)
         (inst aesenc x0 x1)
         (inst movdqa x1 (round-key 13))
         (inst aesenc x0 x1)
         (inst movdqa x1 (round-key 14))
         (emit-label last-round)
         (inst aesenclast x0 x1)
         (inst movdqa (buffer-mem ciphertext ciphertext-start) x0)))))

#+x86-64
(define-vop (aes-ni-decrypt)
  (:translate ironclad::aes-ni-decrypt)
  (:policy :fast-safe)
  (:args (ciphertext :scs (descriptor-reg))
         (ciphertext-start :scs (unsigned-reg))
         (plaintext :scs (descriptor-reg))
         (plaintext-start :scs (unsigned-reg))
         (round-keys :scs (descriptor-reg))
         (n-rounds :scs (unsigned-reg)))
  (:arg-types simple-array-unsigned-byte-8
              unsigned-num
              simple-array-unsigned-byte-8
              unsigned-num
              simple-array-unsigned-byte-32
              unsigned-num)
  (:temporary (:sc double-reg) x0 x1)
  (:generator 1000
     (flet ((buffer-mem (base offset)
              (make-ea :qword
                       :base base
                       :index offset
                       :disp (- (* n-word-bytes vector-data-offset)
                                other-pointer-lowtag)))
            (round-key (i)
              (make-ea :qword
                       :base round-keys
                       :disp (+ (- (* n-word-bytes vector-data-offset)
                                   other-pointer-lowtag)
                                (* 16 i)))))
       (let ((last-round (gen-label)))
         (inst movdqa x0 (buffer-mem ciphertext ciphertext-start))
         (inst movdqa x1 (round-key 0))
         (inst pxor x0 x1)
         (inst movdqa x1 (round-key 1))
         (inst aesdec x0 x1)
         (inst movdqa x1 (round-key 2))
         (inst aesdec x0 x1)
         (inst movdqa x1 (round-key 3))
         (inst aesdec x0 x1)
         (inst movdqa x1 (round-key 4))
         (inst aesdec x0 x1)
         (inst movdqa x1 (round-key 5))
         (inst aesdec x0 x1)
         (inst movdqa x1 (round-key 6))
         (inst aesdec x0 x1)
         (inst movdqa x1 (round-key 7))
         (inst aesdec x0 x1)
         (inst movdqa x1 (round-key 8))
         (inst aesdec x0 x1)
         (inst movdqa x1 (round-key 9))
         (inst aesdec x0 x1)
         (inst movdqa x1 (round-key 10))
         (inst cmp n-rounds 10)
         (inst jmp :z last-round)
         (inst aesdec x0 x1)
         (inst movdqa x1 (round-key 11))
         (inst aesdec x0 x1)
         (inst movdqa x1 (round-key 12))
         (inst cmp n-rounds 12)
         (inst jmp :z last-round)
         (inst aesdec x0 x1)
         (inst movdqa x1 (round-key 13))
         (inst aesdec x0 x1)
         (inst movdqa x1 (round-key 14))
         (emit-label last-round)
         (inst aesdeclast x0 x1)
         (inst movdqa (buffer-mem plaintext plaintext-start) x0)))))

#+x86-64
(define-vop (fast-blake2s-mixing)
  (:translate ironclad::fast-blake2s-mixing)
  (:policy :fast-safe)
  (:args (va :scs (unsigned-reg) :target ra)
         (vb :scs (unsigned-reg) :target rb)
         (vc :scs (unsigned-reg) :target rc)
         (vd :scs (unsigned-reg) :target rd)
         (x :scs (unsigned-reg))
         (y :scs (unsigned-reg)))
  (:arg-types unsigned-num
              unsigned-num
              unsigned-num
              unsigned-num
              unsigned-num
              unsigned-num)
  (:results (ra :scs (unsigned-reg) :from (:argument 0))
            (rb :scs (unsigned-reg) :from (:argument 1))
            (rc :scs (unsigned-reg) :from (:argument 2))
            (rd :scs (unsigned-reg) :from (:argument 3)))
  (:result-types unsigned-num
                 unsigned-num
                 unsigned-num
                 unsigned-num)
  (:generator 1000
    (let ((va (reg-in-size va :dword))
          (vb (reg-in-size vb :dword))
          (vc (reg-in-size vc :dword))
          (vd (reg-in-size vd :dword))
          (x (reg-in-size x :dword))
          (y (reg-in-size y :dword))
          (ra (reg-in-size ra :dword))
          (rb (reg-in-size rb :dword))
          (rc (reg-in-size rc :dword))
          (rd (reg-in-size rd :dword)))
      (move ra va)
      (move rb vb)
      (move rc vc)
      (move rd vd)
      (inst add ra rb)
      (inst add ra x)
      (inst xor rd ra)
      (inst ror rd 16)
      (inst add rc rd)
      (inst xor rb rc)
      (inst ror rb 12)
      (inst add ra rb)
      (inst add ra y)
      (inst xor rd ra)
      (inst ror rd 8)
      (inst add rc rd)
      (inst xor rb rc)
      (inst ror rb 7))))
)                        ; PROGN
