#include "cp_types.h"
#include "cp_proto.h"

/* Given circles (zN,rN) and (zS,rS) on sphere, return Mobius
which centers them at N and S poles, resp. 

   If rE>0, then normalization is included to move circle (zE,rE)
to one centered at 1 on the sphere. Otherwise, normalization is
one that first gives N and S circles the same radius, then
applies mobius z-->factor*z. (Default factor=1 means N and S should 
end up with same spherical radius.) 

   Strategy is to first find mobius MM centering N, S at their
respective poles; then work out normalization. */

int NS_mobius(complex zN,complex zS,complex zE,
	      double rN,double rS,double rE,double factor,
	      int *errflag,Mobius *Mob)
{
  int dilation=0;
  complex zzN,zzE,zzS,tmp,s,T_ez2,ew1,ez1,ez2,One,coef;
  complex Sctr,Nctr,NS,ES,EN;
  Mobius M1,M2,MM;
  double t,a,b,phi,argz,argw,lam,rrN,rrE,rrS,dist,denom;
  double V[3],T[3],C[3],Srad,Nrad,Nmod,Smod;

  if (fabs(rE)<m_toler) dilation=1; /* No 'east' circle given */
  if (dilation && factor<=0.0) factor=1.0;

  /* check for data problems */
  NS=csub(zN,zS);
  ES=csub(zE,zS);
  EN=csub(zE,zN);

  if ((!dilation && (cAbs(ES)<m_toler || cAbs(EN)<m_toler))
      || cAbs(NS)<m_toler)
    goto BOMB;
  
  dist=s_dist(zN,zS);
  MM.flip=1;

  /* N/S essentially antipodal already -- just need rotations */
  if (fabs(dist-M_PI)<m_toler)
    {
      if (fabs(zN.im)<m_toler) /* don't need anything */
	MM=ident_mob();
      else if (fabs(zN.im-M_PI)<m_toler) /* interchange poles by inverting */
	{
	  MM.a.re=MM.a.im=0.0;
	  MM.b.re=1.0;MM.b.im=0.0;
	  MM.c=MM.b;
	  MM.d=MM.a;
	}
      else /* zN to origin, zS to infinity */
	{
          denom=1/sin(zN.im); 
          Nctr.re=cos(zN.re)*denom;
	  Nctr.im=sin(zN.re)*denom;
	  denom=1/sin(zS.im);
	  Sctr.re=cos(zS.re)*denom;
	  Sctr.im=sin(zS.re)*denom;

	  MM.a.re=1.0;MM.a.im=0.0;
	  MM.b.re=-Nctr.re;MM.b.im=-Nctr.im;
	  MM.c=MM.a;
	  MM.d.re=-Sctr.re;MM.d.im=-Sctr.im;
	}
    }

  /* Non-antipodal: involves interesting geometry due to fact that 
sph circle centers are not preserved under Mobious. */
  /* Consider the plane containing the origin, zN, and zS. 
Intersection with sphere gives disc bounded by great circle thru
zN and zS; we treat this as the unit disc and locate points via
angular arguments (sph distance). Going counterclockwise, 
locate pts w2, zS, w1, z1, zN, z2, where the w's and z's are, 
resp., on the circles (zS,rS) and (zN,rN). Let g be the hyp 
geodesic orthogonal to both the geodesics w1 to w2 and z1 to z2. 
The endpoints of g on the unit circle are the precise points which 
we want to send to N and S. They are on the fixed point set of 
anticonformal involution of the disc interchanging z1, z2 and 
w1, w2. */

  /* In the disc, the idea is to put w2 at 1 as base for measuring
the others using angles (sph distances). Then move to the upper 
half-plane with T, where T(1)=infinity, T(w1)=0, and T(z1)=1. In
the half-plane, g is orthog to the imaginary axis and to the
geo through T(z1)=1 and T(z2)>1 (draw a picture!). g ends at
points +/- sqrt(T(z2)). Carry these back to unit circle under 
T^{-1} and then onto sphere using angle and tangent direction
from zS to zN. */

  else 
    {
      /* locate points by putting w2 at 1, progress through 
	 ew1, ez1, ez2 counterclockwise using distances radii
	 info. */
      ew1=cexp(1.0,2*rS);
      ez1=cexp(1.0,dist+rS-rN);
      ez2=cexp(1.0,dist+rS+rN);
      /* trans to upper half-plane is 
	 T(z)=coef*fac=[(ez1-1.0)/(ez1-ew1)]*[(z-ew1)/(z-1.0)],
	 and we only need T(ez2), which should be positive.*/
      One.re=1.0;One.im=0.0;
      coef=cdiv(csub(ez1,One),csub(ez1,ew1));
      tmp=cdiv(csub(ez2,ew1),csub(ez2,One));
      T_ez2=cmult(coef,tmp);
      lam=sqrt(T_ez2.re);
      /* pts corresp to ends of g on real line are lam and -lam.
	 Have to map these back to unit circle under inverse of T, 
	 up to factor this is 
	 TT(z)=(-z+ew1*coef)/(-z+coef),
	 and then find their arguments. */
      s.re=-lam;s.im=0.0;
      tmp=cdiv(cadd(s,cmult(ew1,coef)),cadd(s,coef));
      if ((argz=(Arg(tmp)-rS))<-M_PI) argz += 2.0*M_PI;
      s.re=lam;
      tmp=cdiv(cadd(s,cmult(ew1,coef)),cadd(s,coef));
      if ((argw=(Arg(tmp)-rS))<0) argw += 2.0*M_PI;
      /* argw and argz are the sph distance from zS on circle to
	 desired pts between w1 and w2 and between z1 and z2,
	 resp. */
      s_pt_to_vec(zS,V); /* vector pointing to zS */
      tangent(zS,zN,T); /* tangent vector at zS in direction of zN */

      /* here's the point, first in vec form, then projected to
	 eucl plane, which should go to the South pole.
	 (only need a radius for its sign -- tell if we want
	 the outside.) */
      C[0]=V[0]*cos(argw)+T[0]*sin(argw);
      C[1]=V[1]*cos(argw)+T[1]*sin(argw);
      C[2]=Srad=V[2]*cos(argw)+T[2]*sin(argw);
      denom=sqrt(C[0]*C[0]+C[1]*C[1]);
      if(denom<m_toler) /* at one of the poles */
	{Smod=Sctr.re=Sctr.im=0.0;}
      else
	{
	  Smod=denom/(1.0+C[2]); /* modulus of eucl center */
	  Sctr.re=C[0]*Smod/denom;
	  Sctr.im=C[1]*Smod/denom;
	}

      /* now the point intended to go to the North pole */
      C[0]=V[0]*cos(argz)+T[0]*sin(argz);
      C[1]=V[1]*cos(argz)+T[1]*sin(argz);
      C[2]=Nrad=V[2]*cos(argz)+T[2]*sin(argz);
      denom=sqrt(C[0]*C[0]+C[1]*C[1]);
      if(denom<m_toler) /* at one of the poles */
	{Nmod=Nctr.re=Nctr.im=0.0;}
      else
	{
	  Nmod=denom/(1.0+C[2]); /* modulus of eucl center */
	  Nctr.re=C[0]*Nmod/denom;
	  Nctr.im=C[1]*Nmod/denom;
	}

      /* Now build appropriate mobius */

      if (Nmod<m_toler || Smod<m_toler) /* at least one at a pole */
	{
	  if (Nmod<m_toler)
	    {
	      if (Smod<m_toler) /* Sctr is at other */
		{
		  if (Nrad<0) /* simply have to invert */
		    {
		      MM.a.re=MM.a.im=0.0;
		      MM.b.re=1.0;MM.b.im=0.0;
		      MM.c.re=1.0;MM.c.im=0.0;
		      MM.d.re=0.0;MM.d.im=0.0;
		    }
		  else MM=ident_mob();
		}
	      else 
		{
		  if (Nrad<0) /* Nctr at south pole, need to
				 invert first, get new Sctr */
		    {
		      M1.a.re=M1.a.im=0.0;
		      M1.b.re=1.0;M1.b.im=0.0;
		      M1.c.re=1.0;M1.c.im=0.0;
		      M1.d.re=M1.d.im=0.0;

		      denom=Sctr.re*Sctr.re+Sctr.im*Sctr.im;
		      Sctr.re=Sctr.re/denom;
		      Sctr.im=-Sctr.im/denom;
		    }
		  else M1=ident_mob();
		  /* move Sctr to infinity */
		  M2.a.re=M2.a.im=0.0;
		  M2.b.re=1.0;M2.b.im=0.0;
		  M2.c.re=1.0;M2.c.im=0.0;
		  M2.d.re=-Sctr.re;M2.d.im=-Sctr.im;

		  MM=matrix_product(M2,M1);
		}
	    }
	  else
	    {
	      if (Srad>0) /* Sctr at north pole, need to
			     invert first, get new Nctr */
		{
		  M1.a.re=M1.a.im=0.0;
		  M1.b.re=1.0;M1.b.im=0.0;
		  M1.c.re=1.0;M1.c.im=0.0;
		  M1.d.re=M1.d.im=0.0;

		  denom=Nctr.re*Nctr.re+Nctr.im*Nctr.im;
		  Nctr.re=Nctr.re/denom;
		  Nctr.im=-Nctr.im/denom;
		}
	      else M1=ident_mob();
	      /* move Nctr to origin */
	      M2.a.re=1.0;M2.a.im=0.0;
	      M2.b.re=-Nctr.re;M2.b.im=Nctr.im;
	      M2.c.re=0.0;M2.c.im=0.0;
	      M2.d.re=1.0;M2.d.im=0.0;

	      MM=matrix_product(M2,M1);
	    }		  
	}
      else /* generic case; move Nctr to origin, Sctr to infinity */
	{
	  MM.a.re=1.0;MM.a.im=0.0;
	  MM.b.re=-Nctr.re;MM.b.im=-Nctr.im;
	  MM.c=MM.a;
	  MM.d.re=-Sctr.re;MM.d.im=-Sctr.im;
	}
    }

  /* Now we have MM; need images of sph circles under MM */
  mobius_of_circle(MM,1,zN,rN,&zzN,&rrN,1);
  mobius_of_circle(MM,1,zS,rS,&zzS,&rrS,1);
  if (!dilation) mobius_of_circle(MM,1,zE,rE,&zzE,&rrE,1);

  /* Have two choices for normalization */

  /* Case 1: Place circle 1 at one (when on sphere). 

     Project pts of E closest to N and to S to the positive
     numbers a and b. Necessary dilation is t where 
     t^2=1/(ab). */

  if (!dilation)
    {
      phi=zzE.im-rrE;
      a=sin(phi)/(1.0+cos(phi));
      phi=zzE.im+rrE;
      b=sin(phi)/(1.0+cos(phi));
      t=sqrt(1/fabs(a*b)); /* ab should be positive, but fabs() is
			      cautionary */

      M1.a=cexp(t,-zzE.re);
      M1.b.re=M1.b.im=0.0;
      M1.c.re=M1.c.im=0.0;
      M1.d.re=1.0;M1.d.im=0.0;
      M1.flip=1;

      (*Mob)=matrix_product(M1,MM);
    }

  /* Case 2: first scale by t to get N/S circles the same size;
     then dilate by factor. */
  else
    {
      a=sin(rrN)/(1.0+cos(rrN));
      phi=M_PI-rrS;
      b=sin(phi)/(1.0+cos(phi));
      t=sqrt(1/fabs(a*b)); /* ab should be positive, but fabs() is
			      cautionary */    

      M1.a.re=t*factor;M1.a.im=0.0;
      M1.b.re=M1.b.im=0.0;
      M1.c.re=M1.c.im=0.0;
      M1.d.re=1.0;M1.d.im=0.0;
      M1.flip=1;

      (*Mob)=matrix_product(M1,MM);
    }
  return 1;

 BOMB:
  (*Mob)=ident_mob();
  (*errflag)++;
  return 1;

} /* NS_mobius */

