/*
 * gui_rest.cpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright (c) 2005-2006  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  @file  gui_rest.cpp   -  included by "gui.cxx" via a FLUID directive.
  
  Implementation of things declared in FLUID (->"gui.h"), but NOT implemented
   via FLUID (-->"gui.cxx"), as well as of things declared in "gui_rest.hpp".
   
  @Note: Von ResponsePlot werden nur die nicht-leeren `which-Kurven geplottet.
   Nur die(se) dargestellten sollten auch durch die "Using Next"-Knoepfe offeriert
   werden. Modus des Aktivierens und Einschaltens: `update_activation_radios_...()
   aktuallisiert Aktivierung unabhaengig vom Einschaltwert, `update_onoff_radios_...() 
   aktuallisiert Einschaltwert unabhaengig von Aktivierung. 
*/

// Includes not realy needed because we are included in gui.cxx
#include "gui.h"			// generated by FLUID
#include "gui_rest.hpp"                 
#include "../bracketing_to_hdr.hpp"     // cpaint_load_image()
#include "../br_core/br_messages.hpp"   // v_alert()


/* DECLARATIONS of LOCAL FUNCTIONS */
static bool     file_exists             (const char* fname);
static char*    get_curve_basepath      (char* path);
static void     message_read_error      (const char* fname);


/* LOCAL VARIABLES */
const int LEN = 256;
static char curve_path_ [LEN] = ".";
static bool overwrite_warning_ = true;


using namespace br;

//============================================================================
//
//  non-member functions  -  declared above or in "gui_rest.h" or in "gui.h"
//
//============================================================================
/*****************************************************************************
  Return true, if file `fname exists, false otherwise
******************************************************************************/
bool 
file_exists (const char* fname)
{
    if (!fname || !*fname) 
      return false;

    FILE* f = fopen (fname,"r");
  
    if (!f) return false;
  
    fclose(f); 
    return true; 
}

/*****************************************************************************
  Removes from `path a ".txt" extension and a trailing "_R|G|B" if any.
   E.g. "crv_R.txt" ---> "crv"
        "crv.txt"   ---> "crv"
        "crv_R.aaa" ---> "crv_R.aaa"  (no ".txt" ext)
******************************************************************************/
char* 
get_curve_basepath (char* path)
{
    char* p = (char*) fl_filename_ext (path);
    if (strcmp(p,".txt") == 0) { 
      *p = '\0';                      // truncates extension
      if (p > path + 1) {             // "_R|G|B" possible?
        --p;                          // pos before '.'
        if ((*p=='R' || *p=='G' || *p=='B') && p[-1]=='_') {
          *(--p) = '\0';              // truncates "_R|G|B"
        }
      }
    }
    return path;
}

/*****************************************************************************
  message_read_error()         Provisonally. Should provide more information.
*****************************************************************************/
void
message_read_error (const char* fname)
{
    fl_alert ("Could not read response curve file \"%s\".", fname);
    //br::v_alert ("Could not read response curve file \"%s\".", fname);
}


/**+*************************************************************************\n
  Saves the response curve `channel of kind `which from `the_Br2Hdr into a
   file `fname. A warning and a "what to do" dialog is opened, if a file of 
   this name already exists. GUI wrapper of Br2Hdr::write_ResponseCurve(..) for
   the sake of this "What to do"-dialog.
******************************************************************************/
bool 
save_response_curve (Br2Hdr::WhichCurves which, int channel, const char* fname)
{
    if (!fname || !*fname) return false;
        
    if (overwrite_warning_ && file_exists(fname)) 
    {
      char txt[256];
      snprintf (txt, 256, "A file \"%s\" already exists.", fl_filename_name(fname));
      switch (fl_choice (txt, "Abbort", "Overwrite", "Overwrite all"))
        {
        case 2:  overwrite_warning_ = false; break;
        case 1:  break;
        default: return false;
        }
    }
    return the_Br2Hdr.write_ResponseCurve (which, channel, fname);
}

/**+*************************************************************************\n
  Opens a file_chooser dialog to save response curve(s) of sort `which from 
   the global `the_Br2Hdr. Depending on `channel_like all channels are saved 
   or a single one. The selected filename is taken as basename in the following
   sense: For the red channel is added a "_R", for the green a "_G", for the 
   blue a "_B" plus an extension ".txt". If the selected filename has already 
   e.g. a trailing "_R.txt", this part is removed.
  DISADVANTAGE: Due to the automatic filename complement it is impossible to
   save a curve under a arbitrary filename. 
  
  @param which: Which kind of response curves (next_use, computed, external).
  @param channel_like: -1: save all curves;  >= 0: save curve for that channel.

  @TODO The overwrite warning (in save_response_curve()) should appear over
   the open file_chooser dialog and in a loop, instead of after closing the
   file_chooser dialog.
******************************************************************************/
void 
save_response_curves_dialog (Br2Hdr::WhichCurves which, int channel_like)
{
    // which==CALCTOR_S nur voruebergehend und wird abgefangen
    IF_FAIL_DO (which < Br2Hdr::CALCTOR_S, return);
    
    if (allWins.do_file_chooser (curve_path_, "*.{txt}", 
        Fl_File_Chooser::CREATE, "Save response curve(s) under basename..."))
    {
      //  Get curve base path form selected path
      char basepath [LEN];
      strncpy (basepath, allWins.fileChooser->value(), LEN);
      get_curve_basepath (basepath);
      //printf("basepath = %s\n", basepath);
      
      overwrite_warning_ = true;
      if (channel_like == 0 || channel_like == -1) {
        snprintf (curve_path_, LEN, "%s_R.txt", basepath);
        save_response_curve (which, 0, curve_path_);
      }
      if (channel_like == 1 || channel_like == -1) {
        snprintf (curve_path_, LEN, "%s_G.txt", basepath);
        save_response_curve (which, 1, curve_path_);
      }
      if (channel_like == 2 || channel_like == -1) {
        snprintf (curve_path_, LEN, "%s_B.txt", basepath);
        save_response_curve (which, 2, curve_path_);
      }
      // last curve_path_ == start point for next call
    }
}

/**+*************************************************************************\n
  Reads a response curve from file `fname for channel `channel and updates 
   the GUI. Used by the load_dialogs as well as the automatic response loading 
   at the programm start (that's why we have to check `channel and `fname). 
   Gives a GUI error message, if a failure occured.
******************************************************************************/
bool 
load_response_curve (int channel, const char* fname)
{
    if (channel < 0 || channel > 2 || !fname || !*fname) 
      return false;
    
    if (! the_Br2Hdr.read_ResponseCurve (channel, fname))
    {
      message_read_error (fl_filename_name(fname));
      return false;
    }
    //  Update the "using next" group and the curve plot; update_onoff_radios()
    //   nessecary, since loaded curves are applied autom. if use_extern==1.
    allWins.response->update_activation_group_Using_Next();  
    allWins.response->update_activation_radios_External();
    allWins.response->update_onoff_radios_Using_Next(); 
    allWins.response->do_which_curves (Br2Hdr::EXTERNAL);  // plot external curves
    return true;
}

/**+*************************************************************************\n
  Opens a file dialog and reads the response curve for `channel. Currently 
   used only in ResponseWinClass, could be included there as callback function.
  Idea: For isUseExternResponse()==TRUE we could apply a loaded curve automat. 
******************************************************************************/
void 
load_response_curve_dialog (int channel)
{
    if (channel < 0 || channel > 2) return;
    
    char title [64];
    char ch[3] = {'R','G','B'};
    sprintf (title, "Load response curve for channel %c", ch[channel]);
    
    if (allWins.do_file_chooser (curve_path_, "*.{txt}", Fl_File_Chooser::SINGLE, title))
    {
      load_response_curve (channel, allWins.fileChooser->value());
      //  Save selected path as start point for next call
      strncpy (curve_path_, allWins.fileChooser->value(), LEN);
    }     
}

/**+*************************************************************************\n
  Opens a file dialog, takes the curve *base* path from the selected path and
   tries to read all three response curves for the completed paths. Currently 
   used only in ResponseWinClass, could be included there as callback function.
******************************************************************************/
void 
load_all_response_curves_dialog()
{
    printf("curve_path_=\"%s\"\n", curve_path_);
    if (allWins.do_file_chooser (curve_path_, "*.{txt}", Fl_File_Chooser::SINGLE,
        "Load all response curves. Choose basename"))
    {
      //  Get curve base path from selected path
      char basepath[LEN];
      strncpy (basepath, allWins.fileChooser->value(), LEN);
      get_curve_basepath (basepath);
      printf("basepath = %s\n", basepath);
      
      snprintf (curve_path_, LEN, "%s_R.txt", basepath);
      load_response_curve (0, curve_path_);
      
      snprintf (curve_path_, LEN, "%s_G.txt", basepath);
      load_response_curve (1, curve_path_);

      snprintf (curve_path_, LEN, "%s_B.txt", basepath);
      load_response_curve (2, curve_path_);
      
      //  Save selected path as start point for next call
      strncpy (curve_path_, allWins.fileChooser->value(), LEN);
    }     
}

  
//============================================================================
//
//  class MainWinClass  -  declared in "gui.h" (<-- "gui.fl")
//
//============================================================================
/**+*************************************************************************\n
  Open a file dialog for images, load the image(s), insert them into the image
   container of `the_Br2Hdr; finally init the calctor.
******************************************************************************/
void 
MainWinClass::cb_open_input_file()
{
    int count = allWins.do_file_chooser (0, "Image Files (*.{tif,jpg,png,bmp,gif})",  
                  Fl_File_Chooser::MULTI, "Open Images(s) for Bracketing to HDR");
        
    printf ("Number of selected files = %d\n", count);
  
    int old_active = the_Br2Hdr.size_active();
  
    //  Load selected images and insert them into `the_Br2Hdr
    for (int i=1; i <= count; i++)
    {
      cpaint_load_image (allWins.fileChooser->value(i), the_Br2Hdr); 
    }
  
    //  If possibly and anything has changed, create new calctor
    if ((the_Br2Hdr.size_active() > 1) && (the_Br2Hdr.size_active() != old_active)) 
      the_Br2Hdr.init_Calctor();        // --> CALCTOR_INIT
}

/**+*************************************************************************\n
  Set mode for solving `Ax=b' in `the_Br2Hdr' and update the gui. Intended for
   usage by read_preferences().
******************************************************************************/
void 
MainWinClass::solve_mode (ResponseSolverBase::SolveMode mode)
{
    the_Br2Hdr.solve_mode (mode);
    
    switch (mode)
    {
    case ResponseSolverBase::AUTO:    mbar_item_solvemode_auto_->setonly();    break;
    case ResponseSolverBase::USE_QR:  mbar_item_solvemode_use_qr_->setonly();  break;
    case ResponseSolverBase::USE_SVD: mbar_item_solvemode_use_svd_->setonly(); break;
    default: ;
    }
}


//============================================================================
//
//  class MainWinMenubarWatcher  -  declared in "gui_rest.hpp".
//
//============================================================================
/**+*************************************************************************\n
  handle_Event()

  De/activation of menubar items accordingly to changes of Br2Hdr::Instance()
   state. The concrete event is not used here by now, we check only some state
   variables at every call.

  Reference to the menu items via FLUID-named items. Per default the menu
   structure as well as these variables are *static* class members. Here ok
   as we have only a single instance. For multiple menubar instances we would
   have to take local copies via `o->copy(o->menu()) ("Extra Code" field of
   the menubar in FLUID), and to refer then to `o->menu() + diff.
******************************************************************************/
void 
MainWinMenubarWatcher::handle_Event (Br2Hdr::Event e)
{
#ifdef BR_DEBUG_RECEIVER
    printf("%4d: MainWinMenubarWatcher(): ",__LINE__);
    EventReceiver::handle_Event(e);
#endif 
    
    //  Shorter names for the menu_item variables
    Fl_Menu_Item* item_Init     = MainWinClass::mbar_item_Init_Calctor_;
    Fl_Menu_Item* item_Response = MainWinClass::mbar_item_Compute_Response_;
    Fl_Menu_Item* item_HDR      = MainWinClass::mbar_item_Compute_HDR_;
    Fl_Menu_Item* item_logHDR   = MainWinClass::mbar_item_Compute_logHDR_;

    if (Br2Hdr::Instance().size_active() > 1) {
      if (! item_Init->active()) {
        item_Init->activate();
      }
    } 
    else {
      if (item_Init->active()) {
        item_Init->deactivate();
      }
    }
    
    if (Br2Hdr::Instance().calctor()) {
      if (! item_Response->active()) {
        item_Response->activate();
        item_HDR->activate();
        item_logHDR->activate();
      }
    } 
    else {
      if (item_Response->active()) {
        item_Response->deactivate();
        item_HDR->deactivate();
        item_logHDR->deactivate();
      }
    }
}


//============================================================================
//
//  class ResponseWinClass  -  declared in "gui.h" resp. "gui.fl"
//
//============================================================================
/**+*************************************************************************\n
  handle_Event()
   Update widget for some events. For EXTERN_RESPONSE and CCD_UPDATED copying 
   (applying) to Br2Hdr's use_next curves has done already by the Manager.
   The EXTERN_RESPONSE event was introduced to allow `use_extern switchings
   also from outside of ResponseWinClass. If outside unused, it could be removed.
   Meanwhile outside used by read_preferences().
******************************************************************************/
void 
ResponseWinClass::handle_Event (Br2Hdr::Event e)
{
#ifdef BR_DEBUG_RECEIVER
    printf("ResponseWinClass::"); EventReceiver::handle_Event(e);
#endif
    switch (e)
    {
    case Br2Hdr::EXTERN_RESPONSE:
        BR_EVENT_HANDLED(e);
        update_activation_group_Using_Next();
        if (the_Br2Hdr.isUseExternResponse()) {
          slider_grid_points_->deactivate();
          slider_smoothing_->deactivate();
          update_onoff_radios_Using_Next();
          do_which_curves (Br2Hdr::USE_NEXT);   // show "use_next" curves
        }
        else {
          slider_grid_points_->activate();
          slider_smoothing_->activate();
          plot_->update();      // "weight fncts to plot" could be changed
        }
        toggle_use_extern_curves_->value (the_Br2Hdr.isUseExternResponse());
        break;        
      
    case Br2Hdr::CCD_UPDATED:       // understood as "new curves COMPUTED"
        BR_EVENT_HANDLED(e);
        update_activation_radios_Computed();
        update_onoff_radios_Using_Next();
        do_which_curves (Br2Hdr::COMPUTED);   // show computed curves
        break;
            
    case Br2Hdr::CALCTOR_DELETED:
        BR_EVENT_HANDLED(e);
        if (plot_->which_curves()==Br2Hdr::CALCTOR_S) 
          plot_->update();
        break;    
            
    case Br2Hdr::CCD_OUTDATED:          // only an "outdated" box is to add
        BR_EVENT_HANDLED(e);
        if (plot_->add_outdated_box())  
          plot_->redraw();
        break;
    
    case Br2Hdr::WEIGHT_CHANGED:
    case Br2Hdr::CALCTOR_INIT:    
        BR_EVENT_HANDLED(e);
        plot_->update();
        break;
        
    default: BR_EVENT_NOT_HANDLED(e);
        break;
    }            
}

/**+*************************************************************************\n
  Update activation of the whole "using next" group. Activate, if any external 
   curve exists OR isUseExternResponse() == True ; else deactivate.
******************************************************************************/
void 
ResponseWinClass::update_activation_group_Using_Next()
{
    if (! the_Br2Hdr.isExternResponseEmpty() || the_Br2Hdr.isUseExternResponse()) 
      group_using_next_->activate();
    else
      group_using_next_->deactivate();
} 

/**+*************************************************************************\n
  Update activation of "using next" buttons for the computed response curves.
   Independently of On/Off values!
******************************************************************************/
void 
ResponseWinClass::update_activation_radios_Computed()
{
    int sum=0;
    for (int i=0; i < 3; i++)
    {
      if (the_Br2Hdr.getResponseCurveComputed(i).is_empty())
        {radio_computed_[i]->deactivate();}
      else
        {radio_computed_[i]->activate();  sum++;}
    }
    //  Activate the "all button" if any curve exists, else deactivate
    if (sum) button_computed_all_->activate();
    else     button_computed_all_->deactivate();
}

/**+*************************************************************************\n
  Update activation of "using next" buttons for the external response curves.
   Independently of On/Off values!
******************************************************************************/
void 
ResponseWinClass::update_activation_radios_External()
{
    int sum=0;
    for (int i=0; i < 3; i++)
    {
      if (the_Br2Hdr.getResponseCurveExtern(i).is_empty())
        {radio_extern_[i]->deactivate();}
      else
        {radio_extern_[i]->activate();  sum++;}
    }
    //  Activate the "all button" if any curve exists, else deactivate 
    if (sum) button_extern_all_->activate();  
    else     button_extern_all_->deactivate();
}

/**+*************************************************************************\n
  Update the On/Off values of all "using next" radio buttons accordingly to the
   state of `the_Br2Hdrs use_next curves. Independently of activation status!
******************************************************************************/
void 
ResponseWinClass::update_onoff_radios_Using_Next()
{
    for (int i=0; i<3; i++) {
      if (! the_Br2Hdr.getResponseCurveUsenext(i).is_empty())
        if (the_Br2Hdr.isUsenextCurveComputed(i)) radio_computed_[i]->setonly();
        else                                      radio_extern_[i]->setonly();
      else {
        radio_computed_[i]->clear();
        radio_extern_[i]->clear();
      }  
    }  
}

/**+*************************************************************************\n
  Set the "plotted curves" choice to value `which, update the plot and update
   the save_menu accordingly.
******************************************************************************/
void 
ResponseWinClass::do_which_curves (Br2Hdr::WhichCurves which)
{
    choice_plot_->value (which);
    plot_->which_curves (which);
    plot_->update();
    update_save_menu();
}

/**+*************************************************************************\n
  Apply a computed response curve (or all) and update the widget.
  @param channel_like: >= 0: apply the curve for that channel; < 0: apply all 
    channels. 
******************************************************************************/
void 
ResponseWinClass::applyResponseCurveComputed (int channel_like)
{
    if (channel_like >= 0) 
      the_Br2Hdr.applyResponseCurveComputed (channel_like);
    else 
      for (int i=0; i < 3; i++) 
        the_Br2Hdr.applyResponseCurveComputed (i);
    
    update_onoff_radios_Using_Next();
    do_which_curves (Br2Hdr::USE_NEXT);   // show the "use_next" curves
}

/**+*************************************************************************\n
  Apply an external response curve (or all) and update the widget.
  @param channel_like: >= 0: apply the curve for that channel; < 0: apply all 
    channels. 
******************************************************************************/
void 
ResponseWinClass::applyResponseCurveExtern (int channel_like)
{
    if (channel_like >= 0)
      the_Br2Hdr.applyResponseCurveExtern (channel_like);
    else 
      for (int i=0; i < 3; i++)
        the_Br2Hdr.applyResponseCurveExtern (i);
      
    update_onoff_radios_Using_Next();
    do_which_curves (Br2Hdr::USE_NEXT);   // show the "use_next" curves
}

/**+*************************************************************************\n
  De/Activate the save_menu accordingly to the current plotted curves.
******************************************************************************/
void 
ResponseWinClass::update_save_menu()
{
    //  For the transient case which==CALCTOR_S we don't provide saving
    if (plot_->which_curves() == Br2Hdr::CALCTOR_S) {
      menubttn_save_->deactivate();
      return;
    }
    if (the_Br2Hdr.isResponseEmpty (plot_->which_curves()))
      menubttn_save_->deactivate();
    else {
      menubttn_save_->activate();
#if 0      
      //  So, if we could use in FLUID array-names for static objects:
      for (int i=0; i<3; i++) 
        if (the_Br2Hdr.getResponseCurve (plot_->which_curves(), i).is_empty())
          //menubttn_save_item_channel_[i]->deactivate();
          menubttn_save_item_channel_[i]->hide();
        else
          //menubttn_save_item_channel_[i]->activate();
          menubttn_save_item_channel_[i]->show();
#else          
      //  Work around a FLUID bug (wrong initialization of static arrays):
      if (the_Br2Hdr.getResponseCurve (plot_->which_curves(), 0).is_empty())
           menubttn_save_item_channel_0->hide();
      else menubttn_save_item_channel_0->show();
      
      if (the_Br2Hdr.getResponseCurve (plot_->which_curves(), 1).is_empty())
           menubttn_save_item_channel_1->hide();
      else menubttn_save_item_channel_1->show();
      
      if (the_Br2Hdr.getResponseCurve (plot_->which_curves(), 2).is_empty())
           menubttn_save_item_channel_2->hide();
      else menubttn_save_item_channel_2->show();
#endif          
    }
}


/**+*************************************************************************\n
  Update widget after the set of external response curves has been cleared.
******************************************************************************/
void 
ResponseWinClass::update_after_clear_response_extern()
{
    update_activation_group_Using_Next(); // could be deactivated
    update_activation_radios_External();  // should be deactivated
    update_onoff_radios_Using_Next();
    do_which_curves (plot_->which_curves()); // previous `which
    // Last is a shortening for update_save_menu() + plot_->update().
}


//============================================================================
//
//  class HistogramWinClass  -  declared in "gui.h".
//
//============================================================================
/**+*************************************************************************\n
  handle_Event()  --  update widget for some events.
******************************************************************************/
void 
HistogramWinClass::handle_Event (Br2Hdr::Event e)
{
#ifdef BR_DEBUG_RECEIVER
    printf("HistogramWinClass::"); EventReceiver::handle_Event(e);
#endif
    switch (e)
    {
    case Br2Hdr::IMAGES_CHANGED:
    case Br2Hdr::IMAGE_LOADED:
        BR_EVENT_HANDLED(e);
        if (plot_->image() >= the_Br2Hdr.size()) 
          plot_->set_image(0);
        build_choice_image();
        plot_->update();
        break;
        
    default: BR_EVENT_NOT_HANDLED(e);
        break;
    }            
}

/**+*************************************************************************\n
  Build a choice menu for as many images as in the_Br2Hdr's image container.
******************************************************************************/
void 
HistogramWinClass::build_choice_image()
{
    choice_image_->clear();              // no redraw()
    if (the_Br2Hdr.size()) 
    {
      for (int i=0; i < the_Br2Hdr.size(); i++) 
      {
        char s[4];
        snprintf(s,4,"%d", i+1);         // begin with "1"
        choice_image_->add(s,0,0);
      }
      choice_image_->value (plot_->image()); // implies a redraw()
    }
    else 
      choice_image_->redraw();           // redraw to clear()
}


// END OF FILE
