/*
 * bracketing_to_hdr.cpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright (c) 2005-2006  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  @file  bracketing_to_hdr.cpp
  
  Main source file of the CinePaint plugin "Bracketing to HDR".
  
  This file contains all CinePaint and Gtk references of the "Bracketing" 
   project. The only two services we need from CPaint are loading an image and
   showing an float32 (HDR) image: wrapped and provided to the rest of the 
   program by the two functions declared in "bracketing_to_hdr.hpp".
  
  Internal Note: 
   `GPrecisionType' is defined in "cinepaint###/lib/wire/precision.h" as an
   enum. There are also precision strings `PRECISION_U8_STRING' etc,
   unfortunately not in an array to have a simple reference to the enums.
    
  @BUG If "Br2HdrManager.hpp" && "AllWindows.hpp" && "bracketings_to_hdr.hpp" 
   are included *after* the CPaint headers it gives the compiler error:
 
    In file included from /usr/include/g++/i586-suse-linux/bits/c++locale.h:44,
                 from /usr/include/g++/iosfwd:46,
                 from /usr/include/g++/bits/stl_algobase.h:70,
                 from /usr/include/g++/vector:67,
                 from br_core/br_Image.hpp:31,
                 from br_core/Br2HdrManager.hpp:19,
                 from bracketing_to_hdr.hpp:11,
                 from bracketing_to_hdr.cpp:49:
    /usr/include/libintl.h:40: error: syntax error before `char'
     
   No idea why, anyhow, thus CPaint headers included behind those.
*/

#include <FL/fl_ask.H>              // fl_alert(), fl_message()
#include <FL/filename.H>            // fl_filename_name()
#include "br_core/br_version.hpp"   // BR_VERSION_LSTRING    
#include "br_core/Rgb.hpp"          // Rgb<>
#include "br_core/br_macros.hpp"    // IF_FAIL_DO() etc.
#include "br_core/Br2HdrManager.hpp"
#include "gui/AllWindows.hpp"       // object `allWins'
#include "bracketing_to_hdr.hpp"    // cpaint_load_image(), cpaint_show_image()
extern "C" {
#include "lib/plugin_main.h"
#include "lib/wire/libtile.h"
#include "plugin_pdb.h"
#include "libgimp/stdplugins-intl.h"
}


//==================================================
//  Info strings (macros) for the CinePaint plugin
//==================================================
#define PLUGIN_NAME          "plug_in_bracketing_to_hdr"
#define PLUGIN_BRIEF         "Merges bracketed exposures into one HDR image."
#define PLUGIN_DESCRIPTION   "Loads bracketed exposures of a (still) scene and merges them into one High Dynamic Range image (radiance map)."
#define PLUGIN_VERSION       BR_VERSION_LSTRING
#define PLUGIN_AUTHOR        "Hartmut Sbosny <hartmut.sbosny@gmx.de>"
#define PLUGIN_COPYRIGHT     "Copyright 2005-2006 Hartmut Sbosny"


//======================================
//  Declaration of the local functions
//======================================
void            query                           (void);
void            run                             (char *    name,
                                                 int       nparams,
                                                 GParam *  param,
                                                 int *     nreturn_vals,
                                                 GParam ** return_vals);

int             cpaint_load_image_via_PDB       (const char* fname);
br::BrImage     u8_from_u16                     (const br::BrImage & img16);
br::BrImage     cpaint_get_BrImage_U8           (int W, int H, gint32 layer_ID);
br::BrImage     cpaint_get_BrImage_U16          (int W, int H, gint32 layer_ID);
void            cpaint_show_rgb_imgbuf          (int W, int H, const uchar* imgbuf,
                                                 GPrecisionType );
const char*     cpaint_precision_name           (GPrecisionType);

template<typename T> 
void            copy_without_alpha              (br::Image &, GimpPixelRgn &);
                                                 

//==========================
//  file-global variables      
//==========================
GPlugInInfo  PLUG_IN_INFO =     // required by `MAIN()'
{
    NULL,    /* init_proc  */
    NULL,    /* quit_proc  */
    query,   /* query_proc */
    run,     /* run_proc   */
};

typedef struct { char name[1024]; }  Vals;
Vals  vals;      // for communication between CPaint and the plugin

static int n_return_vals_;
static int n_args_;


//==========================================
//
//              IMPLEMENTATION 
//
//==========================================
MAIN()      // defined in "lib/plugin_main.h"; requires a global GPlugInfo-
            //  object of name "PLUG_IN_INFO" (defined just above)   

            
/**+*************************************************************************\n
  querry()
******************************************************************************/
void
query ()
{
  static GParamDef args [] =
  {
    { PARAM_INT32,      "run_mode",     "Interactive, non-interactive" },
    { PARAM_STRING,     "directory",    "directory" },
  };
  static GParamDef return_vals [] =
  {
    { PARAM_IMAGE,      "image",        "Output Image" },
  };
  n_args_        = sizeof (args) / sizeof (args[0]);
  n_return_vals_ = sizeof (return_vals) / sizeof (return_vals[0]);

  gimp_install_procedure (PLUGIN_NAME,
                          PLUGIN_BRIEF,
                          PLUGIN_DESCRIPTION,
                          PLUGIN_AUTHOR,
                          PLUGIN_COPYRIGHT,
                          PLUGIN_VERSION,
                          "<Toolbox>/File/New From/Bracketing to HDR",
                          "*",
                          PROC_EXTENSION,
                          n_args_, n_return_vals_,
                          args, return_vals);
  _("Bracketing to HDR");
}

/**+*************************************************************************\n
  run()
******************************************************************************/
void
run (char *    name,            // I - Name of called function resp. plug-in
     int       nparams,         // I - Number of parameters passed in
     GParam *  param,           // I - Parameter values
     int *     nreturn_vals,    // O - Number of return values
     GParam ** return_vals)     // O - Return values
{
  GStatusType    status;        // return status
  GRunModeType   run_mode;
  static GParam  values [2];    // return parameter
  int            result = -1;
 
  status                  = STATUS_SUCCESS;
  run_mode                = (GRunModeType) param[0].data.d_int32;
  values[0].type          = PARAM_STATUS;
  values[0].data.d_status = STATUS_CALLING_ERROR;

  *nreturn_vals           = n_return_vals_;
  *return_vals            = values;

  if (strcmp (name, PLUGIN_NAME) == 0)
    {
      sprintf (vals.name, getenv("PWD"));     // default for `vals'
    
      gimp_get_data (PLUGIN_NAME, &vals);     // get data from last call
      //printf ("nach gimp_get_data(): vals = \"%s\"\n", vals.name);
     
      switch (run_mode)
        {
        case RUN_INTERACTIVE:
            result = allWins.run();
            if (result < 0) 
              {
                status = STATUS_EXECUTION_ERROR;
              }
            break;

        case RUN_NONINTERACTIVE:    // Makes this sense? Not realy.
            if (nparams != n_args_) 
              {
                status = STATUS_CALLING_ERROR;
                break;
              }
            result = allWins.run();  
            if (result < 0) 
              {
                status = STATUS_EXECUTION_ERROR;
                break;
              }
            *nreturn_vals = n_return_vals_ + 1;
            break;
    
        case RUN_WITH_LAST_VALS:
            break;

        default:
            break;
        }
      
      values[1].type = PARAM_IMAGE;
      values[1].data.d_image = result;
    }
  
  values[0].data.d_status = status;
  gimp_set_data (PLUGIN_NAME, &vals, sizeof(Vals));  // data for next call
}


/**+*************************************************************************\n
  u8_from_u16()  --  helper for cpaint_load_image()

  Downsamples a RGB_U16 BrImage into a RGB_U8 image allocated here. Meta data
   of img16 are not copied, they get the default settings.
    
  @param img16: BrImage of type RGB_U16.
  @return  downsampled BrImage of type RGB_U8, created here.

  Open: Metadaten von img16 auch kopieren oder weglassen? Haengt von 
   Verwendungsweise ab. Hier reicht weglassen.
******************************************************************************/
br::BrImage 
u8_from_u16 (const br::BrImage & img16)
{
  //  If input image isn't of U16 type return Null-Image
  IF_FAIL_RETURN (img16.data_type() == br::DATA_U16, br::BrImage());  

  //  Allocate dest image (default settings for meta data)
  br::BrImage  img8 (img16.width(), img16.height(), br::IMAGE_RGB_U8);
  
  Rgb<guint8>*  u8  = (Rgb<guint8>*)  img8.buffer();
  Rgb<guint16>* u16 = (Rgb<guint16>*) img16.buffer();
  
  for (int i=0; i < img8.n_pixel(); i++) 
    {
      u8[i] = u16[i] >> 8;
    }
  return img8;
}

/**+*************************************************************************\n
  cpaint_load_image()

  Load image `fname' via CinePaint and insert it into the Br2HdrManager object
   `mBr2Hdr'.

  @param fname: name of file to load.
  @param mBr2Hdr: Br2HdrManager the image shall be added to. 
  @return  Successful or not?

  Note: Do not leave an unfreed CPaint image!
  Note: Erwaegenswert auch Variante, die ein BrImage returniert und kein 
    Manager-Argument besitzt. Verringerte Abhaengigkeiten merklich. Es muessten
    dann aber Informationen bereitgestellt ob es sich um das erste Bild des
    Containers handelt, und wenn nein, welches die Daten des ersten Bildes
    sind. Also das Manager-Arg. doch nicht so verkehrt. Es sei denn, wir
    lueden erst komplettes Bild und prueften dann.
******************************************************************************/
bool 
cpaint_load_image (const char* fname, br::Br2HdrManager & mBr2Hdr)        
{
  static int            base_W;             // to save the values of the first
  static int            base_H;             //  image
  static gint           base_gray;          // dito
  static GPrecisionType base_precis;        // dito
  static bool           u16_u8_warning = true; // downsampling warning?
  char                  tmp[256];

  printf("%s(\"%s\")...\n", __func__, fname);
  
  int image_ID = cpaint_load_image_via_PDB (fname);
  if (image_ID < 0)  return false;
    
  gint    nlayers  = -1;             // What for?
  gint32* layers   = gimp_image_get_layers (image_ID, &nlayers);
  gint32  layer_ID = layers[0];
  
  GPrecisionType precis = gimp_drawable_precision (layers[0]);
  gint           gray   = gimp_drawable_gray (layers[0]);

  printf ("Image has %d layer(s); layer_ID = %d, precis = %s, gray = %d\n",
      nlayers, layer_ID, cpaint_precision_name(precis), gray);
  
  int W = gimp_image_width (image_ID);
  int H = gimp_image_height (image_ID);
  
  bool ok = false;
  
  //  Tests for every image, be it the first or not...
  if (nlayers > 1) {
    fl_alert ("More than one layer in image\n\nImage \"%s\" ignored", 
        fl_filename_name(fname));
  } 
  else if (nlayers < 1) {
    fl_alert ("No layer in image\n\nImage \"%s\" ignored", 
        fl_filename_name(fname));
  } 
  else if (layers[0] <= 0) {
    fl_alert ("No data found in image\n\nImage \"%s\" ignored", 
        fl_filename_name(fname));
  } 
  else if (gray) {
    fl_alert ("Not an RGB image\n\nImage \"%s\" ignored", 
        fl_filename_name(fname));
  }
  else if (! (precis == PRECISION_U8  ||  
              precis == PRECISION_U16) ) {
    fl_alert ("\"%s\":\n\nSorry, data type '%s' not (yet) supported for input files.\n\nImage \"%s\" ignored",
        fl_filename_name(fname),
        cpaint_precision_name(precis),
        fl_filename_name(fname));
  } 
  //  Now differentiate: first image in container or a following one?
  else if (mBr2Hdr.size() == 0)    // first image  
    {  
      base_W = W;
      base_H = H; 
      base_precis = precis;
      base_gray = gray;
      ok = true;
    }   
  else   // image behind first: check consistency with the first one...
    {
      if (W != base_W || H != base_H) 
        {
          fl_alert ("\"%s\":\n\nDimension of the image to add doesn't fit to the existing image(s):\n\t\"%d x %d\"  vs.  \"%d x %d\".\n\nImage \"%s\" ignored.",
              fl_filename_name(fname), W, H, base_W, base_H,
              fl_filename_name(fname));
        }
      else if ((base_precis != precis) && (precis != PRECISION_U16)) // != U16 weil downgesampelt, veraltet
        {
          fl_alert ("\"%s\":\n\nType of the image to add doesn't fit to the existing image(s):\n\t\"%s\"  vs.  \"%s\".\n\nImage \"%s\" ignored.",
              fl_filename_name(fname),
              cpaint_precision_name(precis),
              cpaint_precision_name(base_precis),
              fl_filename_name(fname));
        } 
      else if (base_gray != gimp_drawable_gray(layers[0]))   // erledigt oben
        {  
          fl_alert ("Other RGB/Gray layout in image.\n\nImage \"%s\" ignored.",
              fl_filename_name(fname));
        } 
      else 
        {
          ok = true;
        }
    }
  
  //  Take over the data..
  if (ok)
    {  
      // If allWinw.main->downsample_U16()==true we offer downsampling to U8
      //  data. To note, that we *must not* scale down each image individually
      //  onto the full 8-bit range, because this would change the relation 
      //  between the various input images. We can only cut "stupidly" to the
      //  8 leading bits. Therefore also converting of single float images to
      //  integers makes no sense, because no reasonable min-max-borders exist.
      //  Would make sense indeed for a *complete* set of float images.
      
      br::BrImage img;   // Null-Image as start
    
      switch (precis)
        {
        case PRECISION_U8:  
          //printf ("uint8\n"); 
          img = cpaint_get_BrImage_U8 (W,H, layer_ID);
          break;
    
        case PRECISION_U16:  
          //printf ("uint16\n"); 
          if (allWins.main->downsample_U16()) {
            if (u16_u8_warning) 
            {
              snprintf(tmp, 256, "\"%s\":  16-bit input data\n\nDownsampling to 8-bit will happen",
                  fl_filename_name(fname));
              int choice = fl_choice (tmp, "Abbort", "Downsample", "No more warn");
              if (choice == 0)        // Abbort
                {      
                  ok = false; 
                  break; 
                }  
              else if (choice == 2)   // "Do no more warn"      
                { 
                  u16_u8_warning = false;
                }  
            }
            img = u8_from_u16 (cpaint_get_BrImage_U16 (W,H, layer_ID));
            //  If it is the first image, also base_precis is now U8
            if (mBr2Hdr.size()==0) base_precis = PRECISION_U8;
          }
          else 
            img = cpaint_get_BrImage_U16 (W,H, layer_ID);            
          break;
    
        case PRECISION_FLOAT:  // currently unreachable
          printf ("f32\n");  ok = false;
          break;
    
        default:  
          //  Might never be reached if tests above are complete
          printf ("\"%s\":  Not supported data type %d: '%s'\n", 
              fl_filename_name(fname), precis, 
              cpaint_precision_name(precis));
          ok = false;
          break;
        }

      // cpaint_get_BrImage() or u8_from_u16() could have returned Null-images:
      ok = !img.is_null();  
              
      //  Set some meta data and add the image to the Br2HdrManager
      if (ok) 
        { 
          img.name (fname);
          img.set_time (1.0);       // z.Z. nichts Besseres
          mBr2Hdr.add_Image (img);
          //mBr2Hdr.report_Images();
        }
    } 
  
  //  Free the image in CinePaint
  gimp_image_delete (image_ID);
  gimp_image_clean_all (image_ID);
  free (layers); 
  
  return ok;
}


/**+*************************************************************************\n
  cpaint_load_image_via_PDB()  --  Helper for cpaint_load_image().

  Asks CPaint for loading the file `fname' and returns its image_ID.
   Function wraps the PDB and "return_vals" stuff.

  @param fname: name of file the CPaint image_ID is asked for
  @return image_ID  (-1 for failure).
******************************************************************************/
int 
cpaint_load_image_via_PDB (const char* fname)
{
  int         image_ID;
  int         n_retvals;
  GimpParam * return_vals;
  
  return_vals = gimp_run_procedure ("gimp_file_load",
                                    & n_retvals,
                                    GIMP_PDB_INT32, GIMP_RUN_INTERACTIVE,
                                    GIMP_PDB_STRING, fname,
                                    GIMP_PDB_STRING, fname,
                                    GIMP_PDB_END);
  
  if (return_vals[0].data.d_status != GIMP_PDB_SUCCESS)
    image_ID = -1;
  else
    image_ID = return_vals[1].data.d_image;
  
  gimp_destroy_params (return_vals, n_retvals);
  return image_ID;
}


/**+*************************************************************************\n
  cpaint_get_BrImage_U8()  --  Helper for cpaint_load_image()

  Allocates a BrImage and fills-in the CPaint image buffer related to 
   `layer_ID'. A possible alpha channel in CPaints image buffer gets removed. 
   Default settings for name and exposure time and all other meta data.
   Type dependence (U8, U16...) concerns the template copy_without_alpha<>.
   Could be done also by a br::ImageType parameter and a switch statement.

  @param W,H: width, height of the layer
  @param layer_ID: as got by cpaint_load_image(). Data type must be U8!
  @return  An allocated and filled BrImage of type RGB_U8
******************************************************************************/
br::BrImage 
cpaint_get_BrImage_U8 (int W, int H, gint32 layer_ID)
{
  //  Allocate the BrImage dest structure.   bad_alloc() possibly!
  br::BrImage  img (W, H, br::IMAGE_RGB_U8);
  
  //  Init the CPaint layer for copying
  GDrawable* drawable = gimp_drawable_get (layer_ID);
  GPixelRgn  pixel_rgn;
  gimp_pixel_rgn_init (&pixel_rgn, drawable, 0,0, W,H, false,false);

  if (gimp_drawable_has_alpha (layer_ID))     
    { 
      //  Have alpha: re-copy
      copy_without_alpha <guint8> (img, pixel_rgn);
    }
  else
    {
      //  No alpha -- fill-in directly into img.buffer()
      gimp_pixel_rgn_get_rect (&pixel_rgn, img.buffer(), 0,0, W,H); DBG
    }    
  
  return img;
}


/**+*************************************************************************\n
  cpaint_get_BrImage_U16()  --  Helper for cpaint_load_image()

  See cpaint_get_BrImage_U8().
******************************************************************************/
br::BrImage 
cpaint_get_BrImage_U16 (int W, int H, gint32 layer_ID)
{
  //  Allocate BrImage dest structure.   bad_alloc() possibly!
  br::BrImage  img (W, H, br::IMAGE_RGB_U16);
  
  //  Init the CPaint pixel region for copying
  GDrawable* drawable = gimp_drawable_get (layer_ID);
  GPixelRgn  pixel_rgn;
  gimp_pixel_rgn_init (&pixel_rgn, drawable, 0,0, W,H, false,false);

  if (gimp_drawable_has_alpha (layer_ID))     
    { 
      //  Have alpha: re-copy  
      copy_without_alpha <guint16> (img, pixel_rgn);
    }
  else  
    { //  No alpha: fill-in directly into img.buffer()
      gimp_pixel_rgn_get_rect (&pixel_rgn, img.buffer(), 0,0, W,H); DBG
    }    
    
  return img;
}

/**+*************************************************************************\n
  copy_without_alpha<>()  --  helper of cpaint_get_BrImage()

  @param <T>: type of the image data (guint8, guint16, ...)
  @param img: RGB<T> br::Image. Extern allocated. The destination.
  @param rgn: CPaint pixel region of RGBA<T> data of same dimension as img.
               The source.
  Copies the data of `rgn' into `img' under removing the alpha channel. 
   img.width() and img.height() must be the dimensions also of `rgn'!
******************************************************************************/
template <typename T>
void 
copy_without_alpha (br::Image & img, GPixelRgn & rgn)
{
    printf("Remove alpha channel...\n");
    int W = img.width();
    int H = img.height();
    Rgba<T>* buf = (Rgba<T>*) malloc (W * H * sizeof(Rgba<T>));

    gimp_pixel_rgn_get_rect (&rgn, (guchar*)buf, 0,0, W,H);
    
    Rgb<T>* imgbuf = (Rgb<T>*) img.buffer();
    
    for (int i=0; i < W*H; i++)
      {
        imgbuf[i].r = buf[i].r;
        imgbuf[i].g = buf[i].g;
        imgbuf[i].b = buf[i].b;    
      }  
    free (buf);
}


/**+*************************************************************************\n
  cpaint_show_rgb_imgbuf()  --  Plot of an RGB buffer without alpha.

  @param W,H: image dimensions width and height
  @param imgbuf: contiguous (W x H)-buffer of RGB data (no alpha)
  @param prec: data type (CinePaint enum)

  Intended for CPaint plot of an RGB-BrImage or an Array2D< Rgb<T> > array.
  For br::Image and TNT::Array2D: width==dim2, height==dim1. Array2D example:

  cpaint_show_rgb_imgbuf (A.dim2(), A.dim1(), (uchar*)A[0], PRECISION_FLOAT);
******************************************************************************/
void 
cpaint_show_rgb_imgbuf (int W, int H, const uchar* imgbuf, GPrecisionType prec)
{
  gint32        image_ID, layer_ID;
  GImageType    image_type;
  GDrawableType drawable_type;

  switch (prec) 
    {
    case PRECISION_U8:                        // 8-bit unsigned int
        image_type    = RGB; 
        drawable_type = RGB_IMAGE; break;
  
    case PRECISION_U16:     
        image_type    = U16_RGB; 
        drawable_type = U16_RGB_IMAGE; break;
  
    case PRECISION_FLOAT:                     // 32-bit float
        image_type    = FLOAT_RGB; 
        drawable_type = FLOAT_RGB_IMAGE; break;
  
    case PRECISION_FLOAT16:
        image_type    = FLOAT16_RGB; 
        drawable_type = FLOAT16_RGB_IMAGE; break;
    
    default:
        fl_alert ("%s: Precision %d ('%s') not supported\n",
            PLUGIN_NAME, prec, cpaint_precision_name(prec));
        return;
    }       
  
  image_ID = gimp_image_new ((guint)W, (guint)H, image_type);

  if (image_ID == -1) 
    {
      fl_alert ("%s: Can't create a new image\n%d x %d, image_type %d",
          PLUGIN_NAME, W, H, image_type);
      return;
    }

  layer_ID = gimp_layer_new (image_ID,
                             "HDR bracketing",
                             (guint)W, (guint)H,
                             drawable_type,
                             100.0,          // opacity
                             NORMAL_MODE );
  
  gimp_image_add_layer (image_ID, layer_ID, 0);   // Wozu?

  GDrawable* drawable = gimp_drawable_get (layer_ID);

  GPixelRgn  pixel_rgn;
  gimp_pixel_rgn_init (&pixel_rgn, drawable, 0,0, W,H, false,false);

  gimp_pixel_rgn_set_rect (&pixel_rgn, (guchar*)imgbuf, 0,0, W,H);

  gimp_drawable_flush  (drawable);
  gimp_drawable_detach (drawable);
  gimp_display_new (image_ID);
}


/**+*************************************************************************\n
  cpaint_show_image()
  Typepruefung koennte neuerdings auch an ImageType abgehandelt werden, zumal
   durch CinePaint ohnehin nur RGB-Puffer (STORING_INTERLEAVE) zu verarbeiten
   sind. Momentaner Mangel der Metavariable `ImageType noch, dass zu wenig
   Varianten erfasst und Fehlermeldungen dadurch zu duerftig sind (IMAGE_NONE
   oder IMAGE_UNKNOWN).
******************************************************************************/
void 
cpaint_show_image (const br::Image & img)
{
  IF_FAIL_DO (!img.is_empty(), return);
  IF_FAIL_DO (img.channels() == 3, return);  // no RGB
  IF_FAIL_DO (img.storing() == br::STORING_INTERLEAVE, return);
  
  GPrecisionType precis;
  
  switch (img.data_type())
    {
    case br::DATA_U8:  precis = PRECISION_U8; break;
    case br::DATA_U16: precis = PRECISION_U16; break;
    case br::DATA_F32: precis = PRECISION_FLOAT; break;
    default:           
        NOT_IMPL_DATA_TYPE (img.data_type()); 
        return;
    }
  
  cpaint_show_rgb_imgbuf (img.width(), img.height(), img.buffer(), precis);
}


/**+*************************************************************************\n
  @return  Name of GPrecisionTypes  --  for messages and warnings.
******************************************************************************/
const char* 
cpaint_precision_name (GPrecisionType precis)
{
  switch (precis)
    {
    case PRECISION_NONE:    return "None";
    case PRECISION_U8:      return "8-bit Unsigned Integer"; 
    case PRECISION_U16:     return "16-bit Unsigned Integer";
    case PRECISION_FLOAT:   return "32-bit IEEE Float";
    case PRECISION_FLOAT16: return "16-bit RnH Short Float";
    case PRECISION_BFP:     return "16-bit Fixed Point 0-2.0";
    default:                return "(unknown)";
    };
}



// END OF FILE
