/*
 * ShiftPanel.cpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright 2005  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  ShiftPanel.cpp  --  rudiment
  
  Klasse kommuniziert (auch schon im Konstruktor!) direkt mit der
  Camera-Instanz, referenziert durch den globalen Zeiger `Br.camera()'. 
  Ab heute (02.06.2005) mache ich sie aber "camera()==0" sicher, dh.
  wenn noch keine Camera initialisiert, wird eben nichts ausgegeben.
*/
#include "br_defs.hpp"

#ifdef BR_WITH_SHIFT


#include <cassert>
#include <cstdio>
#include <cstring>

#include <FL/Fl.H>
#include <FL/Fl_Window.H>
#include <FL/Fl_Menu_Bar.H>
#include <FL/Fl_Button.H>
#include <FL/fl_draw.H>
#include <FL/filename.H>        // fl_filename_relative()
#include "Fl_Table_Row.H"

#include "ShiftPanel.hpp"
#include "Br.hpp"               // `Br', SPUR_BR_EVENT()
//#include  "print_event.hpp"   


#if 0
class ShiftDetailTable : public Fl_Table_Row
{
protected:
    void draw_cell(TableContext context,        // table cell drawing
               int R=0, int C=0, int X=0, int Y=0, int W=0, int H=0);
//    void callback(TableContext context,       // callback for table events
//             int R, int C);

public:
    ShiftDetailTable(int x, int y, int w, int h, const char *l=0) 
      : Fl_Table_Row(x,y,w,h,l)
      { CTOR(l); end(); }          // end() - hier und in ShiftPanel()?
    
    ~ShiftDetailTable() { DTOR(label()) }
};

// Handle drawing all cells in table
void ShiftDetailTable::draw_cell(TableContext context, 
                                 int R, int C, int X, int Y, int W, int H)
{
    //printf("draw_cell(): contex=%d, R=%d, C=%d\n", context,R,X);
    static char s[128];
    sprintf(s, "%d/%d", R, C);      // text for each cell
    int RR=0;   // camera-index; != imgVec-index if inactive images!

    switch ( context )
    {
    case CONTEXT_STARTPAGE:
        fl_font(FL_HELVETICA, 16);
        return;

    case CONTEXT_COL_HEADER:
        fl_push_clip(X, Y, W, H);
        {
          fl_draw_box(FL_THIN_UP_BOX, X, Y, W, H, color());
          fl_color(FL_BLACK);
          switch(C) 
          {
            case 0:  fl_draw("Spot",  X, Y, W, H, FL_ALIGN_CENTER); break;
            case 1:  fl_draw("dx, dy",X, Y, W, H, FL_ALIGN_CENTER); break;
            case 2:  fl_draw("Corr",  X, Y, W, H, FL_ALIGN_CENTER); break;
            default: fl_draw(s,       X, Y, W, H, FL_ALIGN_CENTER);
          }
        }
        fl_pop_clip();
        return;

    case CONTEXT_ROW_HEADER:
        fl_push_clip(X, Y, W, H);
        {
          sprintf(s, "%d", R);
          fl_draw_box(FL_THIN_UP_BOX, X, Y, W, H, color());
          fl_color(FL_BLACK);
          fl_draw(s, X, Y, W, H, FL_ALIGN_CENTER);
        }
        fl_pop_clip();
        return;
    
    case CONTEXT_CELL:
        fl_push_clip(X, Y, W, H);
        {
          // BG COLOR
          fl_color( row_selected(R) ? selection_color() : FL_WHITE);
          fl_rectf(X, Y, W, H);

          // TEXT
          Br.imgVec[R].active ? fl_color(FL_BLACK) : fl_color(FL_DARK3);
          switch (C)
          {
          case 0:
              sprintf(s, "%d", R);
              fl_draw(s, X, Y, W, H, FL_ALIGN_CENTER); break;
          case 1:
              if (!Br.imgVec[R].active) break;
              for (int i=0; i < R; i++) if (Br.imgVec[i].active) RR++;
              sprintf(s, "R: %3d, %3d", Br.camera()->channel_shifts[RR].r.x,
                                        Br.camera()->channel_shifts[RR].r.y);
              fl_draw(s, X,Y+H/3);              
              sprintf(s, "G: %3d, %3d", Br.camera()->channel_shifts[RR].g.x,
                                        Br.camera()->channel_shifts[RR].g.y);
              fl_draw(s, X,Y+2*H/3-1);              
              sprintf(s, "B: %3d, %3d", Br.camera()->channel_shifts[RR].b.x,
                                        Br.camera()->channel_shifts[RR].b.y);
              fl_draw(s, X,Y+H-2); break;            
          case 2:
              if (!Br.imgVec[R].active) break;
              for (int i=0; i < R; i++) if (Br.imgVec[i].active) RR++;
              fl_draw("R:", X,Y+H/3);              
              fl_draw("G:", X,Y+2*H/3-1);              
              fl_draw("B:", X,Y+H-2); break;             
          default:
              fl_draw(s, X, Y, W, H, FL_ALIGN_CENTER);
          }

          // BORDER
          fl_color(FL_LIGHT2); 
          fl_rect(X, Y, W, H);
        }
        fl_pop_clip();
        return;

    default:
        return;
    }
}


class ShiftDetailInfo : public Fl_Window
{ 
  ShiftDetailTable* table_;
public:
  ShiftDetailInfo (int W, int H, const char* la=0);
};
  
ShiftDetailInfo::ShiftDetailInfo (int W, int H, const char* la)
  : Fl_Window (W,H,la)
{
  begin();
    table_ = new ShiftDetailTable(0,0,W,H,la);
    table_ -> selection_color(FL_YELLOW);
    table_ -> type(Fl_Table_Row::SELECT_SINGLE);
    table_ -> when(FL_WHEN_RELEASE);   // handle table events on release
    //table_ -> when(FL_WHEN_CHANGED|FL_WHEN_RELEASE);
    table_ -> rows(1);  // number of spots!
    table_ -> cols(3);
    table_ -> col_width(0,40);
    table_ -> col_width(1,80);
    table_ -> col_width(2,60);
    table_ -> row_height_all(60);
    table_ -> col_header(1);       // enable col header
    table_ -> col_resize(4);       // enable col resizing
    table_ -> row_header(0);       // disable row header
    table_ -> row_resize(4);       // enable row resizing
    //table_ -> callback(table_cb, table_);
    //table_ -> callback((Fl_Callback*)cb_table_, this);
    table_ -> end();
  end();
  show();
}


ShiftDetailInfo* pShiftDetailInfo;
#endif

/**==========================================================
 *
 * ShiftTable
 * 
 * Diese Tabelle praesentiert Daten aus den globalen Objekten Br.imgVec und
 * Br.camera. Im Konstruktionsmoment konstruiert sie sich mit soviel Zeilen
 * wie Bilder in Br.imgVec. (Nach)Laden von Bildern erfordert Zeilenzahl
 * anzupassen! 
 * 
 * Diffizile Fehlerquelle ist, dass die Tabelle soviel Zeilen wie
 * imgVec.size() hat (haben soll), andererseits Shift-Daten nur fuer die
 * in Br.camera eingegangenen Bilder vorliegen, dh. nur fuer die zum camera-
 * Init-Zeitpunkt aktivierten. Ob ein imgVec-Bild mit Index `k' in camera
 * eingegangen ist und welchen Index es dort hat, wird durch Br.used_as(k)
 * gegeben. Es liefert einen Wert < 0, wenn nicht benutzt, sonst den Index.
 * 
 * Ich versuche 'mal Bezuege auf eine uninitialisierte camera abzufangen.
 *
 *==========================================================*/
class ShiftTable : public Fl_Table_Row
{
protected:
    void draw_cell(TableContext context,        // table cell drawing
                   int R=0, int C=0, int X=0, int Y=0, int W=0, int H=0);
public:
    ShiftTable(int X, int Y, int W, int H, const char *la=0); 
    ~ShiftTable() { DTOR(label()) }
    
    void        cb_event ();
    static void cb_event_(Fl_Widget*, ShiftTable* I)
      { I -> cb_event(); }
};

/**-------------------
 * Constructor...
 *--------------------*/
ShiftTable::ShiftTable(int X, int Y, int W, int H, const char *la) 
  : Fl_Table_Row (X,Y,W,H, la)
{ 
  CTOR(la); 
  begin();
    selection_color(FL_YELLOW);
    type(Fl_Table_Row::SELECT_SINGLE);
    callback((Fl_Callback*)cb_event_, this);
    when(FL_WHEN_RELEASE);  // handle table events on release
   
    int frame     = Fl::box_dw(box()), // *after* box decision!
        w_name    = 100,
        w_dx      = 40,
        w_dy      = 40,
        w_cov     = 80,
        w_shift;
    
    // fit the last col into the rest of the table (25 is the width
    //   of the activation button)...
    w_shift = W - (25 + w_name + w_dx + w_dy + w_cov) - frame;
    w_shift = (w_shift > 80) ? w_shift : 80;
    
    cols(5);
    col_header(1);          // enable col header
    col_header_height(25);
    col_width(0,w_name);    // filename
    col_width(1,w_dx);      // x-shift
    col_width(2,w_dy);      // y-shift
    col_width(3,w_cov);     // covariance
    col_width(4,w_shift);   
    col_resize(4);          // enable col resizing
    
    rows(Br.imgVec.size()); // number of images
    row_header(1);          // enable row header
    row_header_width(25);   // width of activation button
    row_height_all(25);
    row_resize(4);          // enable row resizing
  end(); 
}         


// Handle drawing all cells in table
void ShiftTable::draw_cell(TableContext context, 
                           int R, int C, int X, int Y, int W, int H)
{
  //printf("draw_cell(): contex=%d, R=%d, C=%d\n", context,R,X);
  static char s[128];
  int RR;   // camera-index: != imgVec-index if deactivated images

  switch ( context )
  {
  case CONTEXT_STARTPAGE:
      fl_font(FL_HELVETICA, 16);
      return;

  case CONTEXT_COL_HEADER:
      fl_push_clip(X, Y, W, H);
      {
        fl_draw_box(FL_THIN_UP_BOX, X, Y, W, H, color());
        fl_color(FL_BLACK);
        switch(C) 
        {
          case 0:  fl_draw("Name",  X, Y, W, H, FL_ALIGN_CENTER); break;
          case 1:  fl_draw("dx",    X, Y, W, H, FL_ALIGN_CENTER); break;
          case 2:  fl_draw("dy",    X, Y, W, H, FL_ALIGN_CENTER); break;
          case 3:  fl_draw("Correl",X, Y, W, H, FL_ALIGN_CENTER); break;
          case 4:  fl_draw("dx,dy", X, Y, W, H, FL_ALIGN_CENTER); break;
          default: sprintf(s, "%d/%d", R, C);  // something forgotten?
                   fl_draw(s,       X, Y, W, H, FL_ALIGN_CENTER);
        }
      }
      fl_pop_clip();
      return;

  case CONTEXT_ROW_HEADER:
      fl_push_clip(X, Y, W, H);
      {
        sprintf(s, "%d", R+1);  // begin with 1, not 0
        if (Br.imgVec.active(R)) fl_draw_box(FL_DOWN_BOX, X,Y,W,H, FL_GREEN);
        else                     fl_draw_box(FL_UP_BOX,   X,Y,W,H, color());
        fl_color(FL_WHITE); 
        fl_draw(s, X,Y,W,H, FL_ALIGN_CENTER);
      }
      fl_pop_clip();
      return;
    
  case CONTEXT_CELL:
      fl_push_clip(X, Y, W, H);
      {
        // BG COLOR
        Fl_Color c = row_selected(R) ? selection_color() : FL_WHITE;
        if (!Br.imgVec.active(R)) c = fl_color_average(c, FL_GRAY, 0.66f);
        draw_box(FL_THIN_UP_BOX, X,Y,W,H, c);

        // TEXT
        Br.imgVec.active(R) ? fl_color(FL_BLACK) : fl_color(fl_inactive(FL_BLACK));
        switch (C)
        {
        case 0:
            fl_filename_relative(s, Br.imgVec[R].name());
            fl_draw(s, X+3, Y, W, H, FL_ALIGN_LEFT); 
            break;
            
        case 1:
            if (!Br.camera()) break;
            if ((RR = Br.used_as(R)) < 0) break;
            printf("R=%d, RR=%d\n", R, RR);
            snprintf(s, 128, "%d", Br.camera()->x_shift(RR));
            fl_draw(s, X, Y, W, H, FL_ALIGN_CENTER); 
            break;
        
        case 2:
            if (!Br.camera()) break;
            if ((RR = Br.used_as(R)) < 0) break;
            snprintf(s, 128, "%d", Br.camera()->y_shift(RR));
            fl_draw(s, X, Y, W, H, FL_ALIGN_CENTER); 
            break;
            
        case 3:
            if ((RR = Br.used_as(R)) < 0) break;
            fl_font(FL_HELVETICA, 13);
            fl_draw("R:", X,Y+H/3);              
            fl_draw("G:", X,Y+2*H/3-1);              
            fl_draw("B:", X,Y+H-2);              
            fl_font(FL_HELVETICA, 16); 
            break;
        
        case 4:
            if (!Br.camera()) break;
            if ((RR = Br.used_as(R)) < 0) break;
            fl_font(FL_HELVETICA, 13);
            sprintf(s, "%3d, %3d", Br.camera()->channel_shifts[RR].r.x,
                                   Br.camera()->channel_shifts[RR].r.y);
            fl_draw(s, X,Y+H/3);              
            sprintf(s, "%3d, %3d", Br.camera()->channel_shifts[RR].g.x,
                                   Br.camera()->channel_shifts[RR].g.y);
            fl_draw(s, X,Y+2*H/3-1);              
            sprintf(s, "%3d, %3d", Br.camera()->channel_shifts[RR].b.x,
                                   Br.camera()->channel_shifts[RR].b.y);
            fl_draw(s, X,Y+H-2);              
            fl_font(FL_HELVETICA, 16); 
            break;
              
        default:
            sprintf(s, "%d/%d", R, C);    // something forgotten?
            fl_draw(s, X, Y, W, H, FL_ALIGN_CENTER);
        }

        // BORDER  (Example)
        //fl_color(FL_LIGHT2); 
        //fl_rect(X, Y, W, H);
      }
      fl_pop_clip();
      return;

  case CONTEXT_ENDPAGE:
      // draw a box in the "headers corner"; X,Y are in STARTPAGE/ENDPAGE
      //   those of the data table *without* headers, x() and y() are the
      //   outer coords, we need wix, wiy, the (protected) inner coords of
      //   the widget without border (same as X - row_header_width(), 
      //   Y - col_header_height))...
      draw_box(FL_EMBOSSED_BOX, wix, wiy, row_header_width(), col_header_height(), color());
      return;

  default:
      return;
  }
}

/**---------------------------
 * cb_event()  --  callback 
 *----------------------------*/
void ShiftTable::cb_event()
{
  TableContext context = callback_context();
  int R = callback_row();
  //int C = callback_col();
  switch (Fl::event())
  {
  case FL_RELEASE:
      switch (context)
      {
      case CONTEXT_ROW_HEADER:
          if (Br.imgVec.active(R)) Br.deactivate(R);
          else                     Br.activate(R);
          // callback by default already redraws inner table, 
          //   *would* be nessecary only redrawing of row header
          redraw(); 
          break;
      
      default:
          break;
      }
  
  default: ;
  }
}


// typedef enum { SPOT_ONE, SPOT_2x2, SPOT_4_1, SPOT_3x3 }  SPOT_MODE;
// 
// struct { const char* text; SPOT_MODE mode; }
// menuMap_spots[] = {
//   {"1",   SPOT_ONE},
//   {"2x2", SPOT_2x2},
//   {"4+1", SPOT_4_1},
//   {"3x3", SPOT_3x3}
// };

/**================================================================
 *
 * ShiftPanel  -  class 
 *
 *=================================================================*/
//--------------------------------------------------
// menu_menubar (definition of an static element)...  
//--------------------------------------------------
// Fl_Menu_Item ShiftPanel::menu_menubar_[] = {
//   {"&File",  0,0,0, FL_SUBMENU},
//     {"Load (fut)", 0,0,0, FL_MENU_INACTIVE},
//     {"Save (fut)", 0,0,0, FL_MENU_INACTIVE},
//     {"&Close", FL_ALT+'c', (Fl_Callback*)ShiftPanel::cb_close_},
//     {0},
//   {"Spots", 0,0,0, FL_SUBMENU},
//     {menuMap_spots[0].text},
//     {menuMap_spots[1].text},
//     {menuMap_spots[2].text},
//     {menuMap_spots[3].text},
//     {0},
//   {"Compute", 0, (Fl_Callback*)ShiftPanel::cb_compute_},
//   {0}
// };

/**----------------------------------------------------------------
 * Constructor... 
 *   Not end()ed here, so user can simply add() more while creation.
 *   I.e. -- in other words -- user *must* end() it.
 *-----------------------------------------------------------------*/
ShiftPanel::ShiftPanel(int X, int Y, int W, int H, const char* la)
  : Fl_Group (X,Y,W,H, la)
{ 
  CTOR(la)
  
  // in Distributor(en) einloggen...
  Br.distrib_event.login ((Callback*)event_msg_, this);
  
  int y_table  = Y + 11,
      y_butt   = Y + H - 95,
      dy_table = y_butt - y_table - 10;
  
  //begin();
  table_ = new ShiftTable (X+5, y_table, W-10, dy_table); DB_
        
  input_nx_ = new Fl_Int_Input(X+30, Y+H-70, 40, 25, "nx");
  input_nx_ -> callback((Fl_Callback*)cb_input_nx_, this);
    
  input_ny_ = new Fl_Int_Input(X+30, Y+H-40, 40, 25, "ny");
  input_ny_ -> callback((Fl_Callback*)cb_input_ny_, this);
    
  input_mx_ = new Fl_Int_Input(X+120, Y+H-70, 40, 25, "mx");
  input_mx_ -> callback((Fl_Callback*)cb_input_mx_, this);
    
  input_my_ = new Fl_Int_Input(X+120, Y+H-40, 40, 25, "my");
  input_my_ -> callback((Fl_Callback*)cb_input_my_, this);
    
  Fl_Button* b = new Fl_Button (X+270, Y+H-55, 80, 25, "Compute");
  //b -> box(FL_UP_BOX);
  b -> callback((Fl_Callback*)cb_compute_, this);
  
  out_nm_values(); DB_
  
  //callback((Fl_Callback*)cb_window_, this);
  resizable(table_);
  
  //end();  --  to end() by user!
}

/**----------------
 * Destructor...
 *-----------------*/
ShiftPanel::~ShiftPanel() 
{ 
  DTOR(label())
  // aus Distributor(en) ausloggen...
  Br.distrib_event.logout(this); 
}


/**----------------------------------
 * Callback for the "Compute" item...
 *-----------------------------------*/
void ShiftPanel::cb_compute (Fl_Menu_*) 
{
  printf("Compute...\n");
  Br.camera() -> calc_shifts();
  redraw();
//   if (!pShiftDetailInfo) 
//     pShiftDetailInfo = new ShiftDetailInfo(300,200,"Details");
}

/**----------------------------------------------
 * Callback for the n*-m*-inputs... 
 *-----------------------------------------------*/
void ShiftPanel::cb_input_nx (Fl_Int_Input* in) 
{
  int res = atoi(in->value());  
  Br.camera()-> shift_nx = res;     // Wertebereich pruefen?
  //out_nm_values(res);   // in Menus eintragen
}
void ShiftPanel::cb_input_ny (Fl_Int_Input* in) 
{
  int res = atoi(in->value());
  Br.camera()-> shift_ny = res;
  //out_nm_values(res);   // in Menus eintragen
}
void ShiftPanel::cb_input_mx (Fl_Int_Input* in) 
{
  int res = atoi(in->value());
  Br.camera()-> shift_mx = res;
  //out_nm_values(res);   // in Menus eintragen
}
void ShiftPanel::cb_input_my (Fl_Int_Input* in) 
{
  int res = atoi(in->value());
  Br.camera()-> shift_my = res;
  //out_nm_values(res);   // in Menus eintragen
}

void ShiftPanel::out_nm_values()
{
  if (!Br.camera()) return; 
  char s[32];
  snprintf(s, 32, "%d", Br.camera()->shift_nx); input_nx_-> value(s);
  snprintf(s, 32, "%d", Br.camera()->shift_ny); input_ny_-> value(s);
  snprintf(s, 32, "%d", Br.camera()->shift_mx); input_mx_-> value(s);
  snprintf(s, 32, "%d", Br.camera()->shift_my); input_my_-> value(s);
}


/**-----------------------------------------------------------
 * event_msg()  --  Distributor callback
 *
 * ShiftPanel (used as a tab) shall be inactive for camera==0, and
 *   active for camera!=0. So for CAMERA_INIT activate it, if it
 *   was inactive before, for CAMERA_DELETED deactivate it.
 *
 * Weil hier sowohl imgVec- als auch camera-Daten dazustellen (Zeilenzahl
 *   ==imgVec.size(), aber nur die davon in camera haben volle Zeile),
 *   sind hier IMAGES_CHANGED wie CAMERA_INIT Events zu verarbeiten.
 *   Weil z.Z. nach jedem Bildladen (--> IMAGES_CHANGED bzw. IMAGE_LOADED) 
 *   automatisch auch ein init_camera(), kommt dann hier auch jedesmal noch
 *   ein CAMERA_INIT an. Insofern auf beide Ereignisse nahezu identisch
 *   nur zu reagieren ist, wird bei einem Bildladen das Widget quasi
 *   zweimal redraw()ed. Schlimm?
 * Besser die Situation bei einem Br.clear(): Br.clear_camera() produziert
 *   CAMERA_DELETED und Br.clear_container() IMAGES_CHANGED. Auf
 *   ersters ist Widget inaktiv zu schalten, auf zweiters fuer 0 Zeilen
 *   darzustellen. Da ueberschneidet sich nichts.
 * Spaeter kommen die Shift-Sachen raus aus Camera, dann muessen dafuer eigene
 *   Ereignisse her.
 *------------------------------------------------------------*/
void ShiftPanel::event_msg (BracketingCore::Event e)
{
  SPUR_BR_EVENT(("ShiftPanel::%s(%d): ", __func__, e));
  switch (e)
  {
  case BracketingCore::CAMERA_INIT:
      SPUR_BR_EVENT(("%s\n", br_eventnames[e]));  
      if (!active()) activate();
      // Update camera dependent output...
      update();     // currently: outputs the n-m-values
      table()-> redraw(); // e.g. empty lines for images unused in camera
      break;
     
  case BracketingCore::CAMERA_DELETED:
      SPUR_BR_EVENT(("%s\n", br_eventnames[e]));  
      deactivate();
      break;
      
   case BracketingCore::IMAGES_CHANGED:
   case BracketingCore::IMAGE_LOADED:
      SPUR_BR_EVENT(("%s\n", br_eventnames[e]));  
      table()-> rows (Br.size()); // new row number 
      table()-> redraw();
      break;    
           
  default:
      SPUR_BR_EVENT(("not handled\n"));
  }    
}

#endif // BR_WITH_SHIFT

// END OF FILE
