/*
 * Br_Image.hpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright 2005  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
 * Br_Image.hpp 
 * 
 * Content:
 *   enum BR_PRECISION
 *   class Br_Image
 *   class Br_ImageVec
 */
#ifndef Br_Image_hpp
#define Br_Image_hpp


#include <cstdlib>      // free()
#include <vector>

//#define YS_DEBUG
#undef YS_DEBUG
#include "YS/ys_dbg.hpp"


typedef enum 
{ 
  BR_PRECISION_NONE  = 0,
  BR_PRECISION_U8    = 1,
  BR_PRECISION_U16   = 2,
  BR_PRECISION_U32   = 3,
  BR_PRECISION_U64   = 4,
  BR_PRECISION_FL16  = 5,
  BR_PRECISION_FL32  = 6,
  BR_PRECISION_FL64  = 7,
  BR_PRECISION_FIX16 = 8,
  BR_PRECISION_FIX32 = 9,
  BR_PRECISION_UNKNOWN = 10
}
Br_Precision;

const char* const BR_PRECISION_SHORT_STR[] =
{ 
  "None",
  "U8",
  "U16",
  "U32",
  "U64",
  "Fl-16",
  "Fl-32",
  "Fl-64"
  "Fix16",
  "Fix32",
  "[Unknown]"
}; 

const char* const BR_PRECISION_LONG_STR[] =
{ 
  "None",
  "8-bit Unsigned Integer",
  "16-bit Unsigned Integer",
  "32-bit Unsigned Integer",
  "64-bit Unsigned Integer",
  "16-bit RnH Short Float",
  "32-bit IEEE Float",
  "64-bit IEEE Float"
  "16-bit Fixed Point 0-2.0",
  "Fix32",
  "[Unknown]"
}; 


/**=======================================================================
 * Br_Image  --  class  (Struktur fuer Bracketing-relevante Bild-Metadaten)
 * 
 * Klasse, die den C-String kopiert. Koennte man sich sparen durch Uebergang 
 *   zu STL-Strings.
 * Welche Begruendung fuer Wahl von float und double?
 *=======================================================================*/
class Br_Image 
{
  char* name_; 
  int   width_;
  int   height_;

public:
  float  time;      // spaeter vielleicht auch via Funktion
  float  aperture;  // Blende, not yet used
  float  ISO;       // not yet used
  bool   active;
  Br_Precision precision;
  double brightness;
  int    n_low;     // number of points at the lower bound    
  int    n_high;    // number of points at the upper bound
  double r_wrange;  // ratio of points within working range
                    // r_wrange = (n_low + n_high) / (width*height)
    
  Br_Image (const Br_Image & img)       // copy-CTOR
    { SPUR(("Copy-Constr (\"%s\")\n", img.name())); 
      copy_(img); 
    }
  Br_Image (const char* fn    = 0,      // init-CTOR
            int w             = 0, 
            int h             = 0, 
            Br_Precision prec = BR_PRECISION_NONE, 
            float t           = 0, 
            bool act          = false);
  
  ~Br_Image ()                      
    { SPUR(("Destructor: ")); // + message of free_()
      free_(); 
    }
  
  Br_Image & operator= (const Br_Image & img); 
  
  int         width()           const  { return width_;  }
  int         height()          const  { return height_; }
  const char* name()            const  { return name_;   }
  void        name (const char*);
  void        list()            const;
  
private:
  void copy_(const Br_Image & img);
  void free_() 
    { SPUR(("free_(): \"%s\"\n",name_));
      if (name_) free(name_); 
    }
};


/**======================================================================
 * Br_ImageVec  -  Container of Br_Image's
 *
 * NOTE: std::vector kopiert seine Objekte mittels deren Kopierkonstruktoren 
 *   auf den Heap. Wuerde bei C-Strings also nur fuer statische funktionieren,
 *   da nur die Adressen kopiert wuerden. Fuer nichtstatische ist Ueberladen
 *   des Kopierkonstruktors von Br_Image erforderlich -- oder mit STL-
 *   Strings arbeiten. -- Ueberladen inzwischen erfolgt.
 *=======================================================================*/
class Br_ImageVec : public std::vector<Br_Image>
{
public:
  Br_ImageVec() {}         // -> default std::vector constructor
  
  void activate(int i)     { this-> at(i).active = true;  }
  void deactivate(int i)   { this-> at(i).active = false; } 
  bool active(int i) const { return this-> at(i).active;  }
  int  size_active() const;  
  void list()        const;
  void set_times_by_stops(double stops);
};


#endif      // Br_Image_cpp

// END OF FILE
