/*
 * Br.cpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright 2005  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  Br.cpp 
  
  Die beiden virtuellen Funktionen `cursor_wait()' und `cursor_default()' 
    hier angesiedelt, weil hier naeher an den Rechnungen und also besserer
    Einblick, wann etwas "dauert". Die GUI sollte diese Funktionen dann
    passend definieren, vorzugsweise durch fl_cursor(FL_CURSOR_WAIT) und
    fl_cursor(FL_CURSOR_DEFAULT).
  Doch Deine Unentschlossenheit in der `Br'-`BrGUI'-Sache bringt Dich immer 
    mehr Verlegenheit. Weil noch immer mit der globalen Instanz `Br' 
    gearbeitet wird, muessen diese Funktionen schon hier definiert werden. 
    Das bringt hier GUI-Abhaengigkeiten hinein (FLTK-Header), die diese Ebene
    eigentlich nicht haben soll.
  BUG: Das fl_cursor(FL_WAIT) hat in compute_FollowCurve() keinen Effekt,
    erst mit Fl::flush(). No idea. Deshalb das Fl::flush() in cursor_wait().
*/
#include <FL/fl_draw.H>         // fl_cursor()  (temporary)
#include <FL/Fl.H>              // Fl::flush()  (temporary)

#include "Br.hpp"
#include "ProgressInfo.hpp"     // global `__progressinfo' (temporary)


const char* br_eventnames[] = 
{   "NO_EVENT",
    "CAMERA_INIT",
    "CAMERA_OUTDATED",
    "CAMERA_REDATED",
    "CAMERA_DELETED",
    "IMAGE_LOADED",
    "IMAGES_CHANGED",
    "CCD_UPDATED",
    "CCD_OUTDATED",
    "CCD_DELETED",
    "HDR_UPDATED"
}; 

/**====================================================================
 *
 * BracketingCore  --  class 
 *
 * Collects the GUI-independent globals: dataVec, imgVec, camera. 
 *
 *=====================================================================*/

/// Ctor...
BracketingCore::BracketingCore ()      
{   
    // ...default Ctors for imgVec and dataVec     
    
    camera8 = 0;
    camera_uptodate_ = false;
    
    // defaults for numeric parameters...
    refpic_        = 0;
    last_refpic_   = -1;   // mark "no FollowUpCurves computation till now"     
    smoothing_     = 20.0;
    n_grid_points_ = 50;
}

/**---------------------------------------------------------------
 * init_camera()...
 *
 * Free the existing camera instance (if one) and create a new one 
 *   initialized with the data from `imgVec' and `dataVec_u8'.
 * Secure, that Br. after that is in a safe state, i.e. bring all
 *   remaining numerical parameters in Br., which are related to camera,
 *   in consistancy with the new camera. For instance `refpic_'.
 * Currently Camera-Ctor abborts (via assert()) if size_active<2. 
 *   Abfangen entweder hier oder in UI.
 *----------------------------------------------------------------*/
void BracketingCore::init_camera()      
{
    clear_camera();
    camera8 = new Camera<uint8,double> (imgVec, dataVec_u8, 8, __progressinfo); 
    
    // Reset some status variables...
    camera_uptodate_ = true;
    last_refpic_ = -1;    // mark for "no FollowCurves computation till now"
    
    // Secure consistancy of the camera related numerical values...
    if (refpic_ >= camera()->nImages()) 
      refpic_ = camera()->nImages() - 1;
    
    // Broadcast the "camera init" message...
    distrib_event.value (CAMERA_INIT);
    //distrib_event.value (CCD_OUTDATED); // we say: subsumed by CAMERA_INIT
}

void BracketingCore::clear_camera()      
{
    delete camera8;  
    camera8 = 0;
    distrib_event.value (CAMERA_DELETED);
    //distrib_event.value (CCD_DELETED);   // subsumed by CAMERA_DELETED
}

void BracketingCore::clear_container()      
{ 
    imgVec.clear(); 
    dataVec_u8.clear(); 
    distrib_event.value (IMAGES_CHANGED);
}

void BracketingCore::activate (int pic)
{
    imgVec.activate (pic);
    camera_uptodate_ = false; 
    distrib_event.value (CAMERA_OUTDATED);
      // up-to-date valuation open for more intelligence, e.g. a check,
      //  whether the activated ones coincide again this those in camera
}

void BracketingCore::deactivate (int pic)
{
    imgVec.deactivate (pic);
    camera_uptodate_ = false; 
    distrib_event.value (CAMERA_OUTDATED);
      // up-to-date valuation open for more intelligence, see above
}

/**--------------------------------------------------------------
 * used_as()...
 * 
 * @param k: index of an imgVec-image
 * @return: corresponding index in camera if used, -1 if 'not used'
 *   or 'k out of range' or 'cammera not initialized'
 *---------------------------------------------------------------*/
int BracketingCore::used_as (int k) const
{
    //printf("%s(k=%d)...\n", __func__, k);
    if (!camera() || k < 0 || size_t(k) >= size()) return -1;
    for (int i=0; i < camera()->nImages(); i++)
    {
      //printf("array_addr=%p   dataVec=%p\n", camera()->array_addr(i),          
      //    dataVec_u8[k][0]);
      if (camera()->array_addr(i) == dataVec_u8[k][0]) return i;
    }
    return -1;
}

/**--------------------------------------------------------------
 * used()...
 *
 * @param k: index of an imgVec-image
 * @return: true if used in camera, else false
 *---------------------------------------------------------------*/
bool BracketingCore::used (int k) const
{
    return (used_as(k) >= 0);
}

/**--------------------------------------------------------------
 * imgVec_index()
 *
 * @param k: index of an image in Camera
 * @return: corresponding index in Br.imgVec
 *---------------------------------------------------------------*/
int BracketingCore::imgVec_index (int k) const
{
    for (int i=0; i < (int)imgVec.size(); i++)
      if (camera()->array_addr(k) == dataVec_u8[i][0]) return i;
      
    return -1;
}

/**--------------------------------------------------------------
 * refpic()  --  Set `refpic_' and broadcast the change
 *   What's about a range check?
 *---------------------------------------------------------------*/
void BracketingCore::refpic (int pic) 
{  
    if (pic != refpic_)
    {
      refpic_ = pic;
      distrib_refpic.value (pic);
      distrib_event.value (CCD_OUTDATED);
    }
}

/**--------------------------------------------------------------
 * update_camera_times()  --  Updates times in `camera' with those of `imgVec' 
 *
 * TODO: Changing of times outdates possibly existing CCD curves in
 *   `camera'. What do? Removing the curves? Sending an CCD_OUTDATED?
 *---------------------------------------------------------------*/
void BracketingCore::update_camera_times () 
{  
    if (!camera()) return;
    for (int i=0; i < (int)size(); i++) {
       int k = used_as (i);
       if (k >= 0) {
         camera()-> times   [k] = imgVec[i].time;
         camera()-> logtimes[k] = log( imgVec[i].time );
       }
    }
    distrib_event.value (CCD_OUTDATED);
}

/**--------------------------------------------------------------
 * compute_FollowCurves() 
 *
 * Computes follow-up curves for the current `refpic_' value, but only
 *   if it has not be done before. Further a wrapper for the dubios
 *   circumstances in Camera (e.g. "z_average()" surely not the
 *   final name for that purpose).
 *---------------------------------------------------------------*/
void BracketingCore::compute_FollowCurves ()
{
    if (!camera()) return;
    printf("Br::%s(): refpic=%d, last_refpic=%d\n", __func__, refpic_, last_refpic_);
    if (refpic_ != last_refpic_) 
    {
      cursor_wait();
      
      camera()-> z_average (refpic_);
      last_refpic_ = refpic_;
      
      cursor_default();
    }
}

/**--------------------------------------------------------------
 * make_CCD_curve() 
 *---------------------------------------------------------------*/
void BracketingCore::make_CCD_curve ()
{
    cursor_wait();
    
    camera()-> make_CCD_curve (refpic_, n_grid_points_, smoothing_);
    distrib_event.value (CCD_UPDATED);
    
    cursor_default();
}

/**--------------------------------------------------------------
 * calc_HDR() 
 *
 * Uses existing CCD curves, NEEDS it, crashes otherwise! Provided
 *   only for the abstraction "camera.calc_HDR() + Message". Ok?
 *---------------------------------------------------------------*/
Array2D<Rgb<float> >
BracketingCore::calc_HDR ()
{
    cursor_wait();
    
    Array2D<Rgb<float> > A = camera()-> calc_HDR();
    distrib_event.value (HDR_UPDATED);
    
    cursor_default();
    return A;
}

/**--------------------------------------------------------------
 * make_HDR()  --  init() + CCD() + HDR()
 *---------------------------------------------------------------*/
Array2D<Rgb<float> >
BracketingCore::make_HDR ()
{
    init_camera();      // --> CAMERA_INIT
    make_CCD_curve();   // --> CCD_UPDATED
    return calc_HDR();  // --> HDR_UPDATED
};

/**--------------------------------------------------------------
 * complete_HDR()  --  calc all if CCD not exists, else HDR only.
 *
 * Currently not used in the GUI layer as the CCD curve GUI-update
 *   not yet done locally but explicitely in BrGUI::make_CCD_curve().
 *---------------------------------------------------------------*/
Array2D<Rgb<float> >
BracketingCore::complete_HDR ()
{
    if (!camera()-> ccd_curves_ready()) 
      make_CCD_curve();  // --> CCD_UPDATED
    
    return calc_HDR();   // --> HDR_UPDATED
};
 

void BracketingCore::cursor_wait()
{
    fl_cursor(FL_CURSOR_WAIT); 
    Fl::flush();
}
void BracketingCore::cursor_default()
{
    fl_cursor(FL_CURSOR_DEFAULT);
}


// the singular global instance we name "Br"...
BracketingCore Br;       // default Ctor


// END OF FILE
