// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ui/ozone/platform/drm/host/drm_native_display_delegate.h"

#include <stdio.h>

#include "base/logging.h"
#include "base/thread_task_runner_handle.h"
#include "base/threading/thread_restrictions.h"
#include "base/threading/worker_pool.h"
#include "ui/display/types/display_snapshot.h"
#include "ui/display/types/native_display_observer.h"
#include "ui/events/ozone/device/device_event.h"
#include "ui/events/ozone/device/device_manager.h"
#include "ui/ozone/common/display_snapshot_proxy.h"
#include "ui/ozone/common/display_util.h"
#include "ui/ozone/common/gpu/ozone_gpu_messages.h"
#include "ui/ozone/platform/drm/host/display_manager.h"
#include "ui/ozone/platform/drm/host/drm_gpu_platform_support_host.h"

namespace ui {

namespace {

typedef base::Callback<void(const base::FilePath&, base::File)>
    OnOpenDeviceReplyCallback;

void OpenDeviceOnWorkerThread(
    const base::FilePath& path,
    const scoped_refptr<base::TaskRunner>& reply_runner,
    const OnOpenDeviceReplyCallback& callback) {
  base::File file(path, base::File::FLAG_OPEN | base::File::FLAG_READ |
                            base::File::FLAG_WRITE);

  base::File::Info info;
  file.GetInfo(&info);

  CHECK(!info.is_directory);
  CHECK(path.DirName() == base::FilePath("/dev/dri"));

  if (file.IsValid()) {
    reply_runner->PostTask(
        FROM_HERE, base::Bind(callback, path, base::Passed(file.Pass())));
  }
}

class DrmDisplaySnapshotProxy : public DisplaySnapshotProxy {
 public:
  DrmDisplaySnapshotProxy(const DisplaySnapshot_Params& params,
                          DisplayManager* display_manager)
      : DisplaySnapshotProxy(params), display_manager_(display_manager) {
    display_manager_->RegisterDisplay(this);
  }

  ~DrmDisplaySnapshotProxy() override {
    display_manager_->UnregisterDisplay(this);
  }

 private:
  DisplayManager* display_manager_;  // Not owned.

  DISALLOW_COPY_AND_ASSIGN(DrmDisplaySnapshotProxy);
};

}  // namespace

DrmNativeDisplayDelegate::DrmNativeDisplayDelegate(
    DrmGpuPlatformSupportHost* proxy,
    DeviceManager* device_manager,
    DisplayManager* display_manager,
    const base::FilePath& primary_graphics_card_path)
    : proxy_(proxy),
      device_manager_(device_manager),
      display_manager_(display_manager),
      primary_graphics_card_path_(primary_graphics_card_path),
      has_dummy_display_(false),
      weak_ptr_factory_(this) {
  proxy_->RegisterHandler(this);
}

DrmNativeDisplayDelegate::~DrmNativeDisplayDelegate() {
  device_manager_->RemoveObserver(this);
  proxy_->UnregisterHandler(this);
}

void DrmNativeDisplayDelegate::Initialize() {
  device_manager_->AddObserver(this);
  device_manager_->ScanDevices(this);

  if (!displays_.empty())
    return;
  DisplaySnapshot_Params params;
  bool success = false;
  {
    // The file generated by frecon that contains EDID for the 1st display.
    const base::FilePath kEDIDFile("/tmp/display_info.bin");

    // Just read it on current thread as this is necessary information
    // to start. This access only tmpfs, which is fast.
    // TODO(dnicoara|oshima): crbug.com/450886.
    base::ThreadRestrictions::ScopedAllowIO allow_io;
    success = CreateSnapshotFromEDIDFile(kEDIDFile, &params);
  }

  // Fallback to command line if the file doesn't exit or failed to read.
  if (success || CreateSnapshotFromCommandLine(&params)) {
    LOG_IF(ERROR, !success) << "Failed to read display_info.bin.";
    DCHECK_NE(DISPLAY_CONNECTION_TYPE_NONE, params.type);
    displays_.push_back(new DrmDisplaySnapshotProxy(params, display_manager_));
    has_dummy_display_ = true;
  } else {
    LOG(ERROR) << "Failed to obtain initial display info";
  }
}

void DrmNativeDisplayDelegate::GrabServer() {
}

void DrmNativeDisplayDelegate::UngrabServer() {
}

bool DrmNativeDisplayDelegate::TakeDisplayControl() {
  proxy_->Send(new OzoneGpuMsg_TakeDisplayControl());
  return true;
}

bool DrmNativeDisplayDelegate::RelinquishDisplayControl() {
  proxy_->Send(new OzoneGpuMsg_RelinquishDisplayControl());
  return true;
}

void DrmNativeDisplayDelegate::SyncWithServer() {
}

void DrmNativeDisplayDelegate::SetBackgroundColor(uint32_t color_argb) {
  NOTIMPLEMENTED();
}

void DrmNativeDisplayDelegate::ForceDPMSOn() {
}

void DrmNativeDisplayDelegate::GetDisplays(
    const GetDisplaysCallback& callback) {
  get_displays_callback_ = callback;
  // GetDisplays() is supposed to force a refresh of the display list.
  if (!proxy_->Send(new OzoneGpuMsg_RefreshNativeDisplays())) {
    get_displays_callback_.Reset();
    callback.Run(displays_.get());
  }
}

void DrmNativeDisplayDelegate::AddMode(const DisplaySnapshot& output,
                                       const DisplayMode* mode) {
}

void DrmNativeDisplayDelegate::Configure(const DisplaySnapshot& output,
                                         const DisplayMode* mode,
                                         const gfx::Point& origin,
                                         const ConfigureCallback& callback) {
  // The dummy display is used on the first run only. Note: cannot post a task
  // here since there is no task runner.
  if (has_dummy_display_) {
    callback.Run(true);
    return;
  }

  configure_callback_map_[output.display_id()] = callback;

  bool status = false;
  if (mode) {
    status = proxy_->Send(new OzoneGpuMsg_ConfigureNativeDisplay(
        output.display_id(), GetDisplayModeParams(*mode), origin));
  } else {
    status =
        proxy_->Send(new OzoneGpuMsg_DisableNativeDisplay(output.display_id()));
  }

  if (!status)
    OnDisplayConfigured(output.display_id(), false);
}

void DrmNativeDisplayDelegate::CreateFrameBuffer(const gfx::Size& size) {
}

bool DrmNativeDisplayDelegate::GetHDCPState(const DisplaySnapshot& output,
                                            HDCPState* state) {
  NOTIMPLEMENTED();
  return false;
}

bool DrmNativeDisplayDelegate::SetHDCPState(const DisplaySnapshot& output,
                                            HDCPState state) {
  NOTIMPLEMENTED();
  return false;
}

std::vector<ColorCalibrationProfile>
DrmNativeDisplayDelegate::GetAvailableColorCalibrationProfiles(
    const DisplaySnapshot& output) {
  NOTIMPLEMENTED();
  return std::vector<ColorCalibrationProfile>();
}

bool DrmNativeDisplayDelegate::SetColorCalibrationProfile(
    const DisplaySnapshot& output,
    ColorCalibrationProfile new_profile) {
  NOTIMPLEMENTED();
  return false;
}

void DrmNativeDisplayDelegate::AddObserver(NativeDisplayObserver* observer) {
  observers_.AddObserver(observer);
}

void DrmNativeDisplayDelegate::RemoveObserver(NativeDisplayObserver* observer) {
  observers_.RemoveObserver(observer);
}

void DrmNativeDisplayDelegate::OnDeviceEvent(const DeviceEvent& event) {
  if (event.device_type() != DeviceEvent::DISPLAY)
    return;

  switch (event.action_type()) {
    case DeviceEvent::ADD:
      VLOG(1) << "Got display added event for " << event.path().value();
      // The default card is a special case since it needs to be opened early on
      // the GPU process in order to initialize EGL. If it is opened here as
      // well, it will cause a race with opening it in the GPU process and the
      // GPU process may fail initialization.
      // TODO(dnicoara) Remove this when the media stack does not require super
      // early initialization.
      if (event.path() == primary_graphics_card_path_)
        return;

      base::WorkerPool::PostTask(
          FROM_HERE,
          base::Bind(&OpenDeviceOnWorkerThread, event.path(),
                     base::ThreadTaskRunnerHandle::Get(),
                     base::Bind(&DrmNativeDisplayDelegate::OnNewGraphicsDevice,
                                weak_ptr_factory_.GetWeakPtr())),
          false /* task_is_slow */);
      return;
    case DeviceEvent::CHANGE:
      VLOG(1) << "Got display changed event for " << event.path().value();
      break;
    case DeviceEvent::REMOVE:
      VLOG(1) << "Got display removed event for " << event.path().value();
      // It shouldn't be possible to remove this device.
      DCHECK(primary_graphics_card_path_ != event.path())
          << "Got event to remove primary graphics card "
          << event.path().value();
      proxy_->Send(new OzoneGpuMsg_RemoveGraphicsDevice(event.path()));
      break;
  }

  FOR_EACH_OBSERVER(NativeDisplayObserver, observers_,
                    OnConfigurationChanged());
}

void DrmNativeDisplayDelegate::OnNewGraphicsDevice(const base::FilePath& path,
                                                   base::File file) {
  DCHECK(file.IsValid());
  proxy_->Send(new OzoneGpuMsg_AddGraphicsDevice(
      path, base::FileDescriptor(file.Pass())));

  FOR_EACH_OBSERVER(NativeDisplayObserver, observers_,
                    OnConfigurationChanged());
}

void DrmNativeDisplayDelegate::OnChannelEstablished(
    int host_id,
    scoped_refptr<base::SingleThreadTaskRunner> send_runner,
    const base::Callback<void(IPC::Message*)>& send_callback) {
  device_manager_->ScanDevices(this);
  FOR_EACH_OBSERVER(NativeDisplayObserver, observers_,
                    OnConfigurationChanged());
}

void DrmNativeDisplayDelegate::OnChannelDestroyed(int host_id) {
  // If the channel got destroyed in the middle of a configuration then just
  // respond with failure.
  if (!get_displays_callback_.is_null()) {
    base::ThreadTaskRunnerHandle::Get()->PostTask(
        FROM_HERE,
        base::Bind(&DrmNativeDisplayDelegate::RunUpdateDisplaysCallback,
                   weak_ptr_factory_.GetWeakPtr(), get_displays_callback_));
    get_displays_callback_.Reset();
  }

  for (const auto& pair : configure_callback_map_) {
    base::ThreadTaskRunnerHandle::Get()->PostTask(
        FROM_HERE, base::Bind(pair.second, false));
  }
  configure_callback_map_.clear();
}

bool DrmNativeDisplayDelegate::OnMessageReceived(const IPC::Message& message) {
  bool handled = true;

  IPC_BEGIN_MESSAGE_MAP(DrmNativeDisplayDelegate, message)
  IPC_MESSAGE_HANDLER(OzoneHostMsg_UpdateNativeDisplays, OnUpdateNativeDisplays)
  IPC_MESSAGE_HANDLER(OzoneHostMsg_DisplayConfigured, OnDisplayConfigured)
  IPC_MESSAGE_UNHANDLED(handled = false)
  IPC_END_MESSAGE_MAP()

  return handled;
}

void DrmNativeDisplayDelegate::OnUpdateNativeDisplays(
    const std::vector<DisplaySnapshot_Params>& displays) {
  has_dummy_display_ = false;
  displays_.clear();
  for (size_t i = 0; i < displays.size(); ++i)
    displays_.push_back(
        new DrmDisplaySnapshotProxy(displays[i], display_manager_));

  if (!get_displays_callback_.is_null()) {
    base::ThreadTaskRunnerHandle::Get()->PostTask(
        FROM_HERE,
        base::Bind(&DrmNativeDisplayDelegate::RunUpdateDisplaysCallback,
                   weak_ptr_factory_.GetWeakPtr(), get_displays_callback_));
    get_displays_callback_.Reset();
  }
}

void DrmNativeDisplayDelegate::OnDisplayConfigured(int64_t display_id,
                                                   bool status) {
  auto it = configure_callback_map_.find(display_id);
  if (it != configure_callback_map_.end()) {
    base::ThreadTaskRunnerHandle::Get()->PostTask(
        FROM_HERE, base::Bind(it->second, status));
    configure_callback_map_.erase(it);
  }
}

void DrmNativeDisplayDelegate::RunUpdateDisplaysCallback(
    const GetDisplaysCallback& callback) const {
  callback.Run(displays_.get());
}

}  // namespace ui
