// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

/**
 * Loads a thumbnail using provided url. In CANVAS mode, loaded images
 * are attached as <canvas> element, while in IMAGE mode as <img>.
 * <canvas> renders faster than <img>, however has bigger memory overhead.
 *
 * @param {Entry} entry File entry.
 * @param {ThumbnailLoader.LoaderType=} opt_loaderType Canvas or Image loader,
 *     default: IMAGE.
 * @param {Object=} opt_metadata Metadata object.
 * @param {string=} opt_mediaType Media type.
 * @param {Array<ThumbnailLoader.LoadTarget>=} opt_loadTargets The list of load
 *     targets in preferential order. The default value is [CONTENT_METADATA,
 *     EXTERNAL_METADATA, FILE_ENTRY].
 * @param {number=} opt_priority Priority, the highest is 0. default: 2.
 * @constructor
 */
function ThumbnailLoader(entry, opt_loaderType, opt_metadata, opt_mediaType,
    opt_loadTargets, opt_priority) {
  var loadTargets = opt_loadTargets || [
    ThumbnailLoader.LoadTarget.CONTENT_METADATA,
    ThumbnailLoader.LoadTarget.EXTERNAL_METADATA,
    ThumbnailLoader.LoadTarget.FILE_ENTRY
  ];

  this.mediaType_ = opt_mediaType || FileType.getMediaType(entry);
  this.loaderType_ = opt_loaderType || ThumbnailLoader.LoaderType.IMAGE;
  this.metadata_ = opt_metadata;
  this.priority_ = (opt_priority !== undefined) ? opt_priority : 2;
  this.transform_ = null;

  /**
   * @type {?ThumbnailLoader.LoadTarget}
   * @private
   */
  this.loadTarget_ = null;

  if (!opt_metadata) {
    this.thumbnailUrl_ = entry.toURL();  // Use the URL directly.
    this.loadTarget_ = ThumbnailLoader.LoadTarget.FILE_ENTRY;
    return;
  }

  this.fallbackUrl_ = null;
  this.thumbnailUrl_ = null;
  if (opt_metadata.external && opt_metadata.external.customIconUrl)
    this.fallbackUrl_ = opt_metadata.external.customIconUrl;

  for (var i = 0; i < loadTargets.length; i++) {
    switch (loadTargets[i]) {
      case ThumbnailLoader.LoadTarget.CONTENT_METADATA:
        if (opt_metadata.thumbnail && opt_metadata.thumbnail.url) {
          this.thumbnailUrl_ = opt_metadata.thumbnail.url;
          this.transform_ =
              opt_metadata.thumbnail && opt_metadata.thumbnail.transform;
          this.loadTarget_ = ThumbnailLoader.LoadTarget.CONTENT_METADATA;
        }
        break;
      case ThumbnailLoader.LoadTarget.EXTERNAL_METADATA:
        if (opt_metadata.external && opt_metadata.external.thumbnailUrl &&
            (!opt_metadata.external.present || !FileType.isImage(entry))) {
          this.thumbnailUrl_ = opt_metadata.external.thumbnailUrl;
          this.loadTarget_ = ThumbnailLoader.LoadTarget.EXTERNAL_METADATA;
        }
        break;
      case ThumbnailLoader.LoadTarget.FILE_ENTRY:
        if (FileType.isImage(entry) || FileType.isRaw(entry)) {
          this.thumbnailUrl_ = entry.toURL();
          this.transform_ =
              opt_metadata.media && opt_metadata.media.imageTransform;
          this.loadTarget_ = ThumbnailLoader.LoadTarget.FILE_ENTRY;
        }
        break;
      default:
        assertNotReached('Unkonwn load type: ' + loadTargets[i]);
    }
    if (this.thumbnailUrl_)
      break;
  }

  if (!this.thumbnailUrl_ && this.fallbackUrl_) {
    // Use fallback as the primary thumbnail.
    this.thumbnailUrl_ = this.fallbackUrl_;
    this.fallbackUrl_ = null;
  } // else the generic thumbnail based on the media type will be used.
}

/**
 * In percents (0.0 - 1.0), how much area can be cropped to fill an image
 * in a container, when loading a thumbnail in FillMode.AUTO mode.
 * The specified 30% value allows to fill 16:9, 3:2 pictures in 4:3 element.
 * @type {number}
 */
ThumbnailLoader.AUTO_FILL_THRESHOLD = 0.3;

/**
 * Type of displaying a thumbnail within a box.
 * @enum {number}
 */
ThumbnailLoader.FillMode = {
  FILL: 0,  // Fill whole box. Image may be cropped.
  FIT: 1,   // Keep aspect ratio, do not crop.
  OVER_FILL: 2,  // Fill whole box with possible stretching.
  AUTO: 3   // Try to fill, but if incompatible aspect ratio, then fit.
};

/**
 * Optimization mode for downloading thumbnails.
 * @enum {number}
 */
ThumbnailLoader.OptimizationMode = {
  NEVER_DISCARD: 0,    // Never discards downloading. No optimization.
  DISCARD_DETACHED: 1  // Canceled if the container is not attached anymore.
};

/**
 * Type of element to store the image.
 * @enum {number}
 */
ThumbnailLoader.LoaderType = {
  IMAGE: 0,
  CANVAS: 1
};

/**
 * Load target of ThumbnailLoader.
 * @enum {string}
 */
ThumbnailLoader.LoadTarget = {
  // e.g. Drive thumbnail, FSP thumbnail.
  EXTERNAL_METADATA: 'externalMetadata',
  // e.g. EXIF thumbnail.
  CONTENT_METADATA: 'contentMetadata',
  // Image file itself.
  FILE_ENTRY: 'fileEntry'
};

/**
 * Maximum thumbnail's width when generating from the full resolution image.
 * @const
 * @type {number}
 */
ThumbnailLoader.THUMBNAIL_MAX_WIDTH = 500;

/**
 * Maximum thumbnail's height when generating from the full resolution image.
 * @const
 * @type {number}
 */
ThumbnailLoader.THUMBNAIL_MAX_HEIGHT = 500;

/**
 * Returns the target of loading.
 * @return {?ThumbnailLoader.LoadTarget}
 */
ThumbnailLoader.prototype.getLoadTarget = function() {
  return this.loadTarget_;
};

/**
 * Loads and attaches an image.
 *
 * @param {Element} box Container element.
 * @param {ThumbnailLoader.FillMode} fillMode Fill mode.
 * @param {ThumbnailLoader.OptimizationMode=} opt_optimizationMode Optimization
 *     for downloading thumbnails. By default optimizations are disabled.
 * @param {function(Image, Object)=} opt_onSuccess Success callback,
 *     accepts the image and the transform.
 * @param {function()=} opt_onError Error callback.
 * @param {function()=} opt_onGeneric Callback for generic image used.
 */
ThumbnailLoader.prototype.load = function(box, fillMode, opt_optimizationMode,
    opt_onSuccess, opt_onError, opt_onGeneric) {
  opt_optimizationMode = opt_optimizationMode ||
      ThumbnailLoader.OptimizationMode.NEVER_DISCARD;

  if (!this.thumbnailUrl_) {
    // Relevant CSS rules are in file_types.css.
    box.setAttribute('generic-thumbnail', this.mediaType_);
    if (opt_onGeneric) opt_onGeneric();
    return;
  }

  this.cancel();
  this.canvasUpToDate_ = false;
  this.image_ = new Image();
  this.image_.onload = function() {
    this.attachImage(box, fillMode);
    if (opt_onSuccess)
      opt_onSuccess(this.image_, this.transform_);
  }.bind(this);
  this.image_.onerror = function() {
    if (opt_onError)
      opt_onError();
    if (this.fallbackUrl_) {
      this.thumbnailUrl_ = this.fallbackUrl_;
      this.fallbackUrl_ = null;
      this.load(box, fillMode, opt_optimizationMode, opt_onSuccess);
    } else {
      box.setAttribute('generic-thumbnail', this.mediaType_);
    }
  }.bind(this);

  if (this.image_.src) {
    console.warn('Thumbnail already loaded: ' + this.thumbnailUrl_);
    return;
  }

  // TODO(mtomasz): Smarter calculation of the requested size.
  var wasAttached = box.ownerDocument.contains(box);
  var modificationTime = this.metadata_ &&
                         this.metadata_.filesystem &&
                         this.metadata_.filesystem.modificationTime &&
                         this.metadata_.filesystem.modificationTime.getTime();
  this.taskId_ = ImageLoaderClient.loadToImage(
      this.thumbnailUrl_,
      this.image_,
      {
        maxWidth: ThumbnailLoader.THUMBNAIL_MAX_WIDTH,
        maxHeight: ThumbnailLoader.THUMBNAIL_MAX_HEIGHT,
        cache: true,
        priority: this.priority_,
        timestamp: modificationTime
      },
      function() {},
      function() {
        this.image_.onerror(new Event('load-error'));
      }.bind(this),
      function() {
        if (opt_optimizationMode ==
            ThumbnailLoader.OptimizationMode.DISCARD_DETACHED &&
            !box.ownerDocument.contains(box)) {
          // If the container is not attached, then invalidate the download.
          return false;
        }
        return true;
      });
};

/**
 * Loads thumbnail as data url. If data url of thumbnail can be fetched from
 * metadata, this fetches it from it. Otherwise, this tries to load it from
 * thumbnail loader.
 * Compared with ThumbnailLoader.load, this method does not provide a
 * functionality to fit image to a box. This method is responsible for rotating
 * and flipping a thumbnail.
 *
 * @return {!Promise<{data:string, width:number, height:number}>} A promise
 *     which is resolved when data url is fetched.
 */
ThumbnailLoader.prototype.loadAsDataUrl = function() {
  return new Promise(function(resolve, reject) {
    // Load by using ImageLoaderClient.
    var modificationTime = this.metadata_ &&
                           this.metadata_.filesystem &&
                           this.metadata_.filesystem.modificationTime &&
                           this.metadata_.filesystem.modificationTime.getTime();
    ImageLoaderClient.getInstance().load(
        this.thumbnailUrl_,
        function(result) {
          if (result.status === 'success')
            resolve(result);
          else
            reject(result);
        },
        {
          maxWidth: ThumbnailLoader.THUMBNAIL_MAX_WIDTH,
          maxHeight: ThumbnailLoader.THUMBNAIL_MAX_HEIGHT,
          cache: true,
          priority: this.priority_,
          timestamp: modificationTime
        });
  }.bind(this)).then(function(result) {
    if (!this.transform_)
      return result;
    else
      return this.applyTransformToDataUrl_(
          this.transform_, result.data, result.width, result.height);
  }.bind(this));
};

/**
 * Applies transform to data url.
 *
 * @param {{scaleX:number, scaleY:number, rotate90: number}} transform
 *     Transform.
 * @param {string} dataUrl Data url.
 * @param {number} width Width.
 * @param {number} height Height.
 * @return {!Promise<{data:string, width:number, height:number}>} A promise
 *     which is resolved with dataUrl and its width and height.
 * @private
 */
ThumbnailLoader.prototype.applyTransformToDataUrl_ = function(
    transform, dataUrl, width, height) {
  var image = new Image();
  var scaleX = this.transform_.scaleX;
  var scaleY = this.transform_.scaleY;
  var rotate90 = this.transform_.rotate90;

  assert(scaleX === 1 || scaleX === -1);
  assert(scaleY === 1 || scaleY === -1);
  assert(rotate90 === 0 || rotate90 === 1);

  return new Promise(function(resolve, reject) {
    // Decode image for transformation.
    image.onload = resolve;
    image.onerror = reject;
    image.src = dataUrl;
  }).then(function() {
    // Apply transform. Scale transformation should be applied before rotate
    // transformation. i.e. When matrices for scale and rotate are A and B,
    // transformation matrix should be BA.
    var canvas = document.createElement('canvas');
    var context = canvas.getContext('2d');

    canvas.width = rotate90 === 1 ? height : width;
    canvas.height = rotate90 === 1 ? width : height;

    // Rotate 90 degree at center.
    if (rotate90 === 1) {
      context.translate(height, 0);
      context.rotate(Math.PI / 2);
    }

    // Flip X and Y.
    context.translate(scaleX === -1 ? width : 0, scaleY === -1 ? height : 0);
    context.scale(scaleX, scaleY);

    context.drawImage(image, 0, 0);

    return {
      data: canvas.toDataURL('image/png'),
      width: canvas.width,
      height: canvas.height
    };
  }.bind(this));
}

/**
 * Cancels loading the current image.
 */
ThumbnailLoader.prototype.cancel = function() {
  if (this.taskId_) {
    this.image_.onload = function() {};
    this.image_.onerror = function() {};
    ImageLoaderClient.getInstance().cancel(this.taskId_);
    this.taskId_ = null;
  }
};

/**
 * @return {boolean} True if a valid image is loaded.
 */
ThumbnailLoader.prototype.hasValidImage = function() {
  return !!(this.image_ && this.image_.width && this.image_.height);
};

/**
 * @return {boolean} True if the image is rotated 90 degrees left or right.
 * @private
 */
ThumbnailLoader.prototype.isRotated_ = function() {
  return this.transform_ && (this.transform_.rotate90 % 2 === 1);
};

/**
 * @return {number} Image width (corrected for rotation).
 */
ThumbnailLoader.prototype.getWidth = function() {
  return this.isRotated_() ? this.image_.height : this.image_.width;
};

/**
 * @return {number} Image height (corrected for rotation).
 */
ThumbnailLoader.prototype.getHeight = function() {
  return this.isRotated_() ? this.image_.width : this.image_.height;
};

/**
 * Load an image but do not attach it.
 *
 * @param {function(boolean)} callback Callback, parameter is true if the image
 *     has loaded successfully or a stock icon has been used.
 */
ThumbnailLoader.prototype.loadDetachedImage = function(callback) {
  if (!this.thumbnailUrl_) {
    callback(true);
    return;
  }

  this.cancel();
  this.canvasUpToDate_ = false;
  this.image_ = new Image();
  this.image_.onload = callback.bind(null, true);
  this.image_.onerror = callback.bind(null, false);

  // TODO(mtomasz): Smarter calculation of the requested size.
  var modificationTime = this.metadata_ &&
                         this.metadata_.filesystem &&
                         this.metadata_.filesystem.modificationTime &&
                         this.metadata_.filesystem.modificationTime.getTime();
  this.taskId_ = ImageLoaderClient.loadToImage(
      this.thumbnailUrl_,
      this.image_,
      {
        maxWidth: ThumbnailLoader.THUMBNAIL_MAX_WIDTH,
        maxHeight: ThumbnailLoader.THUMBNAIL_MAX_HEIGHT,
        cache: true,
        priority: this.priority_,
        timestamp: modificationTime
      },
      function() {},
      function() {
        this.image_.onerror(new Event('load-error'));
      }.bind(this));
};

/**
 * Renders the thumbnail into either canvas or an image element.
 * @private
 */
ThumbnailLoader.prototype.renderMedia_ = function() {
  if (this.loaderType_ !== ThumbnailLoader.LoaderType.CANVAS)
    return;

  if (!this.canvas_)
    this.canvas_ = document.createElement('canvas');

  // Copy the image to a canvas if the canvas is outdated.
  if (!this.canvasUpToDate_) {
    this.canvas_.width = this.image_.width;
    this.canvas_.height = this.image_.height;
    var context = this.canvas_.getContext('2d');
    context.drawImage(this.image_, 0, 0);
    this.canvasUpToDate_ = true;
  }
};

/**
 * Attach the image to a given element.
 * @param {Element} container Parent element.
 * @param {ThumbnailLoader.FillMode} fillMode Fill mode.
 */
ThumbnailLoader.prototype.attachImage = function(container, fillMode) {
  if (!this.hasValidImage()) {
    container.setAttribute('generic-thumbnail', this.mediaType_);
    return;
  }

  this.renderMedia_();
  util.applyTransform(container, this.transform_);
  var attachableMedia = this.loaderType_ === ThumbnailLoader.LoaderType.CANVAS ?
      this.canvas_ : this.image_;

  ThumbnailLoader.centerImage_(
      container, attachableMedia, fillMode, this.isRotated_());

  if (attachableMedia.parentNode !== container) {
    container.textContent = '';
    container.appendChild(attachableMedia);
  }

  if (!this.taskId_)
    attachableMedia.classList.add('cached');
};

/**
 * Gets the loaded image.
 * TODO(mtomasz): Apply transformations.
 *
 * @return {Image|HTMLCanvasElement} Either image or a canvas object.
 */
ThumbnailLoader.prototype.getImage = function() {
  this.renderMedia_();
  return this.loaderType_ === ThumbnailLoader.LoaderType.CANVAS ? this.canvas_ :
      this.image_;
};

/**
 * Update the image style to fit/fill the container.
 *
 * Using webkit center packing does not align the image properly, so we need
 * to wait until the image loads and its dimensions are known, then manually
 * position it at the center.
 *
 * @param {Element} box Containing element.
 * @param {Image|HTMLCanvasElement} img Element containing an image.
 * @param {ThumbnailLoader.FillMode} fillMode Fill mode.
 * @param {boolean} rotate True if the image should be rotated 90 degrees.
 * @private
 */
ThumbnailLoader.centerImage_ = function(box, img, fillMode, rotate) {
  var imageWidth = img.width;
  var imageHeight = img.height;

  var fractionX;
  var fractionY;

  var boxWidth = box.clientWidth;
  var boxHeight = box.clientHeight;

  var fill;
  switch (fillMode) {
    case ThumbnailLoader.FillMode.FILL:
    case ThumbnailLoader.FillMode.OVER_FILL:
      fill = true;
      break;
    case ThumbnailLoader.FillMode.FIT:
      fill = false;
      break;
    case ThumbnailLoader.FillMode.AUTO:
      var imageRatio = imageWidth / imageHeight;
      var boxRatio = 1.0;
      if (boxWidth && boxHeight)
        boxRatio = boxWidth / boxHeight;
      // Cropped area in percents.
      var ratioFactor = boxRatio / imageRatio;
      fill = (ratioFactor >= 1.0 - ThumbnailLoader.AUTO_FILL_THRESHOLD) &&
             (ratioFactor <= 1.0 + ThumbnailLoader.AUTO_FILL_THRESHOLD);
      break;
  }

  if (boxWidth && boxHeight) {
    // When we know the box size we can position the image correctly even
    // in a non-square box.
    var fitScaleX = (rotate ? boxHeight : boxWidth) / imageWidth;
    var fitScaleY = (rotate ? boxWidth : boxHeight) / imageHeight;

    var scale = fill ?
        Math.max(fitScaleX, fitScaleY) :
        Math.min(fitScaleX, fitScaleY);

    if (fillMode !== ThumbnailLoader.FillMode.OVER_FILL)
      scale = Math.min(scale, 1);  // Never overscale.

    fractionX = imageWidth * scale / boxWidth;
    fractionY = imageHeight * scale / boxHeight;
  } else {
    // We do not know the box size so we assume it is square.
    // Compute the image position based only on the image dimensions.
    // First try vertical fit or horizontal fill.
    fractionX = imageWidth / imageHeight;
    fractionY = 1;
    if ((fractionX < 1) === !!fill) {  // Vertical fill or horizontal fit.
      fractionY = 1 / fractionX;
      fractionX = 1;
    }
  }

  function percent(fraction) {
    return (fraction * 100).toFixed(2) + '%';
  }

  img.style.width = percent(fractionX);
  img.style.height = percent(fractionY);
  img.style.left = percent((1 - fractionX) / 2);
  img.style.top = percent((1 - fractionY) / 2);
};
