// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef UI_EVENTS_KEYCODES_KEYBOARD_CODE_CONVERSION_H_
#define UI_EVENTS_KEYCODES_KEYBOARD_CODE_CONVERSION_H_

#include "base/strings/string16.h"
#include "ui/events/events_base_export.h"
#include "ui/events/keycodes/keyboard_codes.h"

namespace ui {

enum class DomCode;
enum class DomKey;

// Helper functions to get the meaning of a Windows key code in a
// platform independent way. It supports control characters as well.
// It assumes a US keyboard layout is used, so it may only be used when there
// is no native event or no better way to get the character.
// For example, if a virtual keyboard implementation can only generate key
// events with key_code and flags information, then there is no way for us to
// determine the actual character that should be generate by the key. Because
// a key_code only represents a physical key on the keyboard, it has nothing
// to do with the actual character printed on that key. In such case, the only
// thing we can do is to assume that we are using a US keyboard and get the
// character according to US keyboard layout definition.
// If a virtual keyboard implementation wants to support other keyboard
// layouts, that may generate different text for a certain key than on a US
// keyboard, a special native event object should be introduced to carry extra
// information to help determine the correct character.
// Take XKeyEvent as an example, it contains not only keycode and modifier
// flags but also group and other extra XKB information to help determine the
// correct character. That's why we can use XLookupString() function to get
// the correct text generated by a X key event (See how is GetCharacter()
// implemented in event_x.cc).
EVENTS_BASE_EXPORT base::char16 GetCharacterFromKeyCode(KeyboardCode key_code,
                                                        int flags);
EVENTS_BASE_EXPORT bool GetMeaningFromKeyCode(KeyboardCode key_code,
                                              int flags,
                                              DomKey* dom_key,
                                              base::char16* character);

// Determine the non-located VKEY corresponding to a located VKEY.
// Most modifier keys have two kinds of KeyboardCode: located (e.g.
// VKEY_LSHIFT and VKEY_RSHIFT), that indentify one of two specific
// physical keys, and non-located (e.g. VKEY_SHIFT) that identify
// only the operation.
EVENTS_BASE_EXPORT KeyboardCode
LocatedToNonLocatedKeyboardCode(KeyboardCode key_code);

// Determine the located VKEY corresponding to a non-located VKEY.
EVENTS_BASE_EXPORT KeyboardCode
NonLocatedToLocatedKeyboardCode(KeyboardCode key_code, DomCode dom_code);

}  // namespace ui

#endif  // UI_EVENTS_KEYCODES_KEYBOARD_CODE_CONVERSION_H_
