// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.net.test;

import android.util.Log;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.HttpVersion;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.DefaultHttpServerConnection;
import org.apache.http.message.BasicHttpResponse;
import org.apache.http.params.BasicHttpParams;
import org.apache.http.params.HttpParams;

import java.io.IOException;
import java.net.Socket;

/** A base class for simple HTTP test servers. */
public abstract class BaseHttpTestServer extends BaseTcpTestServer {
    private static final String TAG = "BaseHttpTestServer";

    /**
     * Create an HTTP test server on the given port.
     *
     * @param serverPort The port to listen on for incoming HTTP connections.
     * @param acceptTimeoutMs The timeout for calls to ServerSocket.accept(), in milliseconds.
     * @throws IOException If the server port can't be bound.
     */
    public BaseHttpTestServer(int serverPort, int acceptTimeoutMs) throws IOException {
        super(serverPort, acceptTimeoutMs);
    }

    /**
     * Handle an incoming connection on |sock|.
     *
     * This will bind the socket, receive the request header (and entity, if applicable),
     * dispatch the request to a handler based on the request method, and send the response
     * generated by the handler.
     *
     * @param sock The socket for the incoming connection.
     * @throws IOException If an error occurs while reading from or writing to the socket.
     */
    @Override
    protected final void handle(Socket sock) throws IOException {
        HttpParams httpParams = getConnectionParams();
        DefaultHttpServerConnection conn = new DefaultHttpServerConnection();
        conn.bind(sock, httpParams);

        try {
            HttpRequest request = conn.receiveRequestHeader();
            if (request instanceof HttpEntityEnclosingRequest) {
                conn.receiveRequestEntity((HttpEntityEnclosingRequest) request);
            }

            HttpResponse response = null;
            switch (request.getRequestLine().getMethod()) {
                case HttpGet.METHOD_NAME:
                    response = handleGet(request);
                    break;
                case HttpPost.METHOD_NAME:
                    response = handlePost((HttpEntityEnclosingRequest) request);
                    break;
                default:
                    response = handleUnsupported(request);
                    break;
            }

            conn.sendResponseHeader(response);
            conn.sendResponseEntity(response);
        } catch (HttpException e) {
            Log.e(TAG, "Error while handling HTTP request", e);
        } finally {
            conn.close();
        }
    }

    /**
     * Returns the parameters used to establish the HTTP connection.
     *
     * The default implementation returns a default BasicHttpParams instance.
     *
     * @return The HttpParams to use to establish the HTTP connection.
     */
    protected HttpParams getConnectionParams() {
        return new BasicHttpParams();
    }

    /**
     * Handle a GET request.
     *
     * The default implementation returns a 405. Override this function if you want to support
     * GET.
     *
     * @param request The GET request to handle.
     * @return The response to the GET request.
     */
    protected HttpResponse handleGet(HttpRequest request) throws HttpException {
        return handleUnsupported(request);
    }

    /**
     * Handle a POST request.
     *
     * The default implementation returns a 405. Override this function if you want to support
     * POST.
     *
     * @param request The POST request to handle.
     * @return The response to the POST request.
     */
    protected HttpResponse handlePost(HttpEntityEnclosingRequest request) throws HttpException {
        return handleUnsupported(request);
    }

    /** Handle an unsupported HTTP request.
     *
     * @param request The unsupported HTTP request.
     * @return A 405 Method Not Allowed response.
     */
    protected HttpResponse handleUnsupported(HttpRequest request) {
        return new BasicHttpResponse(HttpVersion.HTTP_1_1, HttpStatus.SC_METHOD_NOT_ALLOWED, "");
    }
}
