// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CONTENT_RENDERER_MEDIA_RTC_DATA_CHANNEL_HANDLER_H_
#define CONTENT_RENDERER_MEDIA_RTC_DATA_CHANNEL_HANDLER_H_

#include "base/memory/ref_counted.h"
#include "base/memory/scoped_ptr.h"
#include "base/single_thread_task_runner.h"
#include "base/threading/thread_checker.h"
#include "content/common/content_export.h"
#include "third_party/libjingle/source/talk/app/webrtc/peerconnectioninterface.h"
#include "third_party/WebKit/public/platform/WebRTCDataChannelHandler.h"
#include "third_party/WebKit/public/platform/WebRTCDataChannelHandlerClient.h"

namespace content {

// RtcDataChannelHandler is a delegate for the RTC PeerConnection DataChannel
// API messages going between WebKit and native PeerConnection DataChannels in
// libjingle. Instances of this class are owned by WebKit.
// WebKit call all of these methods on the main render thread.
// Callbacks to the webrtc::DataChannelObserver implementation also occur on
// the main render thread.
class CONTENT_EXPORT RtcDataChannelHandler
    : NON_EXPORTED_BASE(public blink::WebRTCDataChannelHandler) {
 public:
  // This object can* be constructed on libjingle's signaling thread and then
  // ownership is passed to the UI thread where it's eventually given to WebKit.
  // The reason we must construct and hook ourselves up as an observer on the
  // signaling thread is to avoid missing out on any state changes or messages
  // that may occur before we've fully connected with webkit.
  // This period is basically between when the ctor is called and until
  // setClient is called.
  // * For local data channels, the object will be construced on the main thread
  //   and we don't have the issue described above.
  RtcDataChannelHandler(
      const scoped_refptr<base::SingleThreadTaskRunner>& main_thread,
      webrtc::DataChannelInterface* channel);
  virtual ~RtcDataChannelHandler();

  // blink::WebRTCDataChannelHandler implementation.
  virtual void setClient(
      blink::WebRTCDataChannelHandlerClient* client) override;
  virtual blink::WebString label() override;
  virtual bool isReliable() override;
  virtual bool ordered() const override;
  virtual unsigned short maxRetransmitTime() const override;
  virtual unsigned short maxRetransmits() const override;
  virtual blink::WebString protocol() const override;
  virtual bool negotiated() const override;
  virtual unsigned short id() const override;
  // TODO(bemasc): Mark |state()| as |override| once https://codereview.chromium.org/782843003/
  // lands in Blink and rolls into Chromium.
  virtual blink::WebRTCDataChannelHandlerClient::ReadyState state() const;
  virtual unsigned long bufferedAmount() override;
  virtual bool sendStringData(const blink::WebString& data) override;
  virtual bool sendRawData(const char* data, size_t length) override;
  virtual void close() override;

  const scoped_refptr<webrtc::DataChannelInterface>& channel() const;

 private:
  void OnStateChange(webrtc::DataChannelInterface::DataState state);
  void OnMessage(scoped_ptr<webrtc::DataBuffer> buffer);
  void RecordMessageSent(size_t num_bytes);

  class CONTENT_EXPORT Observer
      : public NON_EXPORTED_BASE(
            base::RefCountedThreadSafe<RtcDataChannelHandler::Observer>),
        public NON_EXPORTED_BASE(webrtc::DataChannelObserver) {
   public:
    Observer(RtcDataChannelHandler* handler,
        const scoped_refptr<base::SingleThreadTaskRunner>& main_thread,
        webrtc::DataChannelInterface* channel);

    const scoped_refptr<base::SingleThreadTaskRunner>& main_thread() const;
    const scoped_refptr<webrtc::DataChannelInterface>& channel() const;

    // Clears the internal |handler_| pointer so that no further callbacks
    // will be attempted, disassociates this observer from the channel and
    // releases the channel pointer. Must be called on the main thread.
    void Unregister();

   private:
    friend class base::RefCountedThreadSafe<RtcDataChannelHandler::Observer>;
    ~Observer() override;

    // webrtc::DataChannelObserver implementation.
    void OnStateChange() override;
    void OnMessage(const webrtc::DataBuffer& buffer) override;

    void OnStateChangeImpl(webrtc::DataChannelInterface::DataState state);
    void OnMessageImpl(scoped_ptr<webrtc::DataBuffer> buffer);

    RtcDataChannelHandler* handler_;
    const scoped_refptr<base::SingleThreadTaskRunner> main_thread_;
    scoped_refptr<webrtc::DataChannelInterface> channel_;
  };

  scoped_refptr<Observer> observer_;
  base::ThreadChecker thread_checker_;

  blink::WebRTCDataChannelHandlerClient* webkit_client_;
};

}  // namespace content

#endif  // CONTENT_RENDERER_MEDIA_RTC_DATA_CHANNEL_HANDLER_H_
