/*
 * genpolusers old-policy new-users new-policy
 *
 * Given an existing binary policy configuration and a new source users 
 * configuration (post-processed), generate a new binary policy 
 * configuration that is identical to the old one except that it has
 * the new user declarations.  User declarations from the old binary
 * policy are replaced if they also exist in the new source users 
 * configuration or removed otherwise.  Special exceptions are made
 * for system_u and user_u, which can be replaced but not removed as
 * a safety against reloading a user configuration generated from 
 * passwd information that would not include these special SELinux users.
 * This needs to be generalized in some manner.  New users may also be
 * added by including them in the new source users configuration.
 */ 

#include <getopt.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/mman.h>
#include <sepol/policydb.h>
#include <sepol/services.h>
#include <sepol/conditional.h>
#include "queue.h"
#include "checkpolicy.h"

extern policydb_t *policydbp;
extern queue_t id_queue;
extern unsigned int policydb_errors;

extern FILE *yyin;
extern int yyparse(void);
extern void yyrestart(FILE *);

void usage(char *progname)
{
	printf("usage:  %s old-policy new-users new-policy\n", progname);
	exit(1);
}

/* Select users for removal based on whether they were defined in the
   new source users configuration. */
static int select_user(hashtab_key_t key, hashtab_datum_t datum, void *datap)
{
	char *name = key;
	user_datum_t *usrdatum = datum;
	
	if (!usrdatum->defined) {
		/* XXX Hack:  Don't accidentally remove SELinux-only users. */
		if (!strcmp(name, "system_u") || !strcmp(name, "user_u")) {
			printf("Warning!  %s not defined, but not removing.\n", name);
			return 0;
		}
		printf("Removing user %s\n", name);
		return 1;
	}
	return 0;
}

static struct ebitmap free_users;

/* Kill the user entries selected by select_user, and
   record that their slots are free. */
void kill_user(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	user_datum_t *usrdatum;
	struct policydb *pol = p;

	if (key)
		free(key);

	usrdatum = (user_datum_t *) datum;
	ebitmap_set_bit(&free_users, usrdatum->value - 1, 1);

	ebitmap_destroy(&usrdatum->roles);
	free(datum);
	pol->p_users.nprim--;
}

/* Fold user values down to avoid holes generated by removal.
   As the SID table is remapped by the kernel upon a policy reload,
   this is safe for existing SIDs.  But it could be a problem for
   constraints if they refer to the particular user.  */
int remap_users(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	user_datum_t *usrdatum = datum;
	struct policydb *pol = p;
	int i;

	if (usrdatum->value > pol->p_users.nprim) {
		for (i = ebitmap_startbit(&free_users); i < ebitmap_length(&free_users); i++) {
			if (ebitmap_get_bit(&free_users, i)) {
				printf("Remapping user %s (%u -> %u)\n", key, usrdatum->value, i+1);
				usrdatum->value = i+1;
				ebitmap_set_bit(&free_users, i, 0);
				return 0;
			}
		}
	}
	return 0;
}

int main(int argc, char **argv)
{
        policydb_t policydb;
	struct policy_file pf;
	struct stat sb;
	FILE *outfp;
	int fd, rc;
	void *map;

	if (argc != 4) 
		usage(argv[0]);

	fd = open(argv[1], O_RDONLY);
	if (fd < 0) {
		fprintf(stderr, "Can't open '%s':  %s\n",
			argv[1], strerror(errno));
		exit(1);
	}
	if (fstat(fd, &sb) < 0) {
		fprintf(stderr, "Can't stat '%s':  %s\n",
			argv[1], strerror(errno));
		exit(1);
	}
	map = mmap(NULL, sb.st_size, PROT_READ | PROT_WRITE, MAP_PRIVATE, fd, 0);
	if (map == MAP_FAILED) {
		fprintf(stderr, "Can't map '%s':  %s\n",
			argv[1], strerror(errno));
		exit(1);
	}
	pf.type = PF_USE_MEMORY;
	pf.data = map;
	pf.len = sb.st_size;
	if (policydb_read(&policydb,&pf, 0)) {
		fprintf(stderr, "Can't read binary policy from '%s':  %s\n",
			argv[1], strerror(errno));
		exit(1);
	}
	close(fd);

	/* Preserve the policy version of the original policy
	   for the new policy. */
	sepol_set_policyvers(policydb.policyvers);

	yyin = fopen(argv[2], "r");
	if (!yyin) {
		fprintf(stderr, "%s:  unable to open %s\n", argv[0], 
			argv[2]);
		exit(1);
	}
	id_queue = queue_create();
	if (!id_queue) {
		fprintf(stderr, "%s:  out of memory\n", argv[0]);
		exit(1);
	}
	policydbp = &policydb;
	policydb_errors = 0;
	if (yyparse() || policydb_errors) {
		fprintf(stderr, "%s:  error(s) encountered while parsing configuration\n", argv[0]);
		exit(1);
	}
	queue_destroy(id_queue);

	if (policydb_errors) 
		exit(1);

	hashtab_map_remove_on_error(policydb.p_users.table, select_user, kill_user, &policydb);
	hashtab_map(policydb.p_users.table, remap_users, &policydb);

	outfp = fopen(argv[3], "w");
	if (!outfp) {
		perror(argv[3]);
		exit(1);
	}
	pf.type = PF_USE_STDIO;
	pf.fp = outfp;
	rc = policydb_write(&policydb, &pf);
	if (rc) {
		fprintf(stderr, "%s:  error writing %s\n",
			argv[0], argv[3]);
		exit(1);
	}
	fclose(outfp);
	exit(0);
}
