/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2011 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define _GNU_SOURCE /* needed by isblank(), see 'man isblank' */

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <stdlib.h>

#include "cdw_string.h"
#include "cdw_debug.h"

/**
 * \file cdw_string.c
 * \brief Simple string utilities
 *
 * Some utility functions that handle (char *) strings. Some of them you will
 * probably find in a modern C library that tries to implement something
 * more than ANSI C90.
 */



/**
   \brief Return pointer to first char in string not being space, \t, \r or \n

   Returned pointer points to part of string given as an argument, so it
   cannot be free()d.

   Returned pointer may point to string of length = 0 - that means that
   original string consisted only of white characters.

   \p str has to be standard C string ended with '\0'.

   \param str - string which should be trimmed from left side

   \return pointer to first char in argument string str, which is neither space, \t, \n and \r character
   \return NULL if argument is NULL
   \return unchanged argument if argument is an empty string
*/
char *cdw_string_ltrim(char *str)
{
	if (str == NULL) {
		return NULL;
	}

	if (*str == '\0') {
		return (char *) str;
	}

	/* The  strspn()  function calculates the length of the
	   initial segment of first argument which consists entirely
	   of characters in second argument. */
	return str + strspn(str, " \t\r\n");
}





/**
   \brief Remove all spaces, '\t', '\r' or '\n' from tail of string

   Function removes all spaces, '\t', '\n' and '\r' chars from tail of string by
   inserting '\0' char after last char that is not space nor tab character.

   \param str - string to be trimmed at the end, it has to be ended with '\0'

   \return pointer to string trimmed from the right side,
   \return NULL if argument is NULL
   \return unchanged argument if argument is an empty string
 */
char *cdw_string_rtrim(char *str)
{
	if (str == NULL) {
		return NULL;
	}

	if (*str == '\0') {
		return str;
	}

	char *p;
	for (p = str; *p; p++)
		;

	p--;
	while ( (*p == ' ') || (*p == '\t') || (*p == '\r') || (*p == '\n') ) {
		if (p == str) {
			p--;
			break; /* read note below */
		} else {
			p--;
		}

	}

	/* (p == str) condition sooner or later becomes true in case of
	   'str' values being all blank characters; in a moment when this
	   condition is met, 'p--' will move 'p' pointer too far left and
	   will reach memory outside allocated space; checks of value of
	   such memory cells performed in 'while' loop ( (*p == ' ') etc)
	   are in fact invalid reads beyond 'our' memory; valgrind
	   complains about them, so we 'break' the loop to avoid such
	   reads and to avoid littering valgrind log; such reads seem to
	   be harmless on my box, but just to be on a safe side... */

	*++p = '\0';
	return str;
}





/* GPL 2 (or later) code from
gcc-2.95.2/gcc-core-2.95.2.tar.gz/gcc-2.95.2/libiberty/concat.c
*/
char *cdw_string_concat(const char *first, ...)
{
	register size_t length;
	register char *newstr;
	register char *end;
	register const char *arg;
	va_list args;

	/* First compute the size of the result and get sufficient memory. */

	va_start (args, first);

	if (first == NULL) {
		length = 0;
	} else {
		length = strlen (first);
		while ((arg = va_arg (args, const char *)) != NULL) {
			length += strlen (arg);
		}
	}

	newstr = (char *) malloc (length + 1);
	va_end (args);

	/* Now copy the individual pieces to the result string. */

	if (newstr != NULL) {
		va_start (args, first);
		end = newstr;
		if (first != NULL) {
			arg = first;
			while (*arg) {
				*end++ = *arg++;
			}
			while ((arg = va_arg (args, const char *)) != NULL) {
				while (*arg) {
					*end++ = *arg++;
				}
			}
		}
		*end = '\0';
		va_end (args);
	}

	return (newstr);
}





/**
   \brief Append one string to another

   Append 'tail' string to 'head' string. 'head' is reallocated, that is why
   you have to pass a pointer to a string as a first arg.
   Result is a new 'head': pointer to concatenated 'head' and 'tail'.
   Old 'head' is freed, so make sure that 'head is dynamically allocated
   before passing it to append().

   \param head - the string that you want to append sth to
   \param tail - the string that you want to append to tail

   \return CDW_ERROR if some malloc/concat error occurred
   \return CDW_OK if success
 */
cdw_rv_t cdw_string_append(char **head, const char *tail)
{
	char *tmp = cdw_string_concat(*head, tail, NULL);
	if (tmp == NULL) {
		return CDW_ERROR;
	} else {
		free(*head);
		*head = tmp;
		return CDW_OK;
	}
}





/**
   \brief Check if given string doesn't contain any strange characters

   Make sure that string entered by user can't be used to do sth bad
   to/with application. Invalid characters are specified in cdw_string.h
   (see definition of CDW_STRING_UNSAFE_CHARS_STRING).

   Function accepts NULL pointers but then it returns CDW_GEN_ERROR.
   Empty strings ("") are treated as valid strings.

   \p invalid should be a string of length 2. If \p input has any invalid
   chars, first of them will be put into first char of \p invalid.
   \p invalid may be NULL, then function will not produce any output.

   \param string - string to be checked
   \param invalid - place where function will place first unsafe char from input_string

   \return CDW_NO if string contains any char that is not valid
   \return CDW_OK if string contains only valid characters
   \return CDW_GEN_ERROR if input_string is NULL
 */
cdw_rv_t cdw_string_security_parser(const char *string, char *invalid)
{
	if (string == (char *) NULL) {
		cdw_vdm ("ERROR: passing NULL string to the function\n");
		return CDW_ERROR;
	}

	char *c = strpbrk(string, CDW_STRING_UNSAFE_CHARS_STRING);
	if (c == (char *) NULL) {
		/* no unsafe chars */
		return CDW_OK;
	} else {
		if (invalid == (char *) NULL) {
			cdw_vdm ("WARNING: buffer for invalid char is NULL\n");
		} else {
			strncpy(invalid, c, 1);
			invalid[1] = '\0';
		}
		return CDW_NO;
	}
#if 0

	size_t l = strlen(string);
	for (size_t i = 0; i < l; i++) {
		int c = (int) string[i];
		if (isalnum(c) || isblank(c)
			|| string[i] == '-'
		        || string[i] == '~'
			|| string[i] == '.'
			|| string[i] == ','
			|| string[i] == '_'
			|| string[i] == '/'
			|| string[i] == '='
		        || string[i] == '+'
			|| string[i] == ':') {
			;
		} else {
			return CDW_NO;
		}
	}

	return CDW_OK;
#endif
}





/**
   \brief Replace content of target string with content of source string

   Copy string from source to target. If target space is not allocated,
   the function will do this. If target already points to some string,
   old target string will be overwritten, and target pointer may be modified.

   target pointer must be either empty pointer or pointer returned by
   malloc, realloc or calloc.

   *target is set to NULL if there are some (re)allocation errors.

   \param target - pointer to target string that you want to modify
   \param source - source string that you want to copy to target

   \return CDW_ERROR on errors
   \return CDW_OK on success
 */
cdw_rv_t cdw_string_set(char **target, const char *source)
{
	if (source == NULL) {
		return CDW_ERROR;
	}

	size_t len = strlen(source);

	char *tmp = (char *) realloc(*target, len + 1);
	if (tmp == NULL) {
		cdw_vdm ("ERROR: failed to realloc target string \"%s\" (source string = \"%s\")\n", *target, source);
		return CDW_ERROR;
	} else {
		*target = tmp;
	}

	strncpy(*target, source, len + 1);

	return CDW_OK;
}





/**
   \brief Check if given string starts with another string

   Silly wrapper for function that compares two strings. To be more precise
   the function compares only first n=strlen(substring) chars of both strings
   to check if first argument starts with given string (second argument).
   I could use strncmp(), but I want to avoid putting call to strlen() every
   time I need to check what a string starts with.

   You can use this function to check lines read from color configuration file.
   Make sure that 'line' starts with non-white chars. The search is
   case-insensitive (hence 'ci' in name).

   \param line - string that you want to check (to check if it starts with given substring)
   \param substring - field name that you are looking for in given line

   \return true if given line starts with field name,
   \return false otherwise
*/
bool cdw_string_starts_with_ci(const char *line, const char *substring)
{
	if (!strncasecmp(line, substring, strlen(substring))) {
		return true;
	} else {
		return false;
	}
}





/**
   \brief Free all strings in a given table

   Free list of strings that were malloc()ed. Free()d pointers are
   set to (char *) NULL. Table itself is not free()d.

   You can use this function e.g. to free labels of dropdowns.

   \param labels - table of labels that you want to free
   \param n_max - size of table (maximal number of strings to be free()d)
*/
void cdw_string_free_table_of_strings(char *labels[], int n_max)
{
	if (labels != (char **) NULL) {
		for (int i = 0; i < n_max; i++) {
			if (labels[i] != (char *) NULL) {
				free(labels[i]);
				labels[i] = (char *) NULL;
			}
		}
	}

	return;
}





/**
   \brief Create table of labels that represent given numbers

   Using int values from table of ints create table of (char *) strings
   (lables) that represent these ints. Inst should be no longer than
   3 digits.

   Last element of the list is set to (char *) NULL.
   \p labels is a pointer to already alocated table. Table size must be big
   enough to store all labels and one guard element at the end.

   \param labels - table where labels will be stored
   \param ints - input values (in a table) to be converted
   \param n - number of ints to convert, +1 for guard in result table

   \return CDW_ERROR on malloc() error
   \return CDW_OK on success
*/
cdw_rv_t cdw_string_ints2labels_3(char **labels, int ints[], size_t n)
{
	cdw_assert (labels != (char **) NULL, "you should alloc labels table\n");
	cdw_assert (n > 0, "incorect size of labels table: n = %d\n", (int) n);

	size_t i = 0;
	for (i = 0; i < n; i++) {
		labels[i] = (char *) malloc(3 + 1); /* 3 digits + 1 ending '\0' */
		if (labels[i] == NULL) {
			return CDW_ERROR;
		}
		memset(labels[i], ' ', 4);

		snprintf(labels[i], 3 + 1, "%d", ints[i]);

		cdw_sdm ("input int = %d, constructed labels[%d] = \"%s\"\n", ints[i], i, labels[i]);
	}
	/* obligatory ending element (guard) */
	labels[n] = (char *) NULL;

	return CDW_OK;
}





/**
   \brief Map several possible values of option to true/false values

   Function compares value in \p buffer with several defined strings to
   check if value in \p buffer should be treated as 'true' or 'false'.

   If string in \p buffer is recognized (either as having meaning of 'true'
   or 'false'), appropriate bool value is assigned to \p value and CDW_OK
   is returned. Otherwise value of \p value is not changed and CDW_ERROR
   is returned.

   There are several values of \p buffer that can be mapped into 'true',
   and several other values that can be mapped to 'false'. Case of string
   in \p buffer is not significant.

   If \p buffer is NULL function returns CDW_ERROR.

   Remember to remove leading and trailing white chars from \p buffer
   before passing it to the function.

   \param buffer - buffer that you want to check
   \param value - variable that you want to assign true/false to

   \return CDW_OK if value in first argument was recognized
   \return CDW_ERROR if value in first argument was not recognized or first argument is NULL
*/
cdw_rv_t cdw_string_get_bool_value(const char *buffer, bool *value)
{
	if (buffer == (char *) NULL) {
		return CDW_ERROR;
	}

	if (!strcasecmp(buffer, "1") || !strcasecmp(buffer, "yes") || !strcasecmp(buffer, "true")) {
		*value = true;
		return CDW_OK;
	} else if (!strcasecmp(buffer, "0") || !strcasecmp(buffer, "no") || !strcasecmp(buffer, "false")) {
		*value = false;
		return CDW_OK;
	} else {
		return CDW_ERROR;
	}
}





/**
   \brief Check if given string can be printed by ncurses, if not then
   produce printable representation

   Function uses mbstowsc() to check if \p string can be safely printed
   in ncurses window (i.e. if it contains printable characters only).

   If there are any non-printable characters, a pointer to freshly alloced
   printable representation of \p string is returned. Caller is responsible
   for free()ing such string. If \p string contain only printable characters,
   NULL is returned.

   \param string - string to be checked

   \return NULL if input string can be correctly displayed
   \return pointer to printable representation of input string if input string can't be correctly displayed
*/
char *cdw_string_get_printable_if_needed(const char *string)
{
	size_t len = strlen(string);
	wchar_t *wide_string = (wchar_t *) malloc((len + 1) * (sizeof(wchar_t)));
	/* mbstowcs() converts multibyte 'char *' strings into 'wchar_t' strings */
	size_t conv = mbstowcs(wide_string, string, len);
	free(wide_string);
	wide_string = (wchar_t *) NULL;

	if (conv == (size_t) -1) {
		/* there is some non-printable character in "string" */
		char *printable_string = strdup(string);
		if (printable_string == (char *) NULL) {
			cdw_assert (0, "ERROR: failed to strdup() string \"%s\"\n", string);
			return (char *) NULL;
		}
		for (size_t i = 0; i < len; i++) {
			if ((unsigned char) printable_string[i] > 127) {
				printable_string[i] = '?';
			}
		}
		return printable_string;
	} else {
		return (char *) NULL;
	}
}





/*
   - If string has words separated with spaces, return number of characters
     in given range up to end of last word;
   - If string has no spaces, return number of characters in range given by
     chars_max; If string is shorter than chars_max, the function will return
     length of string;
   - If there is a new line character, return number of characters before
     the new line character;
   - If string starts with the delimiting character (NUL, new line character,
     space), the function returns 0;

     The function is a bit too simple, because it can't handle
     strings with two or more consecutive delimiting characters.
*/
int cdw_string_get_words_end(const char *string, int chars_max)
{
	if (string[0] == '\n' || string[0] == '\0' || string[0] == ' ') {
		return 0;
	}

	int last_space = 0;
	int len = (int) strlen(string);
	int i = 0;
	for (i = 0; i < chars_max && i <= len; i++) {
		if (string[i] == ' ') {
			/* remember last position of space in
			   string as you traverse whole string */
			last_space = i;
		} else if (string[i] == '\n' || string[i] == '\0') {
			/* no need to traverse the string further,
			   this is where we must return */
			return i;
		} else {
			;
		}
	}
	/* we get here when one of two things happen: */
	if (last_space == 0) {
		/* no spaces or line breaks in given range of string,
		   return last position in traversed data */
		return i;
	} else {
		/* there was at least one space, its position was saved
		   as we traversed the string */
		return last_space;
	}
}





int cdw_string_wrap_for_form_get_n_lines(const char *message, int n_cols)
{
	char buf[10000];
	int n = 0;
	const char *point = message;

	int n_lines = 0;
	while (1) {
		/* n is a number of characters, not an index */
		n = cdw_string_get_words_end(point, n_cols);
		if (n == 0 && point[n] == '\0') {
			/* this is for empty strings */
			break;
		}
	        snprintf(buf, (size_t) n + 1, "%s", point);

		cdw_sdm ("INFO: line: (%zd chars): \"%s\"\n", n, buf);
		n_lines++;

		if (point[n] == '\0') {
			/* current substring ends with NUL, time to finish */
			break;
		} else {
			point = point + n + 1;
		}
	}

	return n_lines;
}





char *cdw_string_wrap_for_form(const char *message, int n_cols, int n_lines)
{
	int n = 0;
	const char *point = message;
	char *fragment = malloc((size_t) n_cols + 1);
	memset(fragment, ' ', (size_t) n_cols + 1);
	char *new_message = malloc((size_t) (n_cols * n_lines + 1));
	int out_buf_pointer = 0;

	while (1) {
		/* n is a number of characters, not an index */
		n = cdw_string_get_words_end(point, n_cols);
		if (n == 0 && point[n] == '\0') {
			/* this is for empty strings */
			break;
		}

		snprintf(fragment, (size_t) n + 1, "%s", point);
	        sprintf(new_message + out_buf_pointer, "%-*s", (int) n_cols, fragment);
		new_message[out_buf_pointer + n] = ' ';
		cdw_vdm ("INFO: new message = \"%s\"\n", new_message);
		out_buf_pointer += n_cols;

		cdw_sdm ("INFO: line: (%zd chars): \"%s\"\n", n, buf);

		if (point[n] == '\0') {
			/* current substring ends with NUL, time to finish */
			break;
		} else {
			point = point + n + 1;
		}
	}

	/* FIXME: 'out_buf_pointer' or 'out_buf_pointer + 1'? */
	new_message[out_buf_pointer] = '\0';
	cdw_vdm ("INFO: new message = \"%s\"\n", new_message);

	free(fragment);
	fragment = (char *) NULL;

	return new_message;
}




//#define CDW_UNIT_TEST_CODE
#ifdef CDW_UNIT_TEST_CODE


/* *********************** */
/* *** unit tests code *** */
/* *********************** */


static void test_cdw_string_set(void);
static void test_cdw_string_security_parser(void);
static void test_cdw_string_starts_with_ci(void);
static void test_cdw_string_free_table_of_strings(void);
static void test_cdw_string_rtrim(void);
static void test_cdw_string_ltrim(void);
static void test_cdw_string_concat(void);
static void test_cdw_string_append(void);
static void test_cdw_string_get_bool_value(void);
static void test_cdw_string_get_words_end(void);



void cdw_string_run_tests(void)
{
	fprintf(stderr, "testing cdw_string.c\n");

	test_cdw_string_set();

	test_cdw_string_security_parser();
	test_cdw_string_starts_with_ci();

	test_cdw_string_rtrim();
	test_cdw_string_ltrim();

	test_cdw_string_free_table_of_strings();

	test_cdw_string_concat();
	test_cdw_string_append();

	test_cdw_string_get_bool_value();

	test_cdw_string_get_words_end();

	fprintf(stderr, "done\n\n");

	return;
}




void test_cdw_string_set(void)
{
	fprintf(stderr, "\ttesting cdw_string_set()... ");

	int d = 0;

	/* initialization of data */

	/* note that second string is longer than first,
	   and third shorter than second */
	const char *buffer1 = "my small test string, but not too short";
	const char *buffer2 = "Beautiful world, isn't it? A beautiful world behind the firewall. I should go out more often. But not too far from my terminal.";
	const char *buffer3 = "short string";

	char *test_string1 = strdup(buffer1);
	assert(test_string1 != (char *) NULL);

	char *test_string2 = NULL;

	cdw_rv_t crv = CDW_NO;

	/* first test with source string == NULL */
	crv = cdw_string_set(&test_string1, (char *)NULL);
	assert(crv == CDW_ERROR);
	/* in this case first argument should be intact */
	d = strcmp(test_string1, buffer1);
	assert(d == 0);


	/* test with target string already having some content */
	crv = cdw_string_set(&test_string1, buffer2);
	assert(crv == CDW_OK);
	d = strcmp(test_string1, buffer2);
	assert(d == 0);


	/* test with the same target string, but this time set it to much shorter string */
	crv = cdw_string_set(&test_string1, buffer3);
	assert(crv == CDW_OK);
	d = strcmp(test_string1, buffer3);
	assert(d == 0);


	/* set null target to some value */
	crv = cdw_string_set(&test_string2, buffer1);
	assert(crv == CDW_OK);
	d = strcmp(test_string2, buffer1);
	assert(d == 0);


	free(test_string1);
	test_string1 = NULL;

	free(test_string2);
	test_string2 = NULL;

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_string_security_parser(void)
{
	fprintf(stderr, "\ttesting cdw_string_security_parser()... ");

	struct {
		const char *input;
		const char *output;
		cdw_rv_t crv;
	} test_data[] = { { (char *) NULL, (char *) NULL, CDW_ERROR },
			  { "test*test",   "*",   CDW_NO },
			  { "*testtest",   "*",   CDW_NO },
			  { "testtest*",   "*",   CDW_NO },
			  { "test&test",   "&",   CDW_NO },
			  { "&testtest",   "&",   CDW_NO },
			  { "testtest&",   "&",   CDW_NO },
			  { "test!test",   "!",   CDW_NO },
			  { "!testtest",   "!",   CDW_NO },
			  { "testtest!",   "!",   CDW_NO },
			  { "test;test",   ";",   CDW_NO },
			  { ";testtest",   ";",   CDW_NO },
			  { "testtest;",   ";",   CDW_NO },
			  { "test`test",   "`",   CDW_NO },
			  { "`testtest",   "`",   CDW_NO },
			  { "testtest`",   "`",   CDW_NO },
			  { "test\"test",  "\"",  CDW_NO },
			  { "\"testtest",  "\"",  CDW_NO },
			  { "testtest\"",  "\"",  CDW_NO },
			  { "test|test",   "|",   CDW_NO },
			  { "testtest|",   "|",   CDW_NO },
			  { "|testtest",   "|",   CDW_NO },
			  { "test$test",   "$",   CDW_NO },
			  { "testtest$",   "$",   CDW_NO },
			  { "$testtest",   "$",   CDW_NO },
			  /* many unusual chars, but this string is safe */
			  { "test tes t-.,_/ =:598,_/fduf98-. _/  no:", (char *) NULL, CDW_OK },
			  { (char *) NULL, (char *) NULL, CDW_CANCEL }};

	int i = 0;
	while (test_data[i].crv != CDW_CANCEL) {
		char output[2];
		cdw_rv_t crv = cdw_string_security_parser(test_data[i].input, output);
		cdw_assert(crv == test_data[i].crv, "ERROR: failed input test #%d\n", i);
		if (test_data[i].output != (char *) NULL) {
			cdw_assert (!strcmp(test_data[i].output, output), "ERROR: failed output test #%d\n", i);
		}
		i++;
	}

	fprintf(stderr, "OK\n");

	return;
}







void test_cdw_string_starts_with_ci(void)
{
	fprintf(stderr, "\ttesting cdw_string_starts_with_ci()... ");

	bool result;

	result = cdw_string_starts_with_ci("test string one", "test");
	assert(result == true);

	result = cdw_string_starts_with_ci("Test string one", "test");
	assert(result == true);

	result = cdw_string_starts_with_ci("Test string one", "Test");
	assert(result == true);

	result = cdw_string_starts_with_ci("test string one", "Test");
	assert(result == true);


	result = cdw_string_starts_with_ci("ttest string one", "test");
	assert(result == false);

	result = cdw_string_starts_with_ci(" test string one", "test");
	assert(result == false);

	result = cdw_string_starts_with_ci("1test string one", "test");
	assert(result == false);

	result = cdw_string_starts_with_ci("t est string one", "test");
	assert(result == false);

	result = cdw_string_starts_with_ci("one string test", "test");
	assert(result == false);

	fprintf(stderr, "OK\n");

	return;
}


void test_cdw_string_rtrim(void)
{
	fprintf(stderr, "\ttesting cdw_string_rtrim()... ");

	char *result;

	const char *test_strings[] = {
		"test string \t\r\n ",
		"test string \n\t\r ",
		"test string \r\n\t ",

		"test string \t\r\n",
		"test string \n\t\r",
		"test string \r\n\t",

		"test string \t\r \n ",
		"test string \n\t \r ",
		"test string \r\n \t ",

		"test string\t \r\n ",
		"test string\n \t\r ",
		"test string\r \n\t ",

		(char *) NULL };

	char test_buffer[20];

	int i = 0;
	while (test_strings[i] != (char *) NULL) {
		strcpy(test_buffer, test_strings[i]);
		result = cdw_string_rtrim(test_buffer);
		int r = strcmp(result, "test string");
		assert(r == 0);
		size_t l = strlen(result);
		assert(l == 11);
		assert(*(result + 11) == '\0');

		i++;
	}

	result = cdw_string_rtrim(NULL);
	assert(result == NULL);

	char empty[] = "";
	result = cdw_string_rtrim(empty);
	assert(*result == '\0');

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_string_ltrim(void)
{
	fprintf(stderr, "\ttesting cdw_string_ltrim()... ");

	char *result;

	const char *test_strings[] = {
		" \t\r\n test string",
		" \n\t\r test string",
		" \r\n\t test string",

		" \t\r\ntest string",
		" \n\t\rtest string",
		" \r\n\ttest string",

		" \t\r \n test string",
		" \n\t \r test string",
		" \r\n \t test string",

		"\t \r\n test string",
		"\n \t\r test string",
		"\r \n\t test string",
		(char *) NULL };


	int i = 0;
	while (test_strings[i] != (char *) NULL) {
		result = cdw_string_ltrim(test_strings[i]);
		int r = strncmp(result, "test string", strlen("test string"));
		assert(r == 0);
		size_t l = strlen(result);
		assert(l == 11);
		assert(*(result + 11) == '\0');

		i++;
	}

	result = cdw_string_ltrim(NULL);
	assert(result == NULL);

	char empty_string[] = "";
	result = cdw_string_ltrim(empty_string);
	assert(*result == '\0');

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_string_concat(void)
{
	fprintf(stderr, "\ttesting cdw_string_concat()... ");

	const char *substring1 = "four \t\r\n ";

	char *result1 = cdw_string_concat("one ", "two ", "three ", substring1, "five", (char *) NULL);
	assert(result1 != NULL);
	int r1 = strcmp(result1, "one two three four \t\r\n five");
	assert(r1 == 0);
	assert(*(result1 + 27) == '\0');

	char *result2 = cdw_string_concat("one ", "", "three ", substring1, "five", (char *) NULL);
	assert(result2 != NULL);
	int r2 = strcmp(result2, "one three four \t\r\n five");
	assert(r2 == 0);
	assert(*(result2 + 23) == '\0');

	/* this testcase may not be supported by your compiler if it does not
	support strings longer than 509 chars */
	const char *long_string1 = "dddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddd";
	const char *long_string2 = "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee";
	const char *long_string3 = "fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff";
	const char *sum_of_long_strings = "ddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddddeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeefffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff";
	char *result3 = cdw_string_concat(long_string1, long_string2, long_string3, (char *) NULL);
	assert(result3 != NULL);
	int r3 = strcmp(result3, sum_of_long_strings);
	assert(r3 == 0);
	assert(*(result3 + strlen(sum_of_long_strings)) == '\0');

	free(result1);
	free(result2);
	free(result3);

	fprintf(stderr, "OK\n");

	return;
}




void test_cdw_string_append(void)
{
	fprintf(stderr, "\ttesting cdw_string_append()... ");

	char *head = (char *) malloc(strlen("test_string_one") + 1);
	assert(head != NULL);
	strcpy(head, "test_string_one");

	cdw_rv_t crv = cdw_string_append(&head, "I_LOVE_TESTING");
	assert(crv == CDW_OK);
	int r = strcmp(head, "test_string_oneI_LOVE_TESTING");
	assert(r == 0);

	crv = cdw_string_append(&head, "");
	assert(crv == CDW_OK);
	r = strcmp(head, "test_string_oneI_LOVE_TESTING");
	assert(r == 0);

	free(head);

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_string_free_table_of_strings(void)
{
	fprintf(stderr, "\ttesting cdw_string_free_table_of_strings()... ");

#define N_STRINGS_MAX 9
	const char *strings[N_STRINGS_MAX] = {"one", "two", "three", "four", "five", "six", "seven", "eight", "nine"};
	char **table = (char **) malloc((N_STRINGS_MAX + 1) * sizeof (char *)); /* +1 for guard */
	assert(table != (char **) NULL);

	table[N_STRINGS_MAX] = (char *) NULL;
	for (int i = 0; i < N_STRINGS_MAX; i++) {
		table[i] = (char *) NULL;
		table[i] = strdup(*(strings + i));
		assert(table[i] != (char *) NULL);

		cdw_sdm ("INFO: table[%d] = \"%s\"\n", i, table[i]);
	}

	cdw_string_free_table_of_strings(table, N_STRINGS_MAX);

	for (int i = 0; i < N_STRINGS_MAX + 1; i++) { /* +1 - check guard too */
		assert(table[i] == (char *) NULL);
	}
	free(table);
	table = (char **) NULL;

	fprintf(stderr, "OK\n");
	return;
}





void test_cdw_string_get_bool_value(void)
{
	fprintf(stderr, "\ttesting cdw_string_get_bool_value()... ");

	/* remember that cdw_string_get_bool_value() is called with
	   values that are already trimmed */

	/* also remember that value of second arg is not changed when
	   value in first buffer is not recognized */

	bool result = false;
	cdw_rv_t crv = cdw_string_get_bool_value((char *) NULL, &result);
	cdw_assert_test (crv == CDW_ERROR, "ERROR: failed crv test #-1\n");
	cdw_assert_test (result == false, "ERROR: failed bool test #-1\n");

	struct {
	        const char *input_string;
		bool input_bool;
		bool expected_bool;
		cdw_rv_t expected_crv;
	} test_data[] = {
		/* tests with incorrect input */
		{ "0 ",    true,  true,  CDW_ERROR },
		{ "foo",   true,  true,  CDW_ERROR },
		{ "bar",   false, false, CDW_ERROR },
		{ " true", false, false, CDW_ERROR },
		{ "",      true,  true,  CDW_ERROR },

		/* test with correct input meaning 'true' */
		{ "1",     false, true, CDW_OK },
		{ "yes",   false, true, CDW_OK },
		{ "YES",   false, true, CDW_OK },
		{ "yEs",   false, true, CDW_OK },
		{ "true",  false, true, CDW_OK },
		{ "TrUe",  false, true, CDW_OK },
		{ "TRUE",  false, true, CDW_OK },

		/* tests with correct input meaning 'false' */
		{ "0",      true, false, CDW_OK },
		{ "no",     true, false, CDW_OK },
		{ "NO",     true, false, CDW_OK },
		{ "nO",     true, false, CDW_OK },
		{ "false",  true, false, CDW_OK },
		{ "FaLse",  true, false, CDW_OK },
		{ "FALSE",  true, false, CDW_OK },

		{ (char *) NULL, true, true, CDW_OK }};

	int i = 0;
	while (test_data[i].input_string != (char *) NULL) {
		result = test_data[i].input_bool;
		crv = cdw_string_get_bool_value(test_data[i].input_string, &result);
		cdw_assert_test (crv == test_data[i].expected_crv, "ERROR: failed crv test #%d\n", i);
		cdw_assert_test (result == test_data[i].expected_bool, "ERROR: failed bool test #%d\n", i);
		i++;
	}

	fprintf(stderr, "OK\n");
	return;
}



static void test_cdw_string_get_words_end(void)
{
	fprintf(stderr, "\ttesting cdw_string_get_words_end()... ");
	struct {
		const char *string;
		size_t chars_max;
		int expected_value;
	} test_data[] = {
		{ "", 1,   0 },
		{ "", 10,  0 },

		{ "hello world", 1, 1 },
		{ "hello world", 5, 5 },
		{ "hello world", 6, 5 },
		{ "hello world", 7, 5 },

		{ "hello\nworld", 1, 1 },
		{ "hello\nworld", 5, 5 },
		{ "hello\nworld", 6, 5 },
		{ "hello\nworld", 7, 5 },
		{ "hello\nworld", 27, 5 },


		{ "12345678901234567890", 1, 1 },
		{ "12345678901234567890", 10, 10 },
		{ "12345678901234567890", 20, 20 },
		{ "12345678901234567890", 21, 20 },
		{ "12345678901234567890", 22, 20 },
		{ "12345678901234567890", 30, 20 },
		{ "12345678901234567890 12345678901234567890", 30, 20 },
		{ "12345678901234567890 12345678901234567890", 45, 41 },

		{ (char *) NULL, 0, 0 },

	};

	int i = 0;
	while (test_data[i].chars_max != 0) {
		int result = cdw_string_get_words_end(test_data[i].string, test_data[i].chars_max);
		cdw_assert (result == test_data[i].expected_value,
			    "ERROR: test #%d: string = \"%s\", expected value = %zd, result = %zd\n",
			    i, test_data[i].string, test_data[i].expected_value, result);
		i++;
	}

	fprintf(stderr, "OK\n");
	return;
}


#endif /* #ifdef CDW_UNIT_TEST_CODE */
