/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * \mainpage cdw
 *
 * ncurses interface for various cd/dvd tools
 *
 * \author Varkonyi Balazs, Kamil Ignacak
 */

/**
 * \file main.c
 *
 * Content of file:
 * \li main()
 * \li cdw_main_loop(): main loop for handling events/keys
 *
 * cdw_main_loop() calls many functions whose names start with
 * 'command' - these functions are defined in commands.c.
 * This is to keep all details and wrapping code away form main.c.
 */


#ifndef CDW_UNIT_TEST_CODE


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <mcheck.h>

#include <ncursesw/ncurses.h>
#include <ncursesw/menu.h>

#include <dirent.h> /* PATH_MAX */

#include "main.h"
#include "cdw_fs.h"
#include "cdw_help.h"
#include "cdw_config.h"
#include "cdw_config_ui.h"
#include "gettext.h"
#include "cdw_utils.h"
#include "cdw_logging.h"
#include "cdw_ext_tools.h"
#include "cdw_config.h"
#include "cdw_file_manager.h"
#include "cdw_main_window.h"
#include "cdw_ncurses.h"
#include "cdw_burn_disc.h"
#include "cdw_erase_disc.h"
#include "cdw_create_image.h"
#include "cdw_read_disc.h"
#include "cdw_read_disc_info.h"
#include "cdw_widgets.h"
#include "cdw_drive.h"
#include "cdw_debug.h"
#include "cdw_file.h"
#include "cdw_file_picker.h"
#include "cdw_md5sum.h"
#include "cdw_cdio_drives.h"
#include "cdw_sys.h"

extern cdw_config_t global_config;


/* wrapper function prototypes */
static int cdw_main_loop(void);
static void cdw_conditional_exit(void);


int main(int argc, char *argv[])
{
#ifndef NDEBUG
	mtrace();
#endif
	cdw_sys_signal_handlers_init();

	cdw_locale_init();
	/* there is no cdw_locale_clean() to register with atexit() */

	bool support_dvd_rp_dl = false;
	if (argc > 1) {
		int rv = process_commandline_args(argc, argv);
		if (rv == 1) {
			support_dvd_rp_dl = true;
		} else { /* rv == 0 || rv == -1 */
			/* TODO: call clean_before_cdw_exit() here
			   and see how many assertions fail */
			return rv;
		}
	}

	/* initialize ncurses - do this as soon as possible,
	 * some error situations may require use of dialog windows */
	cdw_rv_t crv = cdw_ncurses_init();
	atexit(cdw_ncurses_clean);
	if (crv != CDW_OK) {
		exit(EXIT_FAILURE);
	}

	curses_colors_init_phase1();
	/* there is no curses_colors_clean() to register with atexit() */

	crv = cdw_fs_init();
	atexit(cdw_fs_clean);
	if (crv != CDW_OK) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window;
				      initial configuration of tmp files failed */
				   _("Error occurred when configuring home or tmp directory. Closing."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		exit(EXIT_FAILURE);
	}

	curses_colors_init_phase2();
	/* there is no curses_colors_clean() to register with atexit() */

	{
		/* order of calling these init functions is particularly
		   important; it goes like this: */

		/* cdw_config module reads in hardware configuration
		   from configuration file; from now on "custom device"
		   and "cdw drive" fields have some valid values */
		cdw_config_module_init();
		atexit(cdw_config_module_clean);
		global_config.support_dvd_rp_dl = support_dvd_rp_dl;

		/* cdw_cdio_drives module detects drives in user's system;
		   it seems that it doesn't matter which of the two modules
		   (config / cdio_drives) will be called first */
		cdw_cdio_drives_init();
		atexit(cdw_cdio_drives_clean);

		/* cdw_drive checks data provided by cdw_config and
		   cdw_cdio_drive modules and will be able to return
		   correct value when cdw_drive_get_drive_fullpath()
		   is called */
		cdw_drive_init();
		atexit(cdw_drive_clean);

		/* cdw_cdio needs correct fullpath to currently selected
		   drive - the path is provided by already configured
		   cdw_drive module */
		cdw_cdio_init();
		atexit(cdw_cdio_clean);
	}

	cdw_disc_init();
	/* there is no cdw_disc_clean() to register with atexit() */

	crv = cdw_logging_init();
	atexit(cdw_logging_clean);
	if (crv != CDW_OK) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window;
				      initial configuration of tmp files failed */
				   _("Error occurred when configuring log file. Closing."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		exit(EXIT_FAILURE);
	}

	crv = cdw_ext_tools_init();
	atexit(cdw_ext_tools_clean);
	if (crv == CDW_NO) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window;
				      cdw can't find "which" utility on computer */
				   _("Error occurred when configuring external tools: \"which\" tool is not available on your system. Closing."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		exit(EXIT_FAILURE);
	} else if (crv == CDW_GEN_ERROR) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window;
				      initial configuration of external tools
				      (e.g. cdrecord, mkisofs) failed */
				   _("Unknown error occurred when configuring external tools. Closing."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		exit(EXIT_FAILURE);
	} else { /* CDW_OK */
		;
	}

	crv = cdw_file_manager_init();
	atexit(cdw_file_manager_clean);
	if (crv != CDW_OK) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window;
				      initial configuration of cdw file manager
				      failed */
				   _("Error occurred when configuring file manager. Closing."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		exit(EXIT_FAILURE);
	}

	/* main app window */
	crv = cdw_main_ui_init();
	atexit(cdw_main_ui_clean);
	if (crv != CDW_OK) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window;
				      initial configuration of cdw UI failed */
				   _("Error occurred when configuring user interface files. Closing."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		exit(EXIT_FAILURE);
	}

	/* initially cursor is on first left-hand menu item, but tooltips are
	   displayed only after cursor movement; show tooltip for initial
	   cursor position */
	display_tooltip(0);

	/* main program loop - top level keys handler */
	cdw_main_loop();

	/* clean up of modules is made by functions registered with atexit() */

	return 0;
}





/**
 * \brief Main ncurses events loop - main loop handling keys
 *
 * Function that handles keys when cdw is started and user sits in front of
 * main cdw window: menu on the left, list of files in main area and
 * information area at the bottom. User can select items from left-hand menu
 * or use hotkeys.
 *
 * \return 0
 */
int cdw_main_loop(void)
{
	MENU *menu = cdw_main_ui_get_main_menu();
	WINDOW *window = cdw_main_ui_get_main_window();
	int c = 'a';

	while ((c = wgetch(window))) {
		switch (c) {

		/* ****************************** *
		 * **********  HOT KEYS  ******** *
		 * ****************************** */


		case 'i':
		case 'I': /* write image to optical disc */
			cdw_burn_disc(CDW_TASK_BURN_FROM_IMAGE);
			break;
		case 'f':
		case 'F':
			/* write selected files to optical disc */
			cdw_burn_disc(CDW_TASK_BURN_FROM_FILES);
			break;
		case 'e':
		case 'E':
			cdw_drive_toggle_tray_with_ui_update2();
			break;
		case 'g':
		case 'G':
			/* Copy content of data CD or audio CD to your hard disc */
			cdw_read_disc();
			break;
		case 'r':
		case 'R':
			/* read disc meta information and display basic
			   information in 'Disc info' area */
			cdw_read_disc_info();
		//cdw_main_ui_disc_info_view_update();
			break;
#ifndef NDEBUG /* only for testing purposes */
		case 'v':
		case 'V':
			{
				CDW_TASK_CREATE_TASK(task, CDW_TASK_CHECK_MD5SUM);
				task.verify.tool.id = CDW_TOOL_MD5SUM;
				task.verify.tool.fullpath = "/usr/bin/md5sum";
				task.verify.iso_fullpath = global_config.iso_image_full_path;

				cdw_verify(&task);
				break;
			}
#endif

#if 0
		case 'P':
			{
				char *fullpath = malloc(PATH_MAX);
				if (fullpath == (char *) NULL) {
					cdw_vdm ("ERROR: failed to allocate memory for fullpath\n");
				} else {
					strcpy(fullpath, "/home/acerion/");
					cdw_rv_t crv = cdw_fs_ui_file_picker("file picker", "select a file", &fullpath, R_OK | W_OK, CDW_FS_FILE, CDW_FS_NEW | CDW_FS_EXISTING);
					if (crv == CDW_OK) {
						cdw_vdm ("INFO: file picker returns \"%s\"\n", fullpath);
					} else if (crv == CDW_CANCEL) {
						cdw_vdm ("INFO: pressed escape in file picker, fullpath is \"%s\"\n", fullpath);
					} else {
						cdw_vdm ("INFO: file picker returns CDW_GEN_ERROR, fullpath is \"%s\"\n", fullpath);
					}
					free(fullpath);
					fullpath = (char *) NULL;
				}
				break;
			}
#endif

		case 'q':
		case 'Q':
			/* this function calls exit() only if user confirms closing appication */
			cdw_conditional_exit();
			break;
		case KEY_F(1):
		case 'H':
		case 'h':
		case '?':
			/*  Show small help window */
			show_help_main();
			break;
		case 'l':
		case 'L':
			/* 2TRANS: this is a title of window displaying
			   content of log file */
			cdw_logging_display_log(_("cdw log file"));
			break;
		case 'd':
	        case 'D':
			cdw_read_and_display_disc_info();
		        break;
		case KEY_F(10):
			cdw_main_ui_show_license();
			break;

#if 0 /* not in this release */
#ifdef HAVE_LIBMYSQLCLIENT
			case 'c':
			case 'C':
			/* CDDB */
			cddb_window();
			break;
#endif
#ifdef HAVE_LIBSQLITE
			case 'c':
			case 'C':
			/* CDDB */
			cddb_window();
			break;
#endif
#ifdef HAVE_LIBMYSQLCLIENT
		case 'A':
		case 'a':
			/* Add disk to dic database */
			add_to_dic();
			break;
#endif
#ifdef HAVE_LIBSQLITE
		case 'A':
		case 'a':
			/* Add disk to dic database */
			add_to_dic();
			break;
#endif
#endif

		/* ****************************** *
	         * ******  LEFT-HAND MENU  ****** *
		 * ****************************** */


		case KEY_HOME:
			menu_driver(menu, REQ_FIRST_ITEM);
			display_tooltip(item_index(current_item(menu)));
			break;
		case KEY_END:
			menu_driver(menu, REQ_LAST_ITEM);
			display_tooltip(item_index(current_item(menu)));
			break;
		case KEY_DOWN:
			{
				int n = item_count(menu); /* n includes ending NULL element */
				int ind = item_index(current_item(menu));
				if (n - 1 == ind) {
					menu_driver(menu, REQ_FIRST_ITEM);
				} else {
					menu_driver(menu, REQ_DOWN_ITEM);
				}
				display_tooltip(item_index(current_item(menu)));
			}
			break;
		case KEY_UP:
			{

				int ind = item_index(current_item(menu));
				if (ind == 0) {
					menu_driver(menu, REQ_LAST_ITEM);
				} else {
					menu_driver(menu, REQ_UP_ITEM);
				}
				display_tooltip(item_index(current_item(menu)));
			}
			break;
		case CDW_ENTER:
			{
			int ii = item_index(current_item(menu));
			/* UI_MENU* elements are numbered from 0, so they
			   neatly correspond with menu item numbers */

			if (ii == CDW_MENU_ADD_FILES) { /* Add files to list of files to be written to cd/image */
				cdw_main_ui_add_to_selected_files();

			} else if (ii == CDW_MENU_DELETE_FILES) {
				/* Delete files from list of files to be
				written to cd/image; this function regenerates
				list of files and files info area */
				cdw_main_ui_delete_from_selected_files();

			} else if (ii == CDW_MENU_CREATE_IMAGE) {
				/* Create iso image using selected files */
				cdw_create_image();
			} else if (ii == CDW_MENU_RIP_DISC) {
				/* Copy content of audio / data CD or data DVD to your hard disc */
				cdw_read_disc();
			} else if (ii == CDW_MENU_BURN_FILES) {
				cdw_burn_disc(CDW_TASK_BURN_FROM_FILES);
			} else if (ii == CDW_MENU_BURN_IMAGE) {
				cdw_burn_disc(CDW_TASK_BURN_FROM_IMAGE);
			} else if (ii == CDW_MENU_ERASE_DISC) {
				cdw_erase_disc();
			} else if (ii == CDW_MENU_CONFIG) {
				/* Edit cdw options */
				/* config_window() reads current
				 * configuration from global variable 'struct
				 * conf config', shows user a window that
				 * allows him modify the options, save options
				 * to global variable and to disk file */
				cdw_config_ui_window(true);
			} else if (ii == CDW_MENU_EXIT_CDW) { /* Quit - Exit cdw */
				/* this function calls exit() only if user
				   confirms closing appication */
				cdw_conditional_exit();
			} else {
				;
#if 0 /* not in this release */
				/* Write Audio */
				if ((item_index(current_item(menu)) == 5)) {
					run_command_cdrecord_write_audio();
				}
					/* Copy data CD */
					/* watch out! config.cdrom is disabled in options.c */
					if ((item_index(current_item(menu)) == 6) && (config.cdrom != config.cdrw_device)) {
					run_command_copy_disk();
				}
#endif
			}
			}
			break;
		default:
			break;
		} /* switch(c) */
		cdw_main_ui_main_window_wrefresh();
	} /* while (c = wgetch(main_cdw_win)) */

	return 0;
}






/**
   \brief Exit program if user confirms it

   Ask user if he really wants to quit program and call
   exit(EXIT_SUCCESS) if user confirms.

   This code occurred twice in main loop, so I extracted it. It not only
   re-uses code but also helps avoid translating the same two messages twice.
*/
void cdw_conditional_exit(void)
{
	/* 2TRANS: this is title of dialog window */
	cdw_rv_t crv = cdw_buttons_dialog(_("Please confirm"),
					  /* 2TRANS: this is message in dialog window,
					     user can press OK or Cancel */
					  _("Do you really want to quit cdw?"),
					  CDW_BUTTONS_OK_CANCEL, CDW_COLORS_WARNING);
	if (crv == CDW_OK) {
		exit(EXIT_SUCCESS);
	}
	return;
}



#else /* CDW_UNIT_TEST_CODE is defined */



#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <ncursesw/ncurses.h>

#include "main.h"
#include "config.h"

#include "cdw_dll.h"
#include "cdw_string.h"
#include "cdw_cdll.h"
#include "cdw_fs.h"
#include "cdw_file.h"
#include "cdw_utils.h"
#include "cdw_config.h"
#include "cdw_debug.h"
#include "cdw_which.h"
#include "gettext.h"

static WINDOW *cdw_main_test_window = (WINDOW *) NULL;
WINDOW *cdw_main_test_subwin = (WINDOW *) NULL;



int main(int argc, char **argv)
{
	if (argc > 1) {
		/* if ((strcmp((char *) argv[1], "--help") == 0) || (strcmp((char *) argv[1], "-h") == 0)) { */
		/* 2TRANS: this is message printed in terminal,
		   first %s is name of a testing program, second %s is
		   a name of this software package */
		printf(_("\n%s: testing tool for %s\n\n"), argv[0], PACKAGE);
		/* 2TRANS: this is message printed in terminal,
		   %s is program name; keep square brackets */
		printf(_("Usage: %s [options]\n\n"), argv[0]);
		/* 2TRANS: this is message printed in terminal,
		   a header for options supported by this program */
		printf(_("Options:\n"));
		//printf("  -i                        : start interactive testing\n");
		/* 2TRANS: this is message printed in terminal */
		printf(_("  --help    | -h            : show this screen\n"));
		/* 2TRANS: this is message printed in terminal */
		printf(_("\nInvoke this program without options to run unit tests\n\n"));
	} else {
		fprintf(stderr, "\n");
		/* 2TRANS: this is message printed in terminal;
		   first %s is name of a testing program, second %s is
		   a name of this software package */
		fprintf(stderr, _("%s: %s unit tests facility\n\n"), argv[0], PACKAGE);

		cdw_dll_run_tests();
		cdw_string_run_tests();
		cdw_cdll_run_tests();
		cdw_config_run_tests();
		cdw_fs_run_tests();
		cdw_utils_run_tests();
		cdw_file_run_tests();
		cdw_which_test();

		/* "make check" facility requires this message to be
		   printed on stdout; don't localize it */
		fprintf(stdout, "test result: success\n\n");
	}

	return 0;
}



#endif /* #ifndef CDW_UNIT_TEST_CODE */
