/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <ncursesw/ncurses.h>
#include <ncursesw/menu.h>
#include <ncursesw/form.h>

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <linux/cdrom.h>
#include <fcntl.h>
#include <dirent.h>
#include <nl_types.h>
#include <libintl.h>

#include "gettext.h"
#include "thread.h" /* run_command() */
#include "cdw_widgets.h" /* nice_box() */
#include "log.h"
#include "options.h" /* struct conf */
#include "utils.h"
#include "cdw_tmp_files.h"
#include "processwin.h"
#include "cdw_ui.h"
#include "main.h"
#include "config.h"
#include "cdw_cdio.h" /* some functions used in creating iso image from cd */

#include "commands.h"

/**
 * Top-level functions for actions related to reading or writing CD or
 * image file. These actions are performed when user selects some command
 * available via UI (letf-hand menu buttons).
 */


extern struct conf config;      /* variable holding cdw configuration */
extern char tmp_file_name[];    /* file with graft points used when writing files to CD */


/* Is media currently in drive erasable? */
enum media_erasable_t media_erasable;

/* This variable allows us to find out what task is currently preformed.
 * It is helpful when we want to select some handler, and other existing
 * information may be misleading. */
enum current_command_t current_command;


/* helper functions prototypes */
bool get_multisession_tracks(int *track_start, int *track_end);




/**
 * Create iso image from files selected by user
 *
 * Create command line string (program name + arguments) for creating iso
 * image from selected files. Run the command, show a window with process
 * information to the user.
 *
 * This function calls filelist_to_file() to write
 * selected files' paths to tmp file used later by mkisofs. File image path
 * should be set in preferences (config.tempdir). This function checks if
 * the path is valid.
 *
 * Currently only writing data files is supported.
 *
 * \returns 0 when iso file is created; -1 if operation cancelled, -2 if operation_failed
 */
int run_command_create_image(void)
{
	char createiso_command[500];

	current_command = COMMAND_CREATE_IMAGE;

	if (strlen(config.bootimg)) {
		int boot_exists = check_for_boot_image();
		if (!boot_exists) {
			return -1;
		}
	}

	/* flush current list of selected files to tmp file */
	filelist_to_file();

	int filedes;
	bool image_ok = open_target_image_file(&filedes);
	if (image_ok) { /* can write to image file */
		close(filedes);  /* close - MKISOFS will create it anyway */
		unlink(config.tempdir); /* unlink - open_target_image_file() created it with writing permissions only */

		/* prepare command creating iso image, with writing
		 * image to file (config.tempdir should not be empty or invalid) */
		prepare_createiso_command(createiso_command, config.tempdir);

		/* boot image exists, iso file doesn't already exist (or can be overwritten), so... */
		/* 2TRANS: this is title of dialog window */
		processwin_create(_("Create image"),
				  /* 2TRANS: this is message in dialog window - creating iso image is in progress */
				  _("Creating iso image..."), true);
		run_command(createiso_command);
		/* 2TRANS: this is message in dialog window - iso image file was created */
		processwin_destroy(_("Finished successfully"), true);
	} else { /* cannot/will not write to image (for some reason) */
		/* 2TRANS: this is title of dialog window */
		dialogbox(_("Message"),
			  /* 2TRANS: this is message in dialog window - iso image was not created (for some reason: maybe error or maybe user canceled operation) */
			  _("Image not created"), DIALOG_OK);
	}

	current_command = COMMAND_NONE;

	return 0;
}




/**
 * Write selected iso image to optical disc
 *
 * Create command line string (program name + arguments) for writing iso
 * image to optical disc. Run the command, show a window with process
 * information to the user.
 *
 * File image path should be set in preferences (config.tempdir).
 *
 * Currently only writing data files is supported.
 *
 * \returns 0
 */
int run_command_write_from_image(void)
{
	char command[500];
	current_command = COMMAND_WRITE_FROM_IMAGE;

	sprintf(command, "%s -v speed=%s dev=%s %s", CDRECORD, config.speed, config.scsi, config.other);
	if (strcmp(config.dao, "1") == 0) {
		strcat(command, " -dao");
	}
	if (strcmp(config.dummy, "1") == 0) {
		strcat(command, " -dummy");
	}
	if (strcmp(config.burnproof, "1") == 0) {
		strcat(command, " -driveropts=burnproof");
	}
	sprintf(command, "%s -data %s", command, config.tempdir);

	/* 2TRANS: this is title of dialog window */
	processwin_create(_("Write image to CD"),
			  /* 2TRANS: this is message in dialog window - writing iso image to CD is in progress */
			  _("Writing image to CD..."), true);
	run_command(command);
	/* 2TRANS: this is message in dialog window */
	processwin_destroy(_("Writing finished"), true);

	current_command = COMMAND_NONE;

	return 0;
}




/**
 * Write files selected by user to optical disc
 *
 * Create command line string (program name + arguments) for writing files
 * to optical disc. Run the command, show a window with process
 * information to the user.
 *
 * Run program that will create iso image and write the image to it's stdout.
 * Run another program, that will read iso file (via normal shell
 * pipe '|') from it's stdin and write it to CD disk.
 *
 * In previous versions cdw created symlinks to all files that should be
 * written to iso image, and told program creating iso image to follow
 * these symlinks; current method is to write list of files to tmp file and
 * feed this file to program creating iso image (that is how k3b
 * does it, I think). Look for graft-points in MKISOFS man page.
 *
 * \returns 0 on success, val < 0 on failure
 */
int run_command_write_direct(void)
{
	char command[1000];
	int boot_exists;

	char writetoCD_command[520];
	char tsize_command[270];
	char createiso_command[520];

	current_command = COMMAND_WRITE_DIRECT;
	memset(command, 1000, '\0');

	if (strlen(config.bootimg)) {
		boot_exists = check_for_boot_image();

		if (!boot_exists) {
			current_command = COMMAND_NONE;
			return -2;
		}
	}

	/* flush current list of selected files to tmp file */
	filelist_to_file();

	/* prepare command creating iso image, without writing
	 * image to file (that is why second
	 * parameter - filename - is empty) */
	prepare_createiso_command(createiso_command, "");

	/* prepare command creating tsize - shell variable that will
	 * be used by program writing to disk - read
	 * prepare_writetoCD_command() function for explanation */
	sprintf(tsize_command, "tsize=`%s --print-size -q`", createiso_command);

	/* prepare command reading iso image from stdin and
	 * writing it to CD disk */
	prepare_writetoCD_command(writetoCD_command);

	/* prepare_writetoCD_command does not append input
	 * file - let's do this here: read form stdin (pipe)*/
	sprintf(writetoCD_command, "%s -", writetoCD_command);

	/* final version of 'write direct' command is here: */
	sprintf(command, "%s; %s | %s%c", tsize_command, createiso_command, writetoCD_command, '\0');

	/* 2TRANS: this is title of dialog window */
	processwin_create(_("Write files to CD"),
			  /* 2TRANS: this is message in dialog window: writing selected files to CD is in progress */
			  _("Writing files to CD..."), true);
	run_command(command);
	/* 2TRANS: this is message in dialog window: operation finished with unknown result */
	processwin_destroy(_("Writing files to CD finished"), true);

	current_command = COMMAND_NONE;

	return 0;

}




/*
 * Create process writing audio CD
 */
int run_command_write_audio(void)
{
	char command[500];
	cdw_rv_t rv = CDW_OK;

	current_command = COMMAND_WRITE_AUDIO;

	rv = conditional_volume_label_dialog();

	if (rv == CDW_OK) { /* user entered label: empty or non-empty, but continue in both cases */
		sprintf(command, "%s -v speed=%s dev=%s -audio", CDRECORD, config.speed, config.scsi);
		if (strcmp(config.pad, "1") == 0) {
			strcat(command, " -pad");
		}
		if (strcmp(config.dao, "1") == 0) {
			strcat(command, " -dao");
		}
		if (strcmp(config.dummy, "1") == 0) {
			strcat(command, " -dummy");
		}
		if (strcmp(config.burnproof, "1") == 0) {
			strcat(command, " -driveropts=burnproof");
		}
		sprintf(command, "%s %s/*.wav", command, config.audiodir);
		/* 2TRANS: this is dialog window title (wiriting audio files to CD) */
		processwin_create(_("Write audio"),
				  /* 2TRANS: this is message in dialog window: writing audio CD is in progress */
				  _("Writing audio..."), true);
		run_command(command);
		/* 2TRANS: this is message in dialog window; result of operation is unknown */
		processwin_destroy(_("Writing audio finished"), true);

		/* 2TRANS: this string will be used as title of window displaying log of operation (of writing audio CD) */
		after_event(_("Write audio log"), 1);
	}

	current_command = COMMAND_NONE;

	return 0;
}




/*
 * Create process copying CD disk
 */
int run_command_copy_disk(void)
{
	char command[500];
	int in_fd;

	current_command = COMMAND_COPY_DISK;

	if ((in_fd = open("/dev/cdrom", O_RDONLY)) != -1) {
		close(in_fd);
		sprintf(command, "%s -v speed=%s dev=%s %s", CDRECORD, config.speed, config.scsi, config.other);
		if (strcmp(config.eject, "1") == 0) {
			strcat(command, " -eject");
		}
		if (strcmp(config.dummy, "1") == 0) {
			strcat(command, " -dummy");
		}

		sprintf(command, "%s -isosize %s", command, config.cdrom);
		/* 2TRANS: this is dialog window title */
		processwin_create(_("Copy data CD"), "", true);
		run_command(command);
		/* 2TRANS: this is message in dialog window: result of operation (success/failure) unknown */
		processwin_destroy(_("Copying data CD finished"), true);

		/* 2TRANS: this string will be used as title of window displaying log of operation (of copying CD) */
		after_event(_("Copy data CD log"), 1);
	}

	current_command = COMMAND_NONE;

	return 0;
}




/**
 * Create 'char *' string containing program name and program's argument
 * for creating iso image
 *
 * iso image is created in two different situations:
 * - when user only creates iso image (this is explicit creation,
 *   image file is written to disk)
 * - when user writes directly to CD disk - iso image is piped to
 *   program writing it to cd
 * .
 * Both cases are very similar, only in second case there is no output file,
 * since CD writing program reads from pipe (from its stdin).
 * This function appends output image file name to command (it takes it from
 * config.tempdir).
 *
 * \param char *command - string for program name and it's parameters
 * \param char *iso_file - path (full or relative) to output iso file + file
 *                         name with extension; set parameter to empty string
 *                         it you don't want to write image to file
 *
 * \returns -1 on malloc error, 0 on success
 */
int prepare_createiso_command(char *command, char *iso_file)
{
	int last_sess_start = 0, next_sess_start = 0;
	/* does current disc have any session already? */
	bool ismulti = false;

	/* 'graft points' means text file with list of files to burn AND root
	 * directories on CD that the files will be put into;
	 * see also -path-list argument below */
	sprintf(command, "%s  -graft-points -iso-level 2", MKISOFS);

	/* get some information about possible previous session(s) on disc
	 * currently in drive, but only if we are writing files directly
	 * to cd, othrewise multisession information in image file is wrong;
	 * do this check so early, this way we may avoid running external
	 * process */
	if (strlen(iso_file) == 0) {
		if (strcmp(config.multi, "1") == 0) {
			/* fprintf(stderr, "config.multi = 1\n"); */
			ismulti = get_multisession_tracks(&last_sess_start, &next_sess_start);
		} else {
			/* fprintf(stderr, "config.multi = 0\n"); */
			ismulti = false;
		}
	} else { /* name of output image file is empty string */
		ismulti = false; /* just to be sure */
	}

	/* debug code
	if (ismulti) {
		fprintf(stderr, "tested: ismulti\n");
	} else {
		fprintf(stderr, "tested: is not ismulti\n");
	} */

	/* append multisession track info, but only if we are writing image to
	 * stdout and disk is multisession (already has atleast one session);
	 * '-M' points to file with last session that we want to merge
	 * currently created image */
	if (ismulti)  {
		/* fprintf(stderr, "appending multisession info\n"); */
		sprintf(command, "%s -C %d,%d -M %s ", command, last_sess_start, next_sess_start, config.cdrwdevice);
	}

	if (strcmp(config.joliet, "1") == 0) {
		strcat(command, " -J");
	}
	if (strcmp(config.rockridge, "1") == 0) {
		strcat(command, " -r");
	}

	if (strlen(config.bootimg)) {
		sprintf(command, "%s -b %s -c boot.catalog -no-emul-boot ", command, config.bootimg);
	}

	if (strlen(config.volumeid) == 0) {
		strcpy(config.volumeid, DEFAULT_VOLUMEID);
	}

	if (strlen(iso_file)) { /* write iso file to disk (to iso_file) */
		sprintf(command, "%s -V\"%s\" -o %s -path-list %s ", command, config.volumeid, iso_file, tmp_file_name);
	} else { /* no output iso file - write to stdout, someone will read it */
		sprintf(command, "%s -V\"%s\" -path-list %s ", command, config.volumeid, tmp_file_name);
	}

	sprintf(command, "%s%c", command, '\0');

	return 0;
}




/**
 * Create 'char *' string containing program name and program's argument
 * for writing some input to optical disc
 *
 * This function creates string with program name and all options that
 * are set somewhere in config.* global variable. This function does not
 * append parameter pointing to file (or files) that has to be
 * written - caller must do that himself. It means that this function can
 * be used in two situations: when writing (created previously) image to
 * CD and when writing files directly from filesystem.
 *
 * \param char *command - allocated space for string
 */
void prepare_writetoCD_command(char *command)
{
	if (strlen(config.volumeid) == 0) {
		strcpy(config.volumeid, DEFAULT_VOLUMEID); /* defined in main.h */
	}

	/* all parametrs values are now known, let's build command */
	/* first - program name, write sppeed and writing device */
	sprintf(command, " %s -v speed=%s dev=%s ", CDRECORD, config.speed, config.scsi);

	/* a few obvious options */
	if (strcmp(config.dao, "1") == 0) {
		strcat(command, " -dao ");
	} else {
		strcat(command, " -tao ");
	}

	if (strcmp(config.dummy, "1") == 0) {
		strcat(command, " -dummy ");
	}

	if (strcmp(config.multi, "1") == 0) {
		strcat(command, " -multi ");
	}

	if (strcmp(config.burnproof, "1") == 0) {
		strcat(command, " -driveropts=burnproof ");
	}

	/* tsize parameter - some drives need to know track size before writing to disk;
	 * let's hope that $tsize exists - it should be created by running
	 * MKISOFS --print-size (...) */
	sprintf(command, "%s %s%c ", command, "-tsize=${tsize}s ", '\0');

	/* almost ready - caller must (may?) append path to iso image */

	return;
}




/**
 * Check if optical disc in drive is erasable
 *
 * Check if optical disc in drive is erasable - this can be done with
 * 'wodim -atip dev=/dev/hdx' command retrieving ATIP info.
 *
 * Some drives may not support this.
 *
 * \returns 0 if media is erasable or this information is unavailable, -1 if disc is for sure not erasable
 */
int run_command_check_erasable(void)
{
	char command[500];

	current_command = COMMAND_CHECK_ERASABLE;
	media_erasable = MEDIA_ERASABLE_UNKNOWN;

	sprintf(command, "%s -atip dev=%s", CDRECORD, config.cdrwdevice);
	/* 2TRANS: this is title of dialog window - program is reading metadata from CD */
	processwin_create(_("Checking media"), "", false);
	run_command(command);
	processwin_destroy("", false);

	current_command = COMMAND_NONE;

	if (media_erasable == MEDIA_ERASABLE_YES || media_erasable == MEDIA_ERASABLE_UNKNOWN) {
		return 0;
	} else {
		return -1;
	}
}




/**
 * Blank optical disk using 'all' or 'fast' mode
 *
 * This function does the following:
 *   Prepares blanking command
 *   Calls run_command() to blank disc
 *   If blanking could not be performed, user is informed about this
 * Media should be checked for "erasability" before calling this function.
 * Blanking method (fast/all) sould be selected before calling this function.
 *
 * \returns 0 on success (probably), -1 on failure
 */
int run_command_blank_cd(void)
{
	char command[500];
	current_command = COMMAND_BLANK_CD;

	/* prepare command */
	sprintf(command, "%s -v speed=%s dev=%s blank=%s", CDRECORD, config.speed, config.scsi, config.blank);

	/* 2TRANS: this is title of dialog window */
	processwin_create(_("Blank CD-RW"),
			  /* 2TRANS: this is message in dialog window - operation is in progress */
			  _("Blanking of CD-RW in progress..."), false);

	run_command(command);

	/* it might occur during blanking that 'this media does
	 * not support blanking, ignoring', and media_erasable was set
	 * to 'MEDIA_ERASABLE_NO' in run_command(); let's inform user
	 * about this sad fact */
	if (media_erasable == MEDIA_ERASABLE_NO) {
		processwin_destroy("", false); /* silently close processwin, explanations will follow */
		if ( !strcmp(config.blank, "all") ) {
			/* 2TRANS: this is title of dialog window */
			dialogbox(_("Blanking error"),
				  /* 2TRANS: this is message in dialog window */
				  _("I\'m sorry, this disc cannot be erased."), 0);
		} else { /* config.blank == fast, CDRECORD may suggest using blank=all */
			/* 2TRANS: this is title of dialog window: blanking refers to blanking of CD-RW*/
			dialogbox(_("Blanking error"),
				  /* 2TRANS: this is message in dialog window */
				  _("I\'m sorry, this disc cannot be erased. Please try erasing with option \'all\'"), DIALOG_OK);
		}
		current_command = COMMAND_NONE;
		return -1;
	} else {
		/* looks like everything went OK, but CDRECORD informs about successful blanking
		 * FIXME - add code checking this */
		/* 2TRANS: this is message in dialog window */
		processwin_destroy(_("Blanking of CD-RW finished"), true);
		current_command = COMMAND_NONE;
		return 0;
	}
}





/**
 * Copy content of optical disc to iso image on your hard drive.
 *
 * Currently only copying single session data discs works correctly.
 *
 * \param char *window_title - title of processwin window
 * \param char *window_label - text displayed in processwin during copying of disc
 *
 * \returns 0 on success, -1 on failure
 */
int copy_cd_to_image(char *window_title, char *window_label)
{

	struct disc_information *disc = cdw_cdio_disc_open(config.scsi);
	if (disc == (struct disc_information *) -1) {
		/* 2TRANS: this is title of dialog window */
		dialogbox(_("Message"),
			  /* 2TRANS: this message in dialog window */
			  _("This disc type is not supported."), DIALOG_OK);
		return -1;
	} else if (disc == (struct disc_information *) -2) {
		/* 2TRANS: this is title of dialog window */
		dialogbox(_("Message"),
			  /* 2TRANS: this is message in dialog window */
			  _("Cannot open disc."), DIALOG_OK);
		return -1;
	}

	current_command = COMMAND_COPY_CD_TO_IMAGE;

	if (disc->mode == CDIO_DISC_MODE_CD_DA) {
		/* 2TRANS: this is title of dialog window */
		dialogbox(_("Message"),
			  /* 2TRANS: this is message in dialog window,
			     displayed when user inserts audio cd, but wants
			     to perform operation permitted for data cd */
			  _("This is audio cd. Please insert data cd."), DIALOG_OK);
		cdw_cdio_disc_close();
		return -1;
	}

	/* check if output image file path is valid */
	int filedes;
	bool image_ok = open_target_image_file(&filedes);

	if (image_ok) { /* can write to image file */
		close(filedes);  /* close - MKISOFS will create it anyway */
		unlink(config.tempdir); /* unlink - open_target_image_file() created it with writing permissions only */

		processwin_create(window_title, window_label, true);

		int output_image_file = open(config.tempdir, O_CREAT | O_WRONLY | O_TRUNC, S_IRWXU);
		if (output_image_file  == -1) {
			cdw_cdio_disc_close();
			/* 2TRANS: this is title of dialog window */
			dialogbox(_("Message"),
				  /* 2TRANS: this is message in dialog window,
				     output file is iso file on hdd */
				  _("Error when opening output file"), DIALOG_OK);
			/* 2TRANS: this is message in dialog window */
			processwin_destroy(_("Image not created"), true);
			return -1;
		}

		bool success = cdw_cdio_read_write_tracks_from_disc(disc, output_image_file);

		if (success) {
			/* 2TRANS: this is message in dialog window */
			processwin_destroy(_("Data disc copied succesfuly"), true);
			return 0;
		} else {
			/* 2TRANS: this is message in dialog window. It is
			   displayed when some errors occured during
			   copying of data CD.*/
			processwin_destroy(_("Problems occured when copying tracks"), true);
			return -1;
		}

		/* clean up */
		close(output_image_file);
		cdw_cdio_disc_close();
		curs_set(1); /* turn cursor on */
		current_command = COMMAND_NONE;
	} else { /* cannot/will not write to image (for some reason) */
		/* 2TRANS: this is title of dialog window */
		dialogbox(_("Message"),
			  /* 2TRANS: this is message in dialog window,
			     output file is iso file on hdd */
			  _("Error when opening output file"), DIALOG_OK);
		return -1;
	}
}





/**
 * Get information from mulitisession disc about sessions track numbers
 *
 * Correct values will be retrieved by calling "wodim -msinfo"
 *
 * \param int *last_sess_start - first sector in the last session of the disc
 * \param int *next_sess_start - the starting sector number of the new session
 *
 * \returns 0 if disc is not multisession, 1 if the disc is multisession
 */
bool get_multisession_tracks(int *last_sess_start, int *next_sess_start)
{
	/*
	int cddev;

	if ( (cddev=open(config.cdrwdevice, O_RDONLY | O_NONBLOCK)) != -1 ){
		int rv = ioctl(cddev, CDROM_DRIVE_STATUS, 0);
		close(cddev);
		if (rv == CDS_NO_DISC) {
			fprintf(stderr, "NOCD\n");
		}
	}
	*/
	char msinfo[255];
	char *line;
	FILE *fp;

	bool ismulti = false; /* */

	sprintf(msinfo, "%s dev=%s -msinfo ", CDRECORD, config.scsi);

	/* 2TRANS: this is message in dialog window, displayed when
	   program reads some meta-data from cd */
	processwin_create(_("Load CD info..."), "", false);
	run_command(msinfo);
	processwin_destroy("", false);

	if ( (fp = fopen(config.logfile, "r")) == NULL ) {
		clean_before_cdw_exit();
		/* 2TRANS: this is message in dialog window, displayed when
		   program cannot find log file and exits */
		fprintf(stderr, _("Cannot open config file..."));
		exit(-1);
	}
	if ( (line = (char *) malloc(1024)) == NULL ) {
		return -1;
	}
	*last_sess_start = 0;
	*next_sess_start = 0;
	while ((line = fgets(line, 256, fp)) != NULL) {
		/* fprintf(stderr, "line to scan for session information: %s\n", line); */
		int rv = sscanf(line, "%d,%d", last_sess_start, next_sess_start);

		if (rv == 2) {
			/*
			fprintf(stderr, "%d,%d", *last_sess_start, *next_sess_start);
			fprintf(stderr, " - last session information available\n");
			*/
			ismulti = true;
			break; /* don't check any other lines! */
		} else {
			/*
			fprintf(stderr, "%d,%d", *last_sess_start, *next_sess_start);
			fprintf(stderr, " - no session info in this line, sscanf return value = %d\n\n\n", rv);
			*/
			ismulti = false;
		}
	}

	fclose(fp);
	free(line);
	line = NULL;

	return ismulti;
}




/**
 * Show basic info about disc currently in drive
 *
 * Run cdrecord with -atip argument to obtain some infomration about disc
 * that is currently in drive. The informaton is written to log file.
 */
void run_command_get_cd_info(void)
{
	char command[100];
	sprintf(command, "%s dev=%s -atip", CDRECORD, config.scsi);

	/* 2TRANS: this is message in dialog window, displayed when
	   program reads some meta-data from cd */
	processwin_create(_("Load CD info..."), "", false);
	run_command(command);
	processwin_destroy("", false);

	return;
}



