/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * A few small modifications (C) 2007 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */



#include <ncursesw/ncurses.h>
#include <ncursesw/menu.h>
#include <ncursesw/form.h>

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <linux/cdrom.h>
#include <fcntl.h>
#include <dirent.h>
#include <string.h>
#include <nl_types.h>

#include "gettext.h"
#include <libintl.h>

#include "thread.h" /* run_command() */
#include "cdw_widgets.h" /* nice_box() */
#include "log.h"
#include "options.h" /* struct conf */
#include "utils.h"
#include "cdw_tmp_files.h"
#include "cdw_ui.h"
#include "main.h"
#include "config.h"

#include "commands.h"


/* default disk name */
#define DEFAULT_VOLUMEID "cdrom"

extern WINDOW *mainwin;
extern struct conf config;
extern double size;
extern char tmp_file_name[];

enum media_erasable_t media_erasable;


/*
 * Create process writing files to iso image
 *
 * This function calls filelist_to_file() to write selected files'
 * paths to tmp file used later by mkisofs.
 *
 * \returns 0 when iso file is created; 1 when it is not created
 */
int run_command_create_image(void)
{
	char createiso_command[500];
	int ret = 1;
	FILE *isofile;

	if (strlen(config.bootimg)) {
		int boot_exists = check_for_boot_image();
		if (!boot_exists) {
			return 1;
		}
	}

	/* flush current list of selected files to config file */
	filelist_to_file();

	bool file_set = false;
	while (!file_set) {
		if ((isofile = fopen(config.tempdir, "r")) != NULL) {
			int rv = 0;
			fclose(isofile);
			rv = dialogbox(_("Please confirm"), _("Image file exist!\nDo you wish to overwrite it?"), DIALOG_YES_NO_CANCEL);
			wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
			// select_window(FALSE);
			if (rv == BUTTON_NO) { // do not overwrite, enter new path to iso file
				input_box(_("Path to iso file"), _("Enter path to new iso file"), config.tempdir, 0);
				wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
				continue; // user entered new path - check (again) if the file exists
			} else if (rv == BUTTON_YES) { // yes, overwrite existing iso image
				file_set = true;
			} else { // BUTTON_CANCEL - don't create image
				return 1;
			}
		} else { // file doesn't exist yet, it is safe to creae iso file with such path/name
			file_set = true;
		}
	} // while

	/* prepare command creating iso image, with writing
		* image to file (config.tempdir should not be empty or invalid) */
	prepare_createiso_command(createiso_command, config.tempdir);

	/* boot image exists, iso file doesn't already exist (or can be overwritten), so... */
	// fprintf(stderr, createiso_command);
	run_command(createiso_command, _("Create image"), 0, 1, 1, "");

	wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
	select_window(FALSE);

	return 0;
}




/*
 * Crate process writing image to disk
 *
 * \returns 0
 */
int run_command_write_from_image(void)
{
	char command[500];
	sprintf(command, "%s -v speed=%s dev=%s %s", CDRECORD, config.speed, config.scsi, config.other);
	if (strcmp(config.dao, "1") == 0) {
		strcat(command, " -dao");
	}
	if (strcmp(config.dummy, "1") == 0) {
		strcat(command, " -dummy");
	}
	if (strcmp(config.burnproof, "1") == 0) {
		strcat(command, " -driveropts=burnproof");
	}
	sprintf(command, "%s -data %s", command, config.tempdir);

	// fprintf(stderr, "%s\n", command);

	run_command(command, _("Write image"), 0, 1, 1, "");
	wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
	select_window(FALSE);

	return 0;
}




/*
 * Create process writing selected files directly to CD disk
 *
 * Run program that will create iso image and write it to it's stdout.
 * Run another program, that will read iso file (via normal shell
 * pipe '|') from it's stdin and write it to CD disk.
 *
 * In previous versions cdw created symlinks to all files that are to be
 * written to iso image, and told program creating iso image to follow
 * these symlinks; current method is to write list of files to tmp
 * file and feed this file to program creating iso image (that is how k3b
 * does it, I think).
 *
 * \returns 0 on success, val < 0 on failure
 */
int run_command_write_direct(void)
{
	char command[1000];
	int ret = 1;
	int boot_exists;

	char writetoCD_command[500];
	char tsize_command[255];
	char createiso_command[500];

	if (strlen(config.bootimg)) {
		boot_exists = check_for_boot_image();

		if (!boot_exists) {
			return -2;
		}

	}

	wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
	select_window(FALSE);

	/* flush current list of selected files to config file */
	filelist_to_file();

	/* prepare command creating iso image, without writing
		* image to file (that is why second
		* parameter - filename - is empty) */
	prepare_createiso_command(createiso_command, "");

	/* prepare command creating tsize - shell variable that will
		* be used by program writing to disk - read
		* prepare_writetoCD_command() function for explanation */
	sprintf(tsize_command, "tsize=`%s --print-size -q`", createiso_command);

	/* prepare command reading iso image from stdin and
		* writing it to CD disk */
	prepare_writetoCD_command(writetoCD_command);

	/* prepare_writetoCD_command does not append input
		* file - let's do this here: read form stdin (pipe)*/
	sprintf(writetoCD_command, "%s -", writetoCD_command);

	/* final version of 'write direct' command is here: */
	sprintf(command, "%s; %s | %s", tsize_command, createiso_command, writetoCD_command);

	wrefresh(mainwin);

	run_command(command, _("Write direct"), size, 1, 1, "");

	wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
	select_window(FALSE);

	return 0;

}




/*
 * Create process writing audio CD
 */
int run_command_write_audio(void)
{
	char command[500];
	int ret = 1;
	if (strcmp(config.showvol, "1") == 0) {
		ret = input_box(_("Volume ID"), _("Enter volume label:"), config.volumeid, 0);
		wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
		select_window(TRUE);
	}
	if (ret == 1) {
		sprintf(command, "%s -v speed=%s dev=%s -audio", CDRECORD, config.speed, config.scsi);
		if (strcmp(config.pad, "1") == 0) {
			strcat(command, " -pad");
		}
		if (strcmp(config.dao, "1") == 0) {
			strcat(command, " -dao");
		}
		if (strcmp(config.dummy, "1") == 0) {
			strcat(command, " -dummy");
		}
		if (strcmp(config.burnproof, "1") == 0) {
			strcat(command, " -driveropts=burnproof");
		}
		sprintf(command, "%s %s/*.wav", command, config.audiodir);
		run_command(command, _("Write audio"), size, 1, 1, "");
		after_event("Write audio log", 1);
	}
	wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
	select_window(FALSE);

	return 0;
}




/*
 * Create process copying CD disk to iso image on hard disk
 */
int run_command_copy_disk(void)
{
	char command[500];
	int in_fd;
	if ((in_fd = open("/dev/cdrom", O_RDONLY)) != -1) {
		close(in_fd);
		sprintf(command, "%s -v speed=%s dev=%s %s", CDRECORD, config.speed, config.scsi, config.other);
		if (strcmp(config.eject, "1") == 0) {
			strcat(command, " -eject");
		}
		if (strcmp(config.dummy, "1") == 0) {
			strcat(command, " -dummy");
		}
		sprintf(command, "%s -isosize %s", command, config.cdrom);
		run_command(command, _("Copy data CD"), 0, 1, 1, "");
		after_event("Copy data CD log", 1);
		wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
		select_window(FALSE);
	}

	return 0;

}




/*
 * Create string containing program name and program's argument for creating iso image
 *
 * iso image is created in two different places:
 * - when user only creates iso image (this is explicit creation, file is written to disk)
 * - when user writes directly to CD disk - iso image is piped to writing program
 * .
 * Both cases are very similar, only in second case there is no output file,
 * since CD writing program reads from pipe (from stdin).
 * This function appends output image file name to command (it takes it from config.tempdir).
 *
 * \param char *command - string for program name and it's parameters
 * \param char *iso_file - path (full or relative) to output iso file + file name with extension;
 *                         set parameter to empty string it you don't want to write
 *                         image to file
 *
 * \returns -1 on malloc error, 0 on success
 *
 */
int prepare_createiso_command(char *command, char *iso_file)
{

	int track_start = 0, track_end = 0;
	FILE *fp;
	int ismulti = 0;

	/* 'graft points' means text file with list of files to burn AND root
	 * directories on CD that the files will be put into;
	 * see also -path-list argument below */
	sprintf(command, "%s  -graft-points -iso-level 2", MKISOFS);

	/* FIXME - maybe this should go to external function */
	if (strcmp(config.multi, "1") == 0) {
		/*
    		int cddev;

    		if ( (cddev=open(config.cdrwdevice, O_RDONLY | O_NONBLOCK)) != -1 ){
			int rv = ioctl(cddev, CDROM_DRIVE_STATUS, 0);
			close(cddev);
			if (rv == CDS_NO_DISC) {
				fprintf(stderr, "NOCD\n");
			}
 		}
		*/
		char msinfo[255];
		char *line;
		sprintf(msinfo, "%s dev=%s -msinfo ", CDRECORD, config.scsi);
		run_command(msinfo, _("Load CD info..."), 0, 0, 0, "");

		if ( (fp = fopen(config.logfile, "r")) == NULL ) {
			clean_before_cdw_exit();
			fprintf(stderr, _("Cannot open config file..."));
			exit(-1);
		}
		if ( (line = (char *) malloc(1024)) == NULL ) {
			return -1;
		}

		while ((line = fgets(line, 256, fp)) != NULL) {
			if (sscanf(line, "%d,%d", &track_start, &track_end) == 2) {
				wrefresh(mainwin);
				ismulti = 1;
			} else {
				ismulti = 0;
			}
		}

		fclose(fp);
		free(line);
		line = NULL;
	}

	/* append multisession track info, but only
	 * if we are writing image to stdout and disk is multisession */
	if ((ismulti == 1) & (!strcmp(iso_file, "")) )  {
		sprintf(command, "%s -C %d,%d -M %s ", command, track_start, track_end, config.cdrwdevice);
	}

	if (strcmp(config.joliet, "1") == 0) {
		strcat(command, " -J");
	}
	if (strcmp(config.rockridge, "1") == 0) {
		strcat(command, " -r");
	}

	if (strlen(config.bootimg)) {
		sprintf(command, "%s -b %s -c boot.catalog -no-emul-boot ", command, config.bootimg);
	}

	if (strlen(config.volumeid) == 0) {
		strcpy(config.volumeid, DEFAULT_VOLUMEID);
	}

	if (strlen(iso_file)) { /* write iso file to disk (to iso_file) */
		sprintf(command, "%s -V\"%s\" -o %s -path-list %s ", command, config.volumeid, iso_file, tmp_file_name);
	} else { /* no output iso file - write to stdout, someone will read it */
		sprintf(command, "%s -V\"%s\" -path-list %s ", command, config.volumeid, tmp_file_name);
	}

	return 0;
}




/*
 * Prepare string with program name and its arguments for writing file to disk
 *
 * This function creates string with program name and all options that are set
 * somewhere in config.* global variable. This function does not append
 * parameter pointing to file that has to be written - caller must do that
 * himself.
 *
 * \param char *command - allocated space for string
 */
void prepare_writetoCD_command(char *command)
{

	if (strlen(config.volumeid) == 0) {
		strcpy(config.volumeid, "cdrom");
	}

	/* all parametrs values are now known, let's build command */
	/* first - program name, write sppeed and writing device */
	sprintf(command, " %s -v speed=%s dev=%s ", CDRECORD, config.speed, config.scsi);


	/* a few obvious options */
	if (strcmp(config.dao, "1") == 0) {
		strcat(command, " -dao");
	}
	if (strcmp(config.dummy, "1") == 0) {
		strcat(command, " -dummy");
	}
	if (strcmp(config.multi, "1") == 0) {
		strcat(command, " -multi -tao");
	}
	if (strcmp(config.burnproof, "1") == 0) {
		strcat(command, " -driveropts=burnproof");
	}

	/* tsize parameter - some drives need to know track size before writing to disk;
	 * let's hope that $tsize exists - it should be created by running
	 * MKISOFS --print-size (...) */
	sprintf(command, "%s %s", command, "-tsize=${tsize}s");

	/* almost ready - caller must (may?) append path to iso image */

	return;
}




/*
 * Check if optical disc in drive is erasable
 *
 * Check if optical disc in drive is erasable - this can be done with
 * 'wodim -atip dev=/dev/hdx' command retrieving ATIP info.
 *
 * Some drives may not support this.
 *
 * \returns 0 if media is erasable or this information is unavailable, -1 if disc is for sure not erasable
 */
int run_command_check_erasable(void)
{
	char command[500];

	media_erasable = MEDIA_ERASABLE_UNKNOWN;

	sprintf(command, "%s -atip dev=%s", CDRECORD, config.cdrwdevice);

	// run ATIP command, 4th argument == 0 -> don't propmpt user for accepting results
	run_command(command, _("Checking media"), 0, 0, 1, "");
	wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
	select_window(FALSE);

	if (media_erasable == MEDIA_ERASABLE_YES || media_erasable == MEDIA_ERASABLE_UNKNOWN) {
		return 0;
	} else {
		return -1;
	}
}




/**
 * Blank optical disk
 *
 * This function does the following:
 *   Prepares blanking command
 *   Calls run_command() to blank disc
 *   If blanking could not be performed, user is informed about this
 * Media should be checked for "erasability" before calling this function.
 * Blanking method (fast/all) sould be selected before calling this function.
 *
 * \returns 0 on success (probably), -1 on failure
 */
int run_command_blank_cd(void)
{
	char command[500];
	// prepare command
	sprintf(command, "%s -v speed=%s dev=%s blank=%s", CDRECORD, config.speed, config.scsi, config.blank);

	// run command
	run_command(command, _("Blank CD-RW"), 0, 1, 0, _("Blanking of CD-RW in progress..."));

	/* it might occur during blanking that 'this media does
	 * not support blanking, ignoring', and media_erasable was set
	 * to 'MEDIA_ERASABLE_NO' in run_command(); let's inform user
	 * about this sad fact */
	if (media_erasable == MEDIA_ERASABLE_NO) {
		if ( !strcmp(config.blank, "all") ) {
			dialogbox(_("Blanking error"), _("I\'m sorry, this disc cannot be erased."), 0);
		} else { /* config.blank == fast, CDRECORD may suggest using blank=all */
			dialogbox(_("Blanking error"), _("I\'m sorry, this disc cannot be erased. Please try erasing with option \'all\'"), 0);
		}
		return -1;
	} else {
		// looks like everything went OK, but CDRECORD informs about successful blanking
		// FIXME - add code checking this
		return 0;
	}
}




