/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * A few small modifications (C) 2007 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <libintl.h>
#include <locale.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <ncursesw/ncurses.h>
#include <ncursesw/menu.h>
#include <ncursesw/form.h>

#include "cdw_ui.h"
#include "options.h"
#include "gettext.h"
#include "color.h"
#include "log.h"
#include "config.h"
#include "main.h"
#include "utils.h"
#include "cdw_tmp_files.h"



#define MENU_ITEMS 8 /* amount of items in main cdw menu on the left */

/* main menu on the left side of main cdw window */
struct curses_menu {
	char add[22];
	char del[22];
	char cre[22];
	// char cpi[22]; not in this release
	char wri[22];
	// char wau[22]; not in this release
	// char cdc[22]; not in this release
	char blk[22];
	char opt[22];
	char qut[22];
	char nul[1];
} buttons;


extern struct conf config;

/* table of pointers to paths */
char **filetable = NULL;

fileitem *filei;

extern int num;
int maxfiles;
int lines, cols;
double size;
int menupos;
ITEM **items = NULL;
ITEM **cditems = NULL;

char *TITLE = NULL;
MENU *menu = NULL;
MENU *cdmenu = NULL;

// WINDOW *menuwin, *mainwin, *addwin, *menuwin_sub, *addwinlist;

WINDOW *mainwin; /* main application window */
WINDOW *addwin;
WINDOW *menuwin_sub;
WINDOW *addwinlist;
WINDOW *menuwin;


void clean_title(void)
{
	if (TITLE != NULL) {
		free(TITLE);
	}

	return;
}



void fill_info(void)
{
	wrefresh(menuwin_sub);
	if (strcmp(config.fromimage, "1") == 0)
		mvwprintw(menuwin_sub, LINES - 12, 2, _("Write from image"));
	else
		mvwprintw(menuwin_sub, LINES - 12, 2, _("Write direct    "));
	mvwprintw(menuwin_sub, LINES - 11, 2, _("Speed: %sX "), config.speed);
	if (strcmp(config.multi, "1") == 0)
		mvwprintw(menuwin_sub, LINES - 10, 2, _("Multi-session "));
	else
		mvwprintw(menuwin_sub, LINES - 10, 2, _("Single session"));
	wrefresh(menuwin_sub);
}




/* 
 * Simple file selector window
 *
 * File/directory selection window with one list of files. Allows browsing 
 * tree of directories using keyboard.
 *
 * Uses global variables addwinlist and filei.
 *
 * \param char *dir
 * \param ind width - width ow list window
 * \param int height - height of list window
 *
 */
void load_dir(char *dir, int width, int height)
{
	int i, n;
	struct dirent **eps;

	werase(addwinlist);
	n = scandir(dir, &eps, (void *) 0, alphasort);
	filei = (fileitem *) malloc(n * sizeof(fileitem));

	if (n > 0) {
		fill_filelist(dir, &filei, eps, n);
	}

	/* FIXME - maybe rest of this function should be put into one of branches of if? */

	/* number of items on directory listing;
	 * 2 = current dir ('.') + parent dir ('..') */
	maxfiles = n - 2;
	for (i = 0; i < maxfiles + 1; i++) {
		if (filei[i].type == 0) {
			wattrset(addwinlist, A_BOLD | COLOR_PAIR(2));
			mvwprintw(addwinlist, i, 0, "/%.*s", getmaxx(addwinlist) - 7, filei[i].filename);
			mvwprintw(addwinlist, i, getmaxx(addwinlist) - 5, "DIR");
		}
		if (filei[i].type == 1) {
			wattrset(addwinlist, COLOR_PAIR(2));
			mvwprintw(addwinlist, i, 0, "%.*s", getmaxx(addwinlist) - 7, filei[i].filename);
			mvwprintw(addwinlist, i, getmaxx(addwinlist) - 5, "%4d%s", (filei[i].size / 1024 / 1024 > 1 ? filei[i].size / 1024 / 1024 : filei[i].size / 1024),
				  (filei[i].size / 1024 / 1024 > 1 ? "M" : "K"));
		}

	}

	for (i = 0; i < n; i++){
		free(eps[i]);
		eps[i] = NULL;
	}
	free(eps);
	eps = NULL;

	nice_box(addwin, _("Add files"), dir, width);

	wrefresh(addwin);
}




/*
 * Display one item (and its attributes) from directory listing list 
 *
 * Display in given row one item from directory listing list, set its 
 * font/background properly, display additional attributes in the same row.
 *
 * Set color of direcrory listing item based on its type (dir or regular file).
 *
 * \param int pos - position on list (position in window)
 * \param fileitem file_p - file item with file properties (name, type, size)
 * \param int isreverse - if = 0 then black background and light font, 
 *                         otherwise reverse attributes applied
 */
void highlight(int pos, fileitem file_p, int isreverse)
{
	if (file_p.type == 0) {
		if (isreverse == 0)
			wattrset(addwinlist, A_REVERSE | COLOR_PAIR(2));
		else
			wattrset(addwinlist, A_BOLD | COLOR_PAIR(2));
		mvwprintw(addwinlist, pos, 0, "/%.*s", getmaxx(addwinlist) - 7, file_p.filename);
	}
	if (file_p.type == 1) {
		if (isreverse == 0)
			wattrset(addwinlist, A_REVERSE | COLOR_PAIR(2));
		else
			wattrset(addwinlist, COLOR_PAIR(2));
		mvwprintw(addwinlist, pos, 0, "%.*s", getmaxx(addwinlist) - 7, file_p.filename);
	}
	wrefresh(addwinlist);
}




/*
 * Show window that allows user to select files and directories
 *
 * Show window that allows user to select files and directories
 * that will be put into CD disc.
 *
 * \param int width
 * \param int height
 * \param int y
 * \param int x
 *
 */
void put_addwin(int width, int height, int y, int x)
{
	char path[2000];
	struct stat stbuf;
	char name[1000];
	char *pos;
	int c;
	int rv;
	int screenpos = 0; // cursor position on addwinlist
	char *linkfile = malloc(2000);

	menupos = 0; // position of cursor on whole list of files, so it may be larger than y size of window
	addwin = newwin(width, height, y, x);
	keypad(addwin, TRUE);
	wbkgd(addwin, A_NORMAL | COLOR_PAIR(2));
	wattrset(addwin, A_NORMAL | COLOR_PAIR(2));
	addwinlist = derwin(addwin, width - 2, height - 2, 1, 1);

	nice_box(addwin, _("Add files"), "", 0);

	sprintf(path, "%s", getenv("HOME"));
	load_dir(path, width, height);
	highlight(0, filei[menupos], 0);
	while ((c = wgetch(addwin)) != 27) { /* escape char */
		switch (c) {
		/* First item */
		case KEY_HOME:
			if (menupos > 0) {
				menupos = 0;
				screenpos = 0;
				werase(addwinlist);
				scroll_addlist(addwinlist, menupos, filei, maxfiles);
				highlight(screenpos, filei[menupos], 0);
				// wrefresh(addwinlist);
			}
			break;
		/* Last item */
		case KEY_END:
			if (menupos < maxfiles) { /* maxfiles is number of files/dirs in current dir */
				menupos = maxfiles - (maxfiles % getmaxy(addwinlist));
				werase(addwinlist);
				scroll_addlist(addwinlist, menupos, filei, maxfiles);
				menupos = maxfiles;
				screenpos = menupos % getmaxy(addwinlist);
				highlight(screenpos, filei[menupos], 0);
				// refresh(addwinlist);
			}
			break;
		/* Next item */
		case KEY_DOWN:
			if (menupos < maxfiles) {
				menupos++;
				if ((menupos % (getmaxy(addwinlist)) != 0) || (menupos == 0)) {
					if (menupos < getmaxy(addwinlist))
						screenpos = menupos;
					else
						screenpos = menupos % getmaxy(addwinlist);
					highlight(screenpos - 1, filei[menupos - 1], 1);
				} else {
					werase(addwinlist);
					scroll_addlist(addwinlist, menupos, filei, maxfiles);
					screenpos = menupos % getmaxy(addwinlist);
				}
				highlight(screenpos, filei[menupos], 0);
				wrefresh(addwinlist);
			}
			break;
		/* Prev item */
		case KEY_UP:
			if (menupos > 0) {
				menupos--;
				if ((menupos % (getmaxy(addwinlist)) != getmaxy(addwinlist) - 1) || (menupos == 0)) {
					if (menupos < getmaxy(addwinlist))
						screenpos = menupos;
					else
						screenpos = menupos % getmaxy(addwinlist);
					highlight(screenpos + 1, filei[menupos + 1], 1);
				} else {
					werase(addwinlist);
					scroll_addlist(addwinlist, menupos, filei, maxfiles);
					screenpos = getmaxy(addwinlist) - 1;
				}
				highlight(screenpos, filei[menupos], 0);
				wrefresh(addwinlist);
			}
			break;
		case KEY_PPAGE: /* Previous Page, Page Up */
			if (menupos > 0) {
				if ( (menupos == 0) || (menupos < getmaxy(addwinlist)) ) { // top of dir listing, do not scroll
					highlight(screenpos, filei[menupos], 1);
					menupos = 0; // we are on top of dir listing, so there is only one place to move cursor and file index
					screenpos = 0;
				} else { 
					menupos -= getmaxy(addwinlist); // move file index exactly one screensize up
					// screenpos = screenpos; // cursor does not change its position
					werase(addwinlist);
					scroll_addlist(addwinlist, menupos, filei, maxfiles);
				}
				highlight(screenpos, filei[menupos], 0);
				wrefresh(addwinlist);
			}
			break;
		case KEY_NPAGE: /* Next Page, Page Down */
			if ((maxfiles - menupos) > getmaxy(addwinlist) - 1) { // (remainder of) dir listing is long enough to move it by whole page 
				highlight(screenpos, filei[menupos], 1);
				menupos += getmaxy(addwinlist);
				// screenpos = screenpos;
				werase(addwinlist);
				scroll_addlist(addwinlist, menupos, filei, maxfiles);
				highlight(screenpos, filei[menupos], 0);
				wrefresh(addwinlist);
			} else { // move cursor to last item on the list, display reminder of the list
				/* it almost works
				screenpos = maxfiles % getmaxy(addwinlist);
				menupos = maxfiles;
				werase(addwinlist);
				scroll_addlist(addwinlist, menupos, filei, maxfiles);
				highlight(screenpos, filei[menupos], 0);
				wrefresh(addwinlist);
				*/
			}
			break;
		/* Chdir */
		case 10:
			memset(name, '\0', sizeof(name));
			strcpy(name, path);
			strcat(name, "/");
			strcat(name, filei[menupos].filename);
			if (stat(name, &stbuf) == -1) {
				clean_before_cdw_exit();
				fprintf(stderr, _("File: %s not found!"), name);
				exit(-1);
			}
			if ((stbuf.st_mode & S_IFMT) == S_IFDIR) {
				strcpy(linkfile, path);
				if (strcmp(filei[menupos].filename, "..") == 0) {
					if ((pos = (char *) strrchr((char *) path, '/')) != NULL)
						memset(pos, '\0', 1);
					if (strlen(path) <= 0)
						strcat(path, "/");
				} else {
					if (path[strlen(path) - 1] != '/')
						strcat(path, "/");
					strcat(path, filei[menupos].filename);
				}
				if (access(path, R_OK | X_OK) == -1) {
					strcpy(path, linkfile);
				} else {
					free(filei);
					filei = NULL;
					menupos = 0;
					load_dir(path, width, height);
					highlight(0, filei[menupos], 0);
				}
			}
			break;
		/* Add item to list of files/directories that
		 * are going to be put into CD disk */
		case ' ':
			/* copy full path of selected file to linkfile */
			strcpy(linkfile, path); /* path = full current dir path */
			strcat(linkfile, "/");  
			strcat(linkfile, filei[menupos].filename); /* name of selected file/dir */

			/* add this full path to files list */
			rv = filelist_add(linkfile);
			
			nice_box(addwin, _("Add files"), path, width);

			select_window(TRUE);
			break;
		}
		wrefresh(addwinlist);
	}
	free(filei);
	free(linkfile);
	filei = NULL;
	filelist_to_file();
	delwin(addwin);

	return;
}




void put_mainform()
{
	WINDOW *select_sub;

	mainwin = newwin(LINES - 1, COLS - 1, 0, 0);
	keypad(mainwin, TRUE);
	wbkgd(mainwin, A_NORMAL | COLOR_PAIR(1));
	box(mainwin, 0, 0);
	mvwaddch(mainwin, 2, 0, ACS_LTEE);
	mvwhline(mainwin, 2, 1, ACS_HLINE, COLS - 3);
	mvwaddch(mainwin, 2, COLS - 2, ACS_RTEE);
	wattrset(mainwin, A_BOLD | COLOR_PAIR(3));
	mvwprintw(mainwin, 1, (COLS / 2) - (strlen(TITLE) / 2), TITLE);
	wattrset(mainwin, A_NORMAL | COLOR_PAIR(1));

	// *** Main Menu ***
	strcpy(buttons.add, _(" Add files           "));
	strcpy(buttons.del, _(" Delete files        "));
	strcpy(buttons.cre, _(" Create image        "));
	// strcpy(buttons.cpi, _(" Copy CD image       ")); not in this release
	strcpy(buttons.wri, _(" Write data CD       "));
	// strcpy(buttons.wau, _(" Write audio CD      ")); not in this release
	// strcpy(buttons.cdc, _(" Copy data CD        ")); not in this release
	strcpy(buttons.blk, _(" Blank CD-RW         "));
	strcpy(buttons.opt, _(" Options             "));
	strcpy(buttons.qut, _(" Quit                "));

	items = (ITEM **) calloc(MENU_ITEMS, sizeof(ITEM *));

	items[0] = new_item(buttons.add, "");
	items[1] = new_item(buttons.del, "");
	items[2] = new_item(buttons.cre, "");
	// items[3] = new_item(buttons.cpi, "");
	items[3] = new_item(buttons.wri, "");
	// items[5] = new_item(buttons.wau, "");
	// items[6] = new_item(buttons.cdc, "");
	items[4] = new_item(buttons.blk, "");
	items[5] = new_item(buttons.opt, "");
	items[6] = new_item(buttons.qut, "");
	items[7] = new_item((char *) 0, "");
	// *****************

	menu = new_menu((ITEM **) items);
	set_menu_win(menu, mainwin);
	set_menu_sub(menu, derwin(mainwin, LINES - 8, 26, 5, 3));
	set_menu_fore(menu, COLOR_PAIR(4) | A_REVERSE);
	set_menu_back(menu, COLOR_PAIR(4));
	set_menu_mark(menu, "");

	/* 'Selected files' part of screen */
	select_sub = derwin(mainwin, LINES - 7, COLS - 29, 3, 27);

	keypad(select_sub, TRUE);
	nice_box(select_sub, _("Selected files"), "", 0);
	wattrset(select_sub, A_NORMAL | COLOR_PAIR(1));

	mvwaddch(select_sub, LINES - 13, 0, ACS_LTEE);
	mvwhline(select_sub, LINES - 13, 1, ACS_HLINE, COLS - 31);
	mvwaddch(select_sub, LINES - 13, COLS - 30, ACS_RTEE);
	mvwaddch(select_sub, LINES - 11, 0, ACS_LTEE);
	mvwhline(select_sub, LINES - 11, 1, ACS_HLINE, COLS - 31);
	mvwaddch(select_sub, LINES - 11, COLS - 30, ACS_RTEE);

	/* Left hand cdw menu */
	menuwin_sub = derwin(mainwin, LINES - 7, 26, 3, 1);
	nice_box(menuwin_sub, _("Menu"), "", 0);

	mvwaddch(menuwin_sub, LINES - 13, 0, ACS_LTEE);
	mvwhline(menuwin_sub, LINES - 13, 1, ACS_HLINE, 24);
	mvwaddch(menuwin_sub, LINES - 13, 25, ACS_RTEE);
	mvwprintw(menuwin_sub, LINES - 13, 2, "( %s )", _("Info"));
	fill_info();


	wattrset(mainwin, A_BOLD | COLOR_PAIR(2));
	post_menu(menu);
	wrefresh(mainwin);

	return;
}



/*
 * Deallocate all resources relatd to main app window
 *
 * \returns 0
 */
int destroy_mainform(void)
{
	clean_cdmenu();
	clean_menu();
	delwin(mainwin);
	mainwin = NULL;

	return 0;
}




/* 
 * Delete file(s) from list of selected files
 *
 * Delete file(s) from list of selected files. This function will move 
 * cursor to list of currently selected files and let user remove (using 
 * Delete button) remove item(s) from this list.
 */
void delete_files(void)
{
	int c;
	char filetodelete[200] = "\0";

	while (((c = wgetch(mainwin)) != 27) && (num > 0)) {
		switch (c) {
		case KEY_HOME:
			menu_driver(cdmenu, REQ_FIRST_ITEM);
			break;
		case KEY_END:
			menu_driver(cdmenu, REQ_LAST_ITEM);
			break;
		case KEY_DOWN:
			menu_driver(cdmenu, REQ_DOWN_ITEM);
			break;
		case KEY_UP:
			menu_driver(cdmenu, REQ_UP_ITEM);
			break;
		case KEY_DC:
			/* FIXME - merge those two lines */
			strcpy(filetodelete, item_name(current_item(cdmenu))); 
			filelist_remove(filetodelete);

			select_window(TRUE);
			set_menu_fore(cdmenu, COLOR_PAIR(1) | A_REVERSE);
			fflush(stdin);
			break;
		}
		wrefresh(derwin(mainwin, LINES - 8, COLS - 31, 4, 28));
	}
	filelist_to_file();
	return;
}


/*
 * Show window allowing user to select disk size
 */
void size_selector(void)
{
	WINDOW *volwin, *edit;
	int ch = 0;
	char cdsize[20];

	volwin = newwin(10, 30, (LINES - 10) / 2, (COLS - 30) / 2);
	wbkgd(volwin, COLOR_PAIR(2));
	werase(volwin);

	nice_box(volwin, _("CD size"), "", 0);
	
	mvwprintw(volwin, 2, 3, _("Select CD size"));
	mvwprintw(volwin, 4, 5, "1 - 74 Min (650 MB)");
	mvwprintw(volwin, 5, 5, "2 - 80 Min (700 MB)");
	mvwprintw(volwin, 6, 5, "3 - DVD - (4700 MB)");
	mvwprintw(volwin, 7, 5, "4 -");
	edit = derwin(volwin, 1, 4, 7, 9);
	wbkgd(edit, COLOR_PAIR(6));
	werase(edit);
	sprintf(cdsize, "%d", config.cdsize);
	mvwprintw(edit, 0, 0, cdsize);
	switch (config.cdsize) {
	case 657:
		mvwaddch(volwin, 4, 3, '*');
		break;
	case 702:
		mvwaddch(volwin, 5, 3, '*');
		break;
	case 4700:
		mvwaddch(volwin, 6, 3, '*');
		break;
	default:
		mvwaddch(volwin, 7, 3, '*');
		break;
	}
	wrefresh(edit);
	wmove(volwin, 9, 29);
	while ((ch != 27) && (ch != 10)) {
		ch = wgetch(volwin);
		switch (ch) {
		case '1':
			config.cdsize = 657;
			break;
		case '2':
			config.cdsize = 702;
			break;
		case '3':
			config.cdsize = 4700;
			break;
		case '4':
			sprintf(cdsize, "%d", config.cdsize);
			inbox(edit, cdsize);
			config.cdsize = atoi(cdsize);
			break;
		}
		mvwaddch(volwin, 4, 3, ' ');
		mvwaddch(volwin, 5, 3, ' ');
		mvwaddch(volwin, 6, 3, ' ');
		if (config.cdsize == 657)
			mvwaddch(volwin, 4, 3, '*');
		else if (config.cdsize == 702)
			mvwaddch(volwin, 5, 3, '*');
		else
			mvwaddch(volwin, 6, 3, '*');
		wrefresh(volwin);
	}
	write_conf(config);

	delwin(volwin);
	return;
}




/* 
 * Display CD usage summary on bottom of cdw window
 *
 * Display amount of space used by selected files, amount of 
 * space that would stay free disk, and percent of CD disk usage.
 * CD\DVD size is taken from config panel/file.
 *
 * \param double size - size of selected files
 */
void show_disk_usage(double files_size)
{
	int i;
	float percent;

	for (i = 30; i < COLS - 3; i++)
		mvwaddch(mainwin, LINES - 7, i, ' ');
	for (i = 1; i < 30; i++)
		mvwaddch(mainwin, LINES - 6, 29 + i, ACS_BOARD);
	if (files_size <= config.cdsize) {
		percent = (files_size / ((double) config.cdsize / 100));
		for (i = 1; i < (int) ((percent * 30) / 100); i++)
			mvwaddch(mainwin, LINES - 6, 29 + i, ACS_BLOCK);
	} else {
		percent = (float) (100);
		for (i = 1; i < 30; i++)
			mvwaddch(mainwin, LINES - 6, 29 + i, ACS_BLOCK);
	}
	if (config.cdsize - files_size >= 0)
		mvwprintw(mainwin, LINES - 7, 30, _("Used: %2.1f%%, Wasted: %.0f/%d MB"), percent, config.cdsize - files_size, config.cdsize);
	else
		mvwprintw(mainwin, LINES - 7, 30, _("Used: %2.1f%%, Wasted: %d MB"), percent, config.cdsize);

	return;
}




/*
 * Refresh content of main app window
 *
 * Refresh content of main app window - the one containing menu 
 * on the left and selected files/dirs list on the right + some 
 * statistics at the bottom. This may need reading (possibly) 
 * changed content of files-to-cd listing.
 *
 * \param bool real - do you want to refresh content of main 
 *        window by re-reading it from listing on hard-disk 
 *        (may be time-consuming)?
 *
 * \returns 0
 */
int select_window(bool real)
{
	int pos = 0;
	int i;
	double files_size = 0;
	long long int ds = 0;

	WINDOW *select_sub;

	/* Do time-consuming refresh of file list?
	 * It is time consuming, because this is not ordinary curses 
	 * wrefresh(), this code has to allocate some memory and move data from 
	 * one place to another and print this data to screen. */
	if (real == TRUE) { 

		/* clean up previous selected files list and all associated data structures */
		clean_cdmenu();

		/* fill table with selected files paths, return number of paths */
		num = filelist_items(&filetable);

		/* debug code

		if (num > 0) {
			fprintf(stderr, "first path is");
			fprintf(stderr, "\"%s\"\n", *(filetable + num - 1));
		}
		*/

		cditems = (ITEM **) calloc(num + 1, sizeof(ITEM *));

		if (num > 0) {
			int i;
			/* this file list is in reality a ncurses menu, so 
			 * we will use menu **ITEM data structure (cditems) here */
			for (i = 0; i < num; ++i) {
				/* FIXME find a way to save path with file type 
				 * like it was done in original implementation of cdw */
				/*
				if (cdfiles[i]->d_type == DT_DIR) {
					cditems[pos++] = new_item(cdfiles[i]->d_name, " DIR");
				} else if (cdfiles[i]->d_type != DT_DIR) {
					cditems[pos++] = new_item(cdfiles[i]->d_name, "");
				}
				*/

				// FIXME - here, instad of "    ", should be file/DIR info
				cditems[pos++] = new_item( *(filetable + i), "    " );
			}
		}
			
		/* Let's continue constructing ncurses menu cdmenu using 
		 * ncurses data structure **ITEM cditems. Note that this menu
		 * will be constructed even if there are no files */

		/* last element must be NULL - ncurses requirement */
		cditems[num] = (ITEM *) NULL; 
		
		cdmenu = new_menu((ITEM **) cditems);
		set_menu_win(cdmenu, mainwin);
		set_menu_sub(cdmenu, derwin(mainwin, LINES - 12, COLS - 31, 4, 28));
		set_menu_format(cdmenu, LINES - 12, 1);
		set_menu_fore(cdmenu, COLOR_PAIR(1));
		set_menu_back(cdmenu, COLOR_PAIR(1));
		post_menu(cdmenu);

	}

	/* rest of function redraws frames and it's content 
	 * at the bottom of window */

	select_sub = derwin(mainwin, LINES - 10, COLS - 29, 3, 27);
	wattrset(select_sub, A_NORMAL | COLOR_PAIR(1));
	mvwaddch(select_sub, LINES - 13, 0, ACS_LTEE);
	mvwhline(select_sub, LINES - 13, 1, ACS_HLINE, COLS - 31);
	mvwaddch(select_sub, LINES - 13, COLS - 30, ACS_RTEE);
	wattrset(mainwin, A_BOLD | COLOR_PAIR(8));
	size = 0;

	/* print total size of all items in selected files/dirs list and number of items on this list */
	for (i = 30; i < COLS - 3; i++)
		mvwaddch(mainwin, LINES - 9, i, ' ');

	if (num > 0) {
		ds = dirsize(filetable, num);
		files_size = (double) ((ds / 1024) / 1024);
	}
	size = files_size; /* FIXME - size is global variable - find a way to remove it */
	mvwprintw(mainwin, LINES - 9, 30, _("Size: %.0f Mb in %d files"), files_size, num);

	show_disk_usage(files_size);

	wrefresh(mainwin);

	return 0;
}



/**
 * Display widget allowing to enter simple string value
 *
 * Display widget with title and label, allowing user to enter simple 
 * value, like label, path or number. Parameter "mode" puts restrictions
 * on entered value.
 *
 * \param char *title - window title
 * \param char *label - label put into window
 * \param char *prop - string containing entered value
 * \param int mode - widget mode, puts some restrictions on entered value
 *
 * \returns - TBD
 */
int input_box(char *title, char *label, char *prop, int mode)
{
	WINDOW *volwin;
	// WINDOW *edit;
	FIELD *field[1];
	FORM *form1;
	int ret;
	int ch;

	/* dialog window */
	volwin = newwin(10, 50, (LINES - 10) / 2, (COLS - 50) / 2);
	keypad(volwin, TRUE);
	wbkgd(volwin, COLOR_PAIR(2));
	werase(volwin);
	nice_box(volwin, _(title), "", 0);
	mvwprintw(volwin, 3, 3, _(label));

	/* input field 
	 * new_field params: height, width, starty, startx, number of 
	 * offscreen rows and number of additional working buffers */
	field[0] = new_field(1, 44, 0, 0, 0, 0);
	set_field_buffer(field[0], 0, prop);
	set_field_back(field[0], COLOR_PAIR(6));
	set_field_fore(field[0], COLOR_PAIR(6));
	field_opts_off(field[0], O_STATIC); /* stretch to fit entered data */
	set_max_field(field[0], 200); /* limit stretching of the field; FIXME - arbitrary magic number */
	field[1] = NULL;

	/* form containing field. placed in window */
	form1 = new_form(field);
	set_form_win(form1, volwin);
	set_form_sub(form1, derwin(volwin, 1, 44, 5, 2));
	post_form(form1);
	
	set_current_field(form1, field[0]);
	form_driver(form1, REQ_END_LINE);
	wrefresh(volwin);

	ch = 0;
	while ((ch != 27) && (ch != 10)) {
		ch = wgetch(volwin);
		switch (ch) {
		case KEY_LEFT:
			form_driver(form1, REQ_PREV_CHAR);
			break;
		case KEY_RIGHT:
			form_driver(form1, REQ_NEXT_CHAR);
			break;
		case KEY_BACKSPACE:
			form_driver(form1, REQ_DEL_PREV);
			break;
		case KEY_DC:
			form_driver(form1, REQ_DEL_CHAR);
			break;
		case KEY_F(10):
		case 10:
			// return main loop
			form_driver(form1, REQ_VALIDATION);
			sprintf(prop, rtrim(field_buffer(field[0], 0)));
			write_conf(config);
			//return ch;
			break;
		default:
			/* push char ch to form */
			form_driver(form1, ch);
			break;
			wrefresh(volwin);
		}
	}

/*    edit=derwin(volwin, 1, 24, 5, 3);
    wbkgd(edit, COLOR_PAIR(6));
    werase(edit);
    wrefresh(edit);
    ret=inbox(edit, prop);*/

	/* clean up */
	unpost_form(form1);
	free_form(form1);
	free_field(field[0]);
	delwin(volwin);
	wrefresh(volwin);

	if (ch == 27)
		ret = 0;
	else
		ret = 1;
	write_conf(config);
	return ret;
}




/*
 * Small wrapper function - shift content of 'Add files' list in window
 *
 * \param WINDOW *addwinlist - list to scroll
 * \param int menupos - cursor position on whole list of files (index of highlighted file); it has value calculated after key has been hit
 * \param fileitem *filei - current directory listing
 * \param int maxfiles - number of items in current directory listing
 */
void scroll_addlist(WINDOW *addwinlist, int menupos, fileitem *filei, int maxfiles)
{
	int startpos = 0; // index of first file that will be displayed in scrolled window
	int i; /* rows index of addwinlist window */
	
	/* start displaying items starting from item with index equal (menupos - "cursor position");
	 * cursor position is (menupos % getmaxy(addwinlist)) - this is cursor pos. in window */
	startpos = menupos - (menupos % getmaxy(addwinlist));

	
	/* increment window row number and dir listing index at the same time */
	for (i = 0, startpos; 
	     startpos <= maxfiles, i < getmaxy(addwinlist); 
	     i++, startpos++) {
		/* display startpos-th element in i-th row of window */
		highlight(i, filei[startpos], 1); 

		/* file name is displayed, so we could just go to next iteration, 
		 * but let's display some additional file info (in the same row) */
		if (filei[startpos].type == 0) { /* let user know that this is a dir */
			wattrset(addwinlist, A_BOLD | COLOR_PAIR(2));
			mvwprintw(addwinlist, i, getmaxx(addwinlist) - 5, "DIR");
		} else if (filei[startpos].type == 1) { /* show file size (human readable) */
			wattrset(addwinlist, COLOR_PAIR(2));
			mvwprintw(addwinlist, i, getmaxx(addwinlist) - 5, "%4d%s",
					(filei[startpos].size / 1024 / 1024 > 1 ? 
						filei[startpos].size / 1024 / 1024 : 
						filei[startpos].size / 1024), (filei[startpos].size / 1024 / 1024 > 1 ? "M" : "K"));
		}
	}
	return;
}




/*
 * Draw nice box around window, with label on top and (maybe) bottom of window
 *
 * \param WINDOW *window - ncurses window to be boxed
 * \param char *top_string - string that will appear on top of window
 * \param char *bottom_string - string that will appear on bottom of window
 * \param int width - width of window, can be 0 if bottom_string == ""; 
 * 		FIXME - can width can be guessed from window?
 *
 */
void nice_box(WINDOW *window, char *top_string, char *bottom_string, int width)
{

	box(window, 0, 0);
	if (strcmp(bottom_string, "")) {
		// add additinal window information 
		mvwprintw(window, width - 1, 2, "( %s )", bottom_string);
	}
	mvwaddch(window, 0, 2, ACS_RTEE);

	// add window title
	mvwprintw(window, 0, 3, " %s ", top_string);
	waddch(window, ACS_LTEE);

	return;
}




int cdw_title_init(void)
{
	int title_len = strlen(PACKAGE) + 2 + strlen(VERSION) + 9;
	TITLE = (char *) malloc(title_len);
	sprintf(TITLE, "::: %s-%s :::", PACKAGE, VERSION);

	return 0;
}




/*
 * Run common ncurses setup code - simple wrapper
 *
 * \returns 0 on success, -1 if console has too few lines or columns
 */
int cdw_curses_init(void)
{
	initscr();
	start_color();
	cbreak();
	noecho();
	keypad(stdscr, TRUE);

	init_curses_colors();

	lines = LINES - 2;
	if ((LINES < 24) || (COLS < 79)) {
		return -1;
	}

	return 0;

}




/*
 * Remove menu containing paths selected for recording to CD
 *
 * Remove menu in central part of cdw window. This function is 
 * called either when user adds/removes files to list (every time 
 * new list (in fact: ncurses menu) is allocated, so previous 
 * version must be cleaned up) or at the end of the program.
 */
void clean_cdmenu(void)
{
	int i;
	if (cdmenu != NULL) {
		unpost_menu(cdmenu);
		free_menu(cdmenu);
		cdmenu = NULL;

		for (i = 0; i < num; i++) {
			free_item(cditems[i]);
			cditems[i] = NULL;
		}
			
		if (cditems != NULL ){
			free(cditems);
			cditems = NULL;
		}
		if (filetable != NULL) {
			free(filetable);
			filetable = NULL;
		}
	}

	return;
}




/*
 * Remove menu containing cdw commands
 *
 * Remove menu in left part of cdw window. This function is 
 * called at the end of the program.
 */
void clean_menu(void)
{
	int i;
	if (menu != NULL) {
		unpost_menu(menu);
		free_menu(menu);
		menu = NULL;
	
		for (i = 0; i < MENU_ITEMS; ++i){
			free_item(items[i]);
			items[i] = NULL;
		}

		free(items);
		items = NULL;

		/* we don't free cdw menu labels since they are static */
	}	
	return;
}




/*
 * Check for existence of bootimage
 *
 * This functio nwas created because this code appeared twice in main.c.
 * Maybe it should be replaced by generic file_exists(char *file).
 *
 * \returns 0 on failure, 1 on success
 */
int check_for_boot_image(void)
{
	char bootimage[255];
	FILE *isofile;

	int boot_exists;

	sprintf(bootimage, "%s/.cdw/%s", getenv("HOME"), config.bootimg);
	if ((isofile = fopen(bootimage, "r")) == NULL) {
		boot_exists = dialogbox(_("Boot image doesn'n exists!"), _("Boot image error"), 0);
		boot_exists = 0; /* FIXME - why we are assigning 0 here? */
		/// wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
		/// select_window(FALSE);
		return 0;
	} else {
		fclose(isofile);
		return 1;
	}
}




/*
 * Fill list with file names, types and sizes of files in given dir
 *
 * Fill custom data structure: list of items, each containing information 
 * about file: name, size and type. So in fact - extract some information 
 * about current dir that is stored in struct dirent **eps.
 *
 * \param char *dir
 * \param fileitem **filei - allocated list of file information items, big enough to store information about all files in dir
 * \param struct dirent **eps
 * \param int n - number of files in given dir
 *
 * \returns 0 on success, -1 on failure
 */
int fill_filelist(char *dir, fileitem **filei, struct dirent **eps, int n)
{
	struct stat stbuf;
	int pos = 0;
	bool failure = false;
	int rv = 0;

	/* Here we have two for loops - first checks if eps item 
	 * is a DIR - so we will have directories on top of list.
	 * Then second looks for files. Common counter (pos) ensures that we will
	 * check all items of eps. */
	/* FIXME - I'm sure that filei can be handled more elegantly */
	int i;
	for (i = 0; i < n; ++i) {
		if ( (rv = check_fullpath_dir(dir, eps[i]->d_name, &stbuf)) == -1) {
			failure = true;
			break;
		}
		if (((stbuf.st_mode & S_IFMT) == S_IFDIR) && (strcmp(eps[i]->d_name, ".") != 0)) {
			strcpy((*filei)[pos].filename, eps[i]->d_name);
			(*filei)[pos].size = 0;
			(*filei)[pos].type = 0;
			pos++;
		}
	}
	for (i = 0; i < n; ++i) {
		if ( (rv = check_fullpath_dir(dir, eps[i]->d_name, &stbuf)) == -1) {
			failure = true;
			break;
		}
		if ((stbuf.st_mode & S_IFMT) != S_IFDIR) {
			strcpy((*filei)[pos].filename, eps[i]->d_name);
			(*filei)[pos].size = stbuf.st_size;
			(*filei)[pos].type = 1;
			pos ++;
		}
	}
	if (failure) {
		return -1;
	} else {
		return 0;
	}

}




int blank_method_selector(void)
{
	WINDOW *blankwin;
	int ch = 0;

	blankwin = newwin(10, 30, (LINES - 10) / 2, (COLS - 30) / 2);
	wbkgd(blankwin, COLOR_PAIR(2));
	werase(blankwin);
	keypad(blankwin, true); // enable arrow keys and other keys

	nice_box(blankwin, _("Blanking method"), "", 0);
	
	mvwprintw(blankwin, 2, 3, _("Select cd blanking method"));
	mvwprintw(blankwin, 4, 5, "1 - fast");
	mvwprintw(blankwin, 5, 5, "2 - all");


	if ( !strcmp(config.blank, "fast") ) {
		mvwaddch(blankwin, 4, 3, '*');
	} else if ( !strcmp(config.blank, "all") ) {
		mvwaddch(blankwin, 5, 3, '*');
	} else {
		mvwaddch(blankwin, 4, 3, '*');
	}

	while (1) {
		ch = wgetch(blankwin);
		if ( ch == '1' ) {
			strcpy(config.blank, "fast");
		} else if ( ch == '2' ) {
			strcpy(config.blank, "all");
		} else if ( ch == KEY_UP || ch == KEY_DOWN) {
			if ( !strcmp(config.blank, "fast") ) {
				strcpy(config.blank, "all");
				mvwaddch(blankwin, 4, 3, '*');
				mvwaddch(blankwin, 5, 3, ' ');
			} else {
				strcpy(config.blank, "fast");
				mvwaddch(blankwin, 5, 3, '*');
				mvwaddch(blankwin, 4, 3, ' ');
			}
		} else if ( ch == KEY_LEFT || ch == KEY_RIGHT ) { // user might want to use these, but don't react
			;
		} else if ( ch == 27 || ch == 'q' || ch == 'Q') { // escape, quit
			return -1;
		} else if ( ch == 10 ) { // enter
			return 0;
		} else {
			; // FIXME - should this be left empty?
		}
		mvwaddch(blankwin, 4, 3, ' ');
		mvwaddch(blankwin, 5, 3, ' ');

		if ( !strcmp(config.blank, "fast") ) {
			mvwaddch(blankwin, 4, 3, '*');
		} else if ( !strcmp(config.blank, "all") ) {
			mvwaddch(blankwin, 5, 3, '*');
		} else {
			;
		}
		wrefresh(blankwin);
	}
	write_conf(config);

	delwin(blankwin);
	return -1;
}




/*
 * Show dialog window with message and buttons
 *
 * Show dialog window with title and message. Depending on type of dialog 
 * user is presented with one, two or three buttons: value of 
 * parameter 'type' describes them. Function returns value of selected button.
 * Background color depends on 'type' value: it is red for 'OK/Cancel' and
 * 'Yes/No/Cancel' dialogs and gray for 'Ok' dialog.
 *
 * FIXME: 'Enter' behaviour is not described
 *
 * \param char *message - message displayed in dialog window
 * \param char *title - dialog window title
 * \param int type - type of dialog - values defined in header
 * \returns value of pressed button (valid values defined in header)
 */
int dialogbox(char *message, char *title, int type)
{
	WINDOW *dialog;
	int line=3, n, i, c, bg_color;
	
	// initial dialog WINDOW size: maxwidth and maxline
	int maxline=0, maxwidth=30;

	char temp[50];

	int button; // current button
	// initial button = OK/YES
	if (type == DIALOG_YES_NO_CANCEL) {
		button = BUTTON_YES;
	} else { // DIALOG_OK_CANCEL or DIALOG_OK, BUTTON_OK wil be default in both cases
		button = BUTTON_OK;
	}
	
	/* recalculate window size */
	temp[0] = '\0'; 
	n = 0;
	for (i = 0; i < strlen(message); i++){
		if (message[i] != '\n') 
			temp[n++] = message[i];
		if (( message[i] == '\n') || (i == strlen(message) - 1) || ((n > 30) && message[i] == ' ')){
			temp[n] = '\0';
			if (maxwidth < strlen(temp) + 8) {
				maxwidth = strlen(temp) + 8;
			}
			temp[0] = '\0';
			n = 0;
			if (message[i] == '\n') {
				maxline++;
			}
		}
	}

	switch (type){
		case DIALOG_OK:
			bg_color = 2;
			break;
		case DIALOG_OK_CANCEL:
		case DIALOG_YES_NO_CANCEL:
			bg_color = 10;
			break;
	}

	dialog = newwin(9 + maxline, maxwidth, (LINES - (9 + maxline)) / 2, (COLS - maxwidth) / 2);
	wbkgd(dialog, COLOR_PAIR(bg_color));
	werase(dialog);
	nice_box(dialog, title, "", 0);
	keypad(dialog, TRUE);
	
	/* wrap message text so that it fits into dialog window;
	 * print each wrapped line into dialog window */
	temp[0] = '\0'; 
	n = 0;
	for (i = 0; i < strlen(message); i++){
		if (message[i] != '\n') 
			temp[n++] = message[i];
		if ((message[i] == '\n') || (i == strlen(message) - 1) || ((n > 30) && message[i] == ' ')){
			temp[n] = '\0';
			mvwprintw(dialog, line++, (maxwidth - strlen(temp)) / 2, temp);
			temp[0] = '\0';
			n = 0;
		}
	}

	// buttons positions
	// buttons are in one row
	int buttons_row = maxline + 6;
	// 1-button dialog
	int ok_col_1 = (getmaxx(dialog) - strlen(_("Ok"))) / 2;
	// 2-button dialog
	int ok_col_2 = 8;
	int cancel_col_2 = (maxwidth - 7) - strlen(_("Cancel"));
	// 3-button dialog
	int yes_col_3 = 4; 
	int no_col_3 = 14; 
	int cancel_col_3 = (maxwidth - 7) - strlen(_("Cancel"));

	/* draw initial dialog box */
	if (type == DIALOG_OK) {
		wattrset(dialog, COLOR_PAIR(6));	
		mvwprintw(dialog, buttons_row, ok_col_1	, "[ %s ]", _("Ok"));
	}
	if (type == DIALOG_OK_CANCEL) {
		wattrset(dialog, COLOR_PAIR(6));	
		mvwprintw(dialog, buttons_row, ok_col_2, "[ %s ]", _("Ok"));

		wattrset(dialog, COLOR_PAIR(bg_color));	
		mvwprintw(dialog, buttons_row, cancel_col_2, "[ %s ]", _("Cancel"));
	}
	if (type == DIALOG_YES_NO_CANCEL) {
		wattrset(dialog, COLOR_PAIR(6));	
		mvwprintw(dialog, buttons_row, yes_col_3, "[ %s ]", _("Yes"));

		wattrset(dialog, COLOR_PAIR(bg_color));	
		mvwprintw(dialog, buttons_row, no_col_3, "[ %s ]", _("No"));

		wattrset(dialog, COLOR_PAIR(bg_color));	
		mvwprintw(dialog, buttons_row, cancel_col_3, "[ %s ]", _("Cancel"));
	}
	wrefresh(dialog);

	/* handle cursor keys */
	while (((c = mvwgetch(dialog, 8 + maxline, maxwidth - 1)) != 27) && (c != 10)) {
		/* select action according to dialog type,
		 * key pressed and currently selected button */
		if (type == DIALOG_OK_CANCEL) {
			switch(c){
			case KEY_RIGHT:
				if (button == BUTTON_OK){
					wattrset(dialog, COLOR_PAIR(bg_color));	
					mvwprintw(dialog, buttons_row, ok_col_2, "[ %s ]", _("Ok"));
					wattrset(dialog, COLOR_PAIR(6));	
					mvwprintw(dialog, buttons_row, cancel_col_2, "[ %s ]", _("Cancel"));
					button = BUTTON_CANCEL;
				}
				break;
			case KEY_LEFT:
				if (button == BUTTON_CANCEL){
					wattrset(dialog, COLOR_PAIR(6));	
					mvwprintw(dialog, buttons_row, ok_col_2, "[ %s ]", _("Ok"));
					wattrset(dialog, COLOR_PAIR(bg_color));	
					mvwprintw(dialog, buttons_row, cancel_col_2, "[ %s ]", _("Cancel"));
					button = BUTTON_OK;
				}
				break;
			}
		} else 	if (type == DIALOG_YES_NO_CANCEL) {
			switch(c){
			case KEY_RIGHT:
				if (button == BUTTON_YES){
					wattrset(dialog, COLOR_PAIR(bg_color));	
					mvwprintw(dialog, buttons_row, yes_col_3, "[ %s ]", _("Yes"));
					wattrset(dialog, COLOR_PAIR(6));	
					mvwprintw(dialog, buttons_row, no_col_3, "[ %s ]", _("No"));
					button = BUTTON_NO;
				} else 	if (button == BUTTON_NO){
					wattrset(dialog, COLOR_PAIR(bg_color));	
					mvwprintw(dialog, buttons_row, no_col_3, "[ %s ]", _("No"));
					wattrset(dialog, COLOR_PAIR(6));	
					mvwprintw(dialog, buttons_row, cancel_col_3, "[ %s ]", _("Cancel"));
					button = BUTTON_CANCEL;
				} else 	if (button == BUTTON_CANCEL){
					wattrset(dialog, COLOR_PAIR(bg_color));	
					mvwprintw(dialog, buttons_row, cancel_col_3, "[ %s ]", _("Cancel"));
					wattrset(dialog, COLOR_PAIR(6));	
					mvwprintw(dialog, buttons_row, yes_col_3, "[ %s ]", _("Yes"));
					button = BUTTON_YES;
				}
				break;
			case KEY_LEFT:
				if (button == BUTTON_YES){ // yes -> cancel
					wattrset(dialog, COLOR_PAIR(bg_color));	
					mvwprintw(dialog, buttons_row, yes_col_3, "[ %s ]", _("Yes"));
					wattrset(dialog, COLOR_PAIR(6));	
					mvwprintw(dialog, buttons_row, cancel_col_3, "[ %s ]", _("Cancel"));
					button = BUTTON_CANCEL;
				} else if (button == BUTTON_NO){ // no -> yes
					wattrset(dialog, COLOR_PAIR(bg_color));	
					mvwprintw(dialog, buttons_row, no_col_3, "[ %s ]", _("No"));
					wattrset(dialog, COLOR_PAIR(6));	
					mvwprintw(dialog, buttons_row, yes_col_3, "[ %s ]", _("Yes"));
					button = BUTTON_YES;
				} else 	if (button == BUTTON_CANCEL){ // cancel -> no
					wattrset(dialog, COLOR_PAIR(bg_color));	
					mvwprintw(dialog, buttons_row, cancel_col_3, "[ %s ]", _("Cancel"));
					wattrset(dialog, COLOR_PAIR(6));	
					mvwprintw(dialog, buttons_row, no_col_3, "[ %s ]", _("No"));
					button = BUTTON_NO;
				}
				break;
			} // switch
		} // if
		wrefresh(dialog);
	} // while

	if (c == 27) {
		button = 1;
	}

	delwin(dialog);
	wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
	select_window(FALSE);
	return button;
}
