/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <linux/cdrom.h>
#include <sys/ioctl.h>

// #include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <stdio.h>
#include <sys/types.h>
#include <fcntl.h>

#include <ncursesw/ncurses.h>
#include <ncursesw/menu.h>
#include <ncursesw/form.h>

#include "log.h" // dialogbox()
#include "options.h"

extern struct conf config;
extern WINDOW *mainwin;


/*
 * Check for cdrom drive status
 *
 * Check for cdrom drive status using ioctl (with parameter CDROM_DRIVE_STATUS).
 * Documentation on ioctl can be found on linux source code documentation in
 * Documentation/ioctl/cdrom.txt.
 *
 * This function tries to open cdrom device described by config.cdrwdevice.
 *
 * Return codes are mapped to new names in cdrom_ctrl.h, so there is no need 
 * to include linux/cdrom.h again in *c files.
 *
 * \returns normal ioctl return codes:
 * 	            CDS_NO_INFO         Information not available.
 *                  CDS_NO_DISC
 *                  CDS_TRAY_OPEN
 *                  CDS_DRIVE_NOT_READY
 *                  CDS_DISC_OK
 *                  -1                  error
 *
 *                  or -2 on cdrom_status() error (cannot open cdrom file)
 */
int cdrom_status(void)
{
	int cddev;

	if ( (cddev=open(config.cdrwdevice, O_RDONLY | O_NONBLOCK)) != -1 ){
		int rv = ioctl(cddev, CDROM_DRIVE_STATUS, 0);
		close(cddev);
		return rv;
 	}

	return -2;
}




void eject_tray(char *device)
{
    int cddev;
    
    if ( (cddev=open(device, O_RDONLY | O_NONBLOCK)) != -1 ){
	ioctl(cddev, CDROMEJECT, 0);
	close(cddev);
    }
}




void close_tray(char *device)
{
    int cddev;
    
    if ( (cddev=open(device, O_RDONLY | O_NONBLOCK)) != -1 ){
	ioctl(cddev, CDROMCLOSETRAY, 0);
	close(cddev);
    }
}





/*
 * Ensure that disc is in drive and drive tray is closed
 *
 * This function prevents program from attempting to access CD
 * when there is no disc in drive. 
 * Uses ioctl() to check drive status and ensure that disc is in drive.
 * Uses dialog box to communicate with user.
 *
 * \returns true if disc is in tray and tray is closed, false if user 
 * 	decided to not put disc in drive or drive don't answer
 *
 */
bool disc_in_drive(void)
{
	int k = 0;
	int i;

	int rvd;

	for (i = 0; i < 10; i++) {
		int rv = cdrom_status();
		switch (rv) {
			case CDS_DISC_OK:
				return true;
			case CDS_NO_DISC:
				rvd = dialogbox("No disc in drive. Please insert disc.", "CDROM error", 3);
				/// wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
				/// select_window(FALSE);
				if (rvd == 0) {
					break; 
				} else {
					return false;
				}
			case CDS_TRAY_OPEN:
				rvd = dialogbox("Please close CDROM tray.", "CDROM error", 3);
				/// wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
				/// select_window(FALSE);
				if (rvd == 0) {
					break; 
				} else {
					return false;
				}
			case CDS_DRIVE_NOT_READY:
				sleep(2);
				k++;
				if (k == 5) {
					dialogbox("CDROM doesn't respond - timeout.", "CDROM error", 0);
					/// wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
					/// select_window(FALSE);
					return false; // timeout
				} else {
					break;
				}
			case CDS_NO_INFO:
				rvd = dialogbox("Can't get CDROM info. Continue?", "CDROM error", 3);
				/// wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
				/// select_window(FALSE);
				if (rvd == 0) {
					return true;
				} else {
					return false;
				}
		} // switch
		wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
		select_window(TRUE);

	} // for
	dialogbox("Please try once more", "CDROM message", 0);
	/// wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
	/// select_window(FALSE);

	return false; // user couldn't decide what to do
}
