/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Pack    pack         Pack
*/

#ifdef _OPENMP
#include <omp.h>
#endif

#include <climits>

#include <cdi.h>

#include "process_int.h"
#include "cdo_vlist.h"
#include "datetime.h"
#include "cdo_default_values.h"


static int
get_type_values(const int datatype, double &tmin, double &tmax)
{
  int status = 0;

  // clang-format off
  switch (datatype)
    {
    case CDI_DATATYPE_INT8:   tmin = -SCHAR_MAX + 1;  tmax = SCHAR_MAX;     break;
    case CDI_DATATYPE_UINT8:  tmin = 0;               tmax = UCHAR_MAX - 1; break;
    case CDI_DATATYPE_INT16:  tmin = -SHRT_MAX + 1;   tmax = SHRT_MAX;      break;
    case CDI_DATATYPE_UINT16: tmin = 0;               tmax = USHRT_MAX - 1; break;
    case CDI_DATATYPE_INT32:  tmin = -INT_MAX + 1;    tmax = INT_MAX;       break;
    case CDI_DATATYPE_UINT32: tmin = 0;               tmax = UINT_MAX - 1;  break;
    default: status = 1; break;
    }
  // clang-format on

  return status;
}

static int
computeScaleAndOffset(const int datatype, const double fmin, const double fmax, double &scale_factor, double &add_offset)
{
  scale_factor = 1.0;
  add_offset = 0.0;

  double tmin, tmax;
  if (get_type_values(datatype, tmin, tmax)) return 1;

  if (IS_NOT_EQUAL(fmin, fmax))
    {
      scale_factor = (fmax - fmin) / (tmax - tmin);
      add_offset = ((fmax + fmin) - scale_factor * (tmin + tmax)) / 2;
    }

  return 0;
}

void *
Pack(void *process)
{
  int nrecs;
  int varID, levelID;
  int datatype = CDI_DATATYPE_INT16;
  DateTimeList dtlist;

  cdoInitialize(process);

  operatorCheckArgc(0);

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  VarList varList;
  varListInit(varList, vlistID1);

  const auto nvars = vlistNvars(vlistID1);

  FieldVector3D vars;

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      constexpr size_t NALLOC_INC = 1024;
      if ((size_t) tsID >= vars.size()) vars.resize(vars.size() + NALLOC_INC);

      dtlist.taxisInqTimestep(taxisID1, tsID);

      fieldsFromVlist(vlistID1, vars[tsID]);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          auto &field = vars[tsID][varID][levelID];
          field.init(varList[varID]);
          cdoReadRecord(streamID1, field);
        }

      tsID++;
    }

  const int nts = tsID;

  if (CdoDefault::DataType != CDI_UNDEFID)
    {
      if (CdoDefault::DataType == CDI_DATATYPE_FLT64 || CdoDefault::DataType == CDI_DATATYPE_FLT32)
        {
          cdoWarning("Changed default output datatype to int16");
          CdoDefault::DataType = datatype;
        }
      else
        {
          datatype = CdoDefault::DataType;
        }
    }

  CdoDefault::DataType = datatype;

  for (varID = 0; varID < nvars; varID++)
    {
      double fmin = 1.e300, fmax = -1.e300;
      size_t nmisspv = 0;

      const auto timetype = varList[varID].timetype;
      const auto missval1 = varList[varID].missval;
      const auto gridsize = varList[varID].gridsize;
      const auto nlevels = varList[varID].nlevels;
      for (levelID = 0; levelID < nlevels; levelID++)
        {
          for (int t = 0; t < nts; t++)
            {
              if (t > 0 && timetype == TIME_CONSTANT) continue;

              auto &field = vars[t][varID][levelID];
              const auto nmiss = field.nmiss;

              if (nmiss) nmisspv += nmiss;

              MinMax mm;
              if (field.memType == MemType::Float)
                mm = nmiss ? varrayMinMaxMV(gridsize, field.vec_f, (float)missval1) : varrayMinMax(gridsize, field.vec_f);
              else
                mm = nmiss ? varrayMinMaxMV(gridsize, field.vec_d, missval1) : varrayMinMax(gridsize, field.vec_d);

              fmin = std::min(fmin, mm.min);
              fmax = std::max(fmax, mm.max);
            }
        }

      vlistDefVarDatatype(vlistID2, varID, datatype);
      const auto missval2 = vlistInqVarMissval(vlistID2, varID);

      if (nmisspv)
        {
          double tmin, tmax;
          if (!get_type_values(datatype, tmin, tmax))
            {
              if (!(missval2 < tmin || missval2 > tmax))
                cdoWarning("new missing value %g is inside data range (%g - %g)!", missval2, tmin, tmax);

              for (levelID = 0; levelID < nlevels; levelID++)
                {
                  for (int t = 0; t < nts; t++)
                    {
                      if (t > 0 && timetype == TIME_CONSTANT) continue;

                      auto &field = vars[t][varID][levelID];
                      if (field.nmiss)
                        {
                          if (field.memType == MemType::Float)
                            {
                              auto &v = field.vec_f;
                              for (size_t i = 0; i < gridsize; ++i)
                                if (DBL_IS_EQUAL(v[i], missval1)) v[i] = missval2;
                            }
                          else
                            {
                              auto &v = field.vec_d;
                              for (size_t i = 0; i < gridsize; ++i)
                                if (DBL_IS_EQUAL(v[i], missval1)) v[i] = missval2;
                            }
                        }
                    }
                }
            }
        }

      // printf("fmin %g fmax %g missval %g\n", fmin, fmax, vlistInqVarMissval(vlistID2, varID));
      double sf, ao;
      if (!computeScaleAndOffset(datatype, fmin, fmax, sf, ao))
        {
          if (varList[varID].memType == MemType::Float)
            {
              sf = (float)sf;
              ao = (float)ao;
            }
          // printf("sf = %g ao = %g \n", sf, ao);
          // printf("smin %g smax %g\n", (fmin - ao)/sf, (fmax -ao)/sf);
          vlistDefVarScalefactor(vlistID2, varID, sf);
          vlistDefVarAddoffset(vlistID2, varID, ao);
        }
    }

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  for (tsID = 0; tsID < nts; tsID++)
    {
      dtlist.taxisDefTimestep(taxisID2, tsID);
      cdoDefTimestep(streamID2, tsID);

      for (varID = 0; varID < nvars; varID++)
        {
          if (tsID > 0 && varList[varID].timetype == TIME_CONSTANT) continue;
          for (levelID = 0; levelID < varList[varID].nlevels; levelID++)
            {
              auto &field = vars[tsID][varID][levelID];
              if (field.hasData())
                {
                  cdoDefRecord(streamID2, varID, levelID);
                  cdoWriteRecord(streamID2, field);
                }
            }
        }
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
