/*
 * exec.c
 *
 * Copyright (C) 2004,2014 Bastian Blank <waldi@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#define _GNU_SOURCE

#include <config.h>

#include <debian-installer.h>

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/poll.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

#include "execute.h"
#include "log.h"
#include "target.h"

const char *const execute_environment_target[] =
{
  "DEBIAN_FRONTEND=noninteractive",
  "PATH=/sbin:/usr/sbin:/bin:/usr/bin",
  NULL
};

const struct execute_io_info execute_io_log_info[] = {
  { 1, execute_io_log_handler, (void *)DI_LOG_LEVEL_OUTPUT },
  { 2, execute_io_log_handler, (void *)LOG_LEVEL_OUTPUT_STDERR },
};
const unsigned int execute_io_log_info_count = 2;

static int internal_di_exec_child (const char *const argv[], const char *const envp[], pid_t pid, di_process_handler *child_prepare_handler, void *child_prepare_user_data, int fd_status)
{
  if (child_prepare_handler)
    if (child_prepare_handler (pid, child_prepare_user_data))
    {
      int status = 255;
      if (write (fd_status, &status, sizeof (int)) != sizeof (int))
        abort ();
      _exit (255);
    }

  execvpe (argv[0], (char *const *) argv, (char *const *) envp);

  int status = -errno;
  if (write (fd_status, &status, sizeof (int)) != sizeof (int))
    abort ();
  _exit (255);
}

static int internal_status_handle (int fd)
{
  struct pollfd pollfds[1] = {
    { fd, POLLIN, 0 }
  };

  while (poll (pollfds, 1, -1) >= 0)
  {
    if (pollfds[0].revents & POLLIN)
    {
      int status;
      if (read (pollfds[0].fd, &status, sizeof (int)) < 0)
        abort ();
      return status;
    }
    if (pollfds[0].revents & POLLHUP)
      return 0;
  }

  abort ();
}

static void internal_io_handle (int fds[][2], const struct execute_io_info io_info[], unsigned int io_info_count)
{
  struct pollfd pollfds[io_info_count];
  FILE *files[io_info_count];

  for (unsigned int i = 0; i < io_info_count; ++i)
  {
    pollfds[i].fd = fds[i][0];
    pollfds[i].events = POLLIN;
    files[i] = fdopen (fds[i][0], "r");
  }

  while (poll (pollfds, io_info_count, -1) >= 0)
  {
    bool exit = false;

    for (unsigned int i = 0; i < io_info_count; ++i)
    {
      if (pollfds[i].revents & POLLIN)
      {
        char line[1024];
        while (fgets (line, sizeof (line), files[i]) != NULL)
          io_info[i].handler (line, strlen (line), io_info[i].user_data);
        exit = true;
      }
    }

    if (exit)
      continue;

    for (unsigned int i = 0; i < io_info_count; ++i)
      if (pollfds[i].revents & POLLHUP)
        exit = true;

    if (exit)
      break;
  }

  for (unsigned int i = 0; i < io_info_count; ++i)
    fclose (files[i]);
}

static int internal_di_exec (const char *const argv[], const char *const envp[], const struct execute_io_info io_info[], unsigned int io_info_count, di_process_handler *child_prepare_handler, void *child_prepare_user_data)
{
  pid_t pid;
  int fd_null, fds_status[2], fds_io[io_info_count][2];
  int errno_saved;
  int ret = 0;

  fd_null = open ("/dev/null", O_RDWR);

  if (pipe (fds_status) < 0)
    abort ();
  fcntl (fds_status[1], F_SETFD, FD_CLOEXEC);

  for (unsigned int i = 0; i < io_info_count; ++i)
  {
    if (pipe (fds_io[i]) < 0)
      abort ();
    fcntl (fds_io[i][0], F_SETFD, FD_CLOEXEC);
    fcntl (fds_io[i][0], F_SETFL, O_NONBLOCK);
  }

  pid = fork ();
  errno_saved = errno;

  // XXX: Flow?
  if (pid <= 0)
  {
    close (fds_status[0]);
    for (unsigned int i = 0; i < io_info_count; ++i)
      close (fds_io[i][0]);
  }

  if (pid == 0)
  {
    dup2 (fd_null, 0);
    for (unsigned int i = 0; i < io_info_count; ++i)
      dup2 (fds_io[i][1], io_info[i].fd);

    internal_di_exec_child (argv, envp, pid, child_prepare_handler, child_prepare_user_data, fds_status[1]);
  }

  close (fd_null);

  if (pid < 0)
  {
    ret = -errno_saved;
    goto out;
  }

  close (fds_status[1]);
  for (unsigned int i = 0; i < io_info_count; ++i)
    close (fds_io[i][1]);

  errno_saved = -1;

  if ((ret = internal_status_handle (fds_status[0])))
    goto out;

  if (io_info_count == 0)
    goto wait;

  internal_io_handle (fds_io, io_info, io_info_count);

wait:
  if (!waitpid (pid, &ret, 0))
    ret = -1;

out:
  close (fds_status[0]);
  for (unsigned int i = 0; i < io_info_count; ++i)
    close (fds_io[i][0]);

  return ret;
}

int execute_io_log_handler (char *buf, size_t n __attribute__ ((unused)), void *user_data)
{ 
  if (buf[n - 1] == '\n')
    buf[n - 1] = 0;
  log_text ((uintptr_t)user_data, "%s", buf);
  return 0;
}

static void execute_status (int status)
{
  log_text (DI_LOG_LEVEL_DEBUG, "Status: %d", status);

  if (status < 0)
    log_text (DI_LOG_LEVEL_CRITICAL, "Execution failed: %s", strerror (-status)); 
  else if (WIFEXITED (status))
    ;
  else if (WIFSIGNALED (status))
    log_text (DI_LOG_LEVEL_CRITICAL, "Execution failed with signal: %s", strsignal (WTERMSIG (status))); 
  else
    log_text (DI_LOG_LEVEL_CRITICAL, "Execution failed with unknown status: %d", status); 
} 

int execute_full (const char *const argv[], const struct execute_io_info io_info[], unsigned int io_info_count)
{
  log_text (DI_LOG_LEVEL_DEBUG, "Execute \"%s …\"", argv[0]);
  int ret = internal_di_exec (argv, (const char *const *) environ, io_info, io_info_count, NULL, NULL);
  execute_status (ret);
  return ret;
}

int execute_target_full (const char *const argv[], const struct execute_io_info io_info[], unsigned int io_info_count)
{
  log_text (DI_LOG_LEVEL_DEBUG, "Execute \"%s …\" in chroot", argv[0]);
  int ret = internal_di_exec (argv, execute_environment_target, io_info, io_info_count, di_exec_prepare_chroot, (void *) target_root);
  execute_status (ret);
  return ret;
}

int execute_sh_full (const char *const command, const struct execute_io_info io_info[], unsigned int io_info_count)
{
  log_text (DI_LOG_LEVEL_DEBUG, "Execute \"%s\"", command);
  const char *const argv[] = { "sh", "-c", command, NULL };
  int ret = internal_di_exec (argv, (const char *const *) environ, io_info, io_info_count, NULL, NULL);
  execute_status (ret);
  return ret;
}

int execute_sh_target_full (const char *const command, const struct execute_io_info io_info[], unsigned int io_info_count)
{
  log_text (DI_LOG_LEVEL_DEBUG, "Execute \"%s\" in chroot", command);
  const char *const argv[] = { "sh", "-c", command, NULL };
  int ret = internal_di_exec (argv, execute_environment_target, io_info, io_info_count, di_exec_prepare_chroot, (void *) target_root);
  execute_status (ret);
  return ret;
}

