/*
 * main.c
 *
 * Copyright (C) 2003 Bastian Blank <waldi@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $LastChangedBy: bastian $
 * $LastChangedDate: 2007-05-13 16:29:34 +0000 (So, 13 Mai 2007) $
 * $LastChangedRevision: 1434 $
 */

#include <config.h>

#include "download.h"
#include "execute.h"
#include "frontend.h"
#include "install.h"
#include "message.h"
#include "log.h"
#include "suite.h"

#include <debian-installer.h>

#include <getopt.h>
#include <libgen.h>
#include <limits.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <stdarg.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

const char *mirror;

#ifdef DEB_ARCH
const char default_arch[] = DEB_ARCH;
#endif
const char default_configdir[] = CONFIGDIR;
const char default_flavour[] = "standard";
const char default_mirror[] = "http://ftp.debian.org/debian";

enum
{
  GETOPT_FIRST = CHAR_MAX + 1,
  GETOPT_ALLOW_UNAUTHENTICATED,
  GETOPT_DEBUG,
  GETOPT_EXCLUDE,
  GETOPT_INCLUDE,
  GETOPT_VARIANT,
  GETOPT_VERSION,
};

static struct option const long_opts[] =
{
  {"allow-unauthenticated", no_argument, 0, GETOPT_ALLOW_UNAUTHENTICATED},
  {"arch", required_argument, 0, 'a'},
  {"configdir", required_argument, 0, 'c'},
  {"debug", no_argument, 0, GETOPT_DEBUG},
  {"download-only", no_argument, 0, 'd'},
  {"exclude", required_argument, 0, GETOPT_EXCLUDE},
  {"flavour", required_argument, 0, 'f'},
  {"helperdir", required_argument, 0, 'H'},
  {"include", required_argument, 0, GETOPT_INCLUDE},
  {"keyring", required_argument, 0, 'k'},
  {"quiet", no_argument, 0, 'q'},
  {"suite-config", required_argument, 0, 's'},
  {"variant", required_argument, 0, GETOPT_VARIANT},
  {"verbose", no_argument, 0, 'v'},
  {"help", no_argument, 0, 'h'},
  {"version", no_argument, 0, GETOPT_VERSION},
  {0, 0, 0, 0}
};

char *program_name;

int frontend_download (const char *source, const char *target)
{
  char buf[1024];
  int ret;

  if (!strncmp (mirror, "http://", 7) || !strncmp (mirror, "ftp://", 6))
    snprintf (buf, 1024, "wget -q -O %s %s/%s", target, mirror, source);
  else if (!strncmp (mirror, "file://", 7))
    snprintf (buf, 1024, "cp %s/%s %s", mirror + sizeof ("file://") - 1, source, target);
  ret = execute (buf);

  return WEXITSTATUS (ret);
}

static int check_mirror (void)
{
  if (!strncmp (mirror, "http://", 7))
    return 0;
  else if (!strncmp (mirror, "ftp://", 6))
    return 0;
  else if (!strncmp (mirror, "file://", 7))
    return 0;
  return 1;
}

static int finish_etc (void)
{
  char buf1[PATH_MAX], buf2[PATH_MAX], buf3[PATH_MAX];
  FILE *in, *out;
  size_t len;
  struct stat s;

  log_text (DI_LOG_LEVEL_MESSAGE, "Writing hosts");

  strcpy (buf1, install_root);
  strcat (buf1, "/etc/hosts");

  out = fopen (buf1, "w");
  if (!out)
    return 1;

  if (!fputs ("127.0.0.1 localhost\n", out))
    return 1;

  if (fclose (out))
    return 1;

  strcpy (buf2, install_root);
  strcpy (buf1, "/etc/resolv.conf");
  strcat (buf2, "/etc/resolv.conf");

  if (!stat (buf1, &s))
  {
    log_text (DI_LOG_LEVEL_MESSAGE, "Writing resolv.conf");

    in = fopen (buf1, "r");
    out = fopen (buf2, "w");
    if (!in || !out)
      return 1;

    while (1)
    {
      len = fread (buf3, 1, sizeof (buf3), in);
      if (!len)
        break;
      fwrite (buf3, 1, len, out);
    }

    if (fclose (in) || fclose (out))
      return 1;
  }

  log_text (DI_LOG_LEVEL_MESSAGE, "Writing apt sources.list");

  strcpy (buf1, install_root);
  strcat (buf1, "/etc/apt/sources.list");

  out = fopen (buf1, "w");
  if (!out)
    return 1;

  if (!fprintf (out, "deb %s %s main\n", mirror, suite_name))
    return 1;

  if (fclose (out))
    return 1;

  return 0;
}

static const char *generate_configdir ()
{
#ifdef CONFIGDIR_BINARY
  static char binary_configdir[4096];
  char dir_temp[strlen (program_name) + 1], *dir;

  strcpy (dir_temp, program_name);
  dir = dirname (dir_temp);
  strcpy (binary_configdir, dir);
  strcat (binary_configdir, "/");
  strcat (binary_configdir, default_configdir);
  return binary_configdir;
#else
  return default_configdir;
#endif
}

static void usage (int status)
{
  if (status != 0)
    fprintf (stderr, "Try `%s --help' for more information.\n", program_name);
  else
  {
    fprintf (stdout, "\
Usage: %s [OPTION]... SUITE TARGET [MIRROR]\n\
\n\
", program_name);
    fputs ("\
Mandatory arguments to long options are mandatory for short options too.\n\
", stdout);
    fputs ("\
      --allow-unauthenticated  Ignore if packages can’t be authenticated.\n\
  -a, --arch=ARCH              Set the target architecture.\n\
  -c, --configdir=DIR          Set the config directory.\n\
      --debug                  Enable debug output.\n\
  -d, --download-only          Download packages, but don't perform installation.\n\
      --exclude=A,B,C          Drop packages from the installation list\n\
  -f, --flavour=FLAVOUR        Select the flavour to use.\n\
  -k, --keyring=KEYRING        Use given keyring.\n\
  -H, --helperdir=DIR          Set the helper directory.\n\
      --include=A,B,C          Install extra packages.\n\
  -q, --quiet                  Be quiet.\n\
      --suite-config\n\
  -v, --verbose                Be verbose,\n\
  -h, --help                   Display this help and exit.\n\
      --version                Output version information and exit.\n\
", stdout);
    fputs ("\n\
Defines:\n\
target architecture: " 
#ifdef DEB_ARCH
DEB_ARCH 
#else
"(no default)"
#endif
"\n\
config and helper directory: "
#if CONFIGDIR_BINARY
"path of binary/"
#endif
CONFIGDIR
"\n", stdout);
  }
  exit (status);
}

int frontend_main (int argc, char **argv, char **envp)
{
  int c;
  const char
    *arch = default_arch,
    *configdir,
    *flavour = default_flavour,
    *keyring = NULL,
    *helperdir,
    *suite = NULL,
    *suite_config = NULL,
    *target = NULL;
  bool authentication = true, download_only = false;
  di_slist include = { NULL, NULL }, exclude = { NULL, NULL };
  const char *keyringdirs[] =
  {
    "/usr/share/keyrings",
    NULL, NULL
  };
  static di_packages *packages;
  static di_packages_allocator *allocator;
  static di_slist *list;

  program_name = argv[0];

  configdir = helperdir = keyringdirs[1] = generate_configdir ();

  while ((c = getopt_long (argc, argv, "a:c:df:hH:i:s:qv", long_opts, NULL)) != -1)
  {
    switch (c)
    {
      case 0:
        break;
      case 'a':
        arch = optarg;
        break;
      case 'c':
        configdir = optarg;
        break;
      case 'd':
        download_only = true;
        break;
      case 'f':
        flavour = optarg;
        break;
      case 'h':
        usage (EXIT_SUCCESS);
        break;
      case 'H':
        helperdir = optarg;
        break;
      case 'k':
        keyring = optarg;
        break;
      case 's':
        suite_config = optarg;
        break;
      case 'q':
        if (message_level <= MESSAGE_LEVEL_NORMAL)
          message_level = MESSAGE_LEVEL_QUIET;
        break;
      case 'v':
        if (message_level <= MESSAGE_LEVEL_NORMAL)
          message_level = MESSAGE_LEVEL_VERBOSE;
        break;
      case GETOPT_ALLOW_UNAUTHENTICATED:
        authentication = false;
        break;
      case GETOPT_DEBUG:
        message_level = MESSAGE_LEVEL_DEBUG;
        break;
      case GETOPT_EXCLUDE:
        {
          char *l = strdup (optarg);
          for (char *i = strtok (l, ","); i; i = strtok (NULL, ","))
            di_slist_append (&exclude, i);
        }
        break;
      case GETOPT_INCLUDE:
        {
          char *l = strdup (optarg);
          for (char *i = strtok (l, ","); i; i = strtok (NULL, ","))
            di_slist_append (&include, i);
        }
        break;
      case GETOPT_VARIANT:
        if (!strcmp(optarg, "buildd"))
          flavour = "build";
        else if (!strcmp(optarg, "fakechroot"))
          flavour = "standard";
        else
          log_text(DI_LOG_LEVEL_ERROR, "Invalid flavour");
        break;
      case GETOPT_VERSION:
        fputs (PACKAGE_STRING, stdout);
        fputs ("\n\n\
This is free software; see the source for copying conditions.  There is NO\n\
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n\
", stdout);
        exit (EXIT_SUCCESS);
        break;
      default:
        usage (EXIT_FAILURE);
    }
  }

  if ((argc - optind) <= 0)
  {
    fprintf (stderr, "%s: missing suite argument\n", program_name);
    usage (EXIT_FAILURE);
  }
  else if ((argc - optind) == 1)
  {
    fprintf (stderr, "%s: missing target argument\n", program_name);
    usage (EXIT_FAILURE);
  }
  else if ((argc - optind) > 3)
  {
    fprintf (stderr, "%s: too much arguments\n", program_name);
    usage (EXIT_FAILURE);
  }
  else if ((argc - optind) == 3)
  {
    mirror = argv[optind + 2];
  }
  else
  {
    mirror = default_mirror;
  }

  suite = argv[optind];
  target = argv[optind + 1];

#ifndef DEB_ARCH
  if (!arch)
  {
    fprintf (stderr, "%s: missing architecture\n", program_name);
    usage (EXIT_FAILURE);
  }
#endif

  di_init (basename (program_name));

  if (check_mirror ())
    log_text (DI_LOG_LEVEL_ERROR, "invalid mirror");

  if (!download_only && getuid ())
    log_text (DI_LOG_LEVEL_ERROR, "need root privileges");

  log_init ();

  if (install_target_check (target))
    log_text (DI_LOG_LEVEL_ERROR, "target check");
  if (suite_init (suite, suite_config, arch, flavour, &include, &exclude, configdir, authentication, keyring, keyringdirs))
    log_text (DI_LOG_LEVEL_ERROR, "suite init");
  if (execute_init (envp))
    log_text (DI_LOG_LEVEL_ERROR, "execute init");
  if (download_init ())
    log_text (DI_LOG_LEVEL_ERROR, "download init");
  if (!download_only && install_init (helperdir))
    log_text (DI_LOG_LEVEL_ERROR, "install init");

  if (download (&packages, &allocator, &list))
    log_text (DI_LOG_LEVEL_ERROR, "download");
  if (!download_only && install (packages, allocator, list))
    log_text (DI_LOG_LEVEL_ERROR, "install");

  if (!download_only)
    finish_etc ();

  return 0;
}

