/*
 * main.c
 *
 * Copyright (C) 2003 Bastian Blank <waldi@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $LastChangedBy: bastian $
 * $LastChangedDate: 2004-05-11 22:17:42 +0200 (Tue, 11 May 2004) $
 * $LastChangedRevision: 429 $
 */

#include <config.h>

#include "download.h"
#include "frontend.h"
#include "install.h"
#include "log.h"
#include "packages.h"
#include "prepare.h"
#include "suite.h"

#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/mount.h>
#include <unistd.h>

char install_root[PATH_MAX];
static int count, count_done;

const char *const environment[] = {
  "DEBIAN_FRONTEND=noninteractive",
  "PATH=/usr/bin:/bin:/usr/sbin:/sbin",
  NULL
};

#define BUFSIZE_command 4096

static void install_progress_set (int n)
{
  frontend_progress_set (((float) n/count) * 500 + 500);
}

static int install_execute_io_handler (const char *buf, size_t n __attribute__ ((unused)), void *user_data __attribute__ ((unused)))
{
  log_text (DI_LOG_LEVEL_OUTPUT, "%s", buf);
  return 0;
}

static int install_execute_progress_update (di_packages *packages, char *package, di_package_status status)
{
  di_package *p = di_packages_get_package (packages, package, 0);
  if (p)
  {
    if (p->status < status)
    {
      log_text (DI_LOG_LEVEL_DEBUG, "Updating %s to status %d", p->key.string, status);
      p->status = status;
    }
  }
  else
    return -1;
  return 0;
}

static int install_execute_progress_io_handler (const char *buf, size_t n __attribute__ ((unused)), void *user_data)
{
  di_packages *packages = user_data;
  char buf_package[128];

  install_execute_io_handler (buf, n, NULL);

  if (sscanf (buf, "Unpacking replacement %128s", buf_package) == 1 ||
      sscanf (buf, "Unpacking %128s", buf_package) == 1)
  {
    log_message (LOG_MESSAGE_INFO_INSTALL_UNPACK_PACKAGE, buf_package);
    install_progress_set (++count_done);
    install_execute_progress_update (packages, buf_package, di_package_status_unpacked);
  }
  else if (sscanf (buf, "Setting up %128s", buf_package) == 1)
  {
    log_message (LOG_MESSAGE_INFO_INSTALL_CONFIGURE_PACKAGE, buf_package);
    install_progress_set (++count_done);
    install_execute_progress_update (packages, buf_package, di_package_status_installed);
  }

  return 0;
}

static int install_execute_shell (const char *const cmd, di_io_handler *stdout_handler, di_io_handler *stderr_handler, void *io_user_data, di_process_handler *child_prepare_handler, void *child_prepare_user_data)
{
  const char *const argv[] = { "sh", "-c", cmd, NULL };
  int ret = di_exec_env_full ("/bin/sh", argv, environment, stdout_handler, stderr_handler, io_user_data, NULL, NULL, child_prepare_handler, child_prepare_user_data);
  log_text (DI_LOG_LEVEL_DEBUG, "Return code: %d", ret);
  return ret;
}

int install_execute (const char *command)
{
  log_text (DI_LOG_LEVEL_DEBUG, "Execute \"%s\"", command);
  return install_execute_shell (command, install_execute_io_handler, install_execute_io_handler, NULL, NULL, NULL);
}

static int install_execute_target (const char *command)
{
  log_text (DI_LOG_LEVEL_DEBUG, "Execute \"%s\" in chroot", command);
  return install_execute_shell (command, install_execute_io_handler, install_execute_io_handler, NULL, di_exec_prepare_chroot, install_root);
}

int install_execute_target_do (const char *command)
{
  log_text (DI_LOG_LEVEL_MESSAGE, "Doing something important");
  return install_execute_target (command);
}

static int install_execute_target_progress (const char *command, di_packages *packages)
{
  log_text (DI_LOG_LEVEL_DEBUG, "Execute \"%s\" in chroot", command);
  return install_execute_shell (command, install_execute_progress_io_handler, install_execute_io_handler, packages, di_exec_prepare_chroot, install_root);
}

di_slist *install_list (di_packages *packages, di_packages_allocator *allocator, di_slist *install, di_package_priority priority, di_package_status status)
{
  di_slist *list1, *list2;
  di_slist_node *node;

  list1 = di_slist_alloc ();

  for (node = install->head; node; node = node->next)
  {
    di_package *p = node->data;
    if (p->priority >= priority && p->status < status)
      di_slist_append (list1, p);
  }

  list2 = di_packages_resolve_dependencies (packages, list1, allocator);

  di_slist_free (list1);

  list1 = di_slist_alloc ();

  for (node = list2->head; node; node = node->next)
  {
    di_package *p = node->data;
    if (p->status < status)
      di_slist_append (list1, p);
  }

  di_slist_free (list2);

  return list1;
}

di_slist *install_list_package (di_packages *packages, di_packages_allocator *allocator, char *package, di_package_status status)
{
  di_slist *list1, *list2;
  di_slist_node *node;
  di_package *p;

  list1 = di_slist_alloc ();

  p = di_packages_get_package (packages, package, 0);
  if (!p || p->status >= status)
    return list1;

  di_slist_append (list1, p);

  list2 = di_packages_resolve_dependencies (packages, list1, allocator);

  di_slist_free (list1);

  list1 = di_slist_alloc ();

  for (node = list2->head; node; node = node->next)
  {
    di_package *p = node->data;
    if (p->status < status)
      di_slist_append (list1, p);
  }

  di_slist_free (list2);

  return list1;
}

di_slist *install_list_package_only (di_packages *packages, char *package, di_package_status status)
{
  di_slist *list;
  di_package *p;

  list = di_slist_alloc ();

  p = di_packages_get_package (packages, package, 0);

  if (p && p->status < status)
    di_slist_append (list, p);

  return list;
}

static int install_apt_cleanup_script (const char *name)
{
  char buf[PATH_MAX];

  snprintf (buf, sizeof (buf), "%s/%s", install_root, name);

  if (unlink (buf))
    return 1;

  snprintf (buf, sizeof (buf), "dpkg-divert --remove --rename %s", name);

  if (install_execute_target (buf))
    return 1;
  return 0;
}

int install_apt_cleanup (void)
{
  char buf[PATH_MAX];

  strcpy (buf, install_root);
  strcat (buf, "/etc/apt/sources.list");

  if (unlink (buf))
    return 1;

  if (install_apt_cleanup_script ("/usr/sbin/invoke-rc.d") ||
      install_apt_cleanup_script ("/sbin/start-stop-daemon"))
    return 1;

  strcpy (buf, "apt-get clean");

  if (install_execute_target (buf))
    return 1;

  return 0;
}

int install_apt_install (di_packages *packages, di_slist *install)
{
  char buf[BUFSIZE_command];
  size_t len;
  di_slist_node *node;
  int count = 0;

  strcpy (buf, "apt-get install --yes");
  len = strlen (buf);

  for (node = install->head; node; node = node->next)
  {
    count++;
    di_package *p = node->data;
    len += 1 + p->key.size;
    if (len >= sizeof (buf))
      log_text (DI_LOG_LEVEL_ERROR, "buffer overflow");
    strcat (buf, " ");
    strcat (buf, p->key.string);
  }

  if (count && install_execute_target_progress (buf, packages))
    return 1;

  return 0;
}

static int install_apt_prepare_script (const char *name)
{
  char buf[PATH_MAX];
  FILE *out;

  snprintf (buf, sizeof (buf), "%s/%s.REAL", install_root, name);

  unlink (buf);

  snprintf (buf, sizeof (buf), "dpkg-divert --add --local --rename --divert %s.REAL %s", name, name);

  if (install_execute_target (buf))
    return 1;

  snprintf (buf, sizeof (buf), "%s/%s", install_root, name);

  if (!(out = fopen (buf, "w")) || !fputs ("#!/bin/sh\n", out) || fchmod (fileno (out), 0755) || fclose (out))
    return 1;
  return 0;
}

int install_apt_prepare (void)
{
  char buf[PATH_MAX];
  FILE *out;

  strcpy (buf, install_root);
  strcat (buf, "/etc/apt/sources.list");

  out = fopen (buf, "w");
  if (!out)
    return 1;

  if (!fprintf (out, "deb copy://debootstrap.invalid %s main\n", suite_name))
    return 1;

  if (fclose (out))
    return 1;

  if (install_apt_prepare_script ("/usr/sbin/invoke-rc.d") ||
      install_apt_prepare_script ("/sbin/start-stop-daemon"))
    return 1;

  return 0;
}

int install_count_install (di_slist *install)
{
  int count = 0;
  di_slist_node *node;

  for (node = install->head; node; node = node->next)
    count++;

  return count;
}

static int install_dev_opt (char *buf, const char *name, mode_t mode, dev_t dev, int build)
{
  strcpy (buf, install_root);
  strcat (buf, name);

  unlink (buf);
  if (build)
    return mknod (buf, mode, dev);
  return 0;
}

int install_dev_cleanup (void)
{
  char buf[PATH_MAX];

  if (0)
  {
    install_dev_opt (buf, "/dev/null", 0, 0, 0);
    install_dev_opt (buf, "/dev/.devfsd", 0, 0, 0);
  }
  return 0;
}

int install_dev_prepare (void)
{
  int ret = 0;
  char buf[PATH_MAX];
  mode_t mask;

  mask = umask (0);
  ret |= install_dev_opt (buf, "/dev/null", S_IFCHR | 0666, makedev (1, 3), 1);
  if (0)
    ret |= install_dev_opt (buf, "/dev/.devfsd", S_IFCHR | 0600, makedev (8, 0), 1);
  umask (mask);
  return ret;
}

int install_dpkg_configure (di_packages *packages, int force)
{
  char buf[BUFSIZE_command];

  strcpy (buf, "dpkg --configure -a");
  if (force)
    strcat (buf, " --force-all");

  if (install_execute_target_progress (buf, packages))
    return 1;

  return 0;
}

static int install_dpkg_all (char *buf, size_t bufsize, di_packages *packages, di_slist *install)
{
  char buf1[256];
  size_t len;
  di_slist_node *node;

  len = strlen (buf);

  for (node = install->head; node; node = node->next)
  {
    di_package *p = node->data;
    build_target_deb_root (buf1, sizeof (buf1), package_get_local_filename (p));
    len += 1 + strlen (buf1);
    if (len >= bufsize)
      log_text (DI_LOG_LEVEL_ERROR, "buffer overflow");
    strcat (buf, " ");
    strcat (buf, buf1);
  }

  if (install_execute_target_progress (buf, packages))
    return 1;

  return 0;
}

int install_dpkg_install (di_packages *packages, di_slist *install, int force)
{
  char buf[BUFSIZE_command];

  strcpy (buf, "dpkg -i");
  if (force)
    strcat (buf, " --force-all");

  return install_dpkg_all (buf, sizeof (buf), packages, install);
}

int install_dpkg_unpack (di_packages *packages, di_slist *install)
{
  char buf[BUFSIZE_command];

  strcpy (buf, "dpkg --unpack --force-all");

  return install_dpkg_all (buf, sizeof (buf), packages, install);
}

static int install_extract_file (const char *file)
{
  char buf_file[PATH_MAX];
  char buf[PATH_MAX];

  build_target_deb (buf_file, PATH_MAX, file);
  snprintf (buf, PATH_MAX, "ar -p %s data.tar.gz | tar -xzf -", buf_file);
  return install_execute (buf);
}

int install_extract (di_slist *install)
{
  int count = 0, count_done = 0;
  struct di_slist_node *node;

  for (node = install->head; node; node = node->next)
    count++;

  for (node = install->head; node; node = node->next)
  {
    di_package *p = node->data;
    log_text (DI_LOG_LEVEL_MESSAGE, "Extracting %s", p->package);
    install_extract_file (package_get_local_filename (p));
    count_done++;
    frontend_progress_set (((float) count_done/count) * 50 + 450);
  }

  return 0;
}

int install (di_packages *packages, di_packages_allocator *allocator, di_slist *install)
{
  di_slist_node *node;

  if (chdir (install_root))
    return 1;

  for (node = install->head; node; node = node->next)
    count += 2;

  if (suite_install (packages, allocator, install))
    log_text (DI_LOG_LEVEL_ERROR, "Couldn't install root!");

  return 0;
}

int install_target_check (const char *target)
{
  struct stat statbuf;

  umask (022);

  if (!stat (target, &statbuf))
  {
    if (!(S_ISDIR (statbuf.st_mode)))
      return 1;
  }
  else
  {
    if (mkdir (target, 0755))
      return 1;
  }

  if (!realpath (target, install_root))
    return 1;

  if (install_root[0] != '/')
  {
    char tmp[PATH_MAX];
    if (getcwd (tmp, sizeof (tmp) - strlen (install_root) - 1) == NULL)
      return 1;
    strcat (tmp, "/");
    strcat (tmp, install_root);
    if (!realpath (tmp, install_root))
      return 1;
  }
  return 0;
}

int install_init (void)
{
  if (prepare_install ())
    return 1;
  log_open ();
  return 0;
}

enum mount_status
{
  umounted = 1,
  mounted,
};

static enum mount_status mount_status_proc (enum mount_status status)
{
  static enum mount_status act = umounted;
  enum mount_status ret = act;
  if (status == umounted || status == mounted)
    act = status;
  return ret;
}

int install_mount_proc (void)
{
  if (mount_status_proc (0) == umounted)
  {
    char buf[PATH_MAX];
    snprintf (buf, PATH_MAX, "%s/proc", install_root);
    if (mount ("proc", buf, "proc", 0, 0))
      return 1;
    mount_status_proc (mounted);
  }
  return 0;
}

void install_umount (void)
{
  if (mount_status_proc (0) == mounted)
  {
    char buf[PATH_MAX];
    snprintf (buf, PATH_MAX, "%s/proc", install_root);
    umount (buf);
  }
}

