/**
 * @file ext.c
 * Server extension primitives
 * @author Marko Mkel (msmakela@nic.funet.fi)
 */

/*
 * Copyright  1994-1996 Marko Mkel and Olaf Seibert
 * Copyright  2001 Marko Mkel
 * Original Linux and Commodore 64/128/Vic-20 version by Marko Mkel
 * Ported to the PET and the Amiga series by Olaf Seibert
 * Restructured by Marko Mkel
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "comm.h"
#include "info.h"
#include "ext.h"
#include "mem.h"
#include "run.h"
#include <stdlib.h>
#include <string.h>

/** Some 6502 operation codes */
enum opcodes {
  JSR = 0x20, /**< Jump to subroutine */
  JMP = 0x4c, /**< Unconditional jump */
  LDA = 0xa9, /**< Load Accumulator immediate */
  LDX = 0xa2, /**< Load Index Register X immediate */
  LDY = 0xa0  /**< Load Index Register Y immediate */
};

/** install a server extension
 * @param comm		the communication primitives
 * @param hostinfo	information on the remote host
 * @param code		the extension code to be loaded
 * @param length	length of the code in bytes
 * @param device	device number
 * @param secondary	secondary address
 * @return		zero on success, nonzero on error
 */
int
ext (const struct comm* comm,
     const struct hostinfo* hostinfo,
     const unsigned char* code,
     unsigned length,
     unsigned device,
     unsigned secondary)
{
  /** work area for receiving a jump table */
  unsigned char jmptab[24];
  /** work area for patching the code */
  unsigned char* buf;
  unsigned addr, i, bank, start;

  if (hostinfo->host == P500 ||
      hostinfo->host == B128 ||
      hostinfo->host == B256)
    bank = 15, start = 0x0600;
  else
    bank = 0, start = 0x1800;

  /* copy the jump table from the server */
  if (save (comm, bank, hostinfo->driver,
	    hostinfo->driver + sizeof jmptab, jmptab))
    return 1;

  /* Check the consistency of the jump table */
  for (i = 0; i < sizeof jmptab && jmptab[i] == JMP; i += 3);

  /* All but the last entry must be jumps */
  if (i < (sizeof jmptab) - 3)
    return 2;

  if (i == (sizeof jmptab) - 3) {
    /* The last jump table entry is not a jump; change it to one */
    jmptab[i++] = JMP;
    addr = hostinfo->driver + ((sizeof jmptab) - 3);
    jmptab[i++] = addr;
    jmptab[i] = addr >> 8;
  }

  if (!(buf = malloc (length)))
    return 3; /* out of memory */
  
  /* copy the code in order to patch it */
  memcpy (buf, code, length);

  /* patch the jumps to the jump table */
  for (i = 0; i < length - 2; i++) {
    if ((buf[i] == JSR || buf[i] == JMP) && !buf[i + 2]) {
      addr = buf[i + 1];
      if (addr >= sizeof jmptab || (addr % 3))
	continue;
      buf[++i] = jmptab[addr + 1];
      buf[++i] = jmptab[addr + 2];
    }
  }

  /* find first lda#xx:ldx#xx:ldy#xx and patch it */
  for (i = 0; i < length - 6; i++) {
    if (buf[i] != LDA || buf[i + 2] != LDX || buf[i + 4] != LDY)
      continue;
    if (buf[i + 1] == buf[i + 3] && buf[i + 1] == buf[i + 5])
      buf[i + 3] = device, buf[i + 5] = secondary;
    break;
  }

  /* load the extension */
  if (load (comm, bank, start, start + length, buf)) {
    free (buf);
    return 4; /* load error */
  }

  free (buf);
  /* start the extension */
  if (jump (comm, bank, start))
    return 5;
  return 0;
}
