/*
	Chorus.h
	
	Copyright 2004 Tim Goetze <tim@quitte.de>
	
	http://quitte.de/dsp/

	mono and stereo chorus/flanger units, traditional designs.

*/
/*
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
	02111-1307, USA or point your web browser to http://www.gnu.org.
*/

#ifndef _CHORUS_H_
#define _CHORUS_H_

#include "dsp/Sine.h"
#include "dsp/Delay.h"

class ChorusStub
{
	public:
		double fs;
		d_sample time, width, rate;
		d_sample normal; /* denormal protection */
};

class Chorus
: public ChorusStub
{
	public:
		DSP::Sine lfo;
		DSP::Delay delay;
		DSP::DelayTapA tap;

		template <sample_func_t>
		void one_cycle (int frames);
	
	public:
		d_sample * ports [8];

		static PortInfo port_info [];
		d_sample adding_gain;

		void init (double _fs)
			{
				fs = _fs;
				rate = .15;
				delay.init ((int) (.040 * fs));
				normal = NOISE_FLOOR;
			}

		void activate()
			{
				time = 0;
				width = 0;
				
				rate = *ports[3];
				
				delay.reset();
				tap.reset();

				lfo.set_f (rate, fs, 0);
			}

		void run (int n)
			{
				one_cycle<store_func> (n);
			}
		
		void run_adding (int n)
			{
				one_cycle<adding_func> (n);
			}
};

class StereoChorus
: public ChorusStub
{
	public:
		double fs;
		d_sample rate;
		d_sample phase;

		DSP::Delay delay;

		struct {
			DSP::Sine lfo;
			DSP::DelayTapA tap;
		} left, right;

		template <sample_func_t>
		void one_cycle (int frames);
	
	public:
		d_sample * ports [10];

		static PortInfo port_info [];
		d_sample adding_gain;

		void init (double _fs)
			{
				fs = _fs;
				rate = .15;
				phase = .5; /* pi */

				delay.init ((int) (.040 * fs));
				normal = NOISE_FLOOR;
			}

		void activate()
			{
				time = 0;
				width = 0;

				delay.reset();

				left.tap.reset();
				right.tap.reset();

				left.lfo.set_f (rate, fs, 0);
				right.lfo.set_f (rate, fs, phase * M_PI);
			}

		void run (int n)
			{
				one_cycle<store_func> (n);
			}
		
		void run_adding (int n)
			{
				one_cycle<adding_func> (n);
			}
};

#endif /* _CHORUS_H_ */
