/* The Cantus project.
 * (c)2002, 2003, 2004 by Samuel Abels (spam debain org)
 * This project's homepage is: http://www.debain.org/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "tag2filename.h"

//#define _DEBUG_


/******************************************************************************
 * Constructor/Destructor
 ******************************************************************************/
Tag2Filename::Tag2Filename(CantusHash *pplugindata)
{
  plugindata = pplugindata;
  selected   = NULL;
  
  // Connect all signals with Cantus.
  CantusAddListenerSigCFunc addlistener = (CantusAddListenerSigCFunc)
                  cantushash_get_pointer(plugindata, "Cantus:AddListenerSigC");
  g_return_if_fail(addlistener != NULL);
  listenerids.push_back(
    addlistener("Filelist:Read:Start",
            SigC::slot(*this, &Tag2Filename::on_selection_changed_event)));
  listenerids.push_back(
    addlistener("File:Read:Finished",
            SigC::slot(*this, &Tag2Filename::on_file_read_finished_event)));
  listenerids.push_back(
    addlistener("Filelist:Read:Finished",
            SigC::slot(*this, &Tag2Filename::on_filelist_read_finished_event)));
  listenerids.push_back(
    addlistener("GUI:PluginWidget:Destroyed",
            SigC::slot(*this, &Tag2Filename::on_uiwidget_destroyed_event)));
  
  fields.push_back("Artist");
  fields.push_back("Song");
  fields.push_back("Album");
  fields.push_back("Track");
  fields.push_back("Year");
  fields.push_back("Genre");
}


Tag2Filename::~Tag2Filename(void)
{
#ifdef _DEBUG_
  printf("tag2f(): Tag2Filename::~Tag2Filename(): Called.\n");
#endif
  // Disconnect all signals from Cantus.
  CantusRemoveListenerFunc removelistener = (CantusRemoveListenerFunc)
                  cantushash_get_pointer(plugindata, "Cantus:RemoveListener");
  g_return_if_fail(removelistener != NULL);
  std::list<long>::iterator iter = listenerids.begin();
  while (iter != listenerids.end()) {
    removelistener(*iter);
    iter++;
  }
#ifdef _DEBUG_
  printf("tag2f(): Tag2Filename::~Tag2Filename(): Finished.\n");
#endif
}


/******************************************************************************
 * Public
 ******************************************************************************/
GtkWidget *Tag2Filename::editarea_build(gboolean vertical)
{
  GtkWidget *editareawidget = editarea.build(vertical);
  editarea.signal_button_save_clicked.connect(
             SigC::slot(*this, &Tag2Filename::on_editarea_button_save_clicked));
  editarea.signal_radio_sourcetag_changed.connect(
             SigC::slot(*this, &Tag2Filename::show_first_tag));
  editarea.signal_format_changed.connect(
             SigC::slot(*this, &Tag2Filename::show_first_tag));
  return editareawidget;
}


/******************************************************************************
 * Private
 ******************************************************************************/
/* Called to update the GUI (fill a tag in).
 */
void Tag2Filename::show_first_tag(void)
{
  if (!selected)
    return;
  
  // Get a pointer to all required functions.
  CantusFileInfoGetFunc get_info = (CantusFileInfoGetFunc)
                    cantushash_get_pointer(plugindata, "Cantus:FileInfoGet");
  CantusFileInfoUnlockFunc unlock_info = (CantusFileInfoUnlockFunc)
                    cantushash_get_pointer(plugindata, "Cantus:FileInfoUnlock");
  g_return_if_fail(get_info != NULL || unlock_info != NULL);
  
  const gchar *filename  = (const gchar*)selected->data;
  CantusHash  *info      = get_info(filename);
  std::string  tag       = "ID3V1";
  std::string  fieldname;
  if (editarea.get_check_active("Source:ID3V2:Check"))
    tag = "ID3V2";
  if (editarea.get_check_active("Source:OGG:Check"))
    tag = "OGG";
  
  std::list<const gchar*>::iterator iter = fields.begin();
  while (iter != fields.end()) {
    fieldname = tag + ":" + *iter;
    editarea.set_label_text(*iter,
                            cantushash_get_char(info, fieldname.c_str()));
    iter++;
  }
  std::string newfilename = tag2filename(info, tag);
  unlock_info(filename);
  std::string newbasename = newfilename.substr(newfilename.rfind('/') + 1,
                                               newfilename.length());
  editarea.set_label_text("Filename", newbasename.c_str());
}


/* Called when the selection changed, but the selected files have not yet been
 * read.
 */
void Tag2Filename::on_selection_changed_event(void *pselected)
{
#ifdef _DEBUG_
  printf("tag2f(): Tag2Filename::on_selection_changed_event(): Called.\n");
#endif
  if (pselected && ((GList*)pselected)->next) // Make the buttons inactive if at
    editarea.set_active(FALSE);               // least two files are selected.
  isfirst = TRUE;
  if (selected)
    g_list_free(selected);
  selected = g_list_copy((GList*)pselected);
}


/* The selection has now been read.
 */
void Tag2Filename::on_file_read_finished_event(void *pinfo)
{
#ifdef _DEBUG_
  printf("tag2f(): Tag2Filename::on_file_read_finished_event(): Called.\n");
#endif
  if (!isfirst)
    return;
  
  g_return_if_fail(pinfo != NULL);
  CantusHash *info = (CantusHash*)pinfo;
  
  isfirst = FALSE;
  std::string tag = "ID3V1";
  std::string fieldname;
  if (editarea.get_check_active("Source:ID3V2:Check"))
    tag = "ID3V2";
  if (editarea.get_check_active("Source:OGG:Check"))
    tag = "OGG";
  
  std::list<const gchar*>::iterator iter = fields.begin();
  while (iter != fields.end()) {
    fieldname = tag + ":" + *iter;
    editarea.set_label_text(*iter,
                            cantushash_get_char(info, fieldname.c_str()));
    iter++;
  }
  
  std::string newfilename = tag2filename(info, tag);
  std::string newbasename = newfilename.substr(newfilename.rfind('/') + 1,
                                               newfilename.length());
  editarea.set_label_text("Filename", newbasename.c_str());
}


/* The files that have been selected have now all been read.
 */
void Tag2Filename::on_filelist_read_finished_event(void *trash)
{
#ifdef _DEBUG_
  printf("tag2f(): Tag2Filename::on_filelist_read_finished_event() called.\n");
#endif
  editarea.set_active(TRUE);
}


std::string Tag2Filename::tag2filename(CantusHash *info, std::string tag)
{
  std::string format      = editarea.get_entry_text("Format");
  std::string oldfilename = cantushash_get_char(info, "File:Name");
  std::string directory   = oldfilename;
  std::string filename;
  std::string fieldname;
  const char* trackstr    = NULL;
  
  if (format == "")
    return oldfilename;
  
  directory.resize(directory.rfind('/') + 1);  // Strip the old filename off.
  
  int i = -1;
  while (format[++i] != '\0') {
    /* Not a switch? Then just copy the current char. */
    if (format[i] != '%') {
      filename.append(sizeof(char), format[i]);
      continue;
    }
    
    /* Ok, so it is a switch. Our action depends on the following char. */
    i++;
    switch (format[i]) {
    case '%':
      /* "%%" should only be replaced by a single "%". */
      filename.append("%");
      continue;
      break;
    
    case 'T':   /* Track (Make two digit). */
      fieldname = tag + ":Track";
      trackstr  = cantushash_get_char(info, fieldname.c_str());
      if (strlen(trackstr) < 2)
        filename.append(2 - strlen(trackstr), '0');
      filename.append(trackstr);
      continue;
      break;
    
    case 't':   /* Track. */
      fieldname = tag + ":Track";
      break;
      
    case 'a':   /* Artist. */
      fieldname = tag + ":Artist";
      break;
      
    case 'b':   /* Album. */
      fieldname = tag + ":Album";
      break;
      
    case 'c':   /* Comment. */
      fieldname = tag + ":Comment";
      break;
      
    case 'g':   /* Genre. */
      fieldname = tag + ":Genre";
      break;
      
    case 'y':   /* Year. */
      fieldname = tag + ":Year";
      break;
      
    case 's':   /* Song. */
      fieldname = tag + ":Song";
      break;
      
    case 'x':   /* Trash. */
      continue;
      break;
    
    default:    /* Other switches are not allowed. */
      return oldfilename;
      break;
    }
    
    filename.append(cantushash_get_char(info, fieldname.c_str()));
  }
  
  if (filename == "")
    return oldfilename;
  
  filename = directory
           + filename
           + oldfilename.substr(oldfilename.rfind('.'), oldfilename.length());
  
  if (filename.length() > 1024) {
    g_warning("Tag2Filename::tag2filename(): Filename too long! %s\n",
              filename.c_str());
    return oldfilename;
  }
  
  return filename;
}


void Tag2Filename::on_editarea_button_save_clicked(void)
{
  // Get a pointer to all required functions.
  CantusEmitFunc emit = (CantusEmitFunc)
                    cantushash_get_pointer(plugindata, "Cantus:Emit");
  CantusFileInfoGetFunc get_info = (CantusFileInfoGetFunc)
                    cantushash_get_pointer(plugindata, "Cantus:FileInfoGet");
  CantusFileInfoUnlockFunc unlock_info = (CantusFileInfoUnlockFunc)
                    cantushash_get_pointer(plugindata, "Cantus:FileInfoUnlock");
  g_return_if_fail(emit != NULL || get_info != NULL || unlock_info != NULL);
  
  // Walk through all files, copying the tag fields to the filename.
  GList *item = selected;
  std::string tag = "ID3V1";
  if (editarea.get_check_active("Source:ID3V2:Check"))
    tag = "ID3V2";
  if (editarea.get_check_active("Source:OGG:Check"))
    tag = "OGG";
  while (item) {
    const gchar *filename = (const gchar*)item->data;
    g_assert(filename != NULL);
    CantusHash *info = get_info(filename);
    std::string newfilename = tag2filename(info, tag);
    if (newfilename != "")
      cantushash_set_char(info, "File:Name", newfilename.c_str());
    unlock_info(filename);
    item = item->next;
  }
  
  // Emit the write request in the eventbus.
  GValue arg = {0, };
  g_value_init(&arg, G_TYPE_POINTER);
  g_value_set_pointer(&arg, (gpointer)selected);
  emit("Filelist:Save:Request", &arg);
  g_value_unset(&arg);
}


void Tag2Filename::on_uiwidget_destroyed_event(void *name)
{
#ifdef _DEBUG_
  printf("tag2f(): Tag2Filename::on_uiwidget_destroyed_event(): Called.\n");
#endif
  if (strcmp((gchar*)name, cantushash_get_char(plugindata, "Plugin:Name")) == 0)
    delete this;
}
