/* the cantus project.
 * (c)2002 by Samuel Abels (sam@manicsadness.com)
 * This project's homepage is: http://software.manicsadness.com/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include "support.h"

#include "lib_mpgheader.h"
#include "lib_id3v1.h"
#include "lib_id3v2.3.h"
#include "mp3info.h"
#include "tageditor.h"
#include "gui_tageditor.h"
#include "gui_queue.h"
#include "shared.h"
#include "lib_vorbis.h"


/***************************************************************************************
 * BELOW FOLLOW THE STATICS
 ***************************************************************************************/
// Returns the datapointer to the queueitems data with the given fullfilename
static Mp3Info * g_list_find_matching_fullfilename(GList *haystack, gchar *fullfilename)
{
	GList *haystack_item = NULL;
	Mp3Info *haystack_mp3info = NULL;
	gchar haystack_fullfilename[2048];

	haystack_item = g_list_first(haystack);
	while(haystack_item)
	{
		haystack_mp3info = (Mp3Info *)haystack_item->data;
		haystack_item = g_list_next(haystack_item);
		
		snprintf(haystack_fullfilename, 2047, "%s%s", haystack_mp3info->directory, haystack_mp3info->filename);
		if(strncmp(fullfilename, haystack_fullfilename, 2047)==0)
			return haystack_mp3info;
	}
	return NULL;
}


static gboolean create_shortinfo(gchar *filename, gchar *shortinfo)
{
	Header *mp3header = malloc(sizeof(Header));
	id3Tag *tag = malloc(sizeof(id3Tag));
	
// If we have no valid header, set an error and return.
	if( get_mpgheader(mp3header, filename) != 0 )
	{
		free(mp3header);
		free(tag);
		strncpy(shortinfo, "No valid header found!", 2047);
		return(FALSE);
	}

// Version
	sprintf(shortinfo, "MPEG V%s, Layer %i\nSample rate: %i\nBitrate: %i\n",
		mp3header->version, mp3header->layer, mp3header->frequency, mp3header->bitrate);
// Time
	if(mp3header->seconds-( ((int)mp3header->seconds/60)*60 )<10)
	{
		sprintf(shortinfo, "%sTime: %i:0%i\n", shortinfo,
			(int)mp3header->seconds/60, mp3header->seconds-( ((int)mp3header->seconds/60)*60 ));
	}
	else
	{
		sprintf(shortinfo, "%sTime: %i:%i\n", shortinfo,
			(int)mp3header->seconds/60, mp3header->seconds-( ((int)mp3header->seconds/60)*60 ));
	}
// Stereo Mode
	if(mp3header->mode == 0)
		sprintf(shortinfo, "%sMode: Stereo\n", shortinfo);
	else if(mp3header->mode == 1)
		sprintf(shortinfo, "%sMode: Joint Stereo\n", shortinfo);
	else if(mp3header->mode == 2)
		sprintf(shortinfo, "%sMode: 2 Channel\n", shortinfo);
	else if(mp3header->mode == 3)
		sprintf(shortinfo, "%sMode: Mono\n", shortinfo);
// Copyright
	if(mp3header->copyright)
		sprintf(shortinfo, "%sCopyright: Yes\n", shortinfo);
	else
		sprintf(shortinfo, "%sCopyright: No\n", shortinfo);
// have v1 tag?
	if( get_id3v1_tag (tag, filename) == 0 )
		sprintf(shortinfo, "%sID3V1 Tag present.\n", shortinfo);
// have v2 tag?
	if( get_id3v2_tag (tag, filename) == 0 )
		sprintf(shortinfo, "%sID3V2 Tag present.\n", shortinfo);
#ifdef HAVE_OGG_H
// read ogg tag.
	if( get_vorbis_tag (tag, filename) == 0 )
		sprintf(shortinfo, "%sID3V2 Tag present.\n", shortinfo);
#endif
	
	free(mp3header);
	free(tag);
	return(TRUE);
}
/***************************************************************************************
 * END OF STATICS
 ***************************************************************************************/


/**********************************************************************
 * this will update all informations in the tageditor gui
 * (tags content & mp3header info).
 * mode 0: clear and read again
 * mode 1: merge with the current content
 **********************************************************************/
void tagedit_info_update(gchar *filename, gint whichtag, gint mode)
{
	extern GtkWidget *cantus;
	id3Tag *tag = malloc(sizeof(id3Tag));
	gchar shortinfo[2048];
	gboolean have_tag = FALSE;
	
	memset(tag, 0, sizeof(id3Tag));
	
	gnome_appbar_set_status(GNOME_APPBAR (lookup_widget (cantus, "appbar")), "Reading tag...");

// If we have no valid mpegheader, return.
	if( !create_shortinfo(filename, shortinfo) )
	{
		gtk_label_set_text(GTK_LABEL(lookup_widget(cantus, "label_cantus_tagedit_mpginfo")), shortinfo);
		return;
	}
	
	gtk_label_set_text(GTK_LABEL(lookup_widget(cantus, "label_cantus_tagedit_mpginfo")), shortinfo);
	
// first of all, empty all out (only if in normal view mode,
// otherwise we have to keep all those which are the same in the previous tag)
	if( mode == 0 )
		tageditor_gui_clear();
	
// read v1 tag
	if( (whichtag == 1 || whichtag == 3)
		&& get_id3v1_tag(tag, filename) == 0 )
		have_tag = TRUE;
// read v2 tag
	if( (whichtag == 2 || whichtag == 3)
		&& get_id3v2_tag(tag, filename) == 0 )
		have_tag = TRUE;
#ifdef HAVE_OGG_H
// read ogg tag.
	if( whichtag == 4
		&& get_vorbis_tag(tag, filename) == 0 )
		have_tag = TRUE;
#endif
	
	if( have_tag )
		tageditor_tag_to_gui(tag, mode);

// and put the tag to the form

	free(tag);
	gnome_appbar_set_status(GNOME_APPBAR (lookup_widget (cantus, "appbar")), "");
	return;
}


/**********************************************************************
 * this will write all informations in the tageditor gui to one tag
 **********************************************************************/
void tagedit_tag_set(gchar *filename, gint whichtag)
{
	extern GtkWidget *cantus;
	extern GList *queue;
	id3Tag *v1 = malloc (sizeof(id3Tag));
	id3Tag *v2 = malloc (sizeof(id3Tag));
	gchar bar[4096];
	Mp3Info *mp3info = NULL;

	if( !filename	|| *filename == '\0' || access (filename, F_OK) != 0 ) {
		free (v1);
		free (v2);
		return;
	}
	
	sprintf(bar, "Saving tags: %s", filename);
	gnome_appbar_set_status(GNOME_APPBAR(lookup_widget(cantus, "appbar")), bar);
	while (g_main_iteration(FALSE));
	
  memset(v1, 0, sizeof(id3Tag));
  memset(v2, 0, sizeof(id3Tag));

  // Read the tags to memory.
	if (whichtag == 1 || whichtag == 3)
		get_id3v1_tag (v1, filename);
  // if the v2 tag should be set, overwrite the v1 struct (exept the genre)
	if (whichtag == 2)
		get_id3v2_tag (v2, filename);
#ifdef HAVE_OGG_H
	if ( whichtag == 4 )
		get_vorbis_tag (v1, filename);
#endif
  
  // Get the tag fields from the gui.
	tageditor_gui_to_tag (v1);
	tageditor_gui_to_tag (v2);
	
	mp3info = g_list_find_matching_fullfilename (queue, filename);

// And put the changed tag to the file
	if (access(filename, W_OK) == 0)	{
		if (queue)
			strncpy (mp3info->status, "Tags set!", 255);
		if (strtoucmp (filename + strlen (filename) - 4, ".ogg") == 0 ) {
#ifdef HAVE_OGG_H
			if ( whichtag == 4 && set_vorbis_tag (v1, filename) != 0 && queue)
				strncpy(mp3info->status, "Error!", 255);
#endif
		}
		else {
			if ((whichtag == 1 || whichtag == 3) && set_id3v1_tag (v1, filename) != 0 && queue)
				strncpy(mp3info->status, "Error!", 255);
			if ((whichtag == 2 || whichtag == 3) && set_id3v2_tag (v2, filename) != 0 && queue)
				strncpy(mp3info->status, "Error!", 255);
		}
	}
	else
		if (queue)
			strncpy(mp3info->status, "Locked!", 255);
  
	free(v1);
	free(v2);
	
  // Update the taginfo in the queue.
	strncpy(bar, filename, 2047);
	*(strrchr(bar, '/') + 1) = '\0';
	mp3file_get_info_1(bar, strrchr (filename, '/') + 1, mp3info);
	clist_file_update(GTK_CLIST(lookup_widget (cantus, "clist_cantus_queue")), mp3info);
	clist_file_update(GTK_CLIST(lookup_widget (cantus, "clist_cantus_queueonly_queue")), mp3info);
	
	gnome_appbar_set_status(GNOME_APPBAR(lookup_widget (cantus, "appbar")), "Ready.");
}


/**********************************************************************
 * this will write all informations in the tageditor gui to all tags in queue
 **********************************************************************/
void
tagedit_tag_set_queue(gint whichtag)
{
	extern GList *queue;
	GList *item = NULL;
	Mp3Info *mp3info = NULL;
	gchar fullfilename[2048];
	
	item = g_list_first(queue);
	while(item)
	{
		mp3info = (Mp3Info *)item->data;
		snprintf (fullfilename, 2047, "%s%s", mp3info->directory, mp3info->filename);
		tagedit_tag_set (fullfilename, whichtag);
		item = item->next;
	}
}


/**********************************************************************
 * this will delete one tag
 **********************************************************************/
void
tagedit_tag_delete (Mp3Info *mp3info, gint whichtag)
{
	extern GtkWidget *cantus;
	gchar fullfilename[4096];
	
	gchar status[4096];

	if( mp3info == NULL )
		return;
	
	snprintf(fullfilename, 4095, "%s%s", mp3info->directory, mp3info->filename);
	
	sprintf(status, "Deleting tags: %s", fullfilename);
	gnome_appbar_set_status(GNOME_APPBAR (lookup_widget (cantus, "appbar")), status);
	while(g_main_iteration(FALSE));
	
// And put the changed tag to the file
	if( access(fullfilename, W_OK) == 0 )
	{
		if( strtoucmp(fullfilename + strlen(fullfilename) - 4, ".ogg") == 0 )
		{
#ifdef HAVE_OGG_H
			if( whichtag == 4 )
				del_vorbis_tag(fullfilename);
#endif
		}
		else
		{
			if( whichtag == 1 || whichtag == 3 )
				del_id3v1_tag(fullfilename);
			if( whichtag == 2 || whichtag == 3 )
				del_id3v2_tag(fullfilename);
		}

		mp3file_get_info_1 (mp3info->directory, mp3info->filename, mp3info);
		clist_file_update (GTK_CLIST(lookup_widget(cantus, "clist_cantus_queue")), mp3info);
		clist_file_update (GTK_CLIST(lookup_widget(cantus, "clist_cantus_queueonly_queue")), mp3info);
	}

	gnome_appbar_set_status(GNOME_APPBAR (lookup_widget (cantus, "appbar")), "Ready.");
}


/**********************************************************************
 * this will delete all tags of the files in queue
 **********************************************************************/
void tagedit_tag_delete_queue(gint whichtag)
{
	extern GList *queue;
	GList *item = NULL;
	Mp3Info *mp3info;
	
	item = g_list_first(queue);
	while(item)
	{
		mp3info = (Mp3Info *)item->data;
		
		tagedit_tag_delete(mp3info, whichtag);
		item = item->next;
	}
}


static GList * set_focus_chain(void)
{
	extern GtkWidget *cantus;
	GList *focuschain = NULL;
	GList *tmp_list = NULL;
	GList *item = NULL;
	
// define correct order
	focuschain = g_list_append(focuschain, lookup_widget(cantus, "entry_cantus_tagedit_artist"));
	focuschain = g_list_append(focuschain, lookup_widget(cantus, "entry_cantus_tagedit_song"));
	focuschain = g_list_append(focuschain, lookup_widget(cantus, "entry_cantus_tagedit_album"));
	focuschain = g_list_append(focuschain, lookup_widget(cantus, "entry_cantus_tagedit_year"));
	focuschain = g_list_append(focuschain, lookup_widget(cantus, "entry_cantus_tagedit_track"));
	focuschain = g_list_append(focuschain, lookup_widget(cantus, "entry_cantus_tagedit_genre"));
	focuschain = g_list_append(focuschain, lookup_widget(cantus, "text_cantus_tagedit_comment"));
	
	focuschain = g_list_append(focuschain, lookup_widget(cantus, "check_cantus_tagedit_artist"));
	focuschain = g_list_append(focuschain, lookup_widget(cantus, "check_cantus_tagedit_song"));
	focuschain = g_list_append(focuschain, lookup_widget(cantus, "check_cantus_tagedit_album"));
	focuschain = g_list_append(focuschain, lookup_widget(cantus, "check_cantus_tagedit_year"));
	focuschain = g_list_append(focuschain, lookup_widget(cantus, "check_cantus_tagedit_track"));
	focuschain = g_list_append(focuschain, lookup_widget(cantus, "check_cantus_tagedit_genre"));
	focuschain = g_list_append(focuschain, lookup_widget(cantus, "check_cantus_tagedit_comment"));
	
	focuschain = g_list_append(focuschain, lookup_widget(cantus, "check_cantus_tagedit_single"));
	focuschain = g_list_append(focuschain, lookup_widget(cantus, "button_cantus_tagedit_change"));
	focuschain = g_list_append(focuschain, lookup_widget(cantus, "button_cantus_tagedit_del"));
	
// filter out all insensitive items
	tmp_list = focuschain;
	while (tmp_list)
	{
		if(GTK_WIDGET_IS_SENSITIVE (tmp_list->data)
			&& GTK_WIDGET_DRAWABLE (tmp_list->data)
			&& (GTK_IS_CONTAINER (tmp_list->data) || GTK_WIDGET_CAN_FOCUS (tmp_list->data)))
			tmp_list = tmp_list->next;
		else
		{
			item = tmp_list;
			tmp_list = tmp_list->next;

			focuschain = g_list_remove_link(focuschain, item);
			g_list_free_1(item);
		}
	}
	
// loop the list
//	g_list_last(focuschain)->next = g_list_first(focuschain);
	
	return focuschain;
}


void tagedit_focus_next(GtkWidget *current)
{
	GList *focuschain = NULL;
	GList *item = NULL;

	focuschain = set_focus_chain();
	
// all items inactive?
	if(!focuschain)
		return;
	
// is the item in the chain, and is some item behind it?
	if( (item = g_list_find(focuschain, current)) == NULL
		|| !item->next )
	{
// ok, so were at the end of the list, we take the first item in the list.
		gtk_widget_grab_focus(g_list_first(focuschain)->data);
	}
	else
		gtk_widget_grab_focus(item->next->data);
	
	g_list_free(focuschain);
}


void tagedit_focus_prev(GtkWidget *current)
{
	GList *focuschain = NULL;
	GList *item = NULL;

	focuschain = set_focus_chain();
	
// all items inactive?
	if(!focuschain)
		return;
	
// is the item in the chain, and is some item behind it?
	if( (item = g_list_find(focuschain, current)) == NULL
		|| !item->prev )
	{
// ok, so were at the beginning of the list, we take the last item in the list.
		gtk_widget_grab_focus(g_list_last(focuschain)->data);
	}
	else
		gtk_widget_grab_focus(item->prev->data);
	
	g_list_free(focuschain);
}
