#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <p12.h>
#include <pk11pub.h>
#include <keyhi.h>
#include <nspr.h>
#include <nss.h>
#include <ssl.h>
#include <prerror.h>
#include <base64.h>
#include <pkcs12.h>
#include <p12plcy.h>
#include <cryptohi.h>
#include <secerr.h>
#include <certdb.h>
#include <cert.h>
#include <secder.h>

#include <stdint.h>
#include <iostream>
#include <fstream>
#include <sstream>
#include <string.h>
#include <stdio.h>
#include <vector>
#include <sys/stat.h>
#include <sys/wait.h>
#include <signal.h>
#include <dirent.h>

#ifndef WIN32
#include <unistd.h>
#else
#include <conio.h>
#endif

#include <openssl/pem.h>
#include <openssl/err.h>

#include "nssutil.h"
#include "nssprivkeyinfocodec.h"
#include "nssgetpassword.h"
#include "nsscertextension.h"

#define PORT_ErrorToString(err) PR_ErrorToString((err), PR_LANGUAGE_I_DEFAULT)

#if NSS_VMAJOR >=3 && NSS_VMINOR >=10 && NSS_VPATCH >=0

#endif

/*********************************
 * Structures used in exporting the PKCS 12 blob
 *********************************/

/* A SafeInfo is used for each ContentInfo which makes up the
 * sequence of safes in the AuthenticatedSafe portion of the
 * PFX structure.
 */
struct SEC_PKCS12SafeInfoStr {
    PRArenaPool *arena;

    /* information for setting up password encryption */
    SECItem pwitem;
    SECOidTag algorithm;
    PK11SymKey *encryptionKey;

    /* how many items have been stored in this safe,
     * we will skip any safe which does not contain any
     * items
      */
    unsigned int itemCount;

    /* the content info for the safe */
    SEC_PKCS7ContentInfo *cinfo;

    sec_PKCS12SafeContents *safe;
};


/* An opaque structure which contains information needed for exporting
 * certificates and keys through PKCS 12.
 */
struct SEC_PKCS12ExportContextStr {
    PRArenaPool *arena;
    PK11SlotInfo *slot;
    void *wincx;

    /* integrity information */
    PRBool integrityEnabled;
    PRBool      pwdIntegrity;
    union {
        struct sec_PKCS12PasswordModeInfo pwdInfo;
        struct sec_PKCS12PublicKeyModeInfo pubkeyInfo;
    } integrityInfo;

    /* helper functions */
    /* retrieve the password call back */
    SECKEYGetPasswordKey pwfn;
    void *pwfnarg;

    /* safe contents bags */
    SEC_PKCS12SafeInfo **safeInfos;
    unsigned int safeInfoCount;

    /* the sequence of safes */
    sec_PKCS12AuthenticatedSafe authSafe;

    /* information needing deletion */
    CERTCertificate **certList;
};

typedef struct SEC_PKCS12SafeInfoStr SEC_PKCS12SafeInfo;
typedef struct SEC_PKCS12ExportContextStr SEC_PKCS12ExportContext;


#define NS_CERT_HEADER "-----BEGIN CERTIFICATE-----"
#define NS_CERT_TRAILER "-----END CERTIFICATE-----"

#define NS_CERTREQ_HEADER "-----BEGIN NEW CERTIFICATE REQUEST-----"
#define NS_CERTREQ_TRAILER "-----END NEW CERTIFICATE REQUEST-----"

#define NS_PRIVKEY_HEADER "-----BEGIN PRIVATE KEY-----"
#define NS_PRIVKEY_TRAILER "-----END PRIVATE KEY-----"

namespace AuthN {
namespace NSS {

  //-------Some utility methods---------/

  static std::string nss_error() {
    int len;
    char* text;
    std::string ret;
    if ((len = PR_GetErrorTextLength()) > 0) {
      text = (char*)malloc(len);
      if (PR_GetErrorText(text) > 0)
        ret.append("error string: ").append(text);
      free(text);
    } 
    else
      ret.append("unavailable nss error");
    return ret;
  }

  static char* nss_get_password(PK11SlotInfo* slot, PRBool retry, void *arg) {
    (void)slot; /* unused */
    if(retry || NULL == arg)
      return NULL;
    else
      return (char *)PORT_Strdup((char *)arg);
  }

/*
  template <class T>
  static std::string to_string (const T& t) {
    std::stringstream ss;
    ss << t;
    return ss.str();
  }
*/
  static std::string tostring(unsigned int val) {
    std::stringstream ss;
    ss << val;
    return ss.str();
  }

  static SECStatus file2str(Context& context, const char* infile, std::string& outstr) {
    PRFileInfo info;
    PRFileDesc* in;
    PRStatus status;

    in = PR_Open(infile, PR_RDONLY, 0);
    if(!in) {
      PRErrorCode err = PR_GetError();
      context.LogFormat(Context::LogError, "Failed to open input file %s, error code: %d - %s", infile, err, PORT_ErrorToString(err));
      return SECFailure;
    }

    status = PR_GetOpenFileInfo(in, &info);
    if(status != PR_SUCCESS) {
      PR_Close(in);
      return SECFailure;
    }

    outstr.clear();
    outstr.resize(info.size);
    PRInt32 num = PR_Read(in, (void*)(outstr.c_str()), info.size);
    if (num != info.size) {
      PR_Close(in);
      return SECFailure;
    }

    PR_Close(in);
    return SECSuccess;
  }

  static SECStatus str2file(Context& context, const std::string& instr, const char* outfile) {
    PRFileDesc* out;
    out = PR_Open(outfile, PR_RDWR | PR_CREATE_FILE | PR_TRUNCATE, 00660);
    if(!out) {
      context.LogFormat(Context::LogError, "Failed to open file %s", outfile);
      return SECFailure;
    }
    PR_Write(out, instr.c_str(), instr.size());
    PR_Close(out);
    return SECSuccess;
  }

  static SECStatus secitem2str(Context& context, const SECItem* data, PRBool ascii, std::string& output) {
    SECStatus rv = SECFailure;
    if (ascii) {
      char* buf = BTOA_DataToAscii(data->data, data->len);
      output.clear(); 
      if(buf !=NULL) { output.assign(buf); PORT_Free(buf); }
      rv = SECSuccess;
    }
    else { //raw data
      output.assign((const char*)(data->data), (size_t)(data->len));
      if(output.size() != data->len) {
        context.Log(Context::LogError, "Error writing raw SECItem");
        rv = SECFailure;
      }
      rv = SECSuccess;
    }
    return rv;
  }

  static SECStatus str2secitem(const std::string& instr, SECItem* out) {
    if(instr.empty()) return SECFailure;

    out->data = NULL;
    if(!SECITEM_AllocItem(NULL, out, instr.size())) {
      SECITEM_FreeItem(out, PR_FALSE);
      out->data = NULL;
      return SECFailure;
    }

    memcpy(out->data, instr.c_str(), instr.size());

    return SECSuccess;
  }

  static SECStatus output_cert2str(Context& context, const CERTCertificate* cert, PRBool ascii, std::string& outstr) {
    SECStatus rv = SECFailure;
    std::string out;
    rv = secitem2str(context, &(cert->derCert), ascii, out);
    if(rv == SECSuccess) {
      if(ascii) {
        outstr.append(NS_CERT_HEADER).append("\n").append(out).append("\n")
              .append(NS_CERT_TRAILER).append("\n");
        rv = SECSuccess;
      }
      else {
        outstr.assign(out.c_str(), out.size());
        rv = SECSuccess;
      }
    }
    return rv;
  }

  static SECStatus output_cert2file(Context& context, const CERTCertificate* cert, PRBool ascii, const char* outfile) {
    SECStatus rv = SECFailure;
    std::string outstr;
    rv = output_cert2str(context, cert, ascii, outstr);
    if(rv == SECSuccess) {
      rv = str2file(context, outstr, outfile);
    }
    return rv;
  }

  static SECStatus file2secitem(Context& context, const char* infile, SECItem* out) {
    SECStatus rv;
    std::string outstr;

    rv = file2str(context, infile, outstr);
    if(rv == SECFailure) return SECFailure;

    rv = str2secitem(outstr, out);
 
    return rv;
  }

  static SECStatus readDERfromstr(Context& context, SECItem* der, const std::string& indata, bool ascii) {
    SECStatus rv;
    if (ascii) {
      //Convert ascii to binary
      std::string body;
      std::string::size_type pos1, pos2;
      pos1 = indata.find("-----BEGIN");
      if(pos1 != std::string::npos) {
        pos1 = indata.find_first_of("\r\n", pos1);
        if(pos1 != std::string::npos) {
          pos2 = indata.find("-----END");
          if(pos2 != std::string::npos) 
            body = indata.substr(pos1+1, pos2-pos1-1);
        }
      }
      if(body.empty()) { context.Log(Context::LogError, "Failed to find cert body"); return SECFailure; }
      // Convert to binary
      rv = ATOB_ConvertAsciiToItem(der, (char*)(body.c_str()));
      if(rv) {
        context.Log(Context::LogError, "Failed to convert ascii to binary");
        return SECFailure;
      }

       
/*
      char* data;
      char* body;
      data = (char*)(str.c_str());
      //Remove headers and trailers from data
      if((body = strstr(data, "-----BEGIN")) != NULL) {
        char* trailer = NULL;
        data = body;
        body = PORT_Strchr(data, '\n');
        if (!body)
          body = PORT_Strchr(data, '\r');
        if (body)
          trailer = strstr(++body, "-----END");
        if (trailer != NULL) {
          *trailer = '\0';
        }
        else {
          context.Log(Context::LogError, "Input is without trailer\n");
          return SECFailure;
        }
      }
      else {
        body = data;
      }
      // Convert to binary
      rv = ATOB_ConvertAsciiToItem(der, body);
      if(rv) {
        context.Log(Context::LogError, "Failed to convert ascii to binary");
        return SECFailure;
      }
*/
    }
    else{
      rv = str2secitem(indata, der); 
      if(rv) {
        context.Log(Context::LogError, "Failed to convert data to SECItem");
        return SECFailure;
      }
    }

    return SECSuccess;
  }

  static SECStatus readDERfromfile(Context& context, SECItem* der, const char* infile, bool ascii) {
    SECStatus rv;
    std::string filedata;

    // Read data from file
    rv = file2str(context, infile, filedata);
    if (rv != SECSuccess) {
      context.Log(Context::LogError, "Failed to read data from input file");
      return SECFailure;
    }

    rv = readDERfromstr(context, der, filedata, ascii);
    return rv;
  }


  static SECStatus p12u_SwapUnicodeBytes(SECItem *uniItem) {
    unsigned int i;
    unsigned char a;
    if((uniItem == NULL) || (uniItem->len % 2)) {
        return SECFailure;
    }
    for(i = 0; i < uniItem->len; i += 2) {
        a = uniItem->data[i];
        uniItem->data[i] = uniItem->data[i+1];
        uniItem->data[i+1] = a;
    }
    return SECSuccess;
  }

  static PRBool p12u_ucs2_ascii_conversion_function(PRBool toUnicode,
        unsigned char* inBuf, unsigned int inBufLen,
        unsigned char* outBuf, unsigned int maxOutBufLen,
        unsigned int* outBufLen, PRBool swapBytes) {
    SECItem it;
    SECItem *dup = NULL;
    PRBool ret;
    it.data = inBuf;
    it.len = inBufLen;
    dup = SECITEM_DupItem(&it);
    // If converting Unicode to ASCII, swap bytes before conversion as neccessary.
    if (!toUnicode && swapBytes) {
      if (p12u_SwapUnicodeBytes(dup) != SECSuccess) {
        SECITEM_ZfreeItem(dup, PR_TRUE);
        return PR_FALSE;
      }
    }
    // Perform the conversion.
    ret = PORT_UCS2_UTF8Conversion(toUnicode, dup->data, dup->len,
                                   outBuf, maxOutBufLen, outBufLen);
    if (dup)
      SECITEM_ZfreeItem(dup, PR_TRUE);
    return ret;
  }
  
  bool nssInit(Context& context, const std::string& configdir) {
    SECStatus rv;
    //Initialize NSPR
    PR_Init(PR_USER_THREAD, PR_PRIORITY_NORMAL, 256);
    //Set the PKCS #11 strings for the internal token
    char* db_name = PL_strdup("internal (software)              ");
    PK11_ConfigurePKCS11(NULL,NULL,NULL, db_name, NULL, NULL,NULL,NULL,8,1);
    //Initialize NSS and open the certificate database read-only
    rv = NSS_Initialize(configdir.c_str(), NULL, NULL, "secmod.db", 0);// NSS_INIT_READONLY);
    if (rv != SECSuccess) {
      rv = NSS_NoDB_Init(configdir.c_str());
    }
    if (rv != SECSuccess) { // || NSS_InitTokens() != SECSuccess) {
      NSS_Shutdown();
      context.Log(Context::LogError, "NSS initialization falied on certificate database: " + configdir);
      return false;
    }
    if (NSS_SetDomesticPolicy() != SECSuccess ){
      NSS_Shutdown();
      context.LogFormat(Context::LogError, "NSS set domestic policy failed %s on certificate database %s", nss_error().c_str(), configdir.c_str());
      return false;
    }

    PK11_SetPasswordFunc(nss_get_password_from_console); //(nss_get_password);
    context.Log(Context::LogInfo, "Succeeded to initialize NSS");

    PORT_SetUCS2_ASCIIConversionFunction(p12u_ucs2_ascii_conversion_function);
 
    SEC_PKCS12EnableCipher(PKCS12_RC4_40, 1);
    SEC_PKCS12EnableCipher(PKCS12_RC4_128, 1);
    SEC_PKCS12EnableCipher(PKCS12_RC2_CBC_40, 1);
    SEC_PKCS12EnableCipher(PKCS12_RC2_CBC_128, 1);
    SEC_PKCS12EnableCipher(PKCS12_DES_56, 1);
    SEC_PKCS12EnableCipher(PKCS12_DES_EDE3_168, 1);
    SEC_PKCS12SetPreferredCipher(PKCS12_DES_EDE3_168, 1);

    RegisterDynamicOids(context);

    return true;
  }

  void nssFinish() {
    NSS_Shutdown();
    PL_ArenaFinish();
    PR_Cleanup();
  }

  static bool ReadPrivKeyAttribute(Context& context, SECKEYPrivateKey* key, CK_ATTRIBUTE_TYPE type, std::vector<uint8>* output) {
    SECItem item;
    SECStatus rv;
    rv = PK11_ReadRawAttribute(PK11_TypePrivKey, key, type, &item);
    if (rv != SECSuccess) {
      context.LogFormat(Context::LogError,"Failed to read attribute %x from private key.", type);
      return false;
    }
    output->assign(item.data, item.data + item.len);
    SECITEM_FreeItem(&item, PR_FALSE);
    return true;
  }

  static bool ExportPrivateKey(Context& context, SECKEYPrivateKey* key, std::vector<uint8>* output) {
    PrivateKeyInfoCodec private_key_info(true);

    // Manually read the component attributes of the private key and build up
    // the PrivateKeyInfo.
    if (!ReadPrivKeyAttribute(context, key, CKA_MODULUS, private_key_info.modulus()) ||
      !ReadPrivKeyAttribute(context, key, CKA_PUBLIC_EXPONENT,
          private_key_info.public_exponent()) ||
      !ReadPrivKeyAttribute(context, key, CKA_PRIVATE_EXPONENT,
          private_key_info.private_exponent()) ||
      !ReadPrivKeyAttribute(context, key, CKA_PRIME_1, private_key_info.prime1()) ||
      !ReadPrivKeyAttribute(context, key, CKA_PRIME_2, private_key_info.prime2()) ||
      !ReadPrivKeyAttribute(context, key, CKA_EXPONENT_1, private_key_info.exponent1()) ||
      !ReadPrivKeyAttribute(context, key, CKA_EXPONENT_2, private_key_info.exponent2()) ||
      !ReadPrivKeyAttribute(context, key, CKA_COEFFICIENT, private_key_info.coefficient())) {
      return false;
    }

    return private_key_info.Export(output);
  }

  bool nssExportCertificate(Context& context, const std::string& certname, std::string& certstr, bool ascii) {
    CERTCertificate* find_cert = NULL;
    SECStatus rv = SECFailure;

    find_cert = PK11_FindCertFromNickname((char*)(certname.c_str()), NULL);
    if(find_cert) {
      context.Log(Context::LogError, "Succeeded to get credential");
    } else {
      context.Log(Context::LogError, "Failed to get credential"); return false;
    }

    rv = output_cert2str(context, find_cert, true, certstr);

    if(rv == SECSuccess) return true;
    else return false;
  }

  bool nssExportCertificate(Context& context, const std::string& certname, const char* certfile, bool ascii) {
    CERTCertList* list;
    CERTCertificate* find_cert = NULL;
    CERTCertListNode* node;
    SECStatus rv = SECFailure;

    find_cert = PK11_FindCertFromNickname((char*)(certname.c_str()), NULL);
    //find_cert = CERT_FindCertByNicknameOrEmailAddr(handle, certname);

/*
    list = PK11_ListCerts(PK11CertListAll, NULL);
    for (node = CERT_LIST_HEAD(list); !CERT_LIST_END(node,list);
        node = CERT_LIST_NEXT(node)) {
      CERTCertificate* cert = node->cert;
      const char* nickname = (const char*)node->appData;
      if (!nickname) {
        nickname = cert->nickname;
      }
      if(nickname == NULL) continue;
      if (strcmp(certname.c_str(), nickname) == 0) {
        find_cert = CERT_DupCertificate(cert); break;
      }
    }
    if (list) {
      CERT_DestroyCertList(list);
    }
*/
    if(find_cert) {
      context.Log(Context::LogError, "Succeeded to get credential");
    } else {
      context.Log(Context::LogError, "Failed to get credential"); return false;
    }

    rv = output_cert2file(context, find_cert, true, certfile);

    if(rv == SECSuccess) return true;
    else return false;
  }

  static void p12u_export2str(void *arg, const char *buf, unsigned long len) {
    std::string* outstr = (std::string*)arg;
    if(!outstr) return;
    outstr->append(buf, len);
  }

  static CERTCertificate* FindIssuerCert(CERTCertificate* cert) {
    CERTCertificate* issuercert = NULL;
    SECStatus rv;
    issuercert = CERT_FindCertByName(cert->dbhandle, &cert->derIssuer);
    return issuercert;
  }

  static CERTCertList* RevertChain(CERTCertList* chain) {
    CERTCertListNode* node;
    CERTCertListNode* newnd;
    CERTCertListNode* head;
    CERTCertList* certlist = NULL;
    SECStatus rv;
    certlist = CERT_NewCertList();

    for (node = CERT_LIST_HEAD(chain) ; !CERT_LIST_END(node, chain); node= CERT_LIST_NEXT(node)) {
      head = CERT_LIST_HEAD(certlist);
      newnd = (CERTCertListNode *)PORT_ArenaZAlloc(certlist->arena, sizeof(CERTCertListNode));
      if(newnd == NULL ) return certlist;
      PR_INSERT_BEFORE(&newnd->links, &head->links);
      newnd->cert = CERT_DupCertificate(node->cert);
    }
    return certlist;
  }

  static CERTCertList* FindCertChain(CERTCertificate* cert) {
    CERTCertificate* issuercert = NULL;
    CERTCertList* certlist = NULL;
    SECStatus rv;
    certlist = CERT_NewCertList();
    //certlist = CERT_CertListFromCert(issuercert);
    cert = CERT_DupCertificate(cert);
    CERT_AddCertToListTail(certlist, cert);
    do {
      issuercert = FindIssuerCert(cert);
      if(issuercert) {
        CERT_AddCertToListTail(certlist, issuercert);
        if(CERT_IsCACert(issuercert, NULL)) break;
        cert = issuercert;
      }
      else break;
    } while(true);

    return certlist;
  }


//-----------------Begin of output pk12---------------//

//The following code is copied from nss source code tree, the file
//mozilla/security/nss/lib/pkcs12/p12e.c, and file
//mozilla/security/nss/lib/pkcs12/p12local.c
//We need a new proprietary version of SEC_PKCS12AddCertAndKey which can 
//parse the cert chain of proxy.

//Some methods start with "sec_" is copied from p12local.c, because
//they are not exposed by nss.

/*************Start Here************/

/* Creates a new certificate bag and returns a pointer to it.  If an error
 * occurs NULL is returned.
 */
sec_PKCS12CertBag *
sec_PKCS12NewCertBag(PRArenaPool *arena, SECOidTag certType)
{
    sec_PKCS12CertBag *certBag = NULL;
    SECOidData *bagType = NULL;
    SECStatus rv;
    void *mark = NULL;

    if(!arena) {
        return NULL;
    }

    mark = PORT_ArenaMark(arena);
    certBag = (sec_PKCS12CertBag *)PORT_ArenaZAlloc(arena,
                                                    sizeof(sec_PKCS12CertBag));
    if(!certBag) {
        PORT_ArenaRelease(arena, mark);
        PORT_SetError(SEC_ERROR_NO_MEMORY);
        return NULL;
    }

    bagType = SECOID_FindOIDByTag(certType);
    if(!bagType) {
        PORT_SetError(SEC_ERROR_NO_MEMORY);
        goto loser;
    }

    rv = SECITEM_CopyItem(arena, &certBag->bagID, &bagType->oid);
    if(rv != SECSuccess) {
        PORT_SetError(SEC_ERROR_NO_MEMORY);
        goto loser;
    }
 
    PORT_ArenaUnmark(arena, mark);
    return certBag;

loser:
    PORT_ArenaRelease(arena, mark);
    return NULL;
}

/* Creates a safeBag of the specified type, and if bagData is specified,
 * the contents are set.  The contents could be set later by the calling
 * routine.
 */
sec_PKCS12SafeBag *
sec_PKCS12CreateSafeBag(SEC_PKCS12ExportContext *p12ctxt, SECOidTag bagType,
                        void *bagData)
{
    sec_PKCS12SafeBag *safeBag;
    PRBool setName = PR_TRUE;
    void *mark = NULL;
    SECStatus rv = SECSuccess;
    SECOidData *oidData = NULL;

    if(!p12ctxt) {
        return NULL;
    }

    mark = PORT_ArenaMark(p12ctxt->arena);
    if(!mark) {
        PORT_SetError(SEC_ERROR_NO_MEMORY);
        return NULL;
    }

    safeBag = (sec_PKCS12SafeBag *)PORT_ArenaZAlloc(p12ctxt->arena,
                                                    sizeof(sec_PKCS12SafeBag));
    if(!safeBag) {
        PORT_ArenaRelease(p12ctxt->arena, mark);
        PORT_SetError(SEC_ERROR_NO_MEMORY);
        return NULL;
    }

    /* set the bags content based upon bag type */
    switch(bagType) {
        case SEC_OID_PKCS12_V1_KEY_BAG_ID:
            safeBag->safeBagContent.pkcs8KeyBag =
                (SECKEYPrivateKeyInfo *)bagData;
            break;
        case SEC_OID_PKCS12_V1_CERT_BAG_ID:
            safeBag->safeBagContent.certBag = (sec_PKCS12CertBag *)bagData;
            break;
        case SEC_OID_PKCS12_V1_CRL_BAG_ID:
            safeBag->safeBagContent.crlBag = (sec_PKCS12CRLBag *)bagData;
            break;
        case SEC_OID_PKCS12_V1_SECRET_BAG_ID:
            safeBag->safeBagContent.secretBag = (sec_PKCS12SecretBag *)bagData;
            break;
        case SEC_OID_PKCS12_V1_PKCS8_SHROUDED_KEY_BAG_ID:
            safeBag->safeBagContent.pkcs8ShroudedKeyBag =
                (SECKEYEncryptedPrivateKeyInfo *)bagData;
            break;
        case SEC_OID_PKCS12_V1_SAFE_CONTENTS_BAG_ID:
            safeBag->safeBagContent.safeContents =
                (sec_PKCS12SafeContents *)bagData;
            setName = PR_FALSE;
            break;
        default:
            goto loser;
    }

    oidData = SECOID_FindOIDByTag(bagType);
    if(oidData) {
        rv = SECITEM_CopyItem(p12ctxt->arena, &safeBag->safeBagType, &oidData->oid);
        if(rv != SECSuccess) {
            PORT_SetError(SEC_ERROR_NO_MEMORY);
            goto loser;
        }
    } else {
        goto loser;
    }
    safeBag->arena = p12ctxt->arena;
    PORT_ArenaUnmark(p12ctxt->arena, mark);

    return safeBag;

loser:
    if(mark) {
        PORT_ArenaRelease(p12ctxt->arena, mark);
    }

    return NULL;
}

/* this function converts a password to unicode and encures that the
 * required double 0 byte be placed at the end of the string
 */
PRBool
sec_pkcs12_convert_item_to_unicode(PRArenaPool *arena, SECItem *dest,
                                   SECItem *src, PRBool zeroTerm,
                                   PRBool asciiConvert, PRBool toUnicode)
{
    PRBool success = PR_FALSE;
    if(!src || !dest) {
        PORT_SetError(SEC_ERROR_INVALID_ARGS);
        return PR_FALSE;
    }

    dest->len = src->len * 3 + 2;
    if(arena) {
        dest->data = (unsigned char*)PORT_ArenaZAlloc(arena, dest->len);
    } else {
        dest->data = (unsigned char*)PORT_ZAlloc(dest->len);
    }

    if(!dest->data) {
        dest->len = 0;
        return PR_FALSE;
    }

    if(!asciiConvert) {
        success = PORT_UCS2_UTF8Conversion(toUnicode, src->data, src->len, dest->data,
                                           dest->len, &dest->len);
    } else {
#ifndef IS_LITTLE_ENDIAN
        PRBool swapUnicode = PR_FALSE;
#else
        PRBool swapUnicode = PR_TRUE;
#endif
        success = PORT_UCS2_ASCIIConversion(toUnicode, src->data, src->len, dest->data,
                                            dest->len, &dest->len, swapUnicode);
    }

    if(!success) {
        if(!arena) {
            PORT_Free(dest->data);
            dest->data = NULL;
            dest->len = 0;
        }
        return PR_FALSE;
    }

    if((dest->data[dest->len-1] || dest->data[dest->len-2]) && zeroTerm) {
        if(dest->len + 2 > 3 * src->len) {
            if(arena) {
                dest->data = (unsigned char*)PORT_ArenaGrow(arena,
                                                     dest->data, dest->len,
                                                     dest->len + 2);
            } else {
                dest->data = (unsigned char*)PORT_Realloc(dest->data,
                                                          dest->len + 2);
            }

            if(!dest->data) {
                return PR_FALSE;
            }
        }
        dest->len += 2;
        dest->data[dest->len-1] = dest->data[dest->len-2] = 0;
    }

    return PR_TRUE;
}

/* sec_PKCS12AddAttributeToBag
 * adds an attribute to a safeBag.  currently, the only attributes supported
 * are those which are specified within PKCS 12.
 *
 *      p12ctxt - the export context
 *      safeBag - the safeBag to which attributes are appended
 *      attrType - the attribute type
 *      attrData - the attribute data
 */
SECStatus
sec_PKCS12AddAttributeToBag(SEC_PKCS12ExportContext *p12ctxt,
                            sec_PKCS12SafeBag *safeBag, SECOidTag attrType,
                            SECItem *attrData)
{
    sec_PKCS12Attribute *attribute;
    void *mark = NULL, *dummy = NULL;
    SECOidData *oiddata = NULL;
    SECItem unicodeName = { siBuffer, NULL, 0};
    void *src = NULL;
    unsigned int nItems = 0;
    SECStatus rv;

    if(!safeBag || !p12ctxt) {
        return SECFailure;
    }

    mark = PORT_ArenaMark(safeBag->arena);

    /* allocate the attribute */
    attribute = (sec_PKCS12Attribute *)PORT_ArenaZAlloc(safeBag->arena,
                                                sizeof(sec_PKCS12Attribute));
    if(!attribute) {
        PORT_SetError(SEC_ERROR_NO_MEMORY);
        goto loser;
    }
    /* set up the attribute */
    oiddata = SECOID_FindOIDByTag(attrType);
    if(!oiddata) {
        PORT_SetError(SEC_ERROR_NO_MEMORY);
        goto loser;
    }
    if(SECITEM_CopyItem(p12ctxt->arena, &attribute->attrType, &oiddata->oid) !=
                SECSuccess) {
        PORT_SetError(SEC_ERROR_NO_MEMORY);
        goto loser;
    }

    nItems = 1;
    switch(attrType) {
        case SEC_OID_PKCS9_LOCAL_KEY_ID:
            {
                src = attrData;
                break;
            }
        case SEC_OID_PKCS9_FRIENDLY_NAME:
            {
                if(!sec_pkcs12_convert_item_to_unicode(p12ctxt->arena,
                                        &unicodeName, attrData, PR_FALSE,
                                        PR_FALSE, PR_TRUE)) {
                    goto loser;
                }
                src = &unicodeName;
                break;
            }
        default:
            goto loser;
    }
    /* append the attribute to the attribute value list  */
    attribute->attrValue = (SECItem **)PORT_ArenaZAlloc(p12ctxt->arena,
                                            ((nItems + 1) * sizeof(SECItem *)));
    if(!attribute->attrValue) {
        PORT_SetError(SEC_ERROR_NO_MEMORY);
        goto loser;
    }

    /* XXX this will need to be changed if attributes requiring more than
     * one element are ever used.
     */
    attribute->attrValue[0] = (SECItem *)PORT_ArenaZAlloc(p12ctxt->arena,
                                                          sizeof(SECItem));
    if(!attribute->attrValue[0]) {
        PORT_SetError(SEC_ERROR_NO_MEMORY);
        goto loser;
    }
    attribute->attrValue[1] = NULL;

    rv = SECITEM_CopyItem(p12ctxt->arena, attribute->attrValue[0],
                          (SECItem*)src);
    if(rv != SECSuccess) {
        PORT_SetError(SEC_ERROR_NO_MEMORY);
        goto loser;
    }

    /* append the attribute to the safeBag attributes */
    if(safeBag->nAttribs) {
        dummy = PORT_ArenaGrow(p12ctxt->arena, safeBag->attribs,
                        ((safeBag->nAttribs + 1) * sizeof(sec_PKCS12Attribute *)),
                        ((safeBag->nAttribs + 2) * sizeof(sec_PKCS12Attribute *)));
        safeBag->attribs = (sec_PKCS12Attribute **)dummy;
    } else {
        safeBag->attribs = (sec_PKCS12Attribute **)PORT_ArenaZAlloc(p12ctxt->arena,
                                                2 * sizeof(sec_PKCS12Attribute *));
        dummy = safeBag->attribs;
    }
    if(!dummy) {
        goto loser;
    }

    safeBag->attribs[safeBag->nAttribs] = attribute;
    safeBag->attribs[++safeBag->nAttribs] = NULL;

    PORT_ArenaUnmark(p12ctxt->arena, mark);
    return SECSuccess;

loser:
    if(mark) {
        PORT_ArenaRelease(p12ctxt->arena, mark);
    }

    return SECFailure;
}

/*********************************
 * Routines to handle the exporting of the keys and certificates
 *********************************/

/* creates a safe contents which safeBags will be appended to */
sec_PKCS12SafeContents *
sec_PKCS12CreateSafeContents(PRArenaPool *arena)
{
    sec_PKCS12SafeContents *safeContents;

    if(arena == NULL) {
        return NULL;
    }

    /* create the safe contents */
    safeContents = (sec_PKCS12SafeContents *)PORT_ArenaZAlloc(arena,
                                            sizeof(sec_PKCS12SafeContents));
    if(!safeContents) {
        PORT_SetError(SEC_ERROR_NO_MEMORY);
        goto loser;
    }

    /* set up the internal contents info */
    safeContents->safeBags = NULL;
    safeContents->arena = arena;
    safeContents->bagCount = 0;

    return safeContents;

loser:
    return NULL;
}  

/* appends a safe bag to a safeContents using the specified arena.
 */
SECStatus
sec_pkcs12_append_bag_to_safe_contents(PRArenaPool *arena,
                                       sec_PKCS12SafeContents *safeContents,
                                       sec_PKCS12SafeBag *safeBag)
{
    void *mark = NULL, *dummy = NULL;

    if(!arena || !safeBag || !safeContents) {
        return SECFailure;
    }

    mark = PORT_ArenaMark(arena);
    if(!mark) {
        PORT_SetError(SEC_ERROR_NO_MEMORY);
        return SECFailure;
    }

    /* allocate space for the list, or reallocate to increase space */
    if(!safeContents->safeBags) {
        safeContents->safeBags = (sec_PKCS12SafeBag **)PORT_ArenaZAlloc(arena,
                                                (2 * sizeof(sec_PKCS12SafeBag *)));
        dummy = safeContents->safeBags;
        safeContents->bagCount = 0;
    } else {
        dummy = PORT_ArenaGrow(arena, safeContents->safeBags,
                        (safeContents->bagCount + 1) * sizeof(sec_PKCS12SafeBag *),
                        (safeContents->bagCount + 2) * sizeof(sec_PKCS12SafeBag *));
        safeContents->safeBags = (sec_PKCS12SafeBag **)dummy;
    }

    if(!dummy) {
        PORT_ArenaRelease(arena, mark);
        PORT_SetError(SEC_ERROR_NO_MEMORY);
        return SECFailure;
    }

    /* append the bag at the end and null terminate the list */
    safeContents->safeBags[safeContents->bagCount++] = safeBag;
    safeContents->safeBags[safeContents->bagCount] = NULL;

    PORT_ArenaUnmark(arena, mark);

    return SECSuccess;
}

/* appends a safeBag to a specific safeInfo.
 */
SECStatus
sec_pkcs12_append_bag(SEC_PKCS12ExportContext *p12ctxt,
                      SEC_PKCS12SafeInfo *safeInfo, sec_PKCS12SafeBag *safeBag)
{
    sec_PKCS12SafeContents *dest;
    SECStatus rv = SECFailure;

    if(!p12ctxt || !safeBag || !safeInfo) {
        return SECFailure;
    }

    if(!safeInfo->safe) {
        safeInfo->safe = sec_PKCS12CreateSafeContents(p12ctxt->arena);
        if(!safeInfo->safe) {
            return SECFailure;
        }
    }

    dest = safeInfo->safe;
    rv = sec_pkcs12_append_bag_to_safe_contents(p12ctxt->arena, dest, safeBag);
    if(rv == SECSuccess) {
        safeInfo->itemCount++;
    }

    return rv;
}

/* compute the thumbprint of the DER cert and create a digest info
 * to store it in and return the digest info.
 * a return of NULL indicates an error.
 */
SGNDigestInfo *
sec_pkcs12_compute_thumbprint(SECItem *der_cert)
{
    SGNDigestInfo *thumb = NULL;
    SECItem digest;
    PRArenaPool *temparena = NULL;
    SECStatus rv = SECFailure;

    if(der_cert == NULL)
        return NULL;

    temparena = PORT_NewArena(SEC_ASN1_DEFAULT_ARENA_SIZE);
    if(temparena == NULL) {
        return NULL;
    }

    digest.data = (unsigned char *)PORT_ArenaZAlloc(temparena,
                                                    sizeof(unsigned char) *
                                                    SHA1_LENGTH);
    /* digest data and create digest info */
    if(digest.data != NULL) {
        digest.len = SHA1_LENGTH;
        rv = PK11_HashBuf(SEC_OID_SHA1, digest.data, der_cert->data,
                          der_cert->len);
        if(rv == SECSuccess) {
            thumb = SGN_CreateDigestInfo(SEC_OID_SHA1,
                                         digest.data,
                                         digest.len);
        } else {
            PORT_SetError(SEC_ERROR_NO_MEMORY);
        }
    } else {
        PORT_SetError(SEC_ERROR_NO_MEMORY);
    }

    PORT_FreeArena(temparena, PR_TRUE);

    return thumb;
}

/* SEC_PKCS12AddKeyForCert
 *      Extracts the key associated with a particular certificate and exports
 *      it.
 *
 *      p12ctxt - the export context
 *      safe - the safeInfo to place the key in
 *      nestedDest - the nested safeContents to place a key
 *      cert - the certificate which the key belongs to
 *      shroudKey - encrypt the private key for export.  This value should
 *              always be true.  lower level code will not allow the export
 *              of unencrypted private keys.
 *      algorithm - the algorithm with which to encrypt the private key
 *      pwitem - the password to encrypt the private key with
 *      keyId - the keyID attribute
 *      nickName - the nickname attribute
 */
SECStatus
my_SEC_PKCS12AddKeyForCert(SEC_PKCS12ExportContext *p12ctxt, SEC_PKCS12SafeInfo *safe,
                        void *nestedDest, CERTCertificate *cert,
                        PRBool shroudKey, SECOidTag algorithm, SECItem *pwitem,
                        SECItem *keyId, SECItem *nickName)
{
    void *mark;
    void *keyItem;
    SECOidTag keyType;
    SECStatus rv = SECFailure;
    SECItem nickname = {siBuffer,NULL,0}, uniPwitem = {siBuffer, NULL, 0};
    sec_PKCS12SafeBag *returnBag;

    if(!p12ctxt || !cert || !safe) {
        return SECFailure;
    }

    mark = PORT_ArenaMark(p12ctxt->arena);
    /* retrieve the key based upon the type that it is and
     * specify the type of safeBag to store the key in
     */
    if(!shroudKey) {

        /* extract the key unencrypted.  this will most likely go away */
        SECKEYPrivateKeyInfo *pki = PK11_ExportPrivateKeyInfo(cert,
                                                              p12ctxt->wincx);
        if(!pki) {
            PORT_ArenaRelease(p12ctxt->arena, mark);
            PORT_SetError(SEC_ERROR_PKCS12_UNABLE_TO_EXPORT_KEY);
            return SECFailure;
        }
        keyItem = PORT_ArenaZAlloc(p12ctxt->arena, sizeof(SECKEYPrivateKeyInfo));
        if(!keyItem) {
            PORT_SetError(SEC_ERROR_NO_MEMORY);
            goto loser;
        }
        rv = SECKEY_CopyPrivateKeyInfo(p12ctxt->arena,
                                       (SECKEYPrivateKeyInfo *)keyItem, pki);
        keyType = SEC_OID_PKCS12_V1_KEY_BAG_ID;
        SECKEY_DestroyPrivateKeyInfo(pki, PR_TRUE);
    } else {

        /* extract the key encrypted */
        SECKEYEncryptedPrivateKeyInfo *epki = NULL;
        PK11SlotInfo *slot = NULL;

        if(!sec_pkcs12_convert_item_to_unicode(p12ctxt->arena, &uniPwitem,
                                 pwitem, PR_TRUE, PR_TRUE, PR_TRUE)) {
            PORT_SetError(SEC_ERROR_NO_MEMORY);
            goto loser;
        }
        /* we want to make sure to take the key out of the key slot */
        if(PK11_IsInternal(p12ctxt->slot)) {
            slot = PK11_GetInternalKeySlot();
        } else {
            slot = PK11_ReferenceSlot(p12ctxt->slot);
        }

        epki = PK11_ExportEncryptedPrivateKeyInfo(slot, algorithm,
                                                  &uniPwitem, cert, 1,
                                                  p12ctxt->wincx);
        PK11_FreeSlot(slot);
        if(!epki) {
            PORT_SetError(SEC_ERROR_PKCS12_UNABLE_TO_EXPORT_KEY);
            goto loser;
        }

        keyItem = PORT_ArenaZAlloc(p12ctxt->arena,
                                  sizeof(SECKEYEncryptedPrivateKeyInfo));
        if(!keyItem) {
            PORT_SetError(SEC_ERROR_NO_MEMORY);
            goto loser;
        }
        rv = SECKEY_CopyEncryptedPrivateKeyInfo(p12ctxt->arena,
                                        (SECKEYEncryptedPrivateKeyInfo *)keyItem,
                                        epki);
        keyType = SEC_OID_PKCS12_V1_PKCS8_SHROUDED_KEY_BAG_ID;
        SECKEY_DestroyEncryptedPrivateKeyInfo(epki, PR_TRUE);
    }

    if(rv != SECSuccess) {
        goto loser;
    }
    /* if no nickname specified, let's see if the certificate has a
     * nickname.
     */
    if(!nickName) {
        if(cert->nickname) {
            nickname.data = (unsigned char *)cert->nickname;
            nickname.len = PORT_Strlen(cert->nickname);
            nickName = &nickname;
        }
    }

    /* create the safe bag and set any attributes */
    returnBag = sec_PKCS12CreateSafeBag(p12ctxt, keyType, keyItem);
    if(!returnBag) {
        rv = SECFailure;
        goto loser;
    }

    if(nickName) {
        if(sec_PKCS12AddAttributeToBag(p12ctxt, returnBag,
                                       SEC_OID_PKCS9_FRIENDLY_NAME, nickName)
                                       != SECSuccess) {
            goto loser;
        }
    }

    if(keyId) {
        if(sec_PKCS12AddAttributeToBag(p12ctxt, returnBag, SEC_OID_PKCS9_LOCAL_KEY_ID,
                                       keyId) != SECSuccess) {
            goto loser;
        }
    }

    if(nestedDest) {
        rv = sec_pkcs12_append_bag_to_safe_contents(p12ctxt->arena,
                                          (sec_PKCS12SafeContents*)nestedDest,
                                          returnBag);
    } else {
        rv = sec_pkcs12_append_bag(p12ctxt, safe, returnBag);
    }

loser:

    if (rv != SECSuccess) {
        PORT_ArenaRelease(p12ctxt->arena, mark);
    } else {
        PORT_ArenaUnmark(p12ctxt->arena, mark);
    }

    return rv;
}

/*************End Here************/

  //Our version of SEC_PKCS12AddCert
  static SECStatus my_SEC_PKCS12AddCert(SEC_PKCS12ExportContext* p12ctxt, 
      SEC_PKCS12SafeInfo* safe, void* nestedDest, 
      CERTCertificate* cert, CERTCertDBHandle* certDb, SECItem* keyId) {
    sec_PKCS12CertBag *certBag;
    sec_PKCS12SafeBag *safeBag;
    void *mark;
    CERTCertListNode* node;
    CERTCertList* certlist_tmp = NULL;
    CERTCertList* certlist = NULL;
    SECStatus rv;
    SECItem nick;

    if(!p12ctxt) return SECFailure;

    // Look for the cert chain that issues the proxy
    certlist_tmp = FindCertChain(cert);
    if(!certlist_tmp) {
      //NSSUtilLogger.msg(ERROR, "Failed to find issuer cert for proxy cert"); 
      goto err;
    }
    certlist = RevertChain(certlist_tmp);
/*
    for (i=chain->len-1; i>=0; i--) {
        CERTCertificate *c;
        c = CERT_FindCertByDERCert(handle, &chain->certs[i]);
        for (j=i; j<chain->len-1; j++) printf("  ");
        printf("\"%s\" [%s]\n\n", c->nickname, c->subjectName);
        CERT_DestroyCertificate(c);
    }
*/

    // Compose pk12 data
    for (node = CERT_LIST_HEAD(certlist); !CERT_LIST_END(node,certlist); node= CERT_LIST_NEXT(node)) {
      CERTCertificate* cert = node->cert;
      nick.type = siBuffer;
      nick.data = NULL;
      nick.len  = 0;
      if(!cert) return SECFailure;
      mark = PORT_ArenaMark(p12ctxt->arena);
      certBag = sec_PKCS12NewCertBag(p12ctxt->arena, SEC_OID_PKCS9_X509_CERT);
      if(!certBag) goto err;

      if(SECITEM_CopyItem(p12ctxt->arena, &certBag->value.x509Cert, &cert->derCert) != SECSuccess) {
        goto err;
      }

      //If the certificate has a nickname, set the friendly name to that.
      if(cert->nickname) {
        nick.data = (unsigned char *)cert->nickname;
        nick.len = PORT_Strlen(cert->nickname);
      }
      safeBag = sec_PKCS12CreateSafeBag(p12ctxt, SEC_OID_PKCS12_V1_CERT_BAG_ID,
                                      certBag);
      if(!safeBag) goto err;

      // Add the friendly name and keyId attributes, if necessary
      if(nick.data) {
        if(sec_PKCS12AddAttributeToBag(p12ctxt, safeBag, SEC_OID_PKCS9_FRIENDLY_NAME, &nick) != SECSuccess) {
          goto err;
        }
      }
      if(keyId) {
        if(sec_PKCS12AddAttributeToBag(p12ctxt, safeBag, SEC_OID_PKCS9_LOCAL_KEY_ID, keyId) != SECSuccess) {
          goto err;
        }
      }

      // Append the cert safeBag 
      if(nestedDest) {
        rv = sec_pkcs12_append_bag_to_safe_contents(p12ctxt->arena,
             (sec_PKCS12SafeContents*)nestedDest, safeBag);
      }
      else {
        rv = sec_pkcs12_append_bag(p12ctxt, safe, safeBag);
      }
      if(rv != SECSuccess) goto err;
      PORT_ArenaUnmark(p12ctxt->arena, mark);
    }

    return SECSuccess;
err:
    if(mark) PORT_ArenaRelease(p12ctxt->arena, mark);
    if(certlist_tmp) CERT_DestroyCertList(certlist_tmp);
    if(certlist) CERT_DestroyCertList(certlist);
    return SECFailure;
  }

  //Our version of SEC_PKCS12AddCertAndKey
  SECStatus my_SEC_PKCS12AddCertAndKey(SEC_PKCS12ExportContext *p12ctxt,
                               void *certSafe, void *certNestedDest,
                               CERTCertificate *cert, CERTCertDBHandle *certDb,
                               void *keySafe, void *keyNestedDest,
                               PRBool shroudKey, SECItem *pwitem,
                               SECOidTag algorithm) {
    SECStatus rv = SECFailure;
    SGNDigestInfo *digest = NULL;
    void *mark = NULL;

    if(!p12ctxt || !certSafe || !keySafe || !cert) {
        return SECFailure;
    }

    mark = PORT_ArenaMark(p12ctxt->arena);

    //Generate the thumbprint of the cert to use as a keyId
    digest = sec_pkcs12_compute_thumbprint(&cert->derCert);
    if(!digest) {
      PORT_ArenaRelease(p12ctxt->arena, mark);
      return SECFailure;
    }

    // add the certificate 
    rv = my_SEC_PKCS12AddCert(p12ctxt, (SEC_PKCS12SafeInfo*)certSafe,
                           (SEC_PKCS12SafeInfo*)certNestedDest, cert, certDb,
                           &digest->digest);
    if(rv != SECSuccess) {
        goto loser;
    }

    /* add the key */
    rv = my_SEC_PKCS12AddKeyForCert(p12ctxt, (SEC_PKCS12SafeInfo*)keySafe,
                                 keyNestedDest, cert,
                                 shroudKey, algorithm, pwitem,
                                 &digest->digest, NULL );
    if(rv != SECSuccess) {
        goto loser;
    }

    SGN_DestroyDigestInfo(digest);

    PORT_ArenaUnmark(p12ctxt->arena, mark);
    return SECSuccess;

loser:
    SGN_DestroyDigestInfo(digest);
    PORT_ArenaRelease(p12ctxt->arena, mark);

    return SECFailure;
  }

  bool nssOutputPKCS12(Context& context, const std::string& certname, 
      const char* outfile, const char* slotpw, const char* p12pw) {
    std::string outstr;
    SECStatus rv;
    bool res;
    res = nssOutputPKCS12(context, certname, outstr, slotpw, p12pw);
    if(res) {
      rv = str2file(context, outstr, outfile);
    }
    if(rv == SECSuccess) {
      context.LogFormat(Context::LogVerbose, "Succeeded to output pk12 into file: %s", outfile);
      return true;
    }
    return false;
  }

  bool nssOutputPKCS12(Context& context, const std::string& certname, 
      std::string& outstr, const char* slotpw, const char* p12pw) {
    SEC_PKCS12ExportContext *p12ecx = NULL;
    SEC_PKCS12SafeInfo *keySafe = NULL, *certSafe = NULL;
    SECItem *pwitem = NULL;
    PK11SlotInfo *slot = NULL;
    CERTCertList* certlist = NULL;
    CERTCertListNode* node = NULL;
    CERTCertificate* issuercert = NULL;

    slot = PK11_GetInternalKeySlot();
    if (PK11_Authenticate(slot, PR_TRUE, (void*)slotpw) != SECSuccess) {
      context.LogFormat(Context::LogError, "Failed to authenticate to PKCS11 slot %s", PK11_GetSlotName(slot));
      //NSSUtilLogger.msg(ERROR, "Failed to authenticate to PKCS11 slot %s", PK11_GetSlotName(slot));
      goto err;
    }

    certlist = PK11_FindCertsFromNickname((char*)(certname.c_str()), (void*)slotpw);
    if(!certlist) {
      context.LogFormat(Context::LogError, "Failed to find certs by nickname: %s", certname.c_str());
      //NSSUtilLogger.msg(ERROR, "Failed to find certs by nickname: %s", certname.c_str());
      return false;
    }

    if((SECSuccess != CERT_FilterCertListForUserCerts(certlist)) || CERT_LIST_EMPTY(certlist)) {
      context.LogFormat(Context::LogError, "There is no user cert in the found by nick name %s", certname.c_str());
      //NSSUtilLogger.msg(ERROR, "There is no user cert in the found by nick name %s", certname.c_str());
      return false;
    }

    //Password for the output PKCS12 file.
    if(p12pw != NULL) {
      pwitem = SECITEM_AllocItem(NULL, NULL, PL_strlen(p12pw) + 1);
      memset(pwitem->data, 0, pwitem->len);
      memcpy(pwitem->data, p12pw, pwitem->len);
    }

    if(certlist) {
      CERTCertificate* cert = NULL;
      node = CERT_LIST_HEAD(certlist);
      if(node) cert = node->cert;
      if(cert) slot = cert->slot; 
      //use the slot from the first matching
      //certificate to create the context. This is for keygen
    }
    if(!slot) {
      context.Log(Context::LogError, "Cert does not have a slot");
      //NSSUtilLogger.msg(ERROR, "Cert does not have a slot");
      goto err;
    }

    p12ecx = SEC_PKCS12CreateExportContext(NULL, NULL, slot, (void*)slotpw);
    if(!p12ecx) {
      context.Log(Context::LogError, "Failed to create export context");
      //NSSUtilLogger.msg(ERROR, "Failed to create export context");
      goto err;
    }

    if(pwitem != NULL) {
      if(SEC_PKCS12AddPasswordIntegrity(p12ecx, pwitem, SEC_OID_SHA1) != SECSuccess) {
        
        context.Log(Context::LogError, "PKCS12 add password integrity failed");
        goto err;
      }
    }

    for(node = CERT_LIST_HEAD(certlist); !CERT_LIST_END(node,certlist); node = CERT_LIST_NEXT(node)) {
      CERTCertificate* cert = node->cert;
      if(!cert->slot) {
        context.Log(Context::LogError, "Cert does not have a slot");
        goto err;
      }

      keySafe = SEC_PKCS12CreateUnencryptedSafe(p12ecx);
      if(!SEC_PKCS12IsEncryptionAllowed() || PK11_IsFIPS() || !pwitem) {
        certSafe = keySafe;
      } else {
        certSafe = SEC_PKCS12CreatePasswordPrivSafe(p12ecx, pwitem,
            SEC_OID_PKCS12_V2_PBE_WITH_SHA1_AND_40_BIT_RC2_CBC);
      }

      if(!certSafe || !keySafe) {
        context.Log(Context::LogError, "Failed to create key or cert safe");
        goto err;
      }

      //issuercert = FindIssuerCert(cert);
      /*
      if(my_SEC_PKCS12AddCert(p12ecx, certSafe, NULL, cert,
                CERT_GetDefaultCertDB(), NULL) != SECSuccess) {
        context.Log(Context::LogError, "Failed to add cert and key");
        goto err;
      }
      */
      if(my_SEC_PKCS12AddCertAndKey(p12ecx, certSafe, NULL, cert,
                CERT_GetDefaultCertDB(), keySafe, NULL, !pwitem ? PR_FALSE : PR_TRUE, pwitem,
                SEC_OID_PKCS12_V2_PBE_WITH_SHA1_AND_3KEY_TRIPLE_DES_CBC)
                != SECSuccess) {
        context.Log(Context::LogError, "Failed to add cert and key");
        goto err;
      }
    }
    CERT_DestroyCertList(certlist);
    certlist = NULL;

    if(SEC_PKCS12Encode(p12ecx, p12u_export2str, &outstr)
               != SECSuccess) {
      context.Log(Context::LogError, "Failed to encode PKCS12");
      goto err;
    }
    context.Log(Context::LogInfo, "Succeded to export PKCS12");

    if(pwitem) SECITEM_ZfreeItem(pwitem, PR_TRUE);
    return true;

err:
    if (certlist) {
        CERT_DestroyCertList(certlist);
        certlist = NULL;
    }   
    if(pwitem) {
      SECITEM_ZfreeItem(pwitem, PR_TRUE);
    }
    return false;
  }
//-----------------End of output pk12---------------//

  static SECStatus DeleteCertOnly(Context& context, const char* certname) {
    SECStatus rv;
    CERTCertificate *cert;
    CERTCertDBHandle* handle;

    handle = CERT_GetDefaultCertDB();
    cert = CERT_FindCertByNicknameOrEmailAddr(handle, (char*)certname);
    if(!cert) {
      context.LogFormat(Context::LogInfo, "There is no certificate named %s found, the cert could be removed when generating CSR", certname);
      return SECSuccess;
    }
    rv = SEC_DeletePermCertificate(cert);
    CERT_DestroyCertificate(cert);
    if(rv) {
      context.Log(Context::LogError, "Failed to delete certificate");
    }
    return rv;
  }

  static SECStatus deleteKeyAndCert(Context& context, const char* privkeyname, 
      const char* passwd, bool delete_cert) {
    SECKEYPrivateKeyList* list;
    SECKEYPrivateKeyListNode* node;
    int count = 0;
    SECStatus rv;
    PK11SlotInfo* slot;

    slot = PK11_GetInternalKeySlot();

    if(!privkeyname) context.Log(Context::LogWarning, "The name of the private key to delete is empty");

    if(PK11_NeedLogin(slot)) {
      rv = PK11_Authenticate(slot, PR_TRUE, (void*)passwd);
      if(rv != SECSuccess) {
        context.LogFormat(Context::LogError, "Failed to authenticate to token %s.", PK11_GetTokenName(slot));
        return SECFailure;
      }
    }
 
    list = PK11_ListPrivKeysInSlot(slot, (char *)privkeyname, (void*)passwd);
    if(list == NULL) {
      context.LogFormat(Context::LogInfo, "There is no private with nick name %s exists in nss db", privkeyname);
      return SECFailure;
    }

    for(node=PRIVKEY_LIST_HEAD(list); !PRIVKEY_LIST_END(node,list); node=PRIVKEY_LIST_NEXT(node)) {
      char * keyname;
      static const char orphan[] = { "(orphan)" };
      CERTCertificate* cert;
      keyname = PK11_GetPrivateKeyNickname(node->key);
      if(!keyname || !keyname[0]) {
        PORT_Free((void *)keyname);
        keyname = NULL;
        cert = PK11_GetCertFromPrivateKey(node->key);
        if(cert) {
          if(cert->nickname && cert->nickname[0]) {
            keyname = PORT_Strdup(cert->nickname);
          } 
          else if(cert->emailAddr && cert->emailAddr[0]) {
            keyname = PORT_Strdup(cert->emailAddr);
          }
          CERT_DestroyCertificate(cert);
        }
      }
      if(!keyname || PL_strcmp(keyname, privkeyname)) {
        /* PKCS#11 module returned unwanted keys */
        PORT_Free((void *)keyname);
        continue;
      }
      cert = PK11_GetCertFromPrivateKey(node->key);
      if(cert && delete_cert){
        //Delete the private key and the cert related
        rv = PK11_DeleteTokenCertAndKey(cert, (void*)passwd);
        if(rv != SECSuccess) {
          context.Log(Context::LogError, "Failed to delete private key and cert");
          CERT_DestroyCertificate(cert); continue;
        }
        CERT_DestroyCertificate(cert);
      }
      else{
        //Delete the private key without deleting the cert related
        //rv = PK11_DeleteTokenPrivateKey(node->key, PR_FALSE);
        rv = PK11_DestroyTokenObject(node->key->pkcs11Slot, node->key->pkcs11ID);
        if(rv != SECSuccess) {
          context.Log(Context::LogError, "Failed to delete private key");
          continue;
        }
      }
      if(!keyname) keyname = (char *)orphan;
      if(keyname != (char *)orphan) PORT_Free((void *)keyname);
      count++;
    }
    SECKEY_DestroyPrivateKeyList(list);

    if(count == 0) {
      context.LogFormat(Context::LogWarning, "Can not find key with name: %s", privkeyname);
    }
    if(slot) PK11_FreeSlot(slot);
    return SECSuccess;
  }

  static SECStatus DeleteKeyOnly(Context& context, const char* privkeyname, 
    const char* passwd) {
    return deleteKeyAndCert(context, privkeyname, passwd, false);
  }

  static SECStatus DeleteKeyAndCert(Context& context, const char* privkeyname, 
      const char* passwd) {
    return deleteKeyAndCert(context, privkeyname, passwd, true);
  }

  static SECStatus DeleteCertAndKey(Context& context, const char* certname, 
      const char* passwd) {
    SECStatus rv;
    CERTCertificate* cert;
    PK11SlotInfo* slot;

    slot = PK11_GetInternalKeySlot();
    if(PK11_NeedLogin(slot)) {
      SECStatus rv = PK11_Authenticate(slot, PR_TRUE, (void*)passwd);
      if(rv != SECSuccess) {
        context.LogFormat(Context::LogError, "Failed to authenticate to token %s.", PK11_GetTokenName(slot));
        return SECFailure;
      }
    }
    cert = PK11_FindCertFromNickname((char*)certname, (void*)passwd);
    if(!cert) {
      PK11_FreeSlot(slot);
      return SECFailure;
    }
    rv = PK11_DeleteTokenCertAndKey(cert, (void*)passwd);
    if(rv != SECSuccess) {
      context.LogFormat(Context::LogError, "Failed to delete private key that attaches to cert: %s", certname);
    }
    CERT_DestroyCertificate(cert);
    PK11_FreeSlot(slot);
    return rv;
  }

  static bool InputPrivateKey(Context& context, std::vector<uint8>& output, 
      const std::string& privk_in) {
    PKCS8_PRIV_KEY_INFO* p8info = NULL;
    EVP_PKEY* pkey=NULL;

    BIO* in = NULL;
    in = BIO_new(BIO_s_mem());
    BIO_write(in, privk_in.c_str(), privk_in.length());

    //PW_CB_DATA cb_data;
    //cb_data.password = (passphrase.empty()) ? NULL : (void*)(passphrase.c_str());
    //cb_data.prompt_info = prompt_info.empty() ? NULL : prompt_info.c_str();
    //if(!(pkey = PEM_read_bio_PrivateKey(keybio, NULL, passwordcb, &cb_data))) {
    if(!(pkey = PEM_read_bio_PrivateKey(in, NULL, NULL, NULL))) {
      int reason = ERR_GET_REASON(ERR_peek_error());
      if(reason == PEM_R_BAD_BASE64_DECODE)
        context.Log(Context::LogError, "Can not read PEM private key: probably bad password");
      if(reason == PEM_R_BAD_DECRYPT)
        context.Log(Context::LogError, "Can not read PEM private key: failed to decrypt");
      if(reason == PEM_R_BAD_PASSWORD_READ)
        context.Log(Context::LogError, "Can not read PEM private key: failed to obtain password");
      if(reason == PEM_R_PROBLEMS_GETTING_PASSWORD)
        context.Log(Context::LogError,"Can not read PEM private key: failed to obtain password");
      context.Log(Context::LogError, "Can not read PEM private key");
    }
    BIO_free(in);

    PKCS8_PRIV_KEY_INFO *p8inf = NULL;
    if (pkey) {
      int p8_broken = PKCS8_OK;
      if (!(p8inf = EVP_PKEY2PKCS8_broken(pkey, p8_broken))) {
        context.Log(Context::LogError, "Failed to convert EVP_PKEY to PKCS8");
        EVP_PKEY_free(pkey);
        return false;
      }
      BIO* key = NULL; 
      key = BIO_new(BIO_s_mem());
      i2d_PKCS8_PRIV_KEY_INFO_bio(key, p8inf);
      std::string privk_out;
      for(;;) {
        char s[256];
        int l = BIO_read(key,s,sizeof(s));
        if(l <= 0) break;
        privk_out.append(s,l);
      }
      std::string::iterator it;
      for(it = privk_out.begin() ; it < privk_out.end(); it++) output.push_back(*it);
      BIO_free(key);
    }

    EVP_PKEY_free(pkey);
    PKCS8_PRIV_KEY_INFO_free(p8inf);
    return true;
  }

  static bool OutputPrivateKey(Context& context, const std::vector<uint8>& input, 
      std::string& privk_out) {
    std::stringstream strstream;
    std::vector<uint8>::const_iterator it;
    for(it = input.begin(); it != input.end(); it++) strstream<<(*it);

    BIO* key= NULL;
    PKCS8_PRIV_KEY_INFO* p8info = NULL;
    EVP_PKEY* pkey=NULL;
    key = BIO_new(BIO_s_mem());
    std::string str = strstream.str();
    BIO_write(key, str.c_str(), str.length());

    p8info = d2i_PKCS8_PRIV_KEY_INFO_bio(key, NULL);
    if(p8info == NULL) { context.Log(Context::LogError, "Failed to load privkey"); return false; }
    else context.Log(Context::LogInfo, "Succeeded to load PrivateKeyInfo");
    if(p8info) {
      pkey = EVP_PKCS82PKEY(p8info);
      if(pkey == NULL) { context.Log(Context::LogError, "Failed to convert PrivateKeyInfo to EVP_PKEY"); BIO_free(key); return false; }
      else context.Log(Context::LogInfo, "Succeeded to convert PrivateKeyInfo to EVP_PKEY");
    }
    BIO* out = NULL;
    //out = BIO_new_file (outfile, "wb");
    out = BIO_new(BIO_s_mem());
    //char* passout = "secretpw";
    //PEM_write_bio_PrivateKey(out, pkey, EVP_des_ede3_cbc(), NULL, 0, NULL, passout);
    PEM_write_bio_PrivateKey(out, pkey, NULL, NULL, 0, NULL, NULL);
    privk_out.clear();
    for(;;) {
      char s[256];
      int l = BIO_read(out,s,sizeof(s));
      if(l <= 0) break;
      privk_out.append(s,l);
    }
    BIO_free(key);
    BIO_free(out);
    PKCS8_PRIV_KEY_INFO_free(p8info);
    return true;
  }

  static bool ImportDERPrivateKey(Context& context, PK11SlotInfo* slot, 
      const std::vector<uint8>& input, const std::string& name) {
    SECKEYPrivateKey* privKey;
    SECItem der_private_key_info;
    SECStatus rv;
    der_private_key_info.data = const_cast<unsigned char*>(&input.front());
    der_private_key_info.len = input.size();
    //The private key is set to be used for 
    //key unwrapping, data decryption, and signature generation.
    SECItem nickname;
    nickname.data = (unsigned char*)(name.c_str());
    nickname.len = name.size();
    const unsigned int key_usage = KU_KEY_ENCIPHERMENT | KU_DATA_ENCIPHERMENT |
                                 KU_DIGITAL_SIGNATURE;
    rv =  PK11_ImportDERPrivateKeyInfoAndReturnKey(
      slot, &der_private_key_info, &nickname, NULL, PR_TRUE, PR_FALSE,
      key_usage, NULL, NULL);//&privKey, NULL);
    if(rv != SECSuccess) {
      context.Log(Context::LogError, "Failed to import private key");
      return false;
    }
    else context.Log(Context::LogInfo, "Succeeded to import private key");
    
    return true;
  }

  static bool GenerateKeyPair(Context& context, char* slotpw, 
      SECKEYPublicKey** pubk, SECKEYPrivateKey** privk, 
      std::string& privk_str, int keysize, const std::string& nick_str) {
    PK11RSAGenParams rsaParams;
    rsaParams.keySizeInBits = keysize;
    rsaParams.pe = 0x10001;
    SECStatus rv;

    PK11SlotInfo* slot = NULL;
    slot = PK11_GetInternalKeySlot();
    if(PK11_Authenticate(slot, PR_TRUE, slotpw) != SECSuccess) {
      context.Log(Context::LogError, "Failed to authenticate to key database");
      if(slot) PK11_FreeSlot(slot);
      return false;
    }

    *privk = PK11_GenerateKeyPair(slot, CKM_RSA_PKCS_KEY_PAIR_GEN, &rsaParams,
        pubk, PR_FALSE, PR_FALSE, NULL); 
    //Only by setting the "isPerm" parameter to be false, the private key can be export
    if(*privk != NULL && *pubk != NULL)
      context.Log(Context::LogDebug, "Succeeded to generate public/private key pair");
    else {
      context.Log(Context::LogError, "Failed to generate public/private key pair");
      if(slot) PK11_FreeSlot(slot);
      return false;
    }
    std::vector<uint8> output;
    if(!ExportPrivateKey(context, *privk, &output)) context.Log(Context::LogError, "Failed to export private key");
    OutputPrivateKey(context, output, privk_str);

    ImportDERPrivateKey(context, slot, output, nick_str);

    if(slot) PK11_FreeSlot(slot);
    return true;
  }

  bool nssImportPrivateKey(Context& context, const char* slotpw, 
      const std::string& keystr, const std::string& nick_str) {
    std::vector<uint8> input;
    InputPrivateKey(context, input, keystr);

    PK11SlotInfo* slot = NULL;
    slot = PK11_GetInternalKeySlot();
    // Authentication is required for importing private key 
    if(PK11_Authenticate(slot, PR_TRUE, (void*)slotpw) != SECSuccess) {
      context.Log(Context::LogError, "Failed to authenticate to key database");
      if(slot) PK11_FreeSlot(slot);
      return false;
    }
    DeleteKeyOnly(context, (nick_str.c_str()), slotpw);

    ImportDERPrivateKey(context, slot, input, nick_str);

    if(slot) PK11_FreeSlot(slot);
    return true;
  }

  bool nssImportPrivateKey(Context& context, const char* slotpw, 
      const char* keyfile, const std::string& nick_str) {
    BIO* key = NULL;
    key = BIO_new_file(keyfile, "r");
    std::string key_str;
    for(;;) {
      char s[256];
      int l = BIO_read(key,s,sizeof(s));
      if(l <= 0) break;
      key_str.append(s,l);
    }
    BIO_free_all(key);

    return nssImportPrivateKey(context, slotpw, key_str, nick_str);
  }

  bool nssGenerateCSR(Context& context, const std::string& privkey_name, 
      const std::string& dn, const char* slotpw, const char* outfile, 
      std::string& privk_str, bool ascii) {
    SECStatus rv = SECFailure;
    std::string outstr;
    bool res = nssGenerateCSR(context, privkey_name, dn, slotpw, outstr, privk_str, ascii);
    if(res) {
      rv = str2file(context, outstr, outfile);
    }
    if(rv == SECSuccess) {
      context.LogFormat(Context::LogVerbose, "Succeeded to output the cert req into file: %s", outfile);
      return true;
    }
    else return false;
  }

  bool nssGenerateCSR(Context& context, const std::string& privkey_name, 
      const std::string& dn, const char* slotpw, std::string& outstr, 
      std::string& privk_str, bool ascii) {
    CERTCertificateRequest* req = NULL;
    CERTSubjectPublicKeyInfo* spki;
    SECKEYPrivateKey* privkey = NULL;
    SECKEYPublicKey* pubkey = NULL;
    CERTName* name = NULL;
    int keybits = 1024;
    PRArenaPool* arena;
    SECItem* encoding;
    SECOidTag signAlgTag;
    SECStatus rv;
    SECItem result;

    std::string dn_str;
    dn_str = dn;
    if(dn_str.empty()) dn_str = "CN=bogusname,OU=EMI,O=Grid"; 
    // It seems that nss requires certname for creating cert request,
    // so a bogus name is set here, which is useless, and will be covered
    // by the signer using another name
    name = CERT_AsciiToName((char*)(dn_str.c_str()));
    if(name == NULL) {
      context.Log(Context::LogError, "Failed to create subject name");
      return false;
    }

    //Remove the existing private key and related cert in nss db
    rv = DeleteKeyAndCert(context, (privkey_name.c_str()), slotpw);

    if(!GenerateKeyPair(context, (char*)slotpw, &pubkey, &privkey, privk_str, keybits, privkey_name)) return false;
    //PK11_SetPrivateKeyNickname(privkey, privkey_name.c_str());    

    //privkey = SECKEY_CreateRSAPrivateKey(keybits, &pubkey, NULL);
    spki = SECKEY_CreateSubjectPublicKeyInfo(pubkey);

    req = CERT_CreateCertificateRequest(name, spki, NULL);
    if(req == NULL) {
      context.Log(Context::LogError, "Failed to create certificate request");
    }
    if(pubkey != NULL) {
      SECKEY_DestroyPublicKey(pubkey);
    }
    if(spki != NULL) {
      SECKEY_DestroySubjectPublicKeyInfo(spki);
    }
    if(name) CERT_DestroyName(name);

    //Output the cert request
    arena = PORT_NewArena(DER_DEFAULT_CHUNKSIZE);
    if(!arena ) {
      context.Log(Context::LogError, "Failed to call PORT_NewArena");
      return false;
    }
    //Encode the cert request 
    encoding = SEC_ASN1EncodeItem(arena, NULL, req, SEC_ASN1_GET(CERT_CertificateRequestTemplate));
    CERT_DestroyCertificateRequest(req);
    if (encoding == NULL){
      PORT_FreeArena (arena, PR_FALSE);
      context.Log(Context::LogError, "Failed to encode the cert request with DER format");
      return false;
    }
    //Sign the cert request
    signAlgTag = SEC_GetSignatureAlgorithmOidTag(privkey->keyType, SEC_OID_UNKNOWN);
    if (signAlgTag == SEC_OID_UNKNOWN) {
      PORT_FreeArena (arena, PR_FALSE);
      context.Log(Context::LogError, "Unknow key or hash type");
      return false;
    }
    rv = SEC_DerSignData(arena, &result, encoding->data, encoding->len, privkey, signAlgTag);
    if(rv) {
      PORT_FreeArena (arena, PR_FALSE);
      context.Log(Context::LogError, "Failed to sign the cert request");
      return false;
    }
  
    // Encode cert request with specified format
    std::string out;
    rv = secitem2str(context, &result, ascii, out);
    if(rv == SECSuccess) {
      if(ascii) {
        outstr.append(NS_CERTREQ_HEADER).append("\n").append(out).append("\n")
              .append(NS_CERTREQ_TRAILER).append("\n");
        rv = SECSuccess;
      }
      else {
        outstr.assign(out.c_str(), out.size());
        rv = SECSuccess;
      }
    }

    PORT_FreeArena (arena, PR_FALSE);
    if (privkey) {
        SECKEY_DestroyPrivateKey(privkey);
    }
    context.LogFormat(Context::LogVerbose, "Succeeded to output the cert req: %s", outstr.c_str());
    return true;
  }

  static CERTCertificateRequest* getCertRequest(Context& context, 
      const std::string& indata, bool ascii) {
    CERTCertificateRequest* req = NULL;
    CERTSignedData signed_data;
    PRArenaPool* arena = NULL;
    SECItem req_der;
    SECStatus rv;

    req_der.data = NULL;
    do {
      arena = PORT_NewArena(DER_DEFAULT_CHUNKSIZE);
      if (arena == NULL) { rv = SECFailure; break; }
      rv = readDERfromstr(context, &req_der, indata, ascii);
      if (rv) break;
      req = (CERTCertificateRequest*) PORT_ArenaZAlloc
                (arena, sizeof(CERTCertificateRequest));
      if (!req) { rv = SECFailure; break; }
      req->arena = arena;
      PORT_Memset(&signed_data, 0, sizeof(signed_data));
      rv = SEC_ASN1DecodeItem(arena, &signed_data,
               SEC_ASN1_GET(CERT_SignedDataTemplate), &req_der);
      if (rv) break;
      rv = SEC_ASN1DecodeItem(arena, req,
               SEC_ASN1_GET(CERT_CertificateRequestTemplate), &signed_data.data);
      if (rv) break;
      rv = CERT_VerifySignedDataWithPublicKeyInfo(&signed_data,
                &req->subjectPublicKeyInfo, NULL);
    } while (0);

    if (req_der.data)
      SECITEM_FreeItem(&req_der, PR_FALSE);

    if (rv) {
      context.Log(Context::LogError, "Certificate Request is invalid");
      if (arena) {
        PORT_FreeArena(arena, PR_FALSE);
      }
      req = NULL;
    }
    return req;
  }

  static CERTCertificateRequest* getCertRequest(Context& context, 
      const char* infile, bool ascii) {
    SECStatus rv;
    std::string filedata;

    // Read data from file
    rv = file2str(context, infile, filedata);
    if (rv != SECSuccess) {
      context.Log(Context::LogError, "Failed to read data from input file");
      return NULL;
    }

    return getCertRequest(context, filedata, ascii);
  }



#if !defined(WIN32) && !defined(MACOS)
// CERT_NameFromDERCert
// CERT_IssuerNameFromDERCert
// CERT_SerialNumberFromDERCert
// The above nss methods which we need to use have not been exposed by
// nss.def (the situation does not apply to MACOS, only to Win and Linux,
// strange, maybe *.def is not used by MACOS when packaging).
// Therefore we have two different solutions for Linux and Win.
// For Linux, the three methods are duplicated here.
// For Win, the code duplication does not work ( arcproxy always crashes
// when goes to these three method, the crash is from
// nssutil3.dll!SECOID_GetAlgorithmTag_Util, which I don't know how
// to solve it for now). So the other working solution is to change the
// nss.def file under nss source tree to add these three methods, so that
// these methods can be exposed to external.

  const SEC_ASN1Template SEC_SkipTemplate[] = {
      { SEC_ASN1_SKIP, 0, NULL, 0 }
  };
  SEC_ASN1_CHOOSER_IMPLEMENT(SEC_SkipTemplate)

  // Find the subjectName in a DER encoded certificate
  const SEC_ASN1Template SEC_CertSubjectTemplate[] = {
    { SEC_ASN1_SEQUENCE,
          0, NULL, sizeof(SECItem) },
    { SEC_ASN1_EXPLICIT | SEC_ASN1_OPTIONAL | SEC_ASN1_CONSTRUCTED |
          SEC_ASN1_CONTEXT_SPECIFIC | SEC_ASN1_XTRN | 0,
          0, SEC_ASN1_SUB(SEC_SkipTemplate), 0 },  /* version */
    { SEC_ASN1_SKIP, 0, NULL, 0 },          /* serial number */
    { SEC_ASN1_SKIP, 0, NULL, 0 },          /* signature algorithm */
    { SEC_ASN1_SKIP, 0, NULL, 0 },          /* issuer */
    { SEC_ASN1_SKIP, 0, NULL, 0 },          /* validity */
    { SEC_ASN1_ANY, 0, NULL, 0 },          /* subject */
    { SEC_ASN1_SKIP_REST, 0, NULL, 0 },
    { 0, 0, NULL, 0 }
  };
  SEC_ASN1_CHOOSER_IMPLEMENT(SEC_CertSubjectTemplate)

  //Find the issuerName in a DER encoded certificate
  const SEC_ASN1Template SEC_CertIssuerTemplate[] = {
    { SEC_ASN1_SEQUENCE,
          0, NULL, sizeof(SECItem) },
    { SEC_ASN1_EXPLICIT | SEC_ASN1_OPTIONAL | SEC_ASN1_CONSTRUCTED |
          SEC_ASN1_CONTEXT_SPECIFIC | SEC_ASN1_XTRN | 0,
          0, SEC_SkipTemplate, 0 },  /* version */
    { SEC_ASN1_SKIP, 0, NULL, 0 },          /* serial number */
    { SEC_ASN1_SKIP, 0, NULL, 0 },          /* signature algorithm */
    { SEC_ASN1_ANY, 0, NULL, 0 },          /* issuer */
    { SEC_ASN1_SKIP_REST, 0, NULL, 0 },
    { 0, 0, NULL, 0 }
  };
  SEC_ASN1_CHOOSER_IMPLEMENT(SEC_CertIssuerTemplate)

  //Find the serialNumber in a DER encoded certificate
  const SEC_ASN1Template SEC_CertSerialNumberTemplate[] = {
    { SEC_ASN1_SEQUENCE,
          0, NULL, sizeof(SECItem) },
    { SEC_ASN1_EXPLICIT | SEC_ASN1_OPTIONAL | SEC_ASN1_CONSTRUCTED |
          SEC_ASN1_CONTEXT_SPECIFIC | SEC_ASN1_XTRN | 0,
          0, SEC_SkipTemplate, 0 },  /* version */
    { SEC_ASN1_ANY, 0, NULL, 0 }, /* serial number */
    { SEC_ASN1_SKIP_REST, 0, NULL, 0 },
    { 0, 0, NULL, 0 }
  };
  SEC_ASN1_CHOOSER_IMPLEMENT(SEC_CertSerialNumberTemplate)

  /* Extract the subject name from a DER certificate
     This is a copy from nss code, due to the "undefined reference to" compiling issue
   */
  SECStatus my_CERT_NameFromDERCert(SECItem *derCert, SECItem *derName) {
    int rv;
    PRArenaPool *arena;
    CERTSignedData sd;
    void *tmpptr;

    arena = PORT_NewArena(DER_DEFAULT_CHUNKSIZE);
    if (!arena) return SECFailure;

    PORT_Memset(&sd, 0, sizeof(CERTSignedData));
    rv = SEC_QuickDERDecodeItem(arena, &sd, CERT_SignedDataTemplate, derCert);
    if (rv) goto loser;

    PORT_Memset(derName, 0, sizeof(SECItem));
    rv = SEC_QuickDERDecodeItem(arena, derName, SEC_CertSubjectTemplate, &sd.data);
    if (rv) goto loser;

    tmpptr = derName->data;
    derName->data = (unsigned char*)PORT_Alloc(derName->len);
    if (derName->data == NULL) goto loser;
    PORT_Memcpy(derName->data, tmpptr, derName->len);

    PORT_FreeArena(arena, PR_FALSE);
    return(SECSuccess);

  loser:
    PORT_FreeArena(arena, PR_FALSE);
    return(SECFailure);
  }

  SECStatus my_CERT_IssuerNameFromDERCert(SECItem *derCert, SECItem *derName) {
    int rv;
    PRArenaPool *arena;
    CERTSignedData sd;
    void *tmpptr;

    arena = PORT_NewArena(DER_DEFAULT_CHUNKSIZE);
    if (!arena) return(SECFailure);

    PORT_Memset(&sd, 0, sizeof(CERTSignedData));
    rv = SEC_QuickDERDecodeItem(arena, &sd, SEC_ASN1_GET(CERT_SignedDataTemplate), derCert);
    if (rv) goto loser;

    PORT_Memset(derName, 0, sizeof(SECItem));
    rv = SEC_QuickDERDecodeItem(arena, derName, SEC_ASN1_GET(SEC_CertIssuerTemplate), &sd.data);
    if (rv) goto loser;

    tmpptr = derName->data;
    derName->data = (unsigned char*)PORT_Alloc(derName->len);
    if (derName->data == NULL) goto loser;
    PORT_Memcpy(derName->data, tmpptr, derName->len);

    PORT_FreeArena(arena, PR_FALSE);
    return(SECSuccess);

  loser:
    PORT_FreeArena(arena, PR_FALSE);
    return(SECFailure);
  }

  SECStatus my_CERT_SerialNumberFromDERCert(SECItem *derCert, SECItem *derName) {
    int rv;
    PRArenaPool *arena;
    CERTSignedData sd;
    void *tmpptr;

    arena = PORT_NewArena(DER_DEFAULT_CHUNKSIZE);
    if (!arena) return(SECFailure);

    PORT_Memset(&sd, 0, sizeof(CERTSignedData));
    rv = SEC_QuickDERDecodeItem(arena, &sd, SEC_ASN1_GET(CERT_SignedDataTemplate), derCert);
    if (rv) goto loser;

    PORT_Memset(derName, 0, sizeof(SECItem));
    rv = SEC_QuickDERDecodeItem(arena, derName, SEC_ASN1_GET(SEC_CertSerialNumberTemplate), &sd.data);
    if(rv) goto loser;

    tmpptr = derName->data;
    derName->data = (unsigned char*)PORT_Alloc(derName->len);
    if (derName->data == NULL) goto loser;
    PORT_Memcpy(derName->data, tmpptr, derName->len);

    PORT_FreeArena(arena, PR_FALSE);
    return(SECSuccess);

  loser:
    PORT_FreeArena(arena, PR_FALSE);
    return(SECFailure);
  }
#endif // #if !defined(WIN32) && !defined(MACOS)

  void nssListUserCertificatesInfo(std::list<certInfo>& certInfolist) {
    CERTCertList* list;
    CERTCertificate* find_cert = NULL;
    CERTCertListNode* node;

    list = PK11_ListCerts(PK11CertListAll, NULL);
    for (node = CERT_LIST_HEAD(list); !CERT_LIST_END(node,list);
        node = CERT_LIST_NEXT(node)) {
      CERTCertificate* cert = node->cert;
      const char* nickname = (const char*)node->appData;
      if (!nickname) {
        nickname = cert->nickname;
      }
      if(nickname == NULL) continue;
      PRBool isUser = CERT_IsUserCert(cert);
      if(!isUser) continue;

      certInfo cert_info;
      cert_info.certname = nickname;

      SECStatus rv;
      std::string subject_dn;
      SECItem derSubject;

#if !defined(WIN32) && !defined(MACOS)
      rv = my_CERT_NameFromDERCert(&cert->derCert, &derSubject);
#else
      rv = CERT_NameFromDERCert(&cert->derCert, &derSubject);
#endif

      if(rv == SECSuccess) {
        char* subjectName = CERT_DerNameToAscii(&derSubject);
        subject_dn = subjectName;
        if(subjectName) PORT_Free(subjectName);
        cert_info.subject_dn = subject_dn;
      }

      std::string issuer_dn;
      SECItem derIssuer;

#if !defined(WIN32) && !defined(MACOS)
      rv = my_CERT_IssuerNameFromDERCert(&cert->derCert, &derIssuer);
#else
      rv = CERT_IssuerNameFromDERCert(&cert->derCert, &derIssuer);
#endif
      if(rv == SECSuccess) {
        char* issuerName = CERT_DerNameToAscii(&derIssuer);
        issuer_dn = issuerName;
        if(issuerName) PORT_Free(issuerName);
        cert_info.issuer_dn = issuer_dn;
      }

      cert_info.serial = 0;
      std::string serial;
      SECItem derSerial;

#if !defined(WIN32) && !defined(MACOS)
      rv = my_CERT_SerialNumberFromDERCert (&cert->derCert, &derSerial);
#else
      rv = CERT_SerialNumberFromDERCert (&cert->derCert, &derSerial);
#endif

      if(rv == SECSuccess) {
        SECItem decodedValue;
        decodedValue.data = NULL;
        rv = SEC_ASN1DecodeItem (NULL, &decodedValue,
                                SEC_ASN1_GET(SEC_IntegerTemplate),
                                &derSerial);
        if (rv == SECSuccess) {
          unsigned long res;
          rv = SEC_ASN1DecodeInteger(&decodedValue, &res);
          if(rv == SECSuccess) cert_info.serial = res;
        }
      }

      PRTime notBefore, notAfter;
      rv = CERT_GetCertTimes(cert, &notBefore, &notAfter);
      if(rv == SECSuccess) {
        cert_info.start = AuthN::Utils::Time(notBefore/1000/1000);
        cert_info.end = AuthN::Utils::Time(notAfter/1000/1000);
        certInfolist.push_back(cert_info);
      }
    }
    if (list) {
      CERT_DestroyCertList(list);
    }
  }

  static SECStatus copy_CERTName(CERTName* destName, CERTName* srcName) {
    PRArenaPool* poolp = NULL;
    SECStatus rv;
    if (destName->arena != NULL) {
      poolp = destName->arena;
    } else {
      poolp = PORT_NewArena(SEC_ASN1_DEFAULT_ARENA_SIZE);
    }
    if (poolp == NULL) {
      return SECFailure;
    }
    destName->arena = NULL;
    rv = CERT_CopyName(poolp, destName, srcName);
    destName->arena = poolp;
    return rv;
  }

  bool nssCreateCert(Context& context, const char* csrfile, 
      const std::string& issuername, const char* passwd, 
      const int duration, const std::string& vomsacseq, 
      const char* outfile, bool ascii) {
    SECStatus rv = SECFailure;
    std::string csrstr;
    std::string outstr;
    // Read data from csr file
    rv = file2str(context, csrfile, csrstr);
    if (rv != SECSuccess) {
      context.Log(Context::LogError, "Failed to read data from csr file");
      return false;
    }

    bool res = nssCreateCert(context, csrstr, issuername, passwd, duration, vomsacseq, outstr, ascii);
    if(res) rv = str2file(context, outstr, outfile);
    if(rv == SECSuccess) {
      context.LogFormat(Context::LogVerbose, "Succeeded to output the cert req into file: %s", outfile);
      return true;
    }
    else return false;
  }

  bool nssCreateCert(Context& context, const std::string& csrstr, 
      const std::string& issuername, const char* passwd, 
      const int duration, const std::string& vomsacseq, 
      std::string& outstr, bool ascii) {
    CERTCertDBHandle* certhandle;
    CERTCertificate* issuercert = NULL;
    SECKEYPrivateKey* issuerkey = NULL;
    CERTValidity* validity;
    CERTCertificate* cert = NULL;
    PRExplodedTime extime;
    PRTime now, start, end;
    int serialnum;
    CERTCertificateRequest* req = NULL;
    void* ext_handle; 
    PRArenaPool* arena;
    SECOidTag tag_sigalg;
    SECOidTag tag_hashalg;
    int pathlen = -1;
    char* policylang = (char*)"Inherit all"; //TODO
    char* policy = NULL;//"test policy"; //TODO
    CERTCertExtension** exts;
    SECItem cert_der;
    SECItem derSubject;
    void* dummy;
    SECItem* signed_cert = NULL;
    SECStatus rv = SECSuccess;
    bool ret = false;

    req = getCertRequest(context, csrstr, ascii);
    if(!req) {
      context.LogFormat(Context::LogError, "Failed to parse certificate request from CSR content: %s", csrstr.c_str());
      return false;
    }
    
    certhandle = CERT_GetDefaultCertDB();
    issuercert = CERT_FindCertByNicknameOrEmailAddr(certhandle, (char*)(issuername.c_str()));
    if(!issuercert) {
      context.LogFormat(Context::LogError, "Can not find certificate with name %s", issuername.c_str());
      return false;
    }

    now = PR_Now();
    PR_ExplodeTime(now, PR_GMTParameters, &extime);
    extime.tm_min  -= 5;
    start = PR_ImplodeTime(&extime);
    extime.tm_min +=5;
    extime.tm_hour += duration;
    end = PR_ImplodeTime (&extime);
    validity = CERT_CreateValidity(start, end);
/*
    now = PR_Now();
    PR_ExplodeTime(now, PR_GMTParameters, &extime);
    extime.tm_hour += duration;
    after = PR_ImplodeTime (&extime);
    validity = CERT_CreateValidity (now, after);
*/

    //Subject
#if !defined(WIN32) && !defined(MACOS)
    my_CERT_NameFromDERCert(&issuercert->derCert, &derSubject);
#else
    CERT_NameFromDERCert(&issuercert->derCert, &derSubject);
#endif

    char* subjectName = CERT_DerNameToAscii(&derSubject);
    std::string subname_str = subjectName;

    srand(time(NULL));
    unsigned long random_cn;
    random_cn = rand();
    char* CN_name = NULL;
    CN_name = (char*)malloc(sizeof(long)*4 + 1);
    snprintf(CN_name, sizeof(long)*4 + 1, "%ld", random_cn);
    std::string str = "CN="; str.append(CN_name); str.append(",");
    subname_str.insert(0, str.c_str(), str.length());
    context.LogFormat(Context::LogDebug, "Proxy subject: %s", subname_str.c_str());
    if(CN_name) free(CN_name);
    CERTName* subName = CERT_AsciiToName((char*)(subname_str.c_str()));
    if(subjectName) PORT_Free(subjectName);

    if (validity) {
      if(subName != NULL) {
        rv = copy_CERTName(&req->subject, subName);
      }
      cert = CERT_CreateCertificate(rand(), &issuercert->subject, validity, req);

      CERT_DestroyValidity(validity);
      if(subName) CERT_DestroyName(subName);
    }

    //Extensions
    ext_handle = CERT_StartCertExtensions (cert);
    if (ext_handle == NULL) {
      context.Log(Context::LogError, "Failed to start cert extension");
      goto error;
    }
    if(AddProxyCertInfoExtension(context, ext_handle, pathlen, policylang, policy) != SECSuccess) {
      context.Log(Context::LogError, "Failed to add proxy cert info extension");
      goto error;
    }
    if((!vomsacseq.empty()) && (AddVOMSACSeqExtension(ext_handle, (char*)(vomsacseq.c_str()), vomsacseq.length()) != SECSuccess)) {
      context.Log(Context::LogError, "Failed to add voms AC extension");
      goto error;
    }

    if(req->attributes != NULL &&
       req->attributes[0] != NULL &&
       req->attributes[0]->attrType.data != NULL &&
       req->attributes[0]->attrType.len   > 0    &&
      SECOID_FindOIDTag(&req->attributes[0]->attrType)
            == SEC_OID_PKCS9_EXTENSION_REQUEST) {
      rv = CERT_GetCertificateRequestExtensions(req, &exts);
      if(rv != SECSuccess) goto error;
      rv = CERT_MergeExtensions(ext_handle, exts);
      if (rv != SECSuccess) goto error;
    }
    CERT_FinishExtensions(ext_handle);

    //Sign the certificate
    issuerkey = PK11_FindKeyByAnyCert(issuercert, (char*)passwd);
    if(issuerkey == NULL) {
      context.LogFormat(Context::LogError, "Failed to retrieve private key for proxy issuer: %s", issuername.c_str());
      goto error;
    }
    arena = cert->arena;
    tag_hashalg = SEC_OID_SHA1;
    tag_sigalg = SEC_GetSignatureAlgorithmOidTag(issuerkey->keyType, tag_hashalg);
    if (tag_sigalg == SEC_OID_UNKNOWN) {
      context.Log(Context::LogError, "Unknown key or hash type of issuer");
      goto error;
    }

    rv = SECOID_SetAlgorithmID(arena, &cert->signature, tag_sigalg, 0);
    if(rv != SECSuccess) {
      context.Log(Context::LogError, "Failed to set signature algorithm id");
      goto error;
    }

    *(cert->version.data) = 2;
    cert->version.len = 1;

    cert_der.len = 0;
    cert_der.data = NULL;
    dummy = SEC_ASN1EncodeItem (arena, &cert_der, cert,
                                SEC_ASN1_GET(CERT_CertificateTemplate));
    if (!dummy) {
      context.Log(Context::LogError, "Failed to encode certificate");
      goto error;
    }

    signed_cert = (SECItem *)PORT_ArenaZAlloc(arena, sizeof(SECItem));
    if(signed_cert == NULL) {
      context.Log(Context::LogError, "Failed to allocate item for certificate data");
      goto error;
    }

    rv = SEC_DerSignData(arena, signed_cert, cert_der.data, cert_der.len, issuerkey, tag_sigalg);
    if(rv != SECSuccess) {
      context.Log(Context::LogError, "Failed to sign encoded certificate data");
      //signed_cert will be freed when arena is freed.
      signed_cert = NULL;
      goto error;
    }
    cert->derCert = *signed_cert;

    rv = output_cert2str(context, cert, ascii, outstr);
    if(rv == SECSuccess) { 
      context.LogFormat(Context::LogVerbose, "Succeeded to output certificate: %s", outstr.c_str());
      ret = true;
    }

error:
    if(issuerkey) SECKEY_DestroyPrivateKey(issuerkey);
    if(issuercert) CERT_DestroyCertificate(issuercert);
    if(req) CERT_DestroyCertificateRequest(req);

    return ret;
  } 

  bool nssImportCert(Context& context, const char* slotpw, 
      const std::string& certstr, const std::string& name, 
      const char* trusts, bool ascii) {
    PK11SlotInfo* slot = NULL;
    CERTCertDBHandle* certhandle;
    CERTCertTrust* trust = NULL;
    CERTCertificate* cert = NULL;
    SECItem certder;
    SECItem subj_name;
    SECStatus rv = SECSuccess;

    slot = PK11_GetInternalKeySlot();
    certhandle = CERT_GetDefaultCertDB();

    certder.data = NULL;
    do {
      // Delete the existing cert with the same name
      if(!name.empty())
        rv = DeleteCertOnly(context, name.c_str());

      rv = readDERfromstr(context, &certder, certstr, ascii);
      if(rv != SECSuccess) {
        context.Log(Context::LogError, "Failed to read input certificate file");
        break;
      }
      // It seems CERT_DecodeCertFromPackage can consume both ascii and binary 
      // format of certificate, therefore we may not need to explicitly give
      // if the cert is ascii or not
      cert = CERT_DecodeCertFromPackage((char *)certder.data, certder.len);
      if(cert == NULL) {
        context.LogFormat(Context::LogError, "Failed to get certificate from certificate string:\n%s", certstr.c_str());
        rv = SECFailure; break;
      }

      std::string name_str;
      if(name.empty()) {
        // If the name is not given, then use the "CN" value of the subject as the name
        // Get the subject of the certificate
        subj_name = cert->derSubject;
        char* s_name = CERT_DerNameToAscii(&subj_name);
        std::string subj_str;
        if(s_name) { subj_str = s_name; PORT_Free(s_name); }
        std::string::size_type pos1, pos2;
        pos1 = subj_str.find("CN=");
        pos2 = subj_str.find(",");
        name_str = subj_str.substr(pos1+3, pos2-pos1-3);
        // Delete the existing cert with the same name
        rv = DeleteCertOnly(context, name_str.c_str());
      }
      else name_str = name;

      //Create a cert trust
      trust = (CERTCertTrust *)PORT_ZAlloc(sizeof(CERTCertTrust));
      if(!trust) {
        context.Log(Context::LogError, "Failed to allocate cert trust");
        rv = SECFailure; break;
      }
      rv = CERT_DecodeTrustString(trust, (char*)trusts);
      if(rv) {
        context.Log(Context::LogError, "Failed to decode trust string");
        rv = SECFailure; break;
      }
    
      //Import the certificate
      rv = PK11_ImportCert(slot, cert, CK_INVALID_HANDLE, 
        (char*)(name_str.c_str()), PR_FALSE);
      if(rv != SECSuccess) {
        if(PORT_GetError() == SEC_ERROR_TOKEN_NOT_LOGGED_IN) {
          if(PK11_Authenticate(slot, PR_TRUE, (void*)slotpw) != SECSuccess) {
            context.LogFormat(Context::LogError, "Failed to authenticate to token %s", PK11_GetTokenName(slot));
            rv = SECFailure; break;
          }
          rv = PK11_ImportCert(slot, cert, CK_INVALID_HANDLE, 
            (char*)(name_str.c_str()), PR_FALSE);
          if(rv != SECSuccess) {
            context.Log(Context::LogError, "Failed to add certificate to token or database");
            break;
          }
          else context.Log(Context::LogInfo, "Succeeded to import certificate");
        }
      }
      else context.Log(Context::LogInfo, "Succeeded to import certificate");

      rv = CERT_ChangeCertTrust(certhandle, cert, trust);
      if(rv != SECSuccess) {
        if (PORT_GetError() == SEC_ERROR_TOKEN_NOT_LOGGED_IN) {
          if(PK11_Authenticate(slot, PR_TRUE, (void*)slotpw) != SECSuccess) {
            context.LogFormat(Context::LogError, "Failed to authenticate to token %s", PK11_GetTokenName(slot));
            rv = SECFailure; break;
          }
          rv = CERT_ChangeCertTrust(certhandle, cert, trust);
          if(rv != SECSuccess) {
            context.Log(Context::LogError, "Failed to add certificate to token or database");
            break;
          }
          else context.LogFormat(Context::LogInfo, "Succeeded to change trusts to: %s", trusts);
        }
      }
      else context.LogFormat(Context::LogInfo, "Succeeded to change trusts to: %s", trusts);
    } while (0);

    PK11_FreeSlot(slot);
    CERT_DestroyCertificate (cert);
    PORT_Free(trust);
    PORT_Free(certder.data);
    if(rv == SECSuccess) return true;
    else return false;
  }

  bool nssImportCert(Context& context, const char* slotpw, 
      const char* certfile, const std::string& name, 
      const char* trusts, bool ascii) {
    SECStatus rv;
    std::string certstr;
    // Read data from cert file
    rv = file2str(context, certfile, certstr);
    if (rv != SECSuccess) {      
      context.Log(Context::LogError, "Failed to read data from cert file");
      return false;
    }

    return nssImportCert(context, slotpw, certstr, name, trusts, ascii);
  }

  bool nssImportCertAndPrivateKey(Context& context, const char* slotpw, 
      const std::string& keystr, const std::string& keyname, 
      const std::string& certstr, const std::string& certname, 
      const char* trusts, bool ascii) {
    bool res;
    res = nssImportPrivateKey(context, slotpw, keystr, keyname);
    if(!res) { context.LogFormat(Context::LogError, "Failed to import private key from str: %s", keystr.c_str()); return false; }
    res = nssImportCert(context, slotpw, certstr, certname, trusts, ascii);
    if(!res) { context.LogFormat(Context::LogError, "Failed to import certificate from str: %s", certstr.c_str()); return false; }
    return true;
  }

  bool nssImportCertAndPrivateKey(Context& context, const char* slotpw, 
      const char* keyfile, const std::string& keyname, 
      const char* certfile, const std::string& certname, 
      const char* trusts, bool ascii) { 
    bool res;
    res = nssImportPrivateKey(context, slotpw, keyfile, keyname);
    if(!res) { context.LogFormat(Context::LogError, "Failed to import private key from file: %s", keyfile); return false; }
    res = nssImportCert(context, slotpw, certfile, certname, trusts, ascii);
    if(!res) { context.LogFormat(Context::LogError, "Failed to import certificate from file: %s", certfile); return false; }
    return true;
  }

  static CERTCertificate* getCert(Context& context, const std::string& certdata, 
      PRBool isAscii) {
    CERTCertificate* cert = NULL;
    CERTCertDBHandle* defaultdb = NULL;
    SECStatus rv;
    SECItem item; 

    defaultdb = CERT_GetDefaultCertDB();

    item.data = NULL;
    rv = readDERfromstr(context, &item, certdata, isAscii);
    if(rv != SECSuccess) {
      context.LogFormat(Context::LogError, "Failed to read input certificate file");
      return NULL;
    }
    if(!item.len) {
      context.LogFormat(Context::LogError, "Cert data is empty");
      return NULL;
    }

    cert = CERT_NewTempCertificate(defaultdb, &item, NULL, PR_FALSE, PR_TRUE);
    if(!cert) {
      PRErrorCode err = PR_GetError();
      context.LogFormat(Context::LogError, "Failed to import certificate: %d - %s", err, PORT_ErrorToString(err));
    }
    PORT_Free(item.data);
    return cert;
  }

  static CERTCertificate* getCert(Context& context, const char* certfile, PRBool isAscii) {
    SECStatus rv;
    std::string filedata;

    // Read data from file
    rv = file2str(context, certfile, filedata);
    if (rv != SECSuccess) {
      context.Log(Context::LogError, "Failed to read data from input file");
      return NULL;
    }

    return getCert(context, filedata, isAscii);
  }

  static char* bestCertName(CERTCertificate *cert) {
    if (cert->nickname) return cert->nickname;
    if (cert->emailAddr && cert->emailAddr[0]) return cert->emailAddr;
    return cert->subjectName;
  }

  static void displayVerifyLog(Context& context, CERTVerifyLog* log, PRBool verbose) {
    CERTVerifyLogNode* node   = NULL;
    unsigned int depth = (unsigned int)-1;
    uintptr_t flags  = 0;
    std::string errstr;
    std::string log_str;

    if(log->count > 0) {
      log_str.append("PROBLEM WITH THE CERT CHAIN:\n");
      for(node = log->head; node; node = node->next) {
        if (depth != node->depth) {
          depth = node->depth;
          log_str.append("CERT ").append(tostring(depth)).append(". ")
                 .append(bestCertName(node->cert)).append(" ")
                 .append(depth ? "[Certificate Authority]": "").append(":\n");
          if(verbose) {
            const char* emailAddr;
            emailAddr = CERT_GetFirstEmailAddress(node->cert);
            if (emailAddr) {
              log_str.append("Email Address(es): ");
              do {
                log_str.append(emailAddr).append("\n");
                emailAddr = CERT_GetNextEmailAddress(node->cert, emailAddr);
              } while (emailAddr);
            }
          }
        }
        log_str.append("  ERROR ").append(tostring(node->error)).append(": ").append(PORT_ErrorToString(node->error)).append("\n");
        switch (node->error) {
          case SEC_ERROR_INADEQUATE_KEY_USAGE:
            flags = (uintptr_t)node->arg;
            switch (flags) {
              case KU_DIGITAL_SIGNATURE:
                errstr = "Cert cannot sign.";
                break;
              case KU_KEY_ENCIPHERMENT:
                errstr = "Cert cannot encrypt.";
                break;
              case KU_KEY_CERT_SIGN:
                errstr = "Cert cannot sign other certs.";
                break;
              default:
                errstr = "[unknown usage].";
                break;
            }
          case SEC_ERROR_INADEQUATE_CERT_TYPE:
            flags = (uintptr_t)node->arg;
            switch (flags) {
              case NS_CERT_TYPE_SSL_CLIENT:
              case NS_CERT_TYPE_SSL_SERVER:
                errstr = "Cert cannot be used for SSL.";
                break;
              case NS_CERT_TYPE_SSL_CA:
                errstr = "Cert cannot be used as an SSL CA.";
                break;
              case NS_CERT_TYPE_EMAIL:
                errstr = "Cert cannot be used for SMIME.";
                break;
              case NS_CERT_TYPE_EMAIL_CA:
                errstr = "Cert cannot be used as an SMIME CA.";
                break;
              case NS_CERT_TYPE_OBJECT_SIGNING:
                errstr = "Cert cannot be used for object signing.";
                break;
              case NS_CERT_TYPE_OBJECT_SIGNING_CA:
                errstr = "Cert cannot be used as an object signing CA.";
                break;
              default:
                errstr = "[unknown usage].";
                break;
            }
          case SEC_ERROR_UNKNOWN_ISSUER:
          case SEC_ERROR_UNTRUSTED_ISSUER:
          case SEC_ERROR_EXPIRED_ISSUER_CERTIFICATE:
            errstr = node->cert->issuerName;
            break;
          default:
            break;
        }
        if(!errstr.empty()) log_str.append("    ").append(errstr).append("\n");
      }
    }
    context.Log(Context::LogInfo, log_str);
  }

  static void printName(Context& context, CERTName* name, const char* msg) {
    char* nameStr = NULL;
    std::string str;
    std::string log_str;

    if(!name) {
      PORT_SetError(SEC_ERROR_INVALID_ARGS);
      return;
    }
    if(!name->rdns || !name->rdns[0]) {
      str = "(empty)";
    } else {
      str = nameStr = CERT_NameToAscii(name);
    }
    if(str.empty()) {
      str = "!Invalid AVA!";
    }

    log_str.append(msg).append(": ").append(str);

    PORT_Free(nameStr);
  }

  // CERTRevocationFlags starts to exist from 3.12 (diff between 3.11.9 and 3.12)
#if NSS_VMAJOR >=3 && NSS_VMINOR >=12
  SECStatus configureRevocationParams(CERTRevocationFlags* flags, bool testType, bool methodType, const char* testFlagStr, const char* methodFlagStr) {
    // testType: true --- full chain check, false --- only leaf check
    // methodType: true --- crl check, false --- ocsp check
    // testFlagStr: Sets revocation flags for the test type it follows. 
    //              Possible flags: "testLocalInfoFirst" and "requireFreshInfo".
    // methodFlagStr: Sets revocation flags for the method it follows.
    //              Possible types are "doNotUse", "forbidFetching",
    //              "ignoreDefaultSrc", "requireInfo" and "failIfNoInfo".

    int i;
    static CERTRevocationTests* revTests = NULL;
    PRUint64* revFlags;

    if(testType) revTests = &flags->chainTests;
    else revTests = &flags->leafTests;

    revTests->number_of_preferred_methods = 0;
    revTests->preferred_methods = 0;
    revFlags = revTests->cert_rev_flags_per_method;

    revTests->number_of_defined_methods = cert_revocation_method_count;

    uint testFlags = 0;
    if (testFlagStr) {
      if (PORT_Strstr(testFlagStr, "testLocalInfoFirst"))
        testFlags |= CERT_REV_MI_TEST_ALL_LOCAL_INFORMATION_FIRST;
      else if (PORT_Strstr(testFlagStr, "requireFreshInfo"))
        testFlags |= CERT_REV_MI_REQUIRE_SOME_FRESH_INFO_AVAILABLE;
    }
    revTests->cert_rev_method_independent_flags |= testFlags;

    uint methodFlags = 0;
    if (methodFlagStr) {
      if (!PORT_Strstr(methodFlagStr, "doNotUse"))
        methodFlags |= CERT_REV_M_TEST_USING_THIS_METHOD;
      else if (PORT_Strstr(methodFlagStr, "forbidFetching"))
        methodFlags |= CERT_REV_M_FORBID_NETWORK_FETCHING;
      else if (PORT_Strstr(methodFlagStr, "ignoreDefaultSrc"))
        methodFlags |= CERT_REV_M_IGNORE_IMPLICIT_DEFAULT_SOURCE;
      else if (PORT_Strstr(methodFlagStr, "requireInfo"))
        methodFlags |= CERT_REV_M_REQUIRE_INFO_ON_MISSING_SOURCE;
      else if (PORT_Strstr(methodFlagStr, "failIfNoInfo"))
        methodFlags |= CERT_REV_M_FAIL_ON_MISSING_FRESH_INFO;
    } else {
      methodFlags |= CERT_REV_M_TEST_USING_THIS_METHOD;
    }

    if(methodType) {
      revFlags[cert_revocation_method_crl] = methodFlags;
    } else {
      revFlags[cert_revocation_method_ocsp] = methodFlags;
    }
    return SECSuccess;
  }
#endif

  /* Check if one certificate is the issuer of the other one*/
  static SECStatus checkIssued(CERTCertificate* cert, CERTCertificate* issuer) {
    SECStatus rv;
    void* wincx = NULL;
    if(!cert || !issuer) return SECFailure;

    rv = CERT_VerifySignedData(&cert->signatureWrap, issuer, PR_Now(), wincx);
    if(rv != SECSuccess) {
      if(PORT_GetError() == SEC_ERROR_EXPIRED_CERTIFICATE)
        PORT_SetError(SEC_ERROR_EXPIRED_ISSUER_CERTIFICATE);
      else PORT_SetError(SEC_ERROR_BAD_SIGNATURE);
    }
    return rv;
  }

  /* Find the issuer from a list*/
  static CERTCertificate* findIssuer(CERTCertList* certList, CERTCertificate* cert) {
    CERTCertificate* tmpcert = NULL;
    CERTCertListNode* node;
    if(certList == NULL || CERT_LIST_EMPTY(certList)) return NULL;

    node = CERT_LIST_HEAD(certList);
    while (!CERT_LIST_END(node, certList)) {
      tmpcert = node->cert;
      if(checkIssued(cert, tmpcert) == SECSuccess) break;
      else tmpcert = NULL;
      node = CERT_LIST_NEXT(node);
    }
    return tmpcert;
  }

  /* Shuffle the input cert list with the sequence of issuing, 
   * starting from the issuer that issues the given certificate */
  static CERTCertList* shuffleChain(CERTCertList* certList, CERTCertificate* cert) {
    CERTCertificate* issuer = NULL;
    CERTCertificate* issued = NULL;
    CERTCertList* list;
    if(certList == NULL || CERT_LIST_EMPTY(certList)) return NULL;
    if(cert == NULL) return NULL;   

    issued = cert;
    list = CERT_NewCertList();
    while(1) {
      issuer = findIssuer(certList, issued);
      if(issuer == NULL) break;
      CERT_AddCertToListTail(list, CERT_DupCertificate(issuer));
      issued = issuer;
    }

    return list;
  }

//  function CERT_PKIXVerifyCert was only added since 3.12
#if NSS_VMAJOR >=3 && NSS_VMINOR >=12
  static bool nssPKIXVerifyCert(Context& context, char* slotpw, 
      CERTCertificate* cert, CERTCertList* untrustedCertList, 
      CERTCertList* trustedCertList, SECOidTag* cert_policy) {
    PRBool certFetching = PR_FALSE;
    PRTime time = 0;

    SECStatus rv;
    SECCertificateUsage certUsage = certificateUsageSSLServer;
    CERTVerifyLog log;
    CERTCertificate* issuerCert = NULL;
    CERTCertList* builtChain = NULL;
    PRBool verbose = PR_TRUE;
    bool ret;

    log.arena = PORT_NewArena(512);
    log.head = log.tail = NULL;
    log.count = 0;

    do {
      static CERTValOutParam cvout[4];
      static CERTValInParam cvin[6];
      int inParamIndex = 0;
      static PRUint64 revFlagsLeaf[2];
      static PRUint64 revFlagsChain[2];
      static CERTRevocationFlags rev;

      if ((cert_policy != NULL) && (*cert_policy != SEC_OID_UNKNOWN)) {
        cvin[inParamIndex].type = cert_pi_policyOID;
        cvin[inParamIndex].value.arraySize = 1;
        cvin[inParamIndex].value.array.oids = cert_policy;
        inParamIndex++;
      }

      if (trustedCertList) {
        cvin[inParamIndex].type = cert_pi_trustAnchors;
        cvin[inParamIndex].value.pointer.chain = trustedCertList;
        inParamIndex++;
      }

      // cert_pi_useAIACertFetch only starts to exist from 3.12.1.1 (diff between 3.12 and 3.12.1.1)
#if NSS_VMAJOR >=3 && NSS_VMINOR >=12 && NSS_VPATCH >=1 && NSS_VBUILD >= 1
      cvin[inParamIndex].type = cert_pi_useAIACertFetch;
      cvin[inParamIndex].value.scalar.b = certFetching;
      inParamIndex++;
#endif

      rev.leafTests.cert_rev_flags_per_method = revFlagsLeaf;
      rev.chainTests.cert_rev_flags_per_method = revFlagsChain;
      rv = configureRevocationParams(&rev, false, true, "testLocalInfoFirst", "doNotUse");//"requireFreshInfo", "requireInfo");
      //"forbidFetching", "ignoreDefaultSrc", "requireInfo" and "failIfNoInfo"
      if (rv) {
        context.Log(Context::LogError, "Can not config revocation parameters");
        break;
      }
      cvin[inParamIndex].type = cert_pi_revocationFlags;
      cvin[inParamIndex].value.pointer.revocation = &rev;
      inParamIndex++;

      if (time) {
        cvin[inParamIndex].type = cert_pi_date;
        cvin[inParamIndex].value.scalar.time = time;
        inParamIndex++;
      }

      cvin[inParamIndex].type = cert_pi_end;

      cvout[0].type = cert_po_trustAnchor;
      cvout[0].value.pointer.cert = NULL;
      cvout[1].type = cert_po_certList;
      cvout[1].value.pointer.chain = NULL;

      // setting pointer to CERTVerifyLog. Initialized structure
      // will be used CERT_PKIXVerifyCert
      cvout[2].type = cert_po_errorLog;
      cvout[2].value.pointer.log = &log;

      cvout[3].type = cert_po_end;

      rv = CERT_PKIXVerifyCert(cert, certUsage, cvin, cvout, slotpw);
      //if (rv != SECSuccess) break;
      if(rv != SECSuccess) {
        CERTVerifyLogNode* node   = NULL;
        if(log.count > 0) {
          for(node = log.head; node; node = node->next) {
            if(node->depth == 1) {
              if(node->error == SEC_ERROR_CA_CERT_INVALID) {
/* If the certificate to check is a proxy, then it will arrives here, 
 * because nss itself can not support the verification of proxy certificate,
 * and it will simply treat the proxy signer as invalid CA.
 * Since the "untrustedCertList" given is a issuing chain corresponding to 
 * the proxy, we simply check the last certificate in the untrustedCertList, 
 * which is supposed to be an EEC certificate, if this cert is valid, then
 * the proxy certificate is also treated as valid.
 */
                CERTCertListNode *cert_node, *nd;
                CERTCertificate* cert_to_check = NULL;
                bool r;
                if(untrustedCertList != NULL && !CERT_LIST_EMPTY(untrustedCertList)) {
                  nd = CERT_LIST_HEAD(untrustedCertList);
                  while (!CERT_LIST_END(nd, untrustedCertList)) {
                    cert_node = nd; 
                    nd = CERT_LIST_NEXT(nd);
                  }
                  cert_to_check = cert_node->cert;
      
                  r = nssPKIXVerifyCert(context, slotpw, cert_to_check, NULL, trustedCertList, NULL);
                  if(r) { 
                    ret = true;
                    rv = SECSuccess;
                  };
                };
              } else {
                context.Log(Context::LogError, std::string(PORT_ErrorToString(node->error)));
              }
            };
          };
        };
        break;
      };
                
      issuerCert = cvout[0].value.pointer.cert;
      builtChain = cvout[1].value.pointer.chain;
    } while(0);

    if(rv != SECSuccess /* || log.count > 0*/) {
      CERTVerifyLogNode *node = NULL;
      context.Log(Context::LogError, "Certificate is not valid");

      displayVerifyLog(context, &log, verbose);
      // There are cert refs in the log in case of failure, destroy them.
      for(node = log.head; node; node = node->next) {
        if(node->cert) CERT_DestroyCertificate(node->cert);
      }
      ret = false;
    } else {
      context.Log(Context::LogInfo, "Certificate is valid");
      if(issuerCert) {
        if(verbose) {
          printName(context, &issuerCert->subject, "Root Certificate Subject:");
        }
        CERT_DestroyCertificate(issuerCert);
      }
      if(builtChain) {
        CERTCertListNode *node;
        int count = 0;
        char buff[256];
        if(verbose) {
          for(node = CERT_LIST_HEAD(builtChain); !CERT_LIST_END(node, builtChain);
            node = CERT_LIST_NEXT(node), count++ ) {
            sprintf(buff, "Certificate %d Subject", count + 1);
            printName(context, &node->cert->subject, buff);
          }
        }
        CERT_DestroyCertList(builtChain);
      }
      ret = true;
    }

    // Destroy CERTVerifyLog arena
    PORT_FreeArena(log.arena, PR_FALSE);

    return ret;
  }
#endif

  bool nssVerifyCert(Context& context, const char* slotpw, 
      const char* certfile, bool use_pkix, 
      const std::vector<const char*>& untrusted_certfiles, 
      const std::vector<const char*>& trusted_certfiles, 
      const std::string& cert_policy) {
    std::string cert_str;
    std::vector<std::string> untrusted_certs;
    std::vector<std::string> trusted_certs;
    SECStatus rv = file2str(context, certfile, cert_str);
    if(rv == SECSuccess) {
      for(unsigned int i=0; i<untrusted_certfiles.size(); i++) {
        std::string str;
        rv = file2str(context, untrusted_certfiles[i], str);
        if(rv == SECSuccess) untrusted_certs.push_back(str);
        else { 
          context.LogFormat(Context::LogError, "Failed to get cert from file: %s", untrusted_certfiles[i]); 
          break;
        }
      }
      for(unsigned int i=0; i<trusted_certfiles.size(); i++) {
        std::string str;
        rv = file2str(context, trusted_certfiles[i], str);
        if(rv == SECSuccess) trusted_certs.push_back(str);
        else {  
          context.LogFormat(Context::LogError, "Failed to get cert from file: %s", trusted_certfiles[i]); 
          break;
        }
      }
      return nssVerifyCert(context, slotpw, cert_str, use_pkix, untrusted_certs, trusted_certs, cert_policy); 
    } 
    return false;
  }

  bool nssVerifyCert(Context& context, const char* slotpw, 
      const std::string& certstr, bool use_pkix, 
      const std::vector<std::string>& untrusted_certs, 
      const std::vector<std::string>& trusted_certs, 
      const std::string& cert_policy) {
    CERTCertificate* cert = NULL;
    CERTCertDBHandle* certhandle = NULL;

    PRTime time = 0;
    CERTCertList* trustedCertList = NULL;
    CERTCertList* untrustedCertList = NULL;
    SECStatus rv = SECFailure;
    SECCertificateUsage certUsage = certificateUsageSSLServer;
    CERTVerifyLog log;
    PRBool verbose = PR_TRUE;
    bool ret;

    log.arena = PORT_NewArena(512);
    log.head = log.tail = NULL;
    log.count = 0;

    cert = getCert(context, certstr, PR_TRUE);
    if(CERT_CertTimesValid(cert) == SECFailure) {
      context.Log(Context::LogError, "Certificate is expired"); goto err;
    }

    if(!use_pkix) {
      time = PR_Now();
      certhandle = CERT_GetDefaultCertDB();
      rv = CERT_VerifyCertificate(certhandle, cert, PR_TRUE, certUsage, time, (void*)slotpw, &log, NULL);
      if(rv != SECSuccess || log.count > 0) {
        CERTVerifyLogNode *node = NULL;
        context.Log(Context::LogError, "Certificate is not valid");

        displayVerifyLog(context, &log, verbose);
        // There are cert refs in the log in case of failure, destroy them.
        for(node = log.head; node; node = node->next) {
          if(node->cert) CERT_DestroyCertificate(node->cert);
        }
        ret = false;
      } else {
        context.Log(Context::LogInfo, "Certificate is valid");
        ret = true;
      }

      // Destroy CERTVerifyLog arena
      PORT_FreeArena(log.arena, PR_FALSE);
    }
#if NSS_VMAJOR >=3 && NSS_VMINOR >=12
    else do { //use pkix library to verify the certificate
      if(trusted_certs.size() != 0)  {
        trustedCertList = CERT_NewCertList();
        CERTCertificate* trusted_cert = NULL;
        for(unsigned int i=0; i<trusted_certs.size(); i++) {
          trusted_cert = getCert(context, trusted_certs[i], PR_TRUE);
          CERT_AddCertToListTail(trustedCertList, trusted_cert);
        }
      }

      if(untrusted_certs.size() != 0)  {
        // Check the given untrusted certificates, shuffle them into 
        // the sequence 
        CERTCertificate* untrusted_cert = NULL;
        CERTCertList* list = CERT_NewCertList();
        for(unsigned int i=0; i<untrusted_certs.size(); i++) {
          untrusted_cert = getCert(context, untrusted_certs[i], PR_TRUE);
          CERT_AddCertToListTail(list, untrusted_cert);
        }
        untrustedCertList = shuffleChain(list, cert);
        CERT_DestroyCertList(list);
      }

      SECOidTag oidTag = SEC_OID_UNKNOWN;
      if (!cert_policy.empty()) {
        PRArenaPool* arena;
        SECOidData od;
        memset(&od, 0, sizeof od);
        od.offset = SEC_OID_UNKNOWN;
        od.desc = "User Defined Policy OID";
        od.mechanism = CKM_INVALID_MECHANISM;
        od.supportedExtension = INVALID_CERT_EXTENSION;

        arena = PORT_NewArena(DER_DEFAULT_CHUNKSIZE);
        if (!arena ) {
          context.Log(Context::LogError, "Failed to allocate object, out of memory");
          goto err;
        }
        rv = SEC_StringToOID(arena, &od.oid, cert_policy.c_str(), 0);
        if (rv != SECSuccess) {
          PORT_FreeArena(arena, PR_FALSE);
          context.LogFormat(Context::LogError, "Can not encode oid: %s(%s)", cert_policy.c_str(), PORT_ErrorToString(PORT_GetError()));
          break;
        }
        oidTag = SECOID_AddEntry(&od);
        PORT_FreeArena(arena, PR_FALSE);
        if (oidTag == SEC_OID_UNKNOWN) {
          context.LogFormat(Context::LogError, "Can not add new oid to the dynamic table: %s", cert_policy.c_str());
          rv = SECFailure;
          break;
        }
      }

      ret = nssPKIXVerifyCert(context, (char*)slotpw, cert, untrustedCertList, trustedCertList, &oidTag);

    } while(0);
#else
    context.Log(Context::LogError, "pkix library is not supported");
#endif

  err:
    if (trustedCertList) CERT_DestroyCertList(trustedCertList);
    if (untrustedCertList) CERT_DestroyCertList(untrustedCertList);

    return ret;
  }

  bool nssImportCRL(Context& context, const char* slotpw, 
      const char* crlfile, const std::string& crl_url) {
    SECStatus rv;
    std::string crlstr;
    // Read data from crl file
    rv = file2str(context, crlfile, crlstr);
    if (rv != SECSuccess) {
      context.Log(Context::LogError, "Failed to read data from crl file");
      return false;
    }

    return nssImportCRL(context, slotpw, crlstr, crl_url);
  }

  bool nssImportCRL(Context& context, const char* slotpw, 
      const std::string& crlstr, const std::string& crl_url) {
    CERTSignedCrl* crl = NULL;
    SECItem crlder;
    PK11SlotInfo* slot = NULL;
    int crltype = SEC_CRL_TYPE;
    PRInt32 decodeOptions = CRL_DECODE_DEFAULT_OPTIONS;
    PRInt32 importOptions = CRL_IMPORT_DEFAULT_OPTIONS;
    CERTCertDBHandle* certhandle = NULL;
    SECStatus rv;

    crlder.data = NULL;

    certhandle = CERT_GetDefaultCertDB();
    if(certhandle == NULL) {
      context.Log(Context::LogError, "Failed to open the cert db");
      return false;
    }

    unsigned char* crlbuf;
    crlbuf = (unsigned char*)(crlstr.c_str());

    if ((*crlbuf & 0x1f) == SEC_ASN1_SEQUENCE) //DER encoded
      rv = readDERfromstr(context, &crlder, crlstr, false);
    else rv = readDERfromstr(context, &crlder, crlstr, true); //Ascii encoded
    if(rv != SECSuccess) {
      context.Log(Context::LogError, "Failed to read crl str");
      return false;
    }

    decodeOptions |= CRL_DECODE_DONT_COPY_DER;

    slot = PK11_GetInternalKeySlot();
/*
    if (PK11_NeedLogin(slot)) {
      rv = PK11_Authenticate(slot, PR_TRUE, (void*)slotpw);
    if(rv != SECSuccess)
      if(slot) PK11_FreeSlot(slot);
      return false;
    }
*/
    if(!crl_url.empty())
      crl = PK11_ImportCRL(slot, &crlder, (char*)(crl_url.c_str()), 
            crltype, NULL, importOptions, NULL, decodeOptions);
    else
      crl = PK11_ImportCRL(slot, &crlder, NULL, crltype, NULL, 
            importOptions, NULL, decodeOptions);

    if (!crl) {
      context.LogFormat(Context::LogError, "Failed to import CRL: %s", PORT_ErrorToString(PORT_GetError()));
      rv = SECFailure;
    } else {
      SEC_DestroyCrl (crl);
      rv = SECSuccess;
    }

    if(slot) PK11_FreeSlot(slot);
 
    return ((rv == SECSuccess) ? true : false);
  }


  /* The following code is about how to let openssl code to access pkcs11.
   * It is basically similar with the pkcs11 engine for openssl, but here
   * the nss api is used to access pkcs11, rather than the libpk11 api that
   * is used in current pkcs11 engine for openssl.
   * 1. The reason why we use nss instead of libpk11 is that since we already have 
   * used nss for certificate manipulation, we would still use nss for securing
   * communication, so that we do not need to add another dependency --- libpk11.
   * 2. Alternatively, we can depend on nss library for securing communication, but
   * in that way, we have to maintain the other big part of code in parallel with 
   * current openssl communication code.
   */
  struct RSA_data {
    CERTCertificate* cert;
    Context* context;
    std::string passwd;
  };

  static int nss_rsa_priv_enc(int flen, const unsigned char* from, 
      unsigned char* to, RSA* rsa, int padding) {
    Context* context = NULL;
    CERTCertificate* cert = NULL;
    SECKEYPrivateKey* privkey = NULL;
    char* password = NULL;
    SECItem sigitem;
    SECItem dataitem;
    int signatureLen;
    SECStatus rv = SECFailure;
    int ret;

    RSA_data* rsa_data = (RSA_data*)RSA_get_app_data(rsa);
    if(!rsa_data) {
      std::cerr<<"NSS RSA app data is not set"<<std::endl; return -1;
    }
    context = rsa_data->context;
    cert = rsa_data->cert;
    if(!cert) {
      context->Log(Context::LogError, "NSS CERTCertificate object is not set"); return -1;
    }
    password = (char*)(rsa_data->passwd.c_str());

    if(padding != RSA_PKCS1_PADDING) {
      context->Log(Context::LogError, "Unknow padding type");
      return -1;
    }

    privkey = PK11_FindKeyByAnyCert(cert, password);
    if(!privkey) {
      context->Log(Context::LogError, "Failed to find private key"); return -1;
    }

    signatureLen = PK11_SignatureLen(privkey);
    if(signatureLen <= 0) {
      context->Log(Context::LogError, "The private key is not valid"); 
      goto err;
    }
    sigitem.len = signatureLen;
    sigitem.data = (unsigned char*) PORT_Alloc(signatureLen);

    if(sigitem.data == NULL) {
      goto err;
    }

    dataitem.data = (unsigned char*)from;
    dataitem.len = flen;

    rv = PK11_Sign(privkey, &sigitem, &dataitem);
    if (rv != SECSuccess) {
      PORT_Free(sigitem.data);
      sigitem.data = NULL;
      goto err;
    }

    for(unsigned int i = 0; i < sigitem.len; i++)
      to[i] = sigitem.data[i];
    
  err:
    if(privkey) {
      SECKEY_DestroyPrivateKey(privkey);
    }

    SECITEM_FreeItem(&sigitem, PR_FALSE);

    if(rv == SECSuccess) context->Log(Context::LogVerbose, "RSA_method priv_enc succeeded");

    return rv == SECSuccess ? signatureLen : -1;

  }

  static int nss_rsa_priv_dec(int flen, const unsigned char* from, 
      unsigned char* to, RSA* rsa, int padding) {
    // have not be able to trigger this method
    Context* context = NULL;
    CERTCertificate* cert = NULL;
    SECKEYPrivateKey* privkey = NULL;
    char* password = NULL;
    SECItem decitem = {siBuffer, NULL, 0};
    unsigned char decbuf[1024];
    SECItem dataitem;
    int decLen;
    CK_MECHANISM_TYPE mech = CKM_RSA_PKCS;
    SECStatus rv = SECFailure;
    int ret;

    RSA_data* rsa_data = (RSA_data*)RSA_get_app_data(rsa);
    if(!rsa_data) {
      std::cerr<<"NSS RSA app data is not set"<<std::endl; return -1;
    }
    context = rsa_data->context;
    cert = rsa_data->cert;
    if(!cert) {
      context->Log(Context::LogError, "NSS CERTCertificate object is not set"); return -1;
    }
    password = (rsa_data->passwd.empty()) ? NULL : (char*)(rsa_data->passwd.c_str());

    if(padding != RSA_PKCS1_PADDING) {
      context->Log(Context::LogError, "Unknow padding type");
      return -1;
    }
    mech = CKM_RSA_PKCS;

    privkey = PK11_FindKeyByAnyCert(cert, password);
    if(!privkey) {
      context->Log(Context::LogError, "Failed to find private key"); return -1;
    }

    dataitem.data = (unsigned char*)from;
    dataitem.len = flen;
    decitem.data = decbuf;
    decitem.len = sizeof decbuf;

    rv = PK11_PrivDecryptPKCS1(privkey, decitem.data, (unsigned int*)&decLen,
                               decitem.len, dataitem.data, dataitem.len);

    if (rv == SECSuccess) {
      for(int i = 0; i < decLen; i++)
        to[i] = decitem.data[i];
    }

  err:
    if(privkey) {
      SECKEY_DestroyPrivateKey(privkey);
    }

    if(rv == SECSuccess) context->Log(Context::LogVerbose, "RSA_method priv_dec succeeded");

    return rv == SECSuccess ? decLen : -1;
  }

  static int nss_finish(RSA* rsa) {
    CERTCertificate* cert = NULL;

    RSA_data* rsa_data = (RSA_data*)RSA_get_app_data(rsa);
    if(rsa_data != NULL) {
      cert = rsa_data->cert;
      if(cert != NULL) {
        CERT_DestroyCertificate(cert);
      }
      delete rsa_data;
    }

    RSA_set_app_data (rsa, NULL);

    return 0;
  }

  static RSA_METHOD* initialize_RSA_method(Context* context) {
    RSA_METHOD* rsa_method = NULL;
    rsa_method = (RSA_METHOD*)calloc(1, sizeof(RSA_METHOD));
    if(!rsa_method) { context->Log(Context::LogError, "Failed to allocate memory for RSA_METHOD"); return NULL; }
    rsa_method->name = "EMI RSA Method to access PKCS11 with NSS";
    //rsa_method->rsa_pub_enc = nss_rsa_pub_enc;
    //rsa_method->rsa_pub_dec = nss_rsa_pub_dec;
    rsa_method->rsa_priv_enc = nss_rsa_priv_enc;
    rsa_method->rsa_priv_dec = nss_rsa_priv_dec;
    //rsa_method->init = nss_init;
    rsa_method->finish = nss_finish;
    //rsa_method->flags = RSA_METHOD_FLAG_NO_CHECK;
    //rsa_method->app_data = ;

    return rsa_method;
  }

  X509* getX509_from_NSSCert(Context* context, CERTCertificate* cert) {
    X509* x509 = NULL;
    const unsigned char* data;
    unsigned int len;
    std::string der_str;

    // Get the cert content with ascii format from nss CERTCertificate object
    data = cert->derCert.data;
    len = cert->derCert.len;

    char* buf = BTOA_DataToAscii(data, len);
    std::string str;
    if(buf != NULL) { str.assign(buf); PORT_Free(buf); }

    if(!str.empty()) {
      der_str.append(NS_CERT_HEADER).append("\n").append(str).append("\n").append(NS_CERT_TRAILER).append("\n");
      context->LogFormat(Context::LogVerbose, "Certificate: %s", der_str.c_str());
    }
    else context->Log(Context::LogError, "NSS Certificate is empty");
   
    // Parse the cert content to create openssl X509 object
    if(!der_str.empty()) {
      BIO* bio = BIO_new(BIO_s_mem());
      BIO_write(bio, der_str.c_str(), der_str.size());
      x509 = PEM_read_bio_X509(bio, NULL, NULL, NULL);
      BIO_free(bio);
    }

    return x509;  
  }

  RSA* getRSA_from_NSSCert(Context* context, CERTCertificate* cert) {
    X509* x509 = NULL;
    RSA* rsa = NULL;
    EVP_PKEY* pubkey = NULL;
    RSA_METHOD* my_rsa_method = NULL;
    std::string password;
    RSA_data* rsa_data = NULL;
    bool ok = false;

    x509 = getX509_from_NSSCert(context, cert);
    if(!x509) {
      context->Log(Context::LogError, "Can not get certificate object");
      goto err;
    }

    pubkey = X509_get_pubkey(x509);
    if(pubkey == NULL) {
      context->Log(Context::LogError, "Can not get public key");
      goto err;
    }
    if(pubkey->type != EVP_PKEY_RSA) {
      context->Log(Context::LogError, "Invalid public key algorithm");
      goto err;
    }

    rsa = EVP_PKEY_get1_RSA(pubkey);
    if(rsa == NULL) {
      context->Log(Context::LogError, "Can not get RSA key");
      goto err;
    }

    my_rsa_method = initialize_RSA_method(context);
    if(!my_rsa_method) {
      context->Log(Context::LogError, "Failed to create RSA method"); goto err; 
    }
    if(!RSA_set_method (rsa, my_rsa_method)) {
      context->Log(Context::LogError, "Failed to set RSA method"); goto err;
    }

    //password = "secretpw";  //TODO: password callback
    rsa_data = new RSA_data;
    rsa_data->context = context;
    rsa_data->cert = cert;
    rsa_data->passwd = password;
    if(!RSA_set_app_data(rsa, rsa_data)) {
      context->Log(Context::LogError, "Failed to set app data to RSA"); goto err;
    } 

    rsa->flags |= RSA_FLAG_SIGN_VER;

    ok = true;

  err:
    if(!ok) {
      if (rsa != NULL) {
        RSA_free (rsa);
        rsa = NULL;
      }
      if(my_rsa_method != NULL) {
        free(my_rsa_method); 
        my_rsa_method = NULL;
      }
    }
    if(pubkey != NULL) {
      EVP_PKEY_free(pubkey);
      pubkey = NULL;
    }
    if(x509 != NULL) {
      X509_free(x509);
      x509 = NULL;
    }

    return rsa;
  }

  int SSL_CTX_use_pkcs11 (Context* context, SSL_CTX* ssl_ctx, const std::string& pkcs11_id) {
    X509* x509 = NULL;
    X509* issuerx509 = NULL;
    RSA* rsa = NULL;
    CERTCertificate* cert = NULL;
    CERTCertificate* issuercert = NULL;
    CERTCertificate* currentcert = NULL;
    CERTCertDBHandle* certhandle;
    bool ok = false;   
 
    if(!ssl_ctx || pkcs11_id.empty()) return 0;

    certhandle = CERT_GetDefaultCertDB();
    cert = CERT_FindCertByNicknameOrEmailAddr(certhandle, (char*)(pkcs11_id.c_str()));
    if(!cert) { context->Log(Context::LogError, "Failed to find certificate from nss db"); return 0; }

    x509 = getX509_from_NSSCert(context, cert);
    if(!x509) { context->Log(Context::LogError, "Can not get X509 object"); goto err; } 
    rsa = getRSA_from_NSSCert(context, cert);
    if(!rsa) { context->Log(Context::LogError, "Can not get RSA object"); goto err; }

    if(!SSL_CTX_use_RSAPrivateKey (ssl_ctx, rsa)) {
      context->Log(Context::LogError, "Can not set private key for SSL_CTX"); goto err;
    }
    if(!SSL_CTX_use_certificate (ssl_ctx, x509)) {
      context->Log(Context::LogError, "Can not set certificate for SSL_CTX"); goto err;
    }

    currentcert = cert;
    issuercert = FindIssuerCert(currentcert);
    while(issuercert != NULL) {
      // For the untrusted (non-CA) certificates in the chain,
      // i.e., for the case of certificates that signs proxy
      issuerx509 = getX509_from_NSSCert(context, issuercert);

      if(!issuerx509) goto err;
      if(SSL_CTX_add_extra_chain_cert(ssl_ctx,issuerx509) != 1) {
        X509_free(issuerx509);
        context->Log(Context::LogError, "Can not set cert chain for SSL_CTX"); goto err;
      }
      if(CERT_IsCACert(issuercert, NULL)) {
        //TODO: not sure why CERT_IsCACert gives true for EEC cert
        CERT_DestroyCertificate(issuercert);
        issuercert = NULL;
      }
      else {
        currentcert = issuercert;
        issuercert = FindIssuerCert(currentcert);
        CERT_DestroyCertificate(currentcert);
        currentcert = NULL;
      }
    };

    ok = true;

  err:
    // openssl object has reference count, since SSL_CTX_use* has increased
    // the count, we need to decrease the count, i.e., free the object.
    if(x509 != NULL) {
      X509_free (x509);
      x509 = NULL;
    }
    if(rsa != NULL) {
      RSA_free(rsa);
      rsa = NULL;
    }
    if(issuercert != NULL) {
      CERT_DestroyCertificate(issuercert);
      issuercert = NULL;
    }
    return ok ? 1:0;
  }


//----------
// CertContext Implementation
//----------

  CertContext::CertContext(Context& ctx) : configdir_("./"), 
      trusts_("u,u,u"), context_(&ctx), nss_initialized_(false) {
    nssInit(*context_, configdir_);
    nss_initialized_ = true;
  }

  CertContext::CertContext(Context& ctx, const std::string& configdir) : 
      configdir_(configdir), trusts_("u,u,u"), context_(&ctx),
      nss_initialized_(false)/*, slotpass_("secretpw")*/ {
    nssInit(*context_, configdir_);
    nss_initialized_ = true; 
  }
      
  CertContext::~CertContext() {
    if(nss_initialized_) nssFinish();
  }

  bool CertContext::SetNSSDBPass(const std::string& new_pass, const std::string& old_pass) {
    PK11SlotInfo* slot = NULL;
    slot = PK11_GetInternalKeySlot();
    bool res = false;
    if(old_pass != new_pass) {
      std::string old_p;
      if(old_pass.empty()) old_p = new_pass; // TODO:
      else old_p = old_pass; 
      res = nss_change_password(slot, old_p.c_str(), new_pass.c_str());
    }
    else res = true;
    PK11_FreeSlot(slot);
    return res;
  }

  void CertContext::signRequest(CSRContext& csr, int duration, std::string& cert_str) {
    bool res;
    CertContext* ret = NULL;
    std::string csr_str;
    csr.getRequest(csr_str);

    res = nssCreateCert(*context_, csr_str, certname_, 
      (slotpass_.empty() ? NULL : slotpass_.c_str()), duration, "", cert_str); 
  }

  void CertContext::certToPEM(std::string& out) const {
    if(!cert_str_.empty()) out.assign(cert_str_);
    else nssExportCertificate(*context_, certname_, out, true);
  }

  void CertContext::certToDER(std::string& out) const {
    if(!cert_str_.empty()) {
      char* buf;
      unsigned int len;
      buf = (char *)ATOB_AsciiToData(cert_str_.c_str(), &len);
      if ((buf != NULL) && (len != 0)) {
        out.assign(buf, len);
      }
      if(buf != NULL) PORT_Free(buf);
    }
    else nssExportCertificate(*context_, certname_, out, false);
  }
 
  static bool parse_cert(const std::string& input, std::string& certstr, 
      std::vector<std::string>& certchainstr) {
    std::string::size_type pos1, pos2;
    certstr.clear();
    certchainstr.clear();
    pos1 = input.find(NS_CERT_TRAILER);
    if(pos1 != std::string::npos) {
      pos2 = input.find_first_of("\r\n", pos1);
      if(pos2 != std::string::npos) {
        certstr = input.substr(0, pos2);
        while(1) {
          pos1 = input.find(NS_CERT_HEADER, pos2);
          if(pos1 == std::string::npos) break;
          pos2 = input.find(NS_CERT_TRAILER, pos1);
          if(pos2 == std::string::npos) break;
          pos2 = input.find_first_of("\r\n", pos2);
          if(pos2 == std::string::npos) { 
            certchainstr.push_back(input.substr(pos1)); break;
          }
          else {
            certchainstr.push_back(input.substr(pos1, pos2-pos1));
          }
        }
      }
      else certstr = input;
      return true;
    }
    return false;
  }

  bool CertContext::certFromDER(const std::string& certstr, 
      const std::string& certname, const std::string& trusts) {
    char* buf = BTOA_DataToAscii((const unsigned char*)(certstr.c_str()), certstr.length());
    std::string str;
    if(buf != NULL) { str.assign(buf); PORT_Free(buf); }

    std::string pemstr;
    if(!str.empty()) {
      pemstr.append(NS_CERT_HEADER).append("\n").append(str).append("\n").append(NS_CERT_TRAILER).append("\n");
    }
    else return false;

    //bool res = parse_cert(pemstr, cert_str_, cert_chain_str_);
    cert_str_.clear();
    cert_str_ = pemstr;

    bool res = ImportCertFromStrToDB(certstr, certname, trusts);
    return res;
  }

  bool CertContext::certFromPEM(const std::string& certstr,
      const std::string& certname, const std::string& trusts) {
    bool res = parse_cert(certstr, cert_str_, cert_chain_str_);
    if(res) res = ImportCertFromStrToDB(certstr, certname, trusts);
    return res;
  }

  bool CertContext::certFromStr(const std::string& certstr,
      const std::string& certname, const std::string& trusts) {
    bool res = parse_cert(certstr, cert_str_, cert_chain_str_);
    if(res) res = ImportCertFromStrToDB(certstr, certname, trusts);
    return res;
  }

  bool CertContext::certFromFile(const char* certfile,
      const std::string& certname, const std::string& trusts) {
    std::string cert_str;
    SECStatus rv = file2str(*context_, certfile, cert_str);
    bool res = false;
    if(rv == SECSuccess) res = certFromStr(cert_str, certname, trusts);
    return res;
  }
  
  bool CertContext::keyFromDER(const std::string& keystr, const std::string& keyname) {
    char* buf = BTOA_DataToAscii((const unsigned char*)(keystr.c_str()), keystr.length());
    std::string str;
    if(buf != NULL) { str.assign(buf); PORT_Free(buf); }

    std::string pemstr;
    if(!str.empty()) {
      pemstr.append(NS_PRIVKEY_HEADER).append("\n").append(str).append("\n").append(NS_PRIVKEY_TRAILER).append("\n");
    }
    else return false;

    bool res = nssImportPrivateKey(*context_, 
      (slotpass_.empty() ? NULL : slotpass_.c_str()), pemstr, keyname);
    return res;

  }
  
  bool CertContext::keyFromPEM(const std::string& keystr, const std::string& keyname) {
    bool res = nssImportPrivateKey(*context_, 
      (slotpass_.empty() ? NULL : slotpass_.c_str()), keystr, keyname);
    return res;
  }

  bool CertContext::ImportKeyFromStrToDB(const std::string& keystr, 
      const std::string& keyname) {
    bool res;
    unsigned char* keybuf;
    keybuf = (unsigned char*)(keystr.c_str());

    // a DER encoded key
    if ((*keybuf & 0x1f) == SEC_ASN1_SEQUENCE )
      res = keyFromDER(keystr, keyname);
    else res = keyFromPEM(keystr, keyname);

    return res;
  }

  bool CertContext::ImportKeyFromFileToDB(const char* keyfile, const std::string& keyname) {
    std::string key_str;
    SECStatus rv = file2str(*context_, keyfile, key_str);
    bool res = false;
    if(rv == SECSuccess) res = ImportKeyFromStrToDB(key_str, keyname);
    return res;
  }

  bool CertContext::ImportCertFromStrToDB(const std::string& certstr, 
    const std::string& certname, const std::string& trusts) {
    std::string name;
    if(!certname.empty()) name = certname;
    else name = certname_;

    parse_cert(certstr, cert_str_, cert_chain_str_);

    std::string ts;
    if(!trusts.empty()) ts = trusts;
    else ts = trusts_;

    // It seems nss itself can distinguish the format, see implementation of nssImportCert.
    // So we can simply give a non-ascii, no matter if it is ascii or non-ascii.
    bool res = nssImportCert(*context_, NULL, cert_str_, name, (char*)(ts.c_str()), false);

    certname_ = name;
    return res;
  }

  bool CertContext::ImportCertFromFileToDB(const char* certfile, 
    const std::string& certname, const std::string& trusts) {
    std::string cert_str;
    SECStatus rv = file2str(*context_, certfile, cert_str);
    bool res = false;
    if(rv == SECSuccess) res = ImportCertFromStrToDB(cert_str, certname, trusts);
    return res;
  }

  bool CertContext::ImportCertFromDirToDB(const char* certdir, const std::string& trusts) {
    DIR* dir = NULL;
    struct stat st;

    if(!certdir) return false;
    if(stat(certdir, &st) != 0) return false;
    if(!S_ISDIR(st.st_mode)) return false;

    dir = ::opendir(certdir);
    if(!dir) { return false; }
    for(;;) {
      struct dirent* d = ::readdir(dir);
      if(!d) break;
      if(strcmp(d->d_name,".") == 0) continue;
      if(strcmp(d->d_name,"..") == 0) continue;
      std::string npath = certdir;
      npath = npath + "/" + d->d_name;
      if(npath.find(".0") != std::string::npos) ImportCertFromFileToDB(npath.c_str(), "", trusts);
    }
    ::closedir(dir);
    return true;
  }

  bool CertContext::getCertExtension(int pos, AuthN::Credentials::Extension& ext) const {
    SECStatus rv = GetExtension(certname_, pos, ext);
    if(rv == SECSuccess) return true;
    else return false;
  }

  bool CertContext::getCertExtension(const std::string& name, AuthN::Credentials::Extension& ext) const {
    SECStatus rv = GetExtension(certname_, name, ext);
    if(rv == SECSuccess) return true;
    else return false;
  }

  CertContext::operator bool(void) const {
    CERTCertificate* cert = NULL;
    CERTCertDBHandle* certhandle;

    certhandle = CERT_GetDefaultCertDB();
    cert = CERT_FindCertByNicknameOrEmailAddr(certhandle, (char*)(certname_.c_str()));
    if(!cert) return false;
 
    CERT_DestroyCertificate(cert);   
    return true; 
  }

  bool CertContext::operator!(void) const {
    return !bool();
  }

  Status CertContext::validate(const AuthN::Validator::ValidationMode& mode, AuthN::Context* ctx) {
    std::vector<std::string> trusted_certs;
    std::vector<std::string> untrusted_certs;
    bool use_pkix = true;
    bool res;
    AuthN::Context* validate_ctx = NULL;

    if(ctx!=NULL) validate_ctx = ctx;
    else validate_ctx = context_;

    // If it is a proxy certificate to be verified,
    // then the intermediate signers are treated as untrusted 
    // certificates for pkix interface
    for(unsigned int i=0; i<cert_chain_str_.size(); i++) {
      std::string str = cert_chain_str_[i];
      untrusted_certs.push_back(cert_chain_str_[i]);
    }

    res = nssVerifyCert(*validate_ctx, NULL, cert_str_, use_pkix, untrusted_certs, trusted_certs, "");
    if(res) return Status(0);
    else return Status(-1, "Failed to verify the certificate");
  }


//----------
// CSRContext Implementation
//----------

  CSRContext::CSRContext(Context& ctx, const std::string& configdir, 
    const std::string& passwd) : configdir_(configdir), 
    slotpass_(passwd), context_(&ctx), nss_initialized_(false) { 
    nssInit(*context_, configdir_);
    nss_initialized_ = true;
  }

  void CSRContext::setRequest(const char* csr_filename) {
    SECStatus rv = file2str(*context_, csr_filename, csr_str_);
  }

  void CSRContext::setRequest(const std::string& csrstr) { 
    csr_str_.assign(csrstr);
  }

  CSRContext::~CSRContext() {
    if(nss_initialized_) nssFinish();
  }

  bool CSRContext::createRequest(const std::string& privkey_name) {
    std::string name;
    if(!privkey_name.empty()) name = privkey_name;
    else name = privkey_name_;

    bool res = nssGenerateCSR(*context_, name, "", 
      (slotpass_.empty() ? NULL : slotpass_.c_str()), csr_str_, privkey_str_);
    privkey_name_ = name; 
    return res;   
  }


//----------
// CRLContext Implementation
//----------

  CRLContext::CRLContext(Context& ctx, const std::string& configdir) : 
        configdir_(configdir), context_(&ctx), nss_initialized_(false) {
    nssInit(*context_, configdir_);
    nss_initialized_ = true;
  }

  CRLContext::~CRLContext() {
    if(nss_initialized_) nssFinish();
  }

  bool CRLContext::ImportCRLFromStrToDB(const std::string& crlstr, const std::string& crl_url) {
    return nssImportCRL(*context_, NULL, crlstr, crl_url);
  }

  bool CRLContext::ImportCRLFromFileToDB(const char* crlfile, const std::string& crl_url) {
    return nssImportCRL(*context_, NULL, crlfile, crl_url);
  }

  static int timeout = 10; // 10 seconds timeout
  static char* tmp_crlfile = (char*)"tmpcrl.pem";

  static bool download_crl_timeout(Context& context, const char* crl_url_file, std::string& crlstr) {
    pid_t pid;
    pid = fork();
    if(pid == -1) {
      context.Log(Context::LogError, "fork() failed");
      exit(1);
    }
    else if(pid > 0) {
      context.LogFormat(Context::LogDebug, "The pid of the child is: %d", pid);
      int waittime = 0;
      int stat;
      pid_t wpid;
      do { // Parent process
        wpid = waitpid(pid, &stat, WNOHANG);
        if (wpid == 0) {
          if (waittime < timeout) {
            context.LogFormat(Context::LogDebug, "Parent waiting %d second(s)", waittime);
            sleep(1);
            waittime ++;
          }
          else {
            context.Log(Context::LogDebug, "Killing child process");
            kill(pid, SIGKILL);
          }
        }
      } while (wpid == 0 && waittime <= timeout);
 
      if (WIFEXITED(stat)) {
        context.LogFormat(Context::LogDebug, "Child exited, status=%d", WEXITSTATUS(stat));
        if(WEXITSTATUS(stat)) return false;
        else {
          SECStatus rv;
          rv = file2str(context, tmp_crlfile, crlstr);
          return true;
        }
      }
      else if (WIFSIGNALED(stat)) {
        context.LogFormat(Context::LogDebug, "Child %d was terminated with a status of: %d", pid, WTERMSIG(stat));
        return false;
      }
    }
    else { // Child process
      // wget -q -i $HOME/certificates/emi.crl_url -O crl.pem
      char* args[7];
      args[0] = (char*)"wget";
      args[1] = (char*)"-q";
      args[2] = (char*)"-i";
      args[3] = (char*)crl_url_file;
      args[4] = (char*)"-O";
      args[5] = (char*)tmp_crlfile;
      args[6] = NULL;
      context.LogFormat(Context::LogDebug, "CRL url file: %s", crl_url_file);
      if(execvp(args[0], args) < 0) {
        context.Log(Context::LogError, "Failed to execute command");
        exit(1);
      }
    }
    return true;
  }

  static bool download_crl(Context& context, const char* crl_url_file, std::string& crlstr) {
    int i;
    std::string cmd;
    std::string tmpcrlfile = tmp_crlfile;
    // wget -q -i $HOME/certificates/emi.crl_url -O crl.pem
    cmd = "wget -q -i ";
    cmd.append(crl_url_file).append(" -O ");
    cmd.append(tmpcrlfile);
    std::cout<<"Command: "<<cmd<<std::endl;
    if(system(NULL)) {
      i = system(cmd.c_str());
      if(i != 0) { context.Log(Context::LogError, "Failed to execute command"); return false; }
      SECStatus rv;
      rv = file2str(context, tmpcrlfile.c_str(), crlstr);
    }
    return true;
  }

  static void get_crl_hash(const std::string& crlstr, std::string& hash) {
    //TODO: get hash of crl' issuer
  }

  bool CRLContext::ImportCRLFromDirToDB(const char* crldir) {    
    DIR* dir = NULL;
    struct stat st;

    if(!crldir) return false;
    if(stat(crldir, &st) != 0) return false;
    if(!S_ISDIR(st.st_mode)) return false;

    dir = ::opendir(crldir);
    if(!dir) { return false; }
    for(;;) {
      struct dirent* d = ::readdir(dir);
      if(!d) break;
      if(strcmp(d->d_name,".") == 0) continue;
      if(strcmp(d->d_name,"..") == 0) continue;
      std::string npath = crldir; 
      npath = npath + "/" + d->d_name;
      if(npath.find(".r0") != std::string::npos) ImportCRLFromFileToDB(npath.c_str(), "");
      if(npath.find(".crl_url") != std::string::npos) {
        std::string crlstr;
        if(download_crl_timeout(*context_, npath.c_str(), crlstr)) {
          ImportCRLFromStrToDB(crlstr, "");
        }
      }
    }
    ::closedir(dir);
    return true;
  }

}
}
