/**
 \class CCodecImageSubband

 This derived class from \ref CCodecImage performs forward
 and backward subband encoding. Forward means going from whole image
 to subbands, backwards from subbanded to whole image.
*/
#include "CodecImageSubband.h"

CCodecImageSubband::CCodecImageSubband()
{
   m_pTempData = 0;
   m_Depth = 0;
}

CCodecImageSubband::~CCodecImageSubband()
{
   delete m_pTempData;
   m_pTempData = 0;
   m_Depth = 0;
}

void CCodecImageSubband::Resize(int w, int h)
{
   CCodecImage::Resize(w, h);
   delete m_pTempData;
   m_pTempData = new int16_t[w > h ? w : h];
   m_Depth = 0;
   while ((w > 32 || h > 32) && (w & 1) == 0 && (h & 1) == 0)
   {
     m_Depth++;
     w >>= 1;
     h >>= 1;
   }
}


void CCodecImageSubband::SubbandForward()
{
   int iw, w, ih, h;
   int x, y;
   unsigned int depth;
   int16_t *src, *s2, *dstl, *dsth;
   int16_t a, b;

   if (m_pData == 0 || m_pData->pImage == 0)
     return;
   if (m_pTempData == 0)
     return;
   detach(); // we're about to change
   iw = m_pData->Width;
   ih = m_pData->Height;

   // Do horizontal direction first
   for (y = 0; y < ih; y++)
   {
      s2 = &m_pData->pImage[y * iw];
      w = iw / 2;
      for (depth = 0; depth < m_Depth; depth++)
      {
         src = s2;
         dstl = m_pTempData;
         dsth = &m_pTempData[w];
         for (x = 0; x < w; x++)
         {
            a = *src++;
            b = *src++;
            *dstl = a + b;
            *dsth = a - b;
            dstl++;
            dsth++;
         }
         // copy back
         memcpy(s2, m_pTempData, sizeof(int16_t) * w * 2);
	 // halve dimensions
         w = w / 2;
      } // ..for depth
   } // ..for y

   // Then vertical direction; the procedure is separable
   for (x = 0; x < iw; x++)
   {
      s2 = src = &m_pData->pImage[x];
      // copy to buffer
      dstl = m_pTempData;
      for (y = 0; y < ih; y++)
      {
         *dstl = *src;
         *dstl++;
         src += iw;
      }
#if 0
      for (depth = 0; depth < m_Depth; depth++)
      {

         s2 = src = &m_pData->pImage[x];
         dstl = m_pTempData;
         dsth = &m_pTempData[h2];
         for (y = 0; y < h2; y++)
         {
            a = *src;
            src += iw;
            b = *src;
            src += iw;
            *dstl = a + b;
            *dsth = a - b;
            dstl++;
            dsth++;
         }
         // put back in image; this is a real cache-line killer
         src = s2;
         s2 = m_pTempData;
         for (y = 0; y < h; y++)
         {
            *src = *s2;
            src += iw;
            s2++;
         }
      }
#endif
      h = h / 2;
      /* Halve dimensions, go one step deeper */
   }
}

void CCodecImageSubband::SubbandBackward()
{


}

void CCodecImageSubband::SubbandForwardReferenceImp()
{
   int iw, w, h, w2, h2;
   int x, y;
   unsigned int depth;
   int16_t *src, *s2, *dstl, *dsth;
   int16_t a, b;

   if (m_pData == 0 || m_pData->pImage == 0)
     return;
   detach(); // we're about to change
   iw = m_pData->Width;
   w = iw;
   w2 = w / 2;
   h = m_pData->Height;
   h2 = h / 2;
   if (m_pTempData == 0)
     return;
   // Perform recursively until we would end up with an odd width or height,
   // or an image < 32x32
   depth = 0;
   while (depth < m_Depth)
   {
      // First do horizontal step
      for (y = 0; y < h; y++)
      {
         s2 = src = &m_pData->pImage[y * m_pData->Width];
         dstl = m_pTempData;
         dsth = &m_pTempData[w2];
         for (x = 0; x < w2; x++)
         {
            a = *src++;
            b = *src++;
            *dstl = a + b;
            *dsth = a - b;
            dstl++;
            dsth++;
         }
         memcpy(s2, m_pTempData, sizeof(int16_t) * w);
      }
      // Then do vertical step
      for (x = 0; x < w; x++)
      {
         s2 = src = &m_pData->pImage[x];
         dstl = m_pTempData;
         dsth = &m_pTempData[h2];
         for (y = 0; y < h2; y++)
         {
            a = *src;
            src += iw;
            b = *src;
            src += iw;
            *dstl = a + b;
            *dsth = a - b;
            dstl++;
            dsth++;
         }
         // put back in image; this is a real cache-line killer
         src = s2;
         s2 = m_pTempData;
         for (y = 0; y < h; y++)
         {
            *src = *s2;
            src += iw;
            s2++;
         }
      }
      h = h2;
      h2 = h / 2;
      /* Halve dimensions, go one step deeper */
      w = w2;
      w2 = w / 2;
      depth++;
   }
}



CCodecImageSubband &CCodecImageSubband::operator =(const CCodecImage &src)
{
   CCodecImage *temp;

   // make sure baseclass gets called
   temp = (CCodecImage *)this;
   *temp = src;
   return *this;
}



bool CCodecImageSubband::operator ==(const CCodecImageSubband &comp) const
{
   int t;
   int16_t *p1, *p2;

   if (m_pData == 0 || m_pData->pImage == 0)
     return false;
   if (comp.m_pData == 0 || comp.m_pData->pImage == 0)
     return false;
   if (m_pData->Width != comp.m_pData->Width || m_pData->Height != comp.m_pData->Height)
     return false;

   t = m_pData->Pixels;
   p1 = m_pData->pImage;
   p2 = comp.m_pData->pImage;
   while (t > 0)
   {
      if (*p1 != *p2)
        return false;
      p1++;
      p2++;
      t--;
   }
   return true;
}

