/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

// -- Core stuff
#include <Application.h>
#include <MainWindow.h>
#include <InteractiveViewer.h>
#include <Core.h>
#include <ExtensionManager.h>
#include <stdlib.h>
#include <QDir>

// Different test levels to procede
const int LEVEL1 = 1;// Open application - Open extension - Open component
const int LEVEL2 = 2;// Save component

using namespace camitk;

// CLI stuff
#include "CommandLineOptions.hxx"

// Description:
//This is a test application, it is not intended to be used as a CamiTK application
//-i <file> is the file to open
//-o <path> is the directory where the output file will be saved
//-c <component> is the component extension to be tested (provide full path of the dynamic library)
//-l <level> is the test level to proceed

// usage for this application
void usage(char *appName) {
    std::cerr << appName << std::endl;
    std::cerr << std::endl;
    std::cerr << "Usage: " << appName << " [options]" << std::endl;
    std::cerr << "Build using " << Core::version << std::endl;
    std::cerr << std::endl;
    std::cerr << "Options:" << endl;
    options::print_usage(std::cerr);
}

int main(int argc, char *argv[]) {
    try {
        int end; // End of options.
        options o(argc, argv, end);
        int ret = EXIT_SUCCESS;

        if (   o.help()
                || (o.input() == "" && o.component() == "" && o.level() == "")
           ) {
            std::cout << "Help or no arguments: " << std::endl;
            usage(argv[0]);
        } else if (o.input() == "" || o.component() == "" || o.level() == "") {
            std::cout << "Bad arguments: " << std::endl;
            usage(argv[0]);
        } else {
            std::cout << "Arguments OK: " << std::endl;
            QFileInfo ifile(o.input().c_str());
            QFileInfo icomponent(o.component().c_str());
            QString   odirectory;
            if (o.odir() == "") { //optional argument
                odirectory = QDir::temp().absolutePath();
            } else {
                odirectory  = o.odir().c_str();
            }

            int       level = atoi(o.level().c_str());

            // init the camitk application context
            Application a("camitk-testcomponents", argc, argv, false, false);//No autoload + registerFileExtension false to avoid the execution of this testapp hampered by PopUp message

            // avoid all redirection to console
            MainWindow *defaultMainWindow = dynamic_cast<MainWindow*>(a.getMainWindow());
            defaultMainWindow->redirectToConsole( false );

            std::cout << "Load extension: " << icomponent.absoluteFilePath().toStdString() << std::endl;

            //Load the component extension defined in the command line
            if ( ExtensionManager::loadExtension(ExtensionManager::COMPONENT, icomponent.absoluteFilePath()) ) {
                Component* comp = Application::open(ifile.absoluteFilePath());

                if (comp) {
                    std::cout << "Open component: " << ifile.fileName().toStdString() << " SUCCESS" << std::endl;
                    comp->setFileName(odirectory + "/" + ifile.fileName());
                    if (level >= LEVEL2) {
                        std::cout << "Save component: " << ifile.fileName().toStdString() << std::endl;
                        if ( !Application::save(comp) ) {
                            std::cout << "Save component: " << ifile.fileName().toStdString() << " FAILED" << std::endl;
                            ret = EXIT_FAILURE;
                        }
                    }
                } else {
                    std::cout << "Open component: " << ifile.fileName().toStdString() << " FAILED" << std::endl;
                    ret = EXIT_FAILURE;
                }
            } else {
                std::cout << "Loading component extension: " << icomponent.fileName().toStdString() << " FAILED" << std::endl;
            }
        }
        return ret;
    } catch (camitk::AbortException& e) {
        //QMessageBox::warning(NULL, QString(argv[0]) + " aborted...", "AbortException:" + QString(e.what()));
        std::cout << argv[0] << " aborted..." << std::endl << "camitk AbortException:" << std::endl << e.what() << std::endl;
        return EXIT_FAILURE;
    } catch (std::exception& e) {
        //QMessageBox::warning(NULL, QString(argv[0]) + " aborted...", "AbortException:" + QString(e.what()));
        std::cout << argv[0] << " aborted..." << std::endl << "std AbortException:" << std::endl << e.what() << std::endl;
        return EXIT_FAILURE;
    } catch (...) {
        //QMessageBox::warning(NULL, QString(argv[0]) + " aborted...", "Unknown Exception");
        std::cout << argv[0] << " aborted..." << std::endl << "Unknown Exception" << std::endl;
        return EXIT_FAILURE;
    }

}
