/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef PMManagerDC_H
#define PMManagerDC_H

// camitk
#include <MeshComponent.h>
#include <InterfaceGeometry.h>
#include <Geometry.h>

#include "PMComponentAPI.h"

// pml classes
class PhysicalModel;
#include <pml/Component.h>
#include <pml/MultiComponent.h>
#include <pml/StructuralComponent.h>
class Atom;
class Cell;
#include <pml/RenderingMode.h>

// CamiTK component classes
class ComponentDC;
class MultiComponentDC;
class StructuralComponentDC;
class AtomDC;
class CellDC;

class LoadsManager;
class PMManagerDCPopup;
class AtomDCWidget;

namespace std {
/** definition of a couple (=STL pair) [Component *, ComponentDC *]
 * this associates a Component to its DC
  */
typedef std::pair< ::Component *, ComponentDC *> ComponentDCPair;
/** definition of the association STL map ComponentDCMap.
  * ComponentDCMap associate all the ComponentDC with their Physical Model Components.
  * The key is the Component, so that it is simple to retrieve a DC from the Component
  * (which is supposed to be the most often used functionnality).
  * This map is used in three ways : [Cell,DC], [SC,DC] and [MC,DC]. (yes, it is a shame there are not AC/DC map... Let there be rock
  )
  */
typedef std::map < ::Component *, ComponentDC *> ComponentDCMap;
/** the iterator corresponding to the ComponentDC map */
typedef std::map < ::Component *, ComponentDC *>::iterator ComponentDCMapIterator;

/** definition of a couple (=STL pair) [Atom *, AtomDC *]
 * this associates an atom to its DC
  */
typedef std::pair<Atom *, AtomDC *> AtomDCPair;
/** definition of the association set (=map in STL) AtomDCMap.
  * AtomDCMap associate all the DC with their atom.
  * The key is the atom, so that it is simple to retrieve a DC from its atom
  * (which is supposed to be the most often used functionnality).
  */
typedef std::map <Atom *, AtomDC *> AtomDCMap;
/** the iterator corresponding to the AtomDCMap map */
typedef std::map <Atom *, AtomDC *>::iterator AtomDCMapIterator;
}
/**
 * @ingroup group_cepmodeling_components_physicalmodel
 *
 * @brief
 * The manager of the physical model data.
 *
 * @note
 * Each cell has one and only one CellDC (see StructuralComponentDC constructor code).
 * Each atom has one and only one AtomDC (see StructuralComponentDC constructor code).
 *
 *
 **/
class PHYSICALMODEL_COMPONENT_API PMManagerDC : public camitk::MeshComponent {
    Q_OBJECT
public:
    /// Default constructor: give it the name of the file containing the data (either a pml file or scn file)
    PMManagerDC(const QString &) throw(camitk::AbortException);

    /// Create a manager directly from the PhysicalModel
    PMManagerDC(PhysicalModel *, const QString &);

    /// Default destructor
    virtual ~PMManagerDC();

    /// overloaded method (Also check for Loads modifications)
    bool getModified() const;

    /// set the name of the physical model as well
    virtual void setName(const QString &);

    /// Overriden method so that we actually can build a popup menu with different actions
    virtual QMenu * getPopupMenu(QWidget* parent);

    /// tell the PMDC that another step is done (and set the progress bar correspondingly)
    void progressOneStep();

    /// get the pixmap for physical model DC
    virtual QPixmap getIcon();

    /** create all the point data (scalar value) to display specific information,
     *  for physical model component, a point data is associated to each atom (hence
     *  outside the camitk::Component subclasses, this is renamed "atom data").
     *
     *
     *  SCs that have a surface representation show the atom information as a color,
     *  automatic interpolation in VTK (thanks to point data) allows for nice
     *  color scale display.
     *
     *  Note: this will automatically fills up the AtomDC point data pointers.
     */
    void createPointData();

    /// destroy all the atom data
    void destroyPointData();

    /// @name Physical Model <-> Component
    /// to get a Component corresponding to a PhysicalModel component or structure
    ///@{

    /// get the current managed PhysicalModel
    PhysicalModel *getPhysicalModel();

    /// get the bounding sphere radius (calculated when loaded)
    virtual double getBoundingRadius();

    /// get the current bounds (calculated when loaded: it is called by getBoundingRadius()
    virtual void getBounds(double bounds[6]);

    /// compute the bounding radius using the PhysicalModel atoms' position
    void computeBoundingRadius();

    /// convert PML rendering mode to Camitk rendering mode
    InterfaceGeometry::RenderingModes toDCRenderingMode(::RenderingMode::Mode);

    /// convert Camitk rendering mode to PML rendering mode
    ::RenderingMode::Mode toPMRenderingMode(camitk::InterfaceGeometry::RenderingModes);

    /// get the DC of a particular Component
    ComponentDC * getDC(::Component *);

    /// get the DC of a particular MC
    MultiComponentDC * getDC(MultiComponent *);

    /// get the DC of a particular SC
    StructuralComponentDC * getDC(StructuralComponent *);

    /// get the DC of a particular Cell
    CellDC * getDC(Cell *);

    /// get the DC of a particular atom
    AtomDC * getDC(Atom *);

    /// add the MC/DC pair
    void addMultiComponentDCPair(std::ComponentDCPair);

    /// add the SC/DC pair
    void addStructuralComponentDCPair(std::ComponentDCPair);

    /// add the C/DC pair
    void addCellDCPair(std::ComponentDCPair);

    /// add the A/DC pair
    void addAtomDCPair(std::AtomDCPair);

    /// get the first non null point set (PMManagerDC has not representation), first check the informative components, then the exclusive components
    virtual vtkSmartPointer< vtkPointSet > getPointSet();
    ///@}

    ///@name Misc
    ///@{
    /** get the AtomDCWidget, create one if needed
    * @param adc the AtomDC calling the method (needed for creation or update)
    * @param parent parent widget (needed the first time)
    */
    QWidget *getAtomDCWidget(AtomDC *adc = NULL, QWidget *parent = NULL);

    /// get the load manager
    LoadsManager * getLoadsManager();

    ///@}

private:

    /// build the Physical model dcs
    void buildPhysicalModelDCs();

    /** the method that build the Geometry (this method actually creates an empty geometry as there are no Geometry that
     *    represents a whole Physical Model).
     */
    virtual void initRepresentation();

    /// Number of Step done yet
    unsigned int nrOfDoneSteps;

    /// Total Number of Step
    unsigned int nrOfSteps;

    /// the popup menu
    PMManagerDCPopup * myPopupMenu;

    /// the PMManagerDC icon
    static QPixmap * myPixmap;

    /// the MC / DC map
    std::ComponentDCMap myMCDCMap;

    /// the SC / DC map
    std::ComponentDCMap mySCDCMap;

    /// the Cell / DC map
    std::ComponentDCMap myCDCMap;

    /// the A / DC map
    std::AtomDCMap myADCMap;

    /// the atomdc widget
    AtomDCWidget * myAtomDCWidget;

    /// the physical model managed here
    PhysicalModel * myPM;

    /// the load manager
    LoadsManager * myLoadsManager;

    /// the bounding sphere radius calculated just after the PML is loaded
    double initialBoundingRadius;
};


// -------------------- getPhysicalModel --------------------
inline PhysicalModel *PMManagerDC::getPhysicalModel() {
    return myPM;
}

// -------------------- getBoundingRadius --------------------
inline double PMManagerDC::getBoundingRadius() {
    return initialBoundingRadius;
}

// -------------------- getLoadsManager --------------------
inline LoadsManager * PMManagerDC::getLoadsManager() {
    return myLoadsManager;
}

// -------------------- C / DC Map --------------------
inline ComponentDC * PMManagerDC::getDC(::Component *mc) {
    if (mc->isInstanceOf("StructuralComponent"))
        return (ComponentDC *) getDC(dynamic_cast<StructuralComponent *>(mc));
    else if (mc->isInstanceOf("MultiComponent"))
        return (ComponentDC *) getDC(dynamic_cast<MultiComponent *>(mc));
    else
        return NULL;
}

// -------------------- MC / DC Map --------------------
inline void PMManagerDC::addMultiComponentDCPair(std::ComponentDCPair p) {
    myMCDCMap.insert(p);
}

// -------------------- SC / DC Map --------------------
inline void PMManagerDC::addStructuralComponentDCPair(std::ComponentDCPair p) {
    mySCDCMap.insert(p);
}

// -------------------- Atom / DC Map --------------------
inline void PMManagerDC::addAtomDCPair(std::AtomDCPair p) {
    myADCMap.insert(p);
}

inline AtomDC * PMManagerDC::getDC(Atom *a) {
    if (!a)
        return NULL;
    std::AtomDCMapIterator result = myADCMap.find(a);
    return (result == myADCMap.end()) ? NULL : (result->second);
}


#endif
