/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

// Reading XML
#include <libxml/parser.h>
#include <libxml/tree.h>

#include <stdexcept>
#include <iostream>
#include <fstream>
#include <iomanip>

// Other includes
#include "PhysicalModel.h"
//#include "Object3D.h"
#include "Atom.h"
#include "Cell.h"
#include "CellProperties.h"
#include "MultiComponent.h"
#include "StructuralComponent.h"

#include "AbortException.h"

#include "PhysicalModelVersion.h"

//--------------- Constructor/Destructor ------------------------------
PhysicalModel::PhysicalModel() {
    init();
}

PhysicalModel::PhysicalModel ( const char * fileName, PtrToSetProgressFunction pspf ) throw ( AbortException ) {
    init();
    setProgressFunction = pspf;

    // load from the xml file
    xmlRead ( fileName );
}

// --------------- destructor ---------------
PhysicalModel::~PhysicalModel() {
    clear();
}

// --------------- init ---------------
void PhysicalModel::init() {
    properties = new Properties(this);
    positionPtr = NULL;
    exclusiveComponents = NULL;
    informativeComponents = NULL;
    atoms = NULL;
    setProgressFunction = NULL;
    cellIndexOptimized = true; //always hopeful!
    isModifiedFlag = false;
}

// --------------- setProgress ---------------
void PhysicalModel::setProgress ( const float donePercentage ) {
    if ( setProgressFunction != NULL )
        setProgressFunction ( donePercentage );
}

// --------------- clear ---------------
void PhysicalModel::clear() {
    if ( informativeComponents )
        delete informativeComponents;

    informativeComponents = NULL;

    if ( exclusiveComponents )
        delete exclusiveComponents;

    exclusiveComponents = NULL;

    // delete all atoms at the end (otherwise deletion of cells in informative
    // or exclusive components will try to tell already deleted (unaccessible) atoms
    // that some of their SC is to be removed from their list
    if ( atoms )
        delete atoms;

    atoms = NULL;

    // clean position memory
    delete [] positionPtr;

    // reset all the unique indexes
    AtomProperties::resetUniqueIndex();

    CellProperties::resetUniqueIndex();

    atomMap.clear();

    cellMap.clear();

    delete properties;
    properties = NULL;
}

// --------------- getNumberOfCells   ---------------
unsigned int PhysicalModel::getNumberOfCells() const {
    unsigned int nrOfCells = 0;

    if ( exclusiveComponents )
        nrOfCells += exclusiveComponents->getNumberOfCells();

    if ( informativeComponents )
        nrOfCells += informativeComponents->getNumberOfCells();

    return nrOfCells;
}

// --------------- getPositionPointer ---------------
double * PhysicalModel::getPositionPointer() const {
    return positionPtr;
}

double * PhysicalModel::getPositionPointer (const Atom *a ) const {
    unsigned int idInAtom = a->getIndexInAtoms();

    if (idInAtom>=0)
        // * 3 because the position are stored consequently, using a double for x, y and z (3 double then!)
        return positionPtr + idInAtom*3;
    else {
        // look for atom  #index in the atoms
        Atom *ai = NULL;
        unsigned int i = 0;

        while ( i<atoms->getNumberOfStructures() && (ai != a))  {
            ai = dynamic_cast<Atom *> ( atoms->getStructure ( i ) );
            i++;
        }

        if ( ai == a )
            // the memory allocated to atom #index is (i-1) * 3 (because the position are stored consequently, using a double for x, y and z
            return positionPtr + ( i-1 ) *3;
        else
            return NULL;
    }
}

double * PhysicalModel::getPositionPointer(const unsigned int index) const {
    // look for atom  #index in the atoms
    bool found = false;
    unsigned int i = 0;

    while (i<atoms->getNumberOfStructures() && !found) {
        Atom *a = dynamic_cast<Atom *>(atoms->getStructure(i));
        found = (a->getIndex() == index);
        i++;
    }

    if (found)
        // the memory allocated to atom #index is (i-1) * 3 (because the position are stored consequently, using a double for x, y and z
        return positionPtr + (i-1)*3;
    else
        return NULL;
}

// --------------- optimizeIndexes ---------------
void PhysicalModel::optimizeIndexes ( MultiComponent * mc, unsigned int * index ) {
    Component *c;
    StructuralComponent *sc;
    Cell *cell;

    for ( unsigned int i = 0; i < mc->getNumberOfSubComponents(); i++ ) {
        c = mc->getSubComponent ( i );

        if ( c->isInstanceOf ( "MultiComponent" ) )
            optimizeIndexes ( ( MultiComponent * ) c, index );
        else {
            if ( c->isInstanceOf ( "StructuralComponent" ) ) {
                sc = ( StructuralComponent * ) c;
                // check all cells

                for ( unsigned int j = 0; j < sc->getNumberOfStructures(); j++ ) {
                    if ( sc->getStructure ( j )->isInstanceOf ( "Cell" ) ) {
                        cell = ( Cell * ) sc->getStructure ( j );
                        // if this is the sc that make the cell print its data, change cell index

                        if ( cell->makePrintData ( sc ) ) {
                            cell->setIndex ( *index );
                            *index = ( *index ) + 1;
                        }
                    }
                }
            }
        }
    }
}

void PhysicalModel::optimizeIndexes() {
    // to optimize the indexes: do as if it was a print/read operation (same order
    // and change the cell index (as everyone is linked with ptrs, that should not
    // change anything else

    // first: the atoms
    if ( atoms ) {
        for ( unsigned int i = 0; i < atoms->getNumberOfStructures(); i++ )
            ( ( Atom * ) atoms->getStructure ( i ) )->setIndex ( i );
    }

    // then the cells
    unsigned int newIndex =  0;

    if ( exclusiveComponents ) {
        optimizeIndexes ( exclusiveComponents, &newIndex );
    }

    if ( informativeComponents ) {
        optimizeIndexes ( informativeComponents, &newIndex );
    }

}

// --------------- xmlPrint   ---------------
void PhysicalModel::xmlPrint ( std::ostream &o, bool opt ) {

    // should we optimize the cell indexes ?
    if ( !cellIndexOptimized && opt ) {
        optimizeIndexes();
    }

    // print out the whole thing
    o << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" << std::endl;

    o << "<!-- physical model is a generic representation for 3D physical model (FEM, spring mass network, phymulob...) --> " << std::endl;

    o << "<physicalModel";

    if ( getName() != "" )
        o << "  name=\"" << getName().c_str() << "\"";

    if ( atoms )
        o << "  nrOfAtoms=\"" << atoms->getNumberOfStructures() << "\"" << std::endl;

    if ( exclusiveComponents )
        o << "  nrOfExclusiveComponents=\"" << exclusiveComponents->getNumberOfSubComponents() << "\"" << std::endl;

    if ( informativeComponents )
        o << "  nrOfInformativeComponents=\"" << informativeComponents->getNumberOfSubComponents() << "\"" << std::endl;

    o << "  nrOfCells=\"" << getNumberOfCells() << "\"" << std::endl;

    for(unsigned int i = 0; i<properties->numberOfFields(); i++)
        o << "  " <<  properties->getField(i) <<"=\"" << properties->getString(properties->getField(i)) << "\"" << std::endl;

    o << ">" << std::endl;

    o << "<!-- list of atoms: -->" << std::endl;

    o << "<atoms>" << std::endl;

    if ( atoms )
        atoms->xmlPrint ( o );

    o << "</atoms>" << std::endl;

    o << "<!-- list of exclusive components : -->" << std::endl;

    o << "<exclusiveComponents>" << std::endl;

    if ( exclusiveComponents )
        exclusiveComponents->xmlPrint ( o );

    o << "</exclusiveComponents>" << std::endl;

    o << "<!-- list of informative components : -->" << std::endl;

    o << "<informativeComponents>" << std::endl;

    if ( informativeComponents )
        informativeComponents->xmlPrint ( o );

    o << "</informativeComponents>" << std::endl;

    o << "</physicalModel>" << std::endl;

    // serialized/marshalled -> saved
    isModifiedFlag = false;
}

// --------------- xmlRead   ---------------
void PhysicalModel::xmlRead ( const char * n ) {
    // clear all the current data
    clear();

    // this initialize the library and check potential ABI mismatches
    // between the version it was compiled for and the actual shared
    // library used.
    LIBXML_TEST_VERSION

    // Set the locale to C for using dot as decimal point dispite locale
    // Set utf8 for output to enforce using utf8 strings.
    char * statusOk = setlocale(LC_CTYPE, "C.UTF-8");
    if (statusOk!=NULL)
        statusOk = setlocale(LC_NUMERIC, "C.UTF-8");
    if (statusOk!=NULL)
        statusOk = setlocale(LC_TIME, "C.UTF-8");
    
    // french is: "fr_FR.UTF8"
    if (!statusOk) {
        // try without UTF-8
        statusOk = setlocale(LC_CTYPE,"C");
        if (statusOk!=NULL)
            statusOk = setlocale(LC_NUMERIC, "C");
        if (statusOk!=NULL)
            statusOk = setlocale(LC_TIME, "C");
        if (statusOk==NULL) {
            std::cerr << "Could not set the locale to C. This is mandatory to enforce using dot as decimal separator (platform independency)." << std::endl;
            std::cerr << "This can cause a lot of trouble for XML I/O... Beware of decimal dots..." << std::endl;
        }
    }

    // the resulting document tree
    xmlDocPtr doc;
    //the pointer to the root node of the document
    xmlNodePtr root;

    doc = xmlParseFile ( n );

    if ( doc == NULL ) {
        std::cerr << "Failed to open " << n << std::endl;
        return ;
    }

    root = xmlDocGetRootElement ( doc );

    if ( root == NULL ) {
        std::cerr << "empty document" << std::endl;
        xmlFreeDoc ( doc );
        return ;
    }

    //build the physicalModel, parsing the xml tree
    if ( !parseTree ( root ) ) {
        std::cerr << "failed to read the xml tree" << std::endl;
        xmlFreeDoc ( doc );
        return ;
    }

    //free the xml
    xmlFreeDoc ( doc );

    xmlCleanupParser();

    xmlMemoryDump();

}


// ------------------ parse tree ------------------
bool PhysicalModel::parseTree ( xmlNodePtr root ) {
    if ( xmlStrcmp ( root->name, ( const xmlChar* ) "physicalModel" ) ) {
        std::cerr << "failed to read the physicalModel" << std::endl;
        return false;
    }

    // read the physicalModel name
    xmlChar *pname = xmlGetProp ( root, ( const xmlChar* ) "name" );

    std::string name = "";
    if ( pname )
        name = (char *) pname;
    
    properties = new Properties(this, name );
    
    xmlFree ( pname );

    // parse the physicalModel options (e.g. the physicalModel node attributes) that holds specific proprerties
    xmlAttr * attrs = root->properties;

    xmlNodePtr unknownAttrs = xmlNewNode(NULL, (xmlChar*)("unknownAttrs"));

    while (attrs) {
        const xmlChar * pname = attrs->name;
        xmlChar * pval = attrs->children->content;

        if (pname && xmlStrcmp(pname, (xmlChar*)"name")
                && xmlStrcmp(pname, (xmlChar*)"nrOfAtoms")
                && xmlStrcmp(pname, (xmlChar*)"nrOfExclusiveComponents")
                && xmlStrcmp(pname, (xmlChar*)"nrOfInformativeComponents")
                && xmlStrcmp(pname, (xmlChar*)"nrOfCells")) {
            xmlSetProp(unknownAttrs, pname, pval);
        }

        attrs = attrs->next;
    }

    //transform the unknown attributes to a property field map
    properties->domToFields(unknownAttrs);

    xmlFree(unknownAttrs);

    //get the pointer on atoms
    xmlNodePtr atomsPtr = root->xmlChildrenNode;

    while ( atomsPtr && xmlStrcmp ( atomsPtr->name, ( const xmlChar* ) "atoms" ) )
        atomsPtr = atomsPtr->next ;

    if ( !atomsPtr || xmlStrcmp ( atomsPtr->name, ( const xmlChar* ) "atoms" ) ) {
        std::cerr << "failed to read the atoms" << std::endl;
        return false;
    }
    else {
        // if we found the pointer on atoms, parse the sub tree and build the atoms
        parseAtoms ( atomsPtr );
        // allocate the big memory bunch
        positionPtr = new double[3*atoms->getNumberOfStructures() ];
        // assign all position memory
        double * currentPositionPtr = positionPtr;

        for ( unsigned int i=0; i<atoms->getNumberOfStructures(); i++ ) {
            Atom *a = dynamic_cast<Atom *> ( atoms->getStructure ( i ) );
            a->getProperties()->setPositionPointer ( currentPositionPtr );
            // next position => jump 3 double memory space
            currentPositionPtr = currentPositionPtr + 3;
        }
    }

    //get the pointer on exclusiveComponents
    xmlNodePtr exCompPtr = atomsPtr->next;

    while ( exCompPtr && xmlStrcmp ( exCompPtr->name, ( const xmlChar* ) "exclusiveComponents" ) )
        exCompPtr = exCompPtr->next ;

    if ( !exCompPtr || xmlStrcmp ( exCompPtr->name, ( const xmlChar* ) "exclusiveComponents" ) ) {
        std::cerr << "failed to read the exclusiveComponents" << std::endl;
        return false;
    }
    else {
        //if we found the pointer on exclusiveComponents, parse the sub tree and build the exclusiveComponents
        //exclusiveComponents = new MultiComponent(this);
        xmlNodePtr mcNode = exCompPtr->children;
        //get the pointer on the multicomponent child

        while ( mcNode && xmlStrcmp ( mcNode->name, ( const xmlChar* ) "multiComponent" ) )
            mcNode = mcNode->next ;

        if ( !mcNode ) {
            std::cerr << "error : no exclusive components found." << std::endl;
            return false;
        }

        MultiComponent * mc = new MultiComponent ( this );

        xmlChar *pname = xmlGetProp ( mcNode, ( const xmlChar* ) "name" );

        if ( pname )
            mc->setName ( std::string ( ( char* ) pname ) );

        xmlFree ( pname );

        // parse extra properties to fill the property fields map
        xmlAttr * attrs = mcNode->properties;
        xmlNodePtr unknownAttrs = xmlNewNode(NULL, (xmlChar*)("unknownAttrs"));;

        while (attrs) {
            const xmlChar * pname = attrs->name;
            xmlChar * pval = attrs->children->content;

            if (pname && xmlStrcmp(pname, (xmlChar*)"name")) {
                xmlSetProp(unknownAttrs, pname, pval);
            }

            attrs = attrs->next;
        }

        //transform the unknown attributes to a property field map
        mc->getProperties()->domToFields(unknownAttrs);
        xmlFreeNode(unknownAttrs);

        parseComponents ( mcNode, mc, true );

        setExclusiveComponents ( mc );
    }

    //get the pointer on informativeComponents
    xmlNodePtr infCompPtr = exCompPtr->next;

    while ( infCompPtr && xmlStrcmp ( infCompPtr->name, ( const xmlChar* ) "informativeComponents" ) )
        infCompPtr = infCompPtr->next ;

    if ( !infCompPtr || xmlStrcmp ( infCompPtr->name, ( const xmlChar* ) "informativeComponents" ) ) {
        std::cerr << "failed to read the informativeComponents" << std::endl;
        return false;
    }
    else {
        //if we found the pointer on informativeComponents, parse the sub tree and build the informativeComponents
        informativeComponents = new MultiComponent ( this );
        xmlNodePtr mcNode = infCompPtr->children;
        //get the pointer on the multicomponent child

        while ( mcNode && xmlStrcmp ( mcNode->name, ( const xmlChar* ) "multiComponent" ) )
            mcNode = mcNode->next ;

        if ( !mcNode ) {
            //if no child found, there is no informative components
            delete informativeComponents;
            informativeComponents = NULL;
            return true;
        }

        xmlChar *pname = xmlGetProp ( mcNode, ( const xmlChar* ) "name" );

        if ( pname )
            informativeComponents->setName ( std::string ( ( char* ) pname ) );

        xmlFree ( pname );

        parseComponents ( mcNode, informativeComponents, false );
    }

    return true;
}


// ------------------ parse atoms ------------------
bool PhysicalModel::parseAtoms ( xmlNodePtr atomsRoot ) {
    //parse the content of atoms
    for ( xmlNodePtr child = atomsRoot->xmlChildrenNode; child != NULL; child = child->next ) {
        //get the pointer on structuralComponent
        if ( !xmlStrcmp ( child->name, ( const xmlChar* ) "structuralComponent" ) ) {
            StructuralComponent * sc = new StructuralComponent ( this, child );
            //parse the content of structuralComponent

            for ( xmlNodePtr SCchild = child->xmlChildrenNode; SCchild != NULL; SCchild = SCchild->next ) {
                //get the pointer on nrOfStructures
                if ( !xmlStrcmp ( SCchild->name, ( const xmlChar* ) "nrOfValues" ) ) {
                    xmlChar * prop = xmlGetProp ( SCchild, ( xmlChar* ) ( "value" ) );
                    unsigned int val = atoi ( ( char* ) prop );
                    sc->plannedNumberOfStructures ( val );
                    xmlFree ( prop );
                }

                //get the pointers on atom
                if ( !xmlStrcmp ( SCchild->name, ( const xmlChar* ) "atom" ) ) {
                    Atom * newatom = new Atom ( this, SCchild, sc->getNumberOfStructures() );
                    sc->addStructure ( newatom );
                }
            }

            this->setAtoms ( sc );
        }

        if ( !xmlStrcmp ( child->name, ( const xmlChar* ) "atom" ) ) {
            Atom * newatom = new Atom ( this, child );
            this->addAtom ( newatom );
        }
    }

    return true;
}


// ------------------ parse Components ------------------
bool PhysicalModel::parseComponents ( xmlNodePtr root, Component * father, bool isExclusive ) {
    for ( xmlNodePtr child = root->xmlChildrenNode; child != NULL; child = child->next ) {
        //read and build a MultiComponent
        if ( !xmlStrcmp ( child->name, ( const xmlChar* ) "multiComponent" ) ) {
            MultiComponent * mc = new MultiComponent ( this );
            mc->setExclusive ( isExclusive );

            // parse the name
            xmlChar *pname = xmlGetProp ( child, ( const xmlChar* ) "name" );

            if ( pname )
                mc->setName ( std::string ( ( char* ) pname ) );

            xmlFree ( pname );

            // parse extra properties to fill the property fields map
            xmlAttr * attrs = child->properties;
            xmlNodePtr unknownAttrs = xmlNewNode(NULL, (xmlChar*)("unknownAttrs"));;

            while (attrs) {
                const xmlChar * pname = attrs->name;
                xmlChar * pval = attrs->children->content;

                if (pname && xmlStrcmp(pname, (xmlChar*)"name")) {
                    xmlSetProp(unknownAttrs, pname, pval);
                }

                attrs = attrs->next;
            }

            //transform the unknown attributes to a property field map
            mc->getProperties()->domToFields(unknownAttrs);
            xmlFreeNode(unknownAttrs);

            ( ( MultiComponent* ) father )->addSubComponent ( mc );

            //parse the multicomponent subtree
            parseComponents ( child, mc, isExclusive );
        }

        //read and build a structuralComponent
        if ( !xmlStrcmp ( child->name, ( const xmlChar* ) "structuralComponent" ) ) {
            StructuralComponent * sc = new StructuralComponent ( this, child );
            sc->setExclusive ( isExclusive );

            xmlNodePtr SCchild = child->xmlChildrenNode;

            while ( SCchild != NULL && xmlStrcmp ( SCchild->name, ( const xmlChar* ) "nrOfValues" ) )
                SCchild = SCchild->next;

            if ( SCchild ) {
                xmlChar * prop = xmlGetProp ( SCchild, ( xmlChar* ) ( "value" ) );
                unsigned int val = atoi ( ( char* ) prop );
                sc->plannedNumberOfStructures ( val );
                xmlFree ( prop );
            }

            ( ( MultiComponent* ) father )->addSubComponent ( sc );

            //parse the structuralcomponent subtree

            if ( !parseComponents ( child, sc, isExclusive ) ) {
                std::cerr << ", in structural component \"" << sc->getName() << "\"" << std::endl;
                return false;
            }
        }

        //read and build a cell
        if ( !xmlStrcmp ( child->name, ( const xmlChar* ) "cell" ) ) {
            StructureProperties::GeometricType gtype = StructureProperties::INVALID;
            //find properties node
            xmlNodePtr cchild = child->xmlChildrenNode;

            while ( cchild != NULL && xmlStrcmp ( cchild->name, ( const xmlChar* ) "cellProperties" ) )
                cchild = cchild->next;

            if ( cchild ) {
                //search the type attribute
                xmlChar *ptype = xmlGetProp ( cchild, ( const xmlChar* ) "type" );

                if ( ptype ) {
                    if ( !xmlStrcmp ( ptype, ( const xmlChar* ) "TRIANGLE" ) )
                        gtype = StructureProperties::TRIANGLE;

                    if ( !xmlStrcmp ( ptype, ( const xmlChar* ) "QUAD" ) )
                        gtype = StructureProperties::QUAD;

                    if ( !xmlStrcmp ( ptype, ( const xmlChar* ) "TETRAHEDRON" ) )
                        gtype = StructureProperties::TETRAHEDRON;

                    if ( !xmlStrcmp ( ptype, ( const xmlChar* ) "HEXAHEDRON" ) )
                        gtype = StructureProperties::HEXAHEDRON;

                    if ( !xmlStrcmp ( ptype, ( const xmlChar* ) "LINE" ) )
                        gtype = StructureProperties::LINE;

                    if ( !xmlStrcmp ( ptype, ( const xmlChar* ) "WEDGE" ) )
                        gtype = StructureProperties::WEDGE;

                    if ( !xmlStrcmp ( ptype, ( const xmlChar* ) "PYRAMID" ) )
                        gtype = StructureProperties::PYRAMID;

                    if ( !xmlStrcmp ( ptype, ( const xmlChar* ) "POLY_LINE" ) )
                        gtype = StructureProperties::POLY_LINE;

                    if ( !xmlStrcmp ( ptype, ( const xmlChar* ) "POLY_VERTEX" ) )
                        gtype = StructureProperties::POLY_VERTEX;

                    xmlFree ( ptype );
                }
            }

            Cell * c = new Cell ( this, gtype, child );

            c->setExclusive ( isExclusive );

            if ( cellIndexOptimized ) {
                std::GlobalIndexStructurePair pair ( c->getIndex(), c );
                this->addGlobalIndexCellPair ( pair );
            }

            ( ( StructuralComponent* ) father )->addStructure ( c, false );

            //parse the cell subtree

            if ( !parseComponents ( child, c, isExclusive ) ) {
                std::cerr << ", in cell #" << c->getIndex();
                return false;
            }
        }

        //find the atomRef nodes
        if ( !xmlStrcmp ( child->name, ( const xmlChar* ) "atomRef" ) ) {
            xmlChar * prop = xmlGetProp ( child, ( xmlChar* ) ( "index" ) );
            unsigned int val = atoi ( ( char* ) prop );
            xmlFree ( prop );
            Atom *a = this->getAtom ( val );

            if ( a )
                ( ( StructuralComponent* ) father )->addStructure ( a );
            else {
                std::cerr << "error : cannot find atom ref #" << val;
                return false;
            }
        }

        //find the cellRef nodes
        if ( !xmlStrcmp ( child->name, ( const xmlChar* ) "cellRef" ) ) {
            xmlChar * prop = xmlGetProp ( child, ( xmlChar* ) ( "index" ) );
            unsigned int val = atoi ( ( char* ) prop );
            xmlFree ( prop );
            ( ( StructuralComponent* ) father )->addStructure ( this->getCell ( val ), false );
        }

    }

    return true;
}



// ------------------ getComponentByName ------------------
Component * PhysicalModel::getComponentByName ( const std::string n ) {
    // look for the component in the informative and exclusive component
    Component * foundC;
    foundC = exclusiveComponents->getComponentByName ( n );

    if ( !foundC && informativeComponents )
        foundC = informativeComponents->getComponentByName ( n );

    return foundC;
}

// ----------------------- setAtoms ------------------
void PhysicalModel::setAtoms ( StructuralComponent *sc, bool deleteOld ) {
    Atom *a;

    if ( sc->composedBy() == StructuralComponent::ATOMS ) {
        if ( atoms && deleteOld )
            delete atoms;

        atoms = sc;

        // register all the atoms in the map, and tell the atoms about its new status
        for ( unsigned int i = 0; i < sc->getNumberOfStructures(); i++ ) {
            a = ( Atom * ) sc->getStructure ( i );
            a->getProperties()->setPhysicalModel ( this );
            addGlobalIndexAtomPair ( std::GlobalIndexStructurePair ( a->getIndex(), a ) );
        }
    }
}

// ----------------------- addAtom ------------------
bool PhysicalModel::addAtom ( Atom *newA ) {
    // register the atom in the map if possible
    if ( atoms && addGlobalIndexAtomPair ( std::GlobalIndexStructurePair ( newA->getIndex(), newA ) ) ) {
        // add the atom in the atom structural component
        atoms->addStructure ( newA );
        return true;
    }
    else
        return false; // atom does not have a unique index
}

// ----------------------- addGlobalIndexAtomPair ------------------
bool PhysicalModel::addGlobalIndexAtomPair ( std::GlobalIndexStructurePair p ) {
    std::GlobalIndexStructureMapIterator mapIt;
    // check if the atom's index is unique
    mapIt = atomMap.find ( p.first );

    // if the index was found, one can not add the atom

    if ( mapIt != atomMap.end() )
        return false;

    // if the atom is present in the map then replace the pair <atomIndex, Atom*>
    mapIt = atomMap.begin();

    while ( mapIt != atomMap.end() && mapIt->second != p.second ) {
        mapIt++;
    }

    // if found then remove the pair
    if ( mapIt != atomMap.end() )
        atomMap.erase ( mapIt );

    // insert or re-insert (and return true if insertion was ok)
    return atomMap.insert ( p ).second;
}

// ----------------------- addGlobalIndexCellPair ------------------
bool PhysicalModel::addGlobalIndexCellPair ( std::GlobalIndexStructurePair p ) {
    std::GlobalIndexStructureMapIterator mapIt;
    // check if the cell index is unique
    mapIt = cellMap.find ( p.first );

    // if the index was found, one can not add the cell

    if ( mapIt != cellMap.end() )
        return false;

    // if the cell is present in the map then replace the pair <cellIndex, Cell*>
    mapIt = cellMap.begin();

    while ( mapIt != cellMap.end() && mapIt->second != p.second ) {
        mapIt++;
    }

    // if found then remove the pair
    if ( mapIt != cellMap.end() )
        cellMap.erase ( mapIt );

    // insert or re-insert
    bool insertionOk = cellMap.insert ( p ).second;

    // is that optimized?
    cellIndexOptimized = cellIndexOptimized && ( ( Cell * ) p.second )->getIndex() == optimizedCellList.size();

    if ( cellIndexOptimized )
        optimizedCellList.push_back ( ( Cell * ) p.second );

    // insert or re-insert (and return true if insertion was ok)
    return insertionOk;
}

// ----------------------- setAtomPosition ------------------
void PhysicalModel::setAtomPosition ( Atom *atom, const double pos[3] ) {
    atom->setPosition ( pos );
}

// ----------------------- setExclusiveComponents ------------------
void PhysicalModel::setExclusiveComponents ( MultiComponent *mc ) {
    if ( exclusiveComponents )
        delete exclusiveComponents;

    exclusiveComponents = mc;
    mc->setPhysicalModel(this);
}

// ----------------------- setInformativeComponents ------------------
void PhysicalModel::setInformativeComponents ( MultiComponent *mc ) {
    if ( informativeComponents )
        delete informativeComponents;

    informativeComponents = mc;
    mc->setPhysicalModel(this);
}

// ----------------------- getNumberOfExclusiveComponents ------------------
unsigned int PhysicalModel::getNumberOfExclusiveComponents() const {
    if ( !exclusiveComponents )
        return 0;
    else
        return exclusiveComponents->getNumberOfSubComponents();
}

// ----------------------- getNumberOfInformativeComponents ------------------
unsigned int PhysicalModel::getNumberOfInformativeComponents() const {
    if ( !informativeComponents )
        return 0;
    else
        return informativeComponents->getNumberOfSubComponents();
}

// ----------------------- getNumberOfAtoms ------------------
unsigned int PhysicalModel::getNumberOfAtoms() const {
    if ( !atoms )
        return 0;
    else
        return atoms->getNumberOfStructures();
}

// ----------------------- getExclusiveComponent ------------------
Component * PhysicalModel::getExclusiveComponent ( const unsigned int i ) const {
    if ( !exclusiveComponents )
        return 0;
    else
        return exclusiveComponents->getSubComponent ( i );
}

// ----------------------- getInformativeComponent ------------------
Component * PhysicalModel::getInformativeComponent ( const unsigned int i ) const {
    if ( !informativeComponents )
        return 0;
    else
        return informativeComponents->getSubComponent ( i );
}


// ----------------------- exportAnsysMesh ------------------
void PhysicalModel::exportAnsysMesh ( std::string filename ) {
    //--- Writing nodes
    std::ofstream nodeFile;
    nodeFile.open(( filename + ".node" ).c_str());

    if ( !nodeFile.is_open() ) {
        std::cerr << "Error in PhysicalModel::exportAnsysMesh : unable to create .node output file" << std::endl;
        return;
    }

    for (unsigned int i = 0; i < getNumberOfAtoms(); i++ ) {
        double pos[3];
        //WARNING getAtom(i) do not work if indexes do not follow each others
        Atom* at=(Atom*)(atoms->getStructure(i));
        // WARNING : indexes are in base 1 !!!!
        unsigned int ansysIndex = at->getIndex() + 1;

        // coordinates of this node
        at->getPosition ( pos );

        nodeFile << std::setw(8) << ansysIndex << " ";
        for (unsigned int j=0; j<3; j++)
            nodeFile << std::setprecision(8) << std::setw(3) << std::fixed << std::scientific << pos[j] << "     ";
        nodeFile << std::endl;        
    }

    nodeFile.close();

    //--- Writing elements : exlusive cells
    std::ofstream elemFile;
    elemFile.open( ( filename + ".elem" ).c_str());

    if ( !elemFile.is_open() ) {
        std::cerr << "Error in PhysicalModel::exportAnsysMesh : unable to create .elem output file" << std::endl;
        return;
    }

    int MAT, TYPE;
    Component *elements = this->getComponentByName ( "Elements" );
    for (unsigned int i = 0; i < elements->getNumberOfCells(); i++ ) {

        // get the cell
        Cell * cell = elements->getCell ( i );
        unsigned int ansysIndex = cell->getIndex() + 1;

        switch ( cell->getType() ) {

            case StructureProperties::HEXAHEDRON:

                // I,J,K,L,M,N,O,P,MAT,TYPE,REAL,SECNUM,ESYS,IEL
                //
                // Format hex:
                //   I,J,K,L,M,N,O,P                = indices des noeuds
                //   MAT,TYPE,REAL,SECNUM et ESYS   = attributes numbers
                //   SECNUM                         = beam section number
                //   IEL                            = element number

                MAT  = 1;
                TYPE = 1;

                for (unsigned int k = 0; k < cell->getNumberOfStructures(); k++ ) {
                    elemFile << " " << std::setw(5) << cell->getStructure ( k )->getIndex() + 1 ;
                }
        
                elemFile << " " << std::setw(5) << MAT << " " << std::setw(5) << TYPE << "     1     1     0 " << std::setw(5) << ansysIndex << std::endl;
                break;

            case StructureProperties::WEDGE:

                // I,J,K,L,M,N,O,P,MAT,TYPE,REAL,SECNUM,ESYS,IEL
                //
                // Format prism: on repete les noeuds 3 et 7:
                //   I,J,K,K,M,N,O,O                = indices des noeuds
                //   MAT,TYPE,REAL,SECNUM et ESYS   = attributes numbers
                //   SECNUM                         = beam section number
                //   IEL                            = element number
                MAT  = 1;
                TYPE = 1;

                elemFile << " " << std::setw(5) << cell->getStructure ( 0 )->getIndex() + 1;
                elemFile << " " << std::setw(5) << cell->getStructure ( 1 )->getIndex() + 1;
                elemFile << " " << std::setw(5) << cell->getStructure ( 2 )->getIndex() + 1;
                elemFile << " " << std::setw(5) << cell->getStructure ( 2 )->getIndex() + 1;

                elemFile << " " << std::setw(5) << cell->getStructure ( 3 )->getIndex() + 1;
                elemFile << " " << std::setw(5) << cell->getStructure ( 4 )->getIndex() + 1;
                elemFile << " " << std::setw(5) << cell->getStructure ( 5 )->getIndex() + 1;
                elemFile << " " << std::setw(5) << cell->getStructure ( 5 )->getIndex() + 1;

                elemFile << " " << std::setw(5) << MAT << " " << std::setw(5) << TYPE << "     1     1     0 " << std::setw(5) << ansysIndex << std::endl;

                break;


            case StructureProperties::TETRAHEDRON:

                MAT  = 1;
                TYPE = 1;

                elemFile << " " << std::setw(5) <<  cell->getStructure ( 0 )->getIndex() + 1;
                elemFile << " " << std::setw(5) << cell->getStructure ( 1 )->getIndex() + 1;
                elemFile << " " << std::setw(5) << cell->getStructure ( 2 )->getIndex() + 1;
                elemFile << " " << std::setw(5) << cell->getStructure ( 3 )->getIndex() + 1;               
                elemFile << "     0     0     0     0" ;

                elemFile << " " << std::setw(5) << MAT << " " << std::setw(5) << TYPE << "     1     1     0 " << std::setw(5) << ansysIndex << std::endl;
                break;



            case StructureProperties::QUAD:
                // I,J,K,L,M,N,O,P,MAT,TYPE,REAL,SECNUM,ESYS,IEL
                //
                // Format quad:
                //   I,J,K,K,L,L,L,L                = indices des noeuds
                //   MAT,TYPE,REAL,SECNUM et ESYS   = attributes numbers
                //   SECNUM                         = beam section number
                //   IEL                            = element number

                MAT  = 1;
                TYPE = 1;

                elemFile << " " << std::setw(5) << cell->getStructure ( 0 )->getIndex() + 1;
                elemFile << " " << std::setw(5) << cell->getStructure ( 1 )->getIndex() + 1;
                elemFile << " " << std::setw(5) << cell->getStructure ( 2 )->getIndex() + 1;
                elemFile << " " << std::setw(5) << cell->getStructure ( 2 )->getIndex() + 1;

                elemFile << " " << std::setw(5) << cell->getStructure ( 3 )->getIndex() + 1;
                elemFile << " " << std::setw(5) << cell->getStructure ( 3 )->getIndex() + 1;
                elemFile << " " << std::setw(5) << cell->getStructure ( 3 )->getIndex() + 1;
                elemFile << " " << std::setw(5) << cell->getStructure ( 3 )->getIndex() + 1;

                elemFile << " " << std::setw(5) << MAT << " " << std::setw(5) << TYPE << "     1     1     0 " << std::setw(5) << ansysIndex << std::endl;
                break;


            default:
                std::cerr << "PhysicalModel::exportAnsysMesh : unknown type for cell " << cell->getIndex() + 1 << ", neither HEXAHEDRON, WEDGE, THETRAHEDRON nor QUAD. Cant' export in Patran format." << std::endl;
                continue;
        }

    }

    elemFile.close();
}


// ----------------------- exportPatran ------------------
void PhysicalModel::exportPatran ( std::string filename ) {
    std::ofstream outputFile;
    outputFile.open( filename.c_str());
    if ( !outputFile.is_open() ) {
        std::cerr << "Error in PhysicalModel::exportPatran : unable to create output file" << std::endl;
        return;
    }

    //--- patran header -> mostly useless info in our case...
    outputFile << "25       0       0       1       0       0       0       0       0\n";
    outputFile << "PATRAN File from: " << getName().c_str() << std::endl;
    outputFile << "26       0       0       1   " << this->getNumberOfAtoms() << "    " << this->getExclusiveComponent ( 0 )->getNumberOfCells() << "       3       4      -1\n";
    outputFile << "17-Mar-00   08:00:00         3.0\n";


    //--- Nodes (atoms)

    for (unsigned int i = 0; i < this->getNumberOfAtoms(); i++ ) {
        double pos[3];

        // first line
        // WARNING : indexes are in base 1 !!!!
        outputFile << " 1" << std::setw(8) << getAtom ( i )->getIndex() + 1 << "       0       2       0       0       0       0       0\n";

        // coordinates of this node
        //  fscanf(inputFile, "%d %f %f %f", &j, &x, &y, &z);
        getAtom ( i )->getPosition ( pos );

        // second line : node coordinates
         for (unsigned int j=0; j<3; j++)
            outputFile << std::setprecision(8) << std::setw(16) << std::fixed << std::scientific << pos[j];
        outputFile << " " << std::endl;
        
        // third line : ??
        outputFile << "1G       6       0       0  000000\n";
    }



    //--- Elements : exlusive cells
    for (unsigned int i = 0; i < this->getExclusiveComponent ( 0 )->getNumberOfCells(); i++ ) {
        int typeElement;

        // get the cell
        Cell * cell = this->getExclusiveComponent ( 0 )->getCell ( i );

        switch ( cell->getType() ) {

            case StructureProperties::HEXAHEDRON:
                typeElement = 8;
                break;

            case StructureProperties::WEDGE:
                typeElement = 7;
                break;

            default:
                std::cerr << "PhysicalModel::exportPatran : unknown type for cell " << cell->getIndex() + 1 << ", neither HEXAHEDRON nor WEDGE. Cant' export in Patran format." << std::endl;
                continue;
        }

        // first element line
        outputFile << " 2" << std::setw(8) <<  cell->getIndex() + 1 << std::setw(8) << typeElement << "       2       0       0       0       0       0\n";

        // second element line
        outputFile << std::setw(8) << cell->getNumberOfStructures() << "       0       1       0 0.000000000E+00 0.000000000E+00 0.000000000E+00\n";

        // third element line : list of nodes
        for (unsigned int k = 0; k < cell->getNumberOfStructures(); k++ ) {
            outputFile << std::setw(8) << cell->getStructure ( k )->getIndex() + 1;
        }

        outputFile << "\n";
    }

    //--- final line
    outputFile << "99       0       0       1       0       0       0       0       0\n" ;

    outputFile.close();
}
