/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include "Action.h"
#include "ActionExtension.h"
#include "ActionViewer.h"
#include "Application.h"
#include "Component.h"
#include "ActionWidget.h"
#include "Log.h"

#include <QHBoxLayout>
#include <QGroupBox>
#include <QMessageBox>

namespace camitk {

// -------------------- constructor --------------------
Action::Action(ActionExtension* extension): QObject() {
    this->extension = extension;
    this->component = "";
    qAction = NULL;
    isEmbedded = true;
    actionWidget = NULL;
    autoUpdateProperties = false;
}

// -------------------- destructor --------------------
Action::~Action() {
}

// -------------------- setName --------------------
void Action::setName(QString name) {
    this->name = name;
    setObjectName(name);
}

// -------------------- setDescription --------------------
void Action::setDescription(QString description) {
    this->description = description;
}

// -------------------- setComponent --------------------
void Action::setComponent(QString component) {
    this->component = component;
}

// -------------------- setFamily --------------------
void Action::setFamily(QString family) {
    this->family = family;
}

// -------------------- setTag --------------------
void Action::addTag(QString tag) {
    this->tags.append(tag);
}

// -------------------- setEmbedded --------------------
void Action::setEmbedded(bool isEmbedded) {
    this->isEmbedded = isEmbedded;
}

// -------------------- setIcon --------------------
void Action::setIcon(QPixmap icon) {
    this->icon = icon;
}

// -------------------- getExtensionName --------------------
QString Action::getExtensionName() const {
    return extension->getName();
}

// -------------------- getIcon --------------------
QPixmap Action::getIcon() {
    return icon;
}

// -------------------- getTargets --------------------
const ComponentList Action::getTargets() const {
    return targetComponents; 
}

// -------------------- getWidget --------------------
QWidget* Action::getWidget() {
    // build or update the widget
    if (!actionWidget) {
        // Setting the widget containing the parameters, using the default widget
        actionWidget = new ActionWidget(this);
    }
    else {
        // make sure the widget has updated targets
        dynamic_cast<ActionWidget*>(actionWidget)->updateTargets();
    }

    // make sure the widget updates its parameters
    dynamic_cast<ActionWidget*>(actionWidget)->setAutoUpdateProperty(autoUpdateProperties);
    
    return actionWidget;
}

// -------------------- getQAction --------------------
QAction* Action::getQAction() {
    if (!qAction) {
        // create the corresponding QAction (using the icon, name and descriptions)
        qAction = new QAction(getIcon(), getName(), this);
        qAction->setStatusTip(getDescription());
        qAction->setWhatsThis(getName() + "\n" + getDescription());
        // connect it to the trigger slot
        connect ( qAction, SIGNAL ( triggered() ), this, SLOT ( trigger() ) );
    }

    return qAction;
}

// -------------------- trigger --------------------
Action::ApplyStatus Action::trigger(QWidget * parent) {
    //-- build the list of valid targets
    targetComponents.clear();
    ComponentList selComp = Application::getSelectedComponents();

    foreach (Component * comp, selComp) {
        // check compatibility
        if (comp->isInstanceOf(this->getComponent()))
            targetComponents.append(comp);
    }

    //-- if there are some valid targets or if the action is generic
    if (targetComponents.size() > 0 || getComponent().isEmpty()) {
        if (isEmbedded) {
            // if there are no parents to use use the action viewer
            if (!parent)
                ActionViewer::getActionViewer()->embedActionWidget(this);
            else {
                getWidget()->setParent(parent);
                getWidget()->show();
            }
        }
        else {
            QWidget * w = getWidget();

            if (w) {
                // non embeded, just show/raise the window
                w->show();
            }
            else {
                // non embeded, no widget -> call apply() directly
                return apply();
            }
        }

        return TRIGGERED;
    }
    else {
        QMessageBox::information(parent, "Cannot apply this Action to selection", "Sorry, but the action \"" + getName() + "\" cannot be applied on the currently selected component(s)");
        return ERROR;
    }

}

// -------------------- applyInPipeline --------------------
Action::ApplyStatus Action::applyInPipeline() {
    ApplyStatus status;

	preProcess();
    //-- if there are some valid targets or if the action is generic
    if (targetComponents.size() > 0 || getComponent().isEmpty()) {
		status = apply();
		postProcess();
	}
	else
		status = Action::ABORTED;

    return status;
}

// -------------------- setInputComponents --------------------
void Action::setInputComponents(ComponentList inputs) {
    //-- build the list of valid targets
	targetComponents.clear();
    foreach (Component * comp, inputs) {
        // check compatibility
        if (comp->isInstanceOf(this->getComponent()))
			targetComponents.append(comp);
	}
}

// -------------------- setInputComponent --------------------
void Action::setInputComponent(Component * input) {
    targetComponents.clear();

	// check compatibility
	if (input->isInstanceOf(this->getComponent())) {
		targetComponents.append(input);
    }

}


// -------------------- getOutputComponents --------------------
ComponentList Action::getOutputComponents() {
    return this->outputComponents;
}

// -------------------- getOutputComponent --------------------
Component * Action::getOutputComponent() {
    if (outputComponents.isEmpty())
        return NULL;
    else
        return outputComponents.first();
}

// -------------------- preProcess --------------------
void Action::preProcess() {
	// get track of alive components before applying the action so 
	// that we can deduce from list of components after the action, 
	// the components created by the action.
	this->aliveBeforeComponents = Application::getAllComponents();

	/*
	Useless, just don't use trigger !
	// unselect all components
	ComponentList selComp = Application::getSelectedComponents();
    foreach (Component * comp, selComp) {
        comp->setSelected(false);
    }

    // select the right components (those set by setInputComponent(s)
    foreach (Component * comp, inputComponents) {
        comp->setSelected(true);
    }
	*/
}

// -------------------- postProcess --------------------
void Action::postProcess() {

	outputComponents.clear();
	ComponentList allComp = Application::getAllComponents();
	foreach (Component * comp, allComp) {
		if ((! wasHereBefore(comp)) || (comp->getModified())) {
			outputComponents.append(comp);
		}
	}

	// TODO record history
}


bool Action::wasHereBefore(Component *comp) {
	ComponentList::const_iterator it = aliveBeforeComponents.begin();

	while (( it != aliveBeforeComponents.end()) && comp != (*it)) 
		it++;

	return (it != aliveBeforeComponents.end());
}

// -------------------- getAutoUpdateProperties --------------------
bool Action::getAutoUpdateProperties() const {
  return autoUpdateProperties;
}

// -------------------- setAutoUpdateProperties --------------------
void Action::setAutoUpdateProperties(bool autoUpdateProperties) {
  this->autoUpdateProperties = autoUpdateProperties;
  if (actionWidget)
    dynamic_cast<ActionWidget*>(actionWidget)->setAutoUpdateProperty(autoUpdateProperties);
}

}

