/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include <pml/PhysicalModel.h>

#include "LoadsMovie.h"

#include "LoadsManager.h"
#include "LoadsSimulationDriver.h"
#include "PMManagerDC.h"

#include <Application.h>
#include <InteractiveViewer.h>

#include <QCheckBox>
#include <QMessageBox>
#include <QFileDialog>

//--------------- Constructor ---------------------------------
LoadsMovie::LoadsMovie(LoadsManager * myLoadsManager,  QWidget* parent) : QDialog(parent) {
  ui.setupUi(this);
  myLM = myLoadsManager;

  // get the normal bg color
  bgColor = ui.tLineEdit->palette().color(QPalette::Base);

  // update the display
  simDriver = new LoadsSimulationDriver(myLM, myLM->getLoads());
  connect(simDriver, SIGNAL(doOneStep()), this, SLOT(doOneStep()));

  // init the display
  init();

  // update the display
  updateDisplay(true);

  // reset the bg color
  QPalette palette;
  palette.setColor(QPalette::Base, bgColor);
  ui.tLineEdit->setPalette(palette);
  ui.tMinLineEdit->setPalette(palette);
  ui.tMaxLineEdit->setPalette(palette);

  // set the focus to ui.tLineEdit
  ui.tLineEdit->setFocus();
}

//--------------- Destructor ---------------------------------
LoadsMovie::~LoadsMovie() {
  delete simDriver;
  simDriver = NULL;
}

//--------------- getTime ---------------------------------
double LoadsMovie::getTime() const {
  if (simDriver)
    return simDriver->getTime();
  else
    return 0.0;
}

//--------------- init ---------------------------------
void LoadsMovie::init() {

  // update tMin, tMax, dt
  defaultTMin();
  defaultTMax();
  ui.dtLineEdit->setText(QString::number(simDriver->getDt()));

  if (myLM->getLoads() != NULL) {
    ui.playPushButton->setEnabled(true);
    ui.rewindPushButton->setEnabled(true);
    ui.pausePushButton->setEnabled(true);
    ui.stopPushButton->setEnabled(true);
  } else {
    ui.playPushButton->setEnabled(false);
    ui.rewindPushButton->setEnabled(false);
    ui.pausePushButton->setEnabled(false);
    ui.stopPushButton->setEnabled(false);
  }

  connect(ui.tSlider, SIGNAL(valueChanged(int)), this, SLOT(tSliderChanged(int)));

  connect(ui.closePushButton, SIGNAL(clicked()), this, SLOT(reset()));
  connect(ui.rewindPushButton, SIGNAL(clicked()), this, SLOT(rewind()));
  connect(ui.playPushButton, SIGNAL(clicked()), this, SLOT(play()));
  connect(ui.playOneStepPushButton, SIGNAL(clicked()), this, SLOT(playOneStep()));
  connect(ui.pausePushButton, SIGNAL(clicked()), this, SLOT(pause()));
  connect(ui.stopPushButton, SIGNAL(clicked()), this, SLOT(stop()));
  connect(ui.makeVideoPushButton, SIGNAL(clicked()), this, SLOT(makeVideo()));
  connect(ui.tMaxLineEdit, SIGNAL(textChanged(const QString&)), this, SLOT(tMaxModified()));
  connect(ui.tMaxLineEdit, SIGNAL(returnPressed()), this, SLOT(tMaxChanged()));
  connect(ui.tMinLineEdit, SIGNAL(textChanged(const QString&)), this, SLOT(tMinModified()));
  connect(ui.tMinLineEdit, SIGNAL(returnPressed()), this, SLOT(tMinChanged()));
  connect(ui.dtLineEdit, SIGNAL(textChanged(const QString&)), this, SLOT(dtModified()));
  connect(ui.dtLineEdit, SIGNAL(returnPressed()), this, SLOT(dtChanged()));
  connect(ui.tLineEdit, SIGNAL(textChanged(const QString&)), this, SLOT(tModified()));
  connect(ui.tLineEdit, SIGNAL(returnPressed()), this, SLOT(tChanged()));
  connect(ui.defaultTMaxPushButton, SIGNAL(clicked()), this, SLOT(defaultTMax()));
  connect(ui.defaultTMinPushButton, SIGNAL(clicked()), this, SLOT(defaultTMin()));
}


//--------------- reset -----------------------
void LoadsMovie::reset() {
  simDriver->resetPositions();
  updateDisplay(true);
}

//--------------- pause -----------------------
void LoadsMovie::pause() {
  simDriver->stopTimer();
}

//--------------- playOneStep -----------------------
void LoadsMovie::playOneStep() {
  simDriver->stopTimer();
  doOneStep();
}


//--------------- doOneStep -----------------------
void LoadsMovie::doOneStep(){
  // As this method is called by a timer, it could happen that the update
  // of the display (loading the new position, transforming the 3D graphics)
  // has not finished when the timer starts
  // the method again. This results in a strange state.
  // waitingForFinish act as a flag to say "wait, wait, we are
  // already doing one step here, we can't do many at the same time,
  // take your chance next time".
  // That allows the timer to be regulated by the time taken to do one step.
  static bool waitingToFinish = false;

  if (waitingToFinish)
    return; // bye bye, we are too busy at the moment

  waitingToFinish = true;

  // if t is ok, play it
  if (simDriver->getTime() <= simDriver->getTMax()) {
    // increment dt
    simDriver->setTime(simDriver->getTime() + simDriver->getDt()); // t += dt;
    // display the movements and time display
    updateDisplay(true);
  } else {
    if (ui.loopPushButton->isChecked()) {
      // rewind the movie
      simDriver->rewind();
      updateDisplay(true);
    }
    else
      // t is beyond the last event, stop
      simDriver->stopTimer();
  }

  // release the flag, so that next time this method is called,
  // a new step could be computed
  waitingToFinish = false;

}

//--------------- play -----------------------
void LoadsMovie::play() {
  if (!simDriver->isTimerActive()) {
    // timer wasn't active, action to be done = launch the timer
    simDriver->startTimer();
  } else {
    // timer is active, action to be done = play one step
    doOneStep();
  }
}

//--------------- rewind -----------------------
void LoadsMovie::rewind() {
  simDriver->rewind();
  updateDisplay(true);
}

//--------------- stop -----------------------
void LoadsMovie::stop() {
  simDriver->stopTimer();
  rewind();
}

//---------------  updateDisplay -----------------------
void LoadsMovie::updateDisplay(bool force) {

  // update the 3D display
  simDriver->updateDisplay(force);

  // refresh the time display
  // ui.tSlider: block the signals, only if not already blocked
  bool alreadyBlocked = ui.tSlider->signalsBlocked();

  if (!alreadyBlocked)
    ui.tSlider->blockSignals(true);

  ui.tSlider->setValue((int)(simDriver->getTime()*100.0));

  if (!alreadyBlocked)
    ui.tSlider->blockSignals(false);

  // ui.tLineEdit
  alreadyBlocked = ui.tLineEdit->signalsBlocked();

  if (!alreadyBlocked)
    ui.tLineEdit->blockSignals(true);

  ui.tLineEdit->setText(QString::number(((double)((int)(simDriver->getTime()*100.0))) / 100.0));

  if (!alreadyBlocked)
    ui.tLineEdit->blockSignals(false);
}

//---------------  tChanged -----------------------
void LoadsMovie::tChanged() {
  simDriver->setTime(ui.tLineEdit->text().toDouble());
  // display the movements and time display
  updateDisplay(true);
  // reset the bgd color
  QPalette palette;
  palette.setColor(QPalette::Base, bgColor);
  ui.tLineEdit->setPalette(palette); // bgColor
}

//---------------  tModified -----------------------
void LoadsMovie::tModified() {
  QPalette palette;
  palette.setColor(QPalette::Base, QColor(255, 220, 168));
  ui.tLineEdit->setPalette(palette);
}

//---------------  dtChanged -----------------------
void LoadsMovie::dtChanged() {
  simDriver->setDt(ui.dtLineEdit->text().toDouble());
  simDriver->setRefreshDt(ui.dtLineEdit->text().toDouble());
  // reset the bgd color
  QPalette palette;
  palette.setColor(QPalette::Base, bgColor);
  ui.dtLineEdit->setPalette(palette); // bgColor
  initSlider();
}

//--------------- dtModified -----------------------
void LoadsMovie::dtModified() {
  QPalette palette;
  palette.setColor(QPalette::Base, QColor(255, 220, 168));
  ui.dtLineEdit->setPalette(palette); // QColor(255, 220, 168)
}

//---------------  tMinChanged -----------------------
void LoadsMovie::tMinChanged() {
  simDriver->setTMin(ui.tMinLineEdit->text().toDouble());
  // reset the bgd color
  QPalette palette;
  palette.setColor(QPalette::Base, bgColor);
  ui.tMinLineEdit->setPalette(palette); // bgColor
  initSlider();
}

//--------------- tMinModified  -----------------------
void LoadsMovie::tMinModified() {
  QPalette palette;
  palette.setColor(QPalette::Base, QColor(255, 220, 168));
  ui.tMinLineEdit->setPalette(palette); // QColor(255, 220, 168)
}

//---------------  defaultTMin -----------------------
void LoadsMovie::defaultTMin() {
  // search for the first event
  simDriver->resetTMinToDefault();

  bool alreadyBlocked = ui.tSlider->signalsBlocked();

  if (!alreadyBlocked)
    ui.tSlider->blockSignals(true);

  ui.tMinLineEdit->setText(QString::number(simDriver->getTMin()));

  if (!alreadyBlocked)
    ui.tSlider->blockSignals(false);

  initSlider();
}

//---------------  tMaxChanged -----------------------
void LoadsMovie::tMaxChanged() {
  simDriver->setTMax(ui.tMaxLineEdit->text().toDouble());
  // reset the bgd color
  QPalette palette;
  palette.setColor(QPalette::Base, bgColor);
  ui.tMaxLineEdit->setPalette(palette); // bgColor
  initSlider();
}

//--------------- tMaxModified  -----------------------
void LoadsMovie::tMaxModified() {
  QPalette palette;
  palette.setColor(QPalette::Base, QColor(255, 220, 168));
  ui.tMaxLineEdit->setPalette(palette); // QColor(255, 220, 168)
}

//---------------  defaultTMax -----------------------
void LoadsMovie::defaultTMax() {
  simDriver->resetTMaxToDefault();

  bool alreadyBlocked = ui.tSlider->signalsBlocked();

  if (!alreadyBlocked)
    ui.tSlider->blockSignals(true);

  ui.tMaxLineEdit->setText(QString::number(simDriver->getTMax()));

  if (!alreadyBlocked)
    ui.tSlider->blockSignals(false);

  initSlider();
}

//--------------- tSliderChanged  -----------------------
void LoadsMovie::tSliderChanged(int val) {
  // block the signals, only if not already blocked
  bool alreadyBlocked = ui.tSlider->signalsBlocked();

  if (!alreadyBlocked)
    ui.tSlider->blockSignals(true);

  bool lineEditAlreadyBlocked = ui.tLineEdit->signalsBlocked();

  if (!lineEditAlreadyBlocked)
    ui.tLineEdit->blockSignals(true);

  // if signals are not blocked, then
  // this will automatically call tModified...
  ui.tLineEdit->setText(QString::number(val / 100.0));

  // update the display
  tChanged();

  if (!lineEditAlreadyBlocked)
    ui.tLineEdit->blockSignals(false);

  // unblock the signals
  if (!alreadyBlocked)
    ui.tSlider->blockSignals(false);
}

//--------------- initSlider -----------------------
void LoadsMovie::initSlider() {

  // block the signals, only if not already blocked
  bool alreadyBlocked = ui.tSlider->signalsBlocked();

  if (!alreadyBlocked)
    ui.tSlider->blockSignals(true);

  // change the slider range and value
  ui.tSlider->setRange((int)((simDriver->getTMin() - simDriver->getDt())*100.0), (int)(simDriver->getTMax()*100.0));

  ui.tSlider->setValue((int)(simDriver->getTime()*100.0));

  // change the slider keyboard behaviour
  ui.tSlider->setSingleStep((int)(simDriver->getDt()*100.0));

  ui.tSlider->setPageStep((int)(simDriver->getDt()*1000.0));

  if (!alreadyBlocked)
    ui.tSlider->blockSignals(false);
}


//--------------- makeVideo -----------------------
void LoadsMovie::makeVideo() {
  QDialog * makeVideoDlg = new QDialog(this);
  makeVideoDlg->setWindowTitle("Video Parameters");

  // Make the top-level layout; a vertical box to contain all widgets
  // and sub-layouts.
  QVBoxLayout * dlgLayout = new QVBoxLayout(makeVideoDlg);

  // the directory
  QLabel *dirLabel = new QLabel(makeVideoDlg);
  dirLabel->setText("Output directory:");
  dlgLayout->addWidget(dirLabel);

  QHBoxLayout *dirLayout = new QHBoxLayout();
  dlgLayout->addLayout(dirLayout);

  outputDirNameLineEdit = new QLineEdit(makeVideoDlg);
  outputDirNameLineEdit->setMinimumWidth(70);
  outputDirNameLineEdit->setToolTip("Choose a directory for the output images");
  outputDirNameLineEdit->setWhatsThis("Choose a directory for the output images");
  dirLayout->addWidget(outputDirNameLineEdit);

  QPushButton * dirButton = new QPushButton("...", makeVideoDlg);
  dirLayout->addWidget(dirButton);

  // the base name
  QLabel * baseNameLabel = new QLabel(makeVideoDlg);
  baseNameLabel->setText("File name base:");
  dlgLayout->addWidget(baseNameLabel);

  QLineEdit *baseName = new QLineEdit(makeVideoDlg);
  baseName->setMinimumWidth(70);
  baseName->setText(myLM->getPMManagerDC()->getPhysicalModel()->getName().c_str());
  baseName->setToolTip("Still images wil be written in base-001, base-002, ...");
  baseName->setWhatsThis("Still images wil be written in base-001, base-002, ...");
  dlgLayout->addWidget(baseName);

  // the restart option
  QCheckBox * restartSim = new QCheckBox(makeVideoDlg);
  restartSim->setText("Rewind before making images");
  restartSim->setChecked(true);
  restartSim->setToolTip("Restart the simulation from the beginning\nbefore creating still images");
  restartSim->setWhatsThis("Restart the simulation from the beginning\nbefore creating still images");
  dlgLayout->addWidget(restartSim);

  // then the buttons
  QHBoxLayout *buttonLayout = new QHBoxLayout();
  dlgLayout->addLayout(buttonLayout);
  QPushButton *okButton = new QPushButton("Ok", makeVideoDlg);
  buttonLayout->addWidget(okButton);
  QPushButton *cancelButton = new QPushButton("Cancel", makeVideoDlg);
  buttonLayout->addWidget(cancelButton);

  // connect the buttons
  QObject::connect(dirButton, SIGNAL(clicked()), this, SLOT(chooseOutputDir()));
  QObject::connect(okButton, SIGNAL(clicked()), makeVideoDlg, SLOT(accept()));
  QObject::connect(cancelButton, SIGNAL(clicked()), makeVideoDlg, SLOT(reject()));

  // show the dialog
  makeVideoDlg->adjustSize();

  if (makeVideoDlg->exec() == QDialog::Accepted) {
    // hide the dialog
    this->hide();

    // get the values
    outputDirName = outputDirNameLineEdit->text();
    std::string baseFileName(baseName->text().toStdString().c_str());

    if (restartSim->isChecked()) {
      rewind();
    }

    // make the video
    QString fn;

    unsigned int imageId = 1;

    int nbIter = (int)((simDriver->getTMax() - (simDriver->getTMin() - simDriver->getDt())) / simDriver->getDt()) + 1;

    double percentageDone;

    Application::resetProgressBar();

    while (simDriver->getTime() <= simDriver->getTMax()) {
      fn.sprintf("%s/%s-%04d.bmp", outputDirName.toStdString().c_str(), baseFileName.c_str(), imageId);
      Application::showStatusBarMessage(QString("Saving image %1 in ").arg(imageId)
                                              + fn
                                              + QString(" [%1/%2]").arg(simDriver->getTime()).arg(simDriver->getTMax()), 0);
      percentageDone = ((double) imageId * 100.0) / ((double) nbIter);
      Application::setProgressBarValue((int) percentageDone);
      InteractiveViewer::get3DViewer()->screenshot(fn);
      qApp->processEvents();
      // next one
      imageId++;
      simDriver->setTime(simDriver->getTime() + simDriver->getDt()); // t += dt;
      // display the movements and time display
      updateDisplay(true);
    }

    Application::resetProgressBar();

    // show the dialog again
    this->show();

    // tell the user what happened
    QMessageBox::information(this, "Video Images Ready",
                             QString("Took %1").arg(imageId) + " screenshots\nAll saved in " +
                             outputDirName);
  }
}

//--------------- chooseOutputDir -----------------------
void LoadsMovie::chooseOutputDir() {
  QString dir = QFileDialog::getExistingDirectory(this, "Output Directory For Screenshots...", outputDirNameLineEdit->text());

  if (!dir.isNull()) {
    outputDirName = dir;
    outputDirNameLineEdit->setText(dir);
  }
}
