/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
/**
 * @file OffExtension.cpp
 * @brief Component class for .off files.
 * @author Sonia Selmi
 * @date July 30 2012
 */

#include "OffComponent.h"

#include <sstream>


#include <InteractiveViewer.h>
#include <Log.h>

#include <QFileInfo>
#include <QVector3D>

#include <vtkOBJReader.h>
#include <vtkProperty.h>
#include <vtkUnstructuredGrid.h>
#include <vtkTetra.h>
#include <vtkHexahedron.h>
#include <vtkWedge.h>
#include <vtkPyramid.h>
#include <vtkCellArray.h>

using namespace camitk;

// -------------------- default constructor  --------------------
OffComponent::OffComponent(const QString & file) throw(AbortException) : MeshComponent(file) {

    // use the file name without extension as component name
    setName(QFileInfo(file).baseName());


	vtkPolyData* offMesh	= vtkPolyData::New();
	vtkPoints *points		= vtkPoints::New(); 
	vtkCellArray *polys		= vtkCellArray::New();

	std::ifstream inputFile(file.toStdString().c_str(), std::ios::in);
	std::string line;

	// pass the comments and the first line
    while (std::getline(inputFile, line)) {
		if (!line.compare("#")||!line.compare("OFF"))
            break;
    }

	//read the numbers of vertices and of cells
	int		  nbVertex = 0;
	int		  nbPolygon = 0;
	vtkIdType countVertex = 0;
	double	  pt[3];
	
    std::getline(inputFile, line);
	std::istringstream stream(line, std::istringstream::in);
	stream >> nbVertex;
	stream >> nbPolygon;

	nbElements	= nbPolygon;
	nbPoints	= nbVertex;

	while(countVertex<nbPoints){
		getline(inputFile, line);
		std::istringstream stream(line, std::istringstream::in);
		if (line.compare("#")!=0){
			for (int i = 0; i < 3; i++)
				stream >> pt[i];
			points->InsertPoint(countVertex,pt[0],pt[1],pt[2]);	// insert point
			++countVertex;
		}
	}

	offMesh->SetPoints(points);

	//read connectivity
	vtkIdType eltType = 0;
	vtkIdType pts[3];
	int elemCount = 0;

	while (elemCount<nbElements) {
		std::getline(inputFile, line);
		std::istringstream stream(line, std::istringstream::in);
		if (line.compare("#")!=0){
			stream >> eltType;			// the first character is the number of points of the element
			for (int i = 0; i < eltType; i++)
				stream >> pts[i];		// points coordinates
			polys->InsertNextCell(eltType,pts);	// add the element to the vtkCellArray
			++elemCount;
		}
	}

	offMesh->SetPolys(polys);
  
    offMesh->Update();

    // instanciate the Geometry
    initRepresentation(offMesh);

    // add the represention in the 3D viewer
    setVisibility(InteractiveViewer::get3DViewer(), true);
    
    // initialize the dynamic properties
    //initDynamicProperties();
}

// -------------------- initDynamicProperties --------------------
void OffComponent::initDynamicProperties() {
    // add a dynamic property to manage the surface color
    //setProperty("position point #1", QVector3D(1.0,0.0,0.0));
    vtkUnstructuredGrid* mesh = dynamic_cast< vtkUnstructuredGrid* >(getPointSet().GetPointer());
    vtkSmartPointer<vtkUnsignedCharArray> types = mesh->GetCellTypesArray();
    std::map<unsigned char, int> elementsMap;
    std::map<unsigned char, int>::iterator elementsMapIt;
    
    for (int i=0; i<types->GetNumberOfTuples(); i++) {
      if (!elementsMap.count(types->GetValue(i)))
	elementsMap[types->GetValue(i)] = 0;
      elementsMap[types->GetValue(i)]++;
    }
    
    for (elementsMapIt = elementsMap.begin(); elementsMapIt != elementsMap.end(); elementsMapIt++) {
	switch(elementsMapIt->first) {
	case VTK_TETRA :
	  setProperty("Tetra", elementsMapIt->second); 
	  break;
	case VTK_HEXAHEDRON :
	  setProperty("Hexa", elementsMapIt->second); 
	  break;
	case VTK_WEDGE :
	  setProperty("Prism", elementsMapIt->second); 
	  break;
	case VTK_PYRAMID :
	  setProperty("Pyramid", elementsMapIt->second); 
	  break;
	default :
	  setProperty("Prob", elementsMapIt->second); 
	  break;
	}
    }
}

// -------------------- getNbPoints --------------------
int OffComponent::getNbPoints() {
    return nbPoints;
}

// -------------------- getNbElements --------------------
int OffComponent::getNbElements() {
    return nbElements;
}

