/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
#include "VolumeRenderingAction.h"

#include <QMessageBox>
#include <QString>
#include <QTextStream>
#include <QVariant>

// -- vtk stuff --
#include <vtkImageCast.h>
#include <vtkVolume.h>
#include <vtkVolumeRayCastMapper.h>
#include <vtkPiecewiseFunction.h>
#include <vtkVolumeRayCastCompositeFunction.h>
#include <vtkColorTransferFunction.h>
#include <vtkRenderWindowInteractor.h>
#include <vtkRenderWindow.h>
#include <vtkRenderer.h>
#include <vtkVolumeProperty.h>
#include <vtkPolyData.h>
#include <vtkImageMagnitude.h>

// -- Application --
#include <Application.h>
#include <MeshComponent.h>
#include <InteractiveViewer.h>
#include <RendererWidget.h>
#include <Log.h>

using namespace camitk;


// --------------- constructor -------------------
VolumeRenderingAction::VolumeRenderingAction(ActionExtension * extension) : Action(extension) {
    // Setting name, description and input component
    setName("Volume Rendering");
    setDescription("Volume rendering of 3D medical image using ray casting");
    setComponent("ImageComponent");

    // Setting classification family and tags
    setFamily("Volume rendering");
    addTag("Volume rendering");
    addTag("volume");
    addTag("render");

    this->setProperty("Gray-level Value", QVariant(0.0));
    this->setProperty("Red value", QVariant(0.80));
    this->setProperty("Green value", QVariant(0.31));
    this->setProperty("Blue value", QVariant(0.2));
    this->setProperty("Alpha value", QVariant(0.0));

    this->setProperty("Gray-level Value 2", QVariant(255.0));
    this->setProperty("Red value 2", QVariant(0.5));
    this->setProperty("Green value 2", QVariant(0.5));
    this->setProperty("Blue value 2", QVariant(0.5));
    this->setProperty("Alpha value 2", QVariant(1.0));

    this->setProperty("Gray-level Value 3", QVariant(255.0));
    this->setProperty("Red value 3", QVariant(0.5));
    this->setProperty("Green value 3", QVariant(0.5));
    this->setProperty("Blue value 3", QVariant(0.5));
    this->setProperty("Alpha value 3", QVariant(1.0));


    this->setEmbedded(true);
}

// --------------- destructor -------------------
VolumeRenderingAction::~VolumeRenderingAction() {
}

// --------------- apply -------------------
Action::ApplyStatus VolumeRenderingAction::apply() {
    CAMITK_INFO("VolumeRenderingAction", "apply", "apply volume rendering");
    foreach (Component *comp, getTargets()) {
        ImageComponent * input = dynamic_cast<ImageComponent *> ( comp );
        process(input);
    }
    return SUCCESS;
}

void VolumeRenderingAction::process(ImageComponent * comp) {
    // For this example, the action just pops up a message
    cout << "VolumeRenderingAction::process" << endl;

    // Tableaux pour stocker les valeurs des points pour le lancer de rayon
    double P1[5] = {property("Gray-level Value").toDouble(),
                    property("Red value").toDouble(),
                    property("Green value").toDouble(),
                    property("Blue value").toDouble(),
                    property("Alpha value").toDouble()
                   };

    double P2[5] = {property("Gray-level Value 2").toDouble(),
                    property("Red value 2").toDouble(),
                    property("Green value 2").toDouble(),
                    property("Blue value 2").toDouble(),
                    property("Alpha value 2").toDouble()
                   };

    double P3[5] = {property("Gray-level Value 3").toDouble(),
                    property("Red value 3").toDouble(),
                    property("Green value 3").toDouble(),
                    property("Blue value 3").toDouble(),
                    property("Alpha value 3").toDouble()
                   };
    /*double P2[5] = {60.0, 0.48, 0.31, 0.2, 0.2};
    double P3[5] = {60.0, 0.48, 0.31, 0.2, 0.2};
    double P4[5] = {60.0, 0.48, 0.31, 0.2, 0.2};*/

    // READER
    // reader->Update();

    // FILTER
    // Cast element type of the image of type unigned short for mapper to work on the image
    vtkSmartPointer<vtkImageCast> imageCast = vtkImageCast::New();
    imageCast->SetInput(comp->getImageData()); // reader->GetOutput());
    imageCast->SetOutputScalarTypeToUnsignedShort();

//  // test : convert RGB images
    vtkSmartPointer<vtkImageMagnitude> magnitudeFilter = vtkSmartPointer<vtkImageMagnitude>::New();
    magnitudeFilter->SetInputConnection(imageCast->GetOutputPort());
    magnitudeFilter->Update();

    // MAPPER
    vtkSmartPointer<vtkVolumeRayCastMapper> volumeMapper = vtkSmartPointer<vtkVolumeRayCastMapper>::New();
    volumeMapper->SetInput(magnitudeFilter->GetOutput());
//  volumeMapper->SetInput(imageCast->GetOutput());
    // The volume will be displayed by ray-cast alpha compositing.
    // A ray-cast mapper is needed to do the ray-casting, and a
    // compositing function is needed to do the compositing along the ray.
    vtkSmartPointer<vtkVolumeRayCastCompositeFunction> rayCastFunction = vtkSmartPointer<vtkVolumeRayCastCompositeFunction>::New();
    volumeMapper->SetVolumeRayCastFunction(rayCastFunction);

    // RENDERER
    // Here, there is only one renderer : the one of the volume, but contains many properties
    vtkSmartPointer<vtkRenderer> renderer = vtkSmartPointer<vtkRenderer>::New();

    // 1st property : volume color
    // This function associates a voxel gray level valut to a color one (r, g, b)
    // Function definition
    // f : gray |-> (r, g, b)
    // where gray is the grayscale value (of type unsigned short)
    vtkSmartPointer<vtkColorTransferFunction>volumeColor = vtkSmartPointer<vtkColorTransferFunction>::New();
    // Set the value that the user has entered in the ui sping box
    volumeColor->AddRGBPoint(P1[0], P1[1], P1[2], P1[3]);
    volumeColor->AddRGBPoint(P2[0], P2[1], P2[2], P2[3]);
    volumeColor->AddRGBPoint(P3[0], P3[1], P3[2], P3[3]);
//     volumeColor->AddRGBPoint(P4[0], P4[1], P4[2], P4[3]);

    // 2nd property : volume opacity
    // The opacity transfer function is used to control the opacity of different tissue types.
    // This function associate to a gray scale value a transparancy value (alpha)
    // Function definition
    // f : gray |-> alpha
    // where gray is the grayscale value (of type unsigned short)
    vtkSmartPointer<vtkPiecewiseFunction> volumeScalarOpacity = vtkSmartPointer<vtkPiecewiseFunction>::New();
    volumeScalarOpacity->AddPoint(P1[0], P1[4]);
    volumeScalarOpacity->AddPoint(P2[0], P2[4]);
    volumeScalarOpacity->AddPoint(P3[0], P3[4]);
//     volumeScalarOpacity->AddPoint(P2[0], P2[4]);
//     volumeScalarOpacity->AddPoint(P3[0], P3[4]);
//     volumeScalarOpacity->AddPoint(P4[0], P4[4]);

    // 3rd property : gradient of opacity function function of the distance
    // The gradient opacity function is used to decrease the opacity
    // in the "flat" regions of the volume while maintaining the opacity
    // at the boundaries between tissue types.  The gradient is measured
    // as the amount by which the intensity changes over unit distance.
    // For most medical data, the unit distance is 1mm.
    // This function associates a distance in mm to an opacity.
    // The higher the distance is, the less opacity it has
    // It enables the function to better render the boundaries between different tissues
    vtkSmartPointer<vtkPiecewiseFunction> volumeGradientOpacity = vtkSmartPointer<vtkPiecewiseFunction>::New();
    volumeGradientOpacity->AddPoint(0,   0.0);
    volumeGradientOpacity->AddPoint(90,  0.5);
    volumeGradientOpacity->AddPoint(150, 1.0);

    // Finally : set the properties
    // The VolumeProperty attaches the color and opacity functions to the
    // volume, and sets other volume properties.  The interpolation should
    // be set to linear to do a high-quality rendering.  The ShadeOn option
    // turns on directional lighting, which will usually enhance the
    // appearance of the volume and make it look more "3D".  However,
    // the quality of the shading depends on how accurately the gradient
    // of the volume can be calculated, and for noisy data delete readerthe gradient
    // estimation will be very poor.  The impact of the shading can be
    // decreased by increasing the Ambient coefficient while decreasing
    // the Diffuse and Specular coefficient.  To increase the impact
    // of shading, decrease the Ambient and increase the Diffuse and Specular.
    vtkSmartPointer<vtkVolumeProperty> volumeProperty = vtkSmartPointer<vtkVolumeProperty>::New();
    volumeProperty->SetColor(volumeColor);
    volumeProperty->SetScalarOpacity(volumeScalarOpacity);
    volumeProperty->SetGradientOpacity(volumeGradientOpacity);
    volumeProperty->SetInterpolationTypeToLinear();
    volumeProperty->ShadeOn();
    volumeProperty->SetAmbient(0.4);
    volumeProperty->SetDiffuse(0.6);
    volumeProperty->SetSpecular(0.2);

    // The vtkVolume is a vtkProp3D (like a vtkActor) and controls the position
    // and orientation of the volume in world coordinates.
    vtkSmartPointer<vtkVolume> volume = vtkSmartPointer<vtkVolume>::New();
    volume->SetMapper(volumeMapper);
    volume->SetProperty(volumeProperty);
    volume->Update();

    // Remove old volume from the component and the 3D viewer
    QString volumeName = comp->getImageName() + "_vol_rendered";
    vtkSmartPointer<vtkProp> oldVolume = comp->getArbitrarySlices()->getProp(volumeName);

    if (oldVolume) {
        comp->getArbitrarySlices()->removeProp(volumeName);
        //InteractiveViewer::get3DViewer()->getRendererWidget()->removeProp(oldVolume);
    }

    // Add the new computed volume to the component and display it in the 3D viewer
    comp->getArbitrarySlices()->addProp(volumeName, volume);
	comp->getArbitrarySlices()->getProp(volumeName)->SetVisibility(true);
		
    //InteractiveViewer::get3DViewer()->getRendererWidget()->addProp(volume);
	comp->getArbitrarySlices()->setVisibility(InteractiveViewer::get3DViewer(), true);
	InteractiveViewer::get3DViewer()->refresh();
    
    
    // Create new component
//  new ImageComponent(volumeMapper->GetInput(), comp->getImageName() + "_vr");
//     Application::refresh();

    // Set the camera to compute different rendering position
    /*vtkCamera *camera = renderer->GetActiveCamera();
    double *c = volume->GetCenter();
    camera->SetFocalPoint(c[0], c[1], c[2]);
    camera->SetViewUp(0, 0, 1);
    camera->SetPosition(c[0], c[1]+500, c[2]);*/
    // Finally, add the volume to the renderer

    // WINDOWS
    // Create a new windows and bounds the renderer to this windows
    // renderer->AddViewProp(volume);
//  vtkSmartPointer<vtkRenderWindow> renderWindow = vtkSmartPointer<vtkRenderWindow>::New();
//  renderWindow->AddRenderer(renderer);
//  renderWindow->SetSize(800, 600);
//
//  // INTERACTOR
//  // Interactor lets the users interact with the windows / renderer
//  vtkSmartPointer<vtkRenderWindowInteractor> interactor = vtkSmartPointer<vtkRenderWindowInteractor>::New();
//  interactor->SetRenderWindow( renderWindow );
//
//  // Run the renderer loop (openGL) and display it.
//  // Initialize the event loop and then start it
//  interactor->Initialize();
//  interactor->Start();
}



