(*********************************************************************************)
(*                Cameleon                                                       *)
(*                                                                               *)
(*    Copyright (C) 2004-2008 Institut National de Recherche en Informatique     *)
(*    et en Automatique. All rights reserved.                                    *)
(*                                                                               *)
(*    This program is free software; you can redistribute it and/or modify       *)
(*    it under the terms of the GNU Library General Public License as            *)
(*    published by the Free Software Foundation; either version 2 of the         *)
(*    License, or any later version.                                             *)
(*                                                                               *)
(*    This program is distributed in the hope that it will be useful,            *)
(*    but WITHOUT ANY WARRANTY; without even the implied warranty of             *)
(*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              *)
(*    GNU Library General Public License for more details.                       *)
(*                                                                               *)
(*    You should have received a copy of the GNU Library General Public          *)
(*    License along with this program; if not, write to the Free Software        *)
(*    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA                   *)
(*    02111-1307  USA                                                            *)
(*                                                                               *)
(*    Contact: Maxence.Guesdon@inria.fr                                          *)
(*                                                                               *)
(*********************************************************************************)

(** Using the same syntax highlight through all applications
   which use the LablGtkSourceView library.

   Use the [source_languages_manager] below to get the
   languages to use in your code views.
   The styles of the tags of each language are read from
   the user's configuration file in [~/.mlgtksourceview/<language>].

   The other functions below can be used to make the user
   edit the highlight attributes for each language. *)

val source_languages_manager : GSourceView.source_languages_manager

(** Store the style of the tags of the given language in the
   user's personal files. *)
val store_lang_style : GSourceView.source_language -> unit

(** Read the style of the tags of the given language from
   the user's personal files. If there is no style file
   for this language, it is created and the current styles
   of the language tags are stored. *)
val read_lang_style : GSourceView.source_language -> unit

type source_view_props

(** Store the given source view properties in the user's personal files.*)
val store_sourceview_props : source_view_props -> unit

(** Read the source view properties from the user's personal files.
   If there is no properties file, default properties are returned.*)
val read_sourceview_props : unit -> source_view_props

(** Register a source view so that each time the function {!read_sourceview_props}
   is called, the properties read are applied to every registered view. *)
val register_source_view : GSourceView.source_view -> unit

(** Apply the given source view properties to the given source view. *)
val apply_sourceview_props : GSourceView.source_view -> source_view_props -> unit

(** Apply the given source view properties to the registered views. *)
val apply_sourceview_props_to_registered : source_view_props -> unit

(** {2 Editing the styles} *)

(** This box can be used to make the user set the styles
   of the tags of a language. Use the {!read_lang_style} and
   {!store_lang_style} functions to restore or save the styles.
*)
class lang_style_box : unit ->
  object
    method box : GObj.widget
    method lang : GSourceView.source_language option
    method reset : unit
    method set_lang : GSourceView.source_language option -> unit
  end

(** Sort the given list of languages by name. *)
val sort_languages_by_name :
  GSourceView.source_language list -> GSourceView.source_language list

(** Make the user set his preferences for the tags of the given language.
   Then {!store_lang_style} is used to save the changes or
   {!read_lang_style} is used to restore the original values. *)
val edit_lang_style : ?modal:bool -> GSourceView.source_language -> unit

(** This box can be used to make the user set the styles
   of the tags of the available languages. Use the [save] and
   [restore] methods to save or restore the styles.
*)
class multi_lang_style_box :
  unit ->
  object
    method box : GObj.widget
    method restore : unit
    method save : unit
  end

(** Make the user set his preferences for the tags of the available languages.
   Then {!store_lang_style} is used to save the changes or
   {!read_lang_style} is used to restore the original values, for each language. *)
val edit_available_languages_styles : ?modal:bool -> unit -> unit

(** This box can be used to make the user set source view properties.
   Use the {!read_sourceview_props} and {!store_sourceview_props} functions
   to restore or save the properties. The function in parameter is used
   to apply the properties when a change occurs. For example, you can
   given the {!apply_sourceview_props_to_registered} function to update
   your registered source views when the user changes a property.
*)
class sourceview_props_box :
  (source_view_props -> unit) ->
  object
    val mutable props : source_view_props option
    method box : GObj.widget
    method props : source_view_props option
    method set_props : source_view_props option -> unit
  end

(** Make the user set his source view properties.
   Then {!store_sourceview_props} is used to save the changes or
   {!read_sourceview_props} is used to restore the original values.
   @param preview is the function called to apply the properties when
   the user makes a change or when he closes the window with "Ok".
*)
val edit_sourceview_props :
  ?modal:bool -> ?preview:(source_view_props -> unit) -> unit -> unit
