/**
 * @file oric_e.c
 * Oric tape encoding functions
 * @author Marko Mkel (msmakela@nic.funet.fi)
 */

/* Copyright  2002 Marko Mkel.

   This file is part of C2N, a program for processing data tapes in
   Commodore C2N format and other formats.

   C2N is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   C2N is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
   or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
   License for more details.

   The GNU General Public License is often shipped with GNU software, and
   is generally kept in a file called COPYING or LICENSE.  If you do not
   have a copy of the license, write to the Free Software Foundation,
   59 Temple Place, Suite 330, Boston, MA 02111 USA. */

#include <stdio.h>
#include <stdlib.h>

#include "c2n.h"
#include "oric_e.h"

/** The pulse writer */
static pulse_w_t enc_wr;

/**
 * Encode a stream of synchronization pulses
 * @param count	number of pulses
 */
static void
encodeSync (unsigned count)
{
  if (verbose)
    fprintf (stderr, "encoding a sync mark (%u pulses)\n", count);
  while (count--)
    (*enc_wr) (Short);
}

/**
 * Encode a bit
 * @param b	the bit to be encoded
 */
static void
encodeBit (unsigned b)
{
  (*enc_wr) (b ? Short : Medium);
}

/**
 * Encode a byte marker and a byte
 * @param c	the byte to be encoded
 */
static void
encodeByte (unsigned char c)
{
  register unsigned i, parity;

  /*
   * Write 4 stop bits '1' followed by a '0'.
   * The ROM routines write 3 stop bits, inverting the phase after each byte.
   * However, the ROM is able to read data with 4 stop bits as well.
   */
  for (i = 4; i--; )
    encodeBit (1);
  encodeBit (0);

  /* Write the data, least significant bit first, odd parity */

  for (parity = 1, i = 8; i--; c >>= 1) {
    encodeBit (c & 1);
    if (c & 1)
      parity = !parity;
  }

  encodeBit (parity);
}

/**
 * Encode a data block
 * @param buf	the data buffer
 * @param size	length of the data in bytes
 */
static void
encBlock (const char* buf, unsigned size)
{
  register const char* bufe = buf + size;
  while (buf < bufe)
    encodeByte ((unsigned char) *buf++);
}

/** Oric pulse stream encoder
 * @param in	the data stream
 * @param err	the error reporter
 * @param wr	the pulse stream writer
 * @param begin	number of initial synchronization pulses
 * @param intra	number of intra-block synchronization pulses
 * @return	number of blocks written; 0 on failure
 */
unsigned
encode_oric (FILE* in, pulse_error_t err, pulse_w_t wr,
	     unsigned begin, unsigned intra)
{
  /** current block */
  unsigned block;
  enc_wr = wr;

  for (block = 0;; block++) {
    /** program block header */
    static char header[9];
    int c;
    /* read and skip synchronization bytes */
    while ((c = fgetc (in)) == 0x16);
    if (c == EOF)
      break;
    else if (c != 0x24) {
      if (!err || !(*err) (Unexpected, block, c))
	break;
      continue;
    }
    if (1 != fread (header, sizeof header, 1, in))
      break;
    else {
      /** program data buffer */
      char* buf;
      /** program length in bytes */
      unsigned length;
      /** received data length */
      unsigned readlength;
      /** null-terminated program name */
      static char name[16];
      /** length of the name */
      unsigned namelength;

      length = ((unsigned) (unsigned char) header[5]) |
	((unsigned) (unsigned char) header[4]) << 8;
      length -= ((unsigned) (unsigned char) header[7]) |
	((unsigned) (unsigned char) header[6]) << 8;
      length &= 0xffff;
      length++;

      /* read the file name */
      for (namelength = 0; namelength < sizeof name; )
	if ((c = fgetc (in)) == EOF || !(name[namelength++] = c))
	  break;

      if (c == EOF)
	break;
      else if (c) {
	if (err)
	  (*err) (LongBlock, block, sizeof header + sizeof name);
	break;
      }
	
      if (!(buf = malloc (length))) {
	fputs ("out of memory\n", stderr);
	break;
      }

      block++;
      readlength = fread (buf, 1, length, in);

      if (readlength != length) {
	if (!err || (*err) (ShortBlock, block, length - readlength)) {
	  free (buf);
	  return block;
	}
      }

      if (block)
	(*enc_wr) (Pause);

      /* write the synchronization sequence followed by the header */
      for (c = begin; c--; )
	encodeByte (0x16);
      encodeByte (0x24);
      encBlock (header, sizeof header);
      encBlock (name, namelength);
      encodeSync (intra);
      encBlock (buf, readlength);
      free (buf);
      encodeSync (4);
    }
  }

  (*enc_wr) (Pause);
  return block;
}
